/**MOD+***********************************************************************/
/* Module:    csample1.c                                                     */
/*                                                                           */
/* Purpose:   CPI-C sample application                                       */
/*                                                                           */
/*  COPYRIGHT DATA CONNECTION LIMITED 1991, 1992, 1995                      */
/*                                                                           */
/**MOD-***********************************************************************/

/*****************************************************************************/
/* The transaction programs CSAMPLE1 and CSAMPLE2 enable a user to browse    */
/* thru a file on another system.  The user of CSAMPLE1 is presented with a  */
/* single data block at a time, in hex and character format.  After each     */
/* block a user can request the next block, request the previous block, or   */
/* quit.  CSAMPLE1 (the invoking TP) sends a filename to CSAMPLE2 (the       */
/* invoked TP).  If CSAMPLE2 locates the file, it returns the first block to */
/* CSAMPLE1, otherwise it issues deallocate, and terminates.  If CSAMPLE1    */
/* receives a block it displays it on the screen and waits for a user        */
/* prompt; if it receives an indication that CSAMPLE2 has terminated, it too */
/* terminates.  If the user asks for the next block when CSAMPLE2 has sent   */
/* the last one, CSAMPLE2 wraps to the beginning of file.  Similarly,        */
/* CSAMPLE2 wraps to send the last block if the previous one is requested    */
/* when the first block is displayed.  Neither program attempts to recover   */
/* from errors.  A 'bad' return code from CPIC causes the program to         */
/* terminate with an explanatory message.                                    */
/*                                                                           */
/* Note that the numeric values in the data transferred between the two      */
/* programs are stored as arrays of 4 characters, rather than as integers.   */
/* CSAMPLE1 uses getlong() to get each value from the data block in local    */
/* format.  This avoids problems if the two programs are running on          */
/* different machines with different byte ordering.                          */
/*****************************************************************************/

#define BLOCKSIZ 256       /* size of data blocks received from CSAMPLE2 */
#define ISIZE    4         /* size of an integer - 4 bytes               */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include <cmc.h>

/*****************************************************************************/
/* remove comments from following line to implement debugging to screen      */
/*****************************************************************************/
/* #define DEBUGGING */

#define boolean int


/**STRUCT+********************************************************************/
/* Structure:   BLOCK_RECEIVED                                               */
/*                                                                           */
/* Description: data received from partner application.                      */
/*                                                                           */
/* Note:        all numeric values are stored as arrays of 4 characters.     */
/*****************************************************************************/
typedef struct
{
   char      block_number[ISIZE];       /* block no in range 1 to num_blocks */
   char      num_blocks[ISIZE];         /* size of file in BLOCKSIZ blocks   */
   char      block_size[ISIZE];         /* bytes in this block               */
   char      data_block[BLOCKSIZ];      /* data in block                     */
} BLOCK_RECEIVED;
/**STRUCT-********************************************************************/

unsigned char               conv_id[8];
CM_REQUEST_TO_SEND_RECEIVED rts_rcv;
unsigned char               buffer[sizeof(BLOCK_RECEIVED)];
CM_INT32                    req_len = sizeof(BLOCK_RECEIVED);
CM_DATA_RECEIVED_TYPE       data_rcv;
CM_INT32                    rcv_len;
CM_STATUS_RECEIVED          stat_rcv;
CM_DEALLOCATE_TYPE          deal_type = CM_DEALLOCATE_SYNC_LEVEL;
CM_RETURN_CODE              retcode;
unsigned char               sym_dest_name[9] = "CPICTEST";

long      getlong();
int       main ();
void      initialize ();
void      allocate ();
void      send ();
void      receive ();
void      display_block ();
char      get_prompt ();
void      deallocate ();
void      exit_error ();


/**PROC+**********************************************************************/
/* Name:      getlong                                                        */
/*                                                                           */
/* Purpose:   Get an integer in local format from the data block             */
/*                                                                           */
/* Returns:   Integer value                                                  */
/*                                                                           */
/* Params:    IN      source - character string from data block              */
/*                             representing the integer value                */
/*                                                                           */
/**PROC-**********************************************************************/
long getlong (source)
unsigned char * source;
{
  long      val = 0;     
  int       ii;   
  for ( ii = 0 ; ii < ISIZE ; ii++ )
  {
    val = ( val << 8 ) + source[(ISIZE - 1) - ii];
  }
  return (val);
}


/**PROC+**********************************************************************/
/* Name:      main                                                           */
/*                                                                           */
/* Purpose:   Entry point to application                                     */
/*                                                                           */
/* Returns:   0                                                              */
/*                                                                           */
/* Params:    IN      argc - count of arguments                              */
/*                    argv - vector of arguments                             */
/*                                                                           */
/**PROC-**********************************************************************/
int main (argc, argv)
int  argc;
char *argv[];
{
  char      prompt = ' ';
  char     *fname;

#ifdef DEBUGGING
  printf ("CSAMPLE1: main: top\n");
#endif

  if (argc != 2)
  {
    printf ("Usage %s filename\n", argv[0]);
    exit (-1);
  }
  else
  {
    /*************************************************************************/
    /* Set up the file name                                                  */
    /*************************************************************************/
    fname = argv[1];
  }

  initialize();                                              /* issue CMINIT */
  allocate();                                                /* issue CMALLC */

  /***************************************************************************/
  /* Send the filename to browse to the remote TP                            */
  /***************************************************************************/
  send (fname, strlen(fname));

  do
  {
    receive();
    if (data_rcv == CM_COMPLETE_DATA_RECEIVED)
    {
      display_block (fname, (BLOCK_RECEIVED *) buffer);
    }
    if (stat_rcv == CM_SEND_RECEIVED)
    {
      prompt = get_prompt();
      /***********************************************************************/
      /* do not send Q prompt                                                */
      /***********************************************************************/
      if( prompt == 'F' || prompt == 'B' )
      {
        send (&prompt, 1);
      }
    }
  }
  while (prompt != 'Q');

#ifdef DEBUGGING
  printf ("CSAMPLE1: main: received Q prompt - deallocating now\n");
#endif

  deallocate();
  return( 0 );
}


/**PROC+**********************************************************************/
/* Name:      initialize                                                     */
/*                                                                           */
/* Purpose:   Initialize CPI-C conversation                                  */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void initialize()
{
#ifdef DEBUGGING
  printf ("CSAMPLE1: initialize: top\n");
#endif

  /***************************************************************************/
  /* Issue CMINIT call                                                       */
  /***************************************************************************/
  cminit (conv_id, sym_dest_name, &retcode);

  if  (retcode != CM_OK )
  {
    exit_error(retcode);
  }

#ifdef DEBUGGING
  printf ("CSAMPLE1: initialize: end\n");
#endif

}


/**PROC+**********************************************************************/
/* Name:      allocate                                                       */
/*                                                                           */
/* Purpose:   Issue CPI-C cmallc call.                                       */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Issue CPIC call cmallc.  Exit with message if not CM_OK.       */
/*                                                                           */
/**PROC-**********************************************************************/
void allocate()
{
#ifdef DEBUGGING
  printf ("CSAMPLE1: allocate: top\n");
#endif

  cmallc(conv_id, &retcode);

  if (retcode != CM_OK )
  {
    exit_error (retcode);
  }

#ifdef DEBUGGING
  printf ("CSAMPLE1: allocate: end\n");
#endif
}


/**PROC+**********************************************************************/
/* Name:      send                                                           */
/*                                                                           */
/* Purpose:   Send data to partner application.                              */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN   datastring - data to send                                 */
/*            IN   datalen    - length to send                               */
/*                                                                           */
/* Operation: Issue CPIC call cmsend.  Exit with error message if not CM_OK. */
/*                                                                           */
/**PROC-**********************************************************************/
void send (datastring, datalen)
unsigned char CM_PTR datastring;
CM_INT32     datalen;
{

#ifdef DEBUGGING
  printf ("CSAMPLE1: send: top with datalen %d\n", datalen);
#endif

  cmsend (conv_id, datastring, &datalen, &rts_rcv, &retcode);

  if (retcode != CM_OK)
  {
    exit_error (retcode);
  }

#ifdef DEBUGGING
  printf ("CSAMPLE1: send: end\n");
#endif

}


/**PROC+**********************************************************************/
/* Name:      receive                                                        */
/*                                                                           */
/* Purpose:   Receive data from partner application                          */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Issue cmrcv                                                    */
/*                                                                           */
/**PROC-**********************************************************************/
void receive()
{

#ifdef DEBUGGING
  printf ("CSAMPLE1: receive: top\n");
#endif

  cmrcv (conv_id, buffer, &req_len, &data_rcv, &rcv_len,
         &stat_rcv, &rts_rcv, &retcode);

  if (retcode != CM_OK)
  {
    exit_error (retcode);
  }

#ifdef DEBUGGING
  printf ("CSAMPLE1: receive: end\n");
#endif

  return;
}


/**PROC+**********************************************************************/
/* Name:      display_block                                                  */
/*                                                                           */
/* Purpose:   Display the received block in hex and ASCII characters.        */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN    fname - filename                                         */
/*            IN    r_blockptr - pointer to received data block              */
/*                                                                           */
/**PROC-**********************************************************************/
void display_block (fname, r_blockptr)
char               *fname;
BLOCK_RECEIVED     *r_blockptr;
{
  int   i;
  int   ch;
  char  buf[17];

#ifdef DEBUGGING
  printf ("CSAMPLE1: display_block: top\n");
#endif

  printf("File %s: Size %u blocks.  Block Number %u, size %d\n\n\n", fname,
          getlong(r_blockptr->num_blocks), 
          getlong(r_blockptr->block_number),
          getlong(r_blockptr->block_size) );

  buf[16] = '\0';                                 /* ensure null termination */

  for (i = 0; i < getlong(r_blockptr->block_size); )
  {
    ch = r_blockptr->data_block[i];
    printf ("%2.2X ", ch & 0xFF);

    /*************************************************************************/
    /* check if char is printable                                            */
    /*************************************************************************/
    buf[i % 16] = (ch >= 0x20 && ch <= 0x7E) ?  (char) ch : '.';
    if ((++i % 16) == 0)
    {
      printf ("      %s\n", buf);
    }
  }

  if (i = i % 16)
  {
    buf[i] = '\0';
    for ( ; i < 16; i++ )
    {
       printf ("   ");
    }
    printf ("      %s\n", buf);
  }

#ifdef DEBUGGING
  printf("CSAMPLE1: display_block: end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      get_prompt                                                     */
/*                                                                           */
/* Purpose:   Get user action (next / previous block, or quit).              */
/*                                                                           */
/* Returns:   Prompt character                                               */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
char get_prompt()
{
  int ch;

#ifdef DEBUGGING
  printf ("CSAMPLE1: get_prompt: top\n");
#endif

  printf ("\nEnter F (forward), B (back), or Q (quit): ");

  do
  {
    ch = getchar();
    ch = (char) toupper(ch);
  }
  while (ch != 'F' && ch != 'B' && ch != 'Q');

  printf ("%c\n\n", ch);

#ifdef DEBUGGING
   printf ("CSAMPLE1: get_prompt: end\n");
#endif

   return (ch);
}


/**PROC+**********************************************************************/
/* Name:      deallocate                                                     */
/*                                                                           */
/* Purpose:   Deallocate converstion                                         */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Issue cmsdt, cmdeal.                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void deallocate()
{

#ifdef DEBUGGING
  printf ("CSAMPLE1: deallocate: top\n");
#endif

  cmsdt(conv_id, &deal_type, &retcode);
  if (retcode != CM_OK)
  {
    exit_error(retcode);
  }
  else
  {
    cmdeal(conv_id, &retcode);
    if (retcode != CM_OK)
    {
      exit_error(retcode);
    }
  }

#ifdef DEBUGGING
  printf ("CSAMPLE1: deallocate: end\n");
#endif

}


/**PROC+**********************************************************************/
/* Name:      exit_error                                                     */
/*                                                                           */
/* Purpose:   Display error message and exit application.                    */
/*                                                                           */
/* Returns:   exits with -1                                                  */
/*                                                                           */
/* Params:    IN   rc  - CPI-C return code.                                  */
/*                                                                           */
/* Operation:                                                                */
/*                                                                           */
/**PROC-**********************************************************************/
void exit_error(rc)
CM_INT32 rc;
{

  switch ((int) (rc))
  {
    case CM_OK:
      printf("CSAMPLE1: CPI-C returned CM_OK\n");
      break;
    case CM_ALLOCATE_FAILURE_NO_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_ALLOCATE_FAILURE_NO_RETRY\n");
      break;
    case CM_ALLOCATE_FAILURE_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_ALLOCATE_FAILURE_RETRY\n");
      break;
    case CM_CONVERSATION_TYPE_MISMATCH:
      printf("CSAMPLE1: CPI-C returned CM_CONVERSATION_TYPE_MISMATCH\n");
      break;
    case CM_PIP_NOT_SPECIFIED_CORRECTLY:
      printf("CSAMPLE1: CPI-C returned CM_PIP_NOT_SPECIFIED_CORRECTLY\n");
      break;
    case CM_SECURITY_NOT_VALID:
      printf("CSAMPLE1: CPI-C returned CM_SECURITY_NOT_VALID\n");
      break;
    case CM_SYNC_LEVEL_NOT_SUPPORTED_LU:
      printf("CSAMPLE1: CPI-C returned CM_SYNC_LEVEL_NOT_SUPPORTED_LU\n");
      break;
    case CM_SYNC_LEVEL_NOT_SUPPORTED_PGM:
      printf("CSAMPLE1: CPI-C returned CM_SYNC_LEVEL_NOT_SUPPORTED_PGM\n");
      break;
    case CM_TPN_NOT_RECOGNISED:
      printf("CSAMPLE1: CPI-C returned CM_TPN_NOT_RECOGNISED\n");
      break;
    case CM_TP_NOT_AVAILABLE_NO_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_TP_NOT_AVAILABLE_NO_RETRY\n");
      break;
    case CM_TP_NOT_AVAILABLE_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_TP_NOT_AVAILABLE_RETRY\n");
      break;
    case CM_DEALLOCATED_ABEND:
      printf("CSAMPLE1: CPI-C returned CM_DEALLOCATED_ABEND\n");
      break;
    case CM_DEALLOCATED_NORMAL:
      printf("CSAMPLE1: CPI-C returned CM_DEALLOCATED_NORMAL\n");
      break;
    case CM_PARAMETER_ERROR:
      printf("CSAMPLE1: CPI-C returned CM_PARAMETER_ERROR\n");
      break;
    case CM_PRODUCT_SPECIFIC_ERROR:
      printf("CSAMPLE1: CPI-C returned CM_PRODUCT_SPECIFIC_ERROR\n");
      break;
    case CM_PROGRAM_ERROR_NO_TRUNC:
      printf("CSAMPLE1: CPI-C returned CM_PROGRAM_ERROR_NO_TRUNC\n");
      break;
    case CM_PROGRAM_ERROR_PURGING:
      printf("CSAMPLE1: CPI-C returned CM_PROGRAM_ERROR_PURGING\n");
      break;
    case CM_PROGRAM_ERROR_TRUNC:
      printf("CSAMPLE1: CPI-C returned CM_PROGRAM_ERROR_TRUNC\n");
      break;
    case CM_PROGRAM_PARAMETER_CHECK:
      printf("CSAMPLE1: CPI-C returned CM_PROGRAM_PARAMETER_CHECK\n");
      break;
    case CM_PROGRAM_STATE_CHECK:
      printf("CSAMPLE1: CPI-C returned CM_PROGRAM_STATE_CHECK\n");
      break;
    case CM_RESOURCE_FAILURE_NO_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_RESOURCE_FAILURE_NO_RETRY\n");
      break;
    case CM_RESOURCE_FAILURE_RETRY:
      printf("CSAMPLE1: CPI-C returned CM_RESOURCE_FAILURE_RETRY\n");
      break;
    case CM_UNSUCCESSFUL:
      printf("CSAMPLE1: CPI-C returned CM_UNSUCCESSFUL\n");
      break;
    case CM_DEALLOCATED_ABEND_SVC:
      printf("CSAMPLE1: CPI-C returned CM_DEALLOCATED_ABEND_SVC\n");
      break;
    case CM_DEALLOCATED_ABEND_TIMER:
      printf("CSAMPLE1: CPI-C returned CM_DEALLOCATED_ABEND_TIMER\n");
      break;
    case CM_SVC_ERROR_NO_TRUNC:
      printf("CSAMPLE1: CPI-C returned CM_SVC_ERROR_NO_TRUNC\n");
      break;
    case CM_SVC_ERROR_PURGING:
      printf("CSAMPLE1: CPI-C returned CM_SVC_ERROR_PURGING\n");
      break;
    case CM_SVC_ERROR_TRUNC:
      printf("CSAMPLE1: CPI-C returned CM_SVC_ERROR_TRUNC\n");
      break;
    default:
      printf("CSAMPLE1: CPI-C returned unrecognised return_code %d", (int) rc);
      break;
  }

  exit (-1);
}


