/*REXX*****************************************************************
 *
 * (C) Copyright IBM Corp. 2000 - All Rights Reserved.
 *
 * DISCLAIMER OF WARRANTIES.  The following [enclosed] code is sample
 * code created by IBM Corporation. This sample code is not part of
 * any standard or IBM product and is provided to you solely for the
 * purpose of assisting you in the development of your applications.
 * The code is provided "AS IS", without warranty of any kind.
 * IBM shall not be liable for any damages arising out of your use of
 * the sample code,  even if they have been advised of the possibility
 * of such damages.
 *
 *______________________________________________________________about__
 *
 *  purpose : generate code for RPC client and server from one single
 *            REXX program -> so it can be used as client-server pgm.
 *
 *  syntax  : RXRPCGEN <rpcFileName>
 *
 *  systems : aix, linux, os/2, win32
 *  requires:
 *  author  : Thorsten Schaper, IBM
 *  created : 25 Mar 2000
 *  last mod: 02 Jul 2001
 *
 *__________________________________________________________structure__
 *
 *  <init>
 *
 *  <error / message handler>
 *      noValue:
 *      syntax:
 *      addMsg: procedure expose !. (cls, nr, txt)
 *      outputMsg: procedure expose !. (cls, nr)
 *      exitWith: procedure expose !. (cls, nr)
 *      informationalMessages: procedure expose !.
 *      warningMessages: procedure expose !.
 *      errorMessages: procedure expose !.
 *      severeErrorMessages: procedure expose !.
 *
 *  <main>
 *      main:
 *
 *  <file utility functions>
 *      readFile: procedure expose !. fileContent. (fileName)
 *
 *  <rexx utility functions>
 *      stemCopy: (dest, src)
 *      askYesNo: procedure expose !. (question)
 *
 */

/* ====================================================================
 * ====================================================================
 *                               init
 * ====================================================================
 * ====================================================================
 */
   parse upper source os fn ccl
   parse arg rpcFileName

   !. = '';                    /* big mama of all global data      */

   !.osDep.curOS       = translate(left(os,3))

   !.rxRPCgenDir       = fileSpec('PATH', ccl)

   /* error handling__________________________________________________
    */
   !.msg.inf.0         = 0     /* number of information messages   */
   !.msg.wrn.0         = 0     /* number of warning messages       */
   !.msg.err.0         = 0     /* number of error messages         */
   !.msg.sev.0         = 0     /* number of severe error messages  */

   call informationalMessages
   call warningMessages
   call errorMessages
   call severeErrorMessages

/* ====================================================================
 * ====================================================================
 *                        error / message handler
 * ====================================================================
 * ====================================================================
 */
   SIGNAL ON NOVALUE;
   SIGNAL ON SYNTAX;
   SIGNAL main;

noValue:
   say
   say copies('-', 72);
   say 'Runtime error condition raised:' condition('c');
   say 'Variable    :' condition('d');
   say copies('-', 72);

   return '';

syntax:
   say
   say copies('-', 72);
   say 'Runtime error condition raised:' condition('c');
   say 'Source line#:' SIGL;
   say 'Source line :' sourceLine(SIGL);
   say 'Return code :' RC;
   say 'Error text  :' errorText(RC);
   say copies('-', 72);
   say
   if askYesNo('Do you want to go into interactive trace mode ?') then
      TRACE ?r;
   nop;nop;                    /* surely stop here in case of trace*/
   call exitwith 'E', 2;       /* syntax error runtime condition   */

addMsg: procedure expose !.;
   parse arg cls, nr, txt;

   select
      when cls = 'I' then class = 'INF';
      when cls = 'W' then class = 'WRN';
      when cls = 'E' then class = 'ERR';
      when cls = 'S' then class = 'SEV';
      otherwise
         signal errOcc;
   end
   if \dataType(nr, 'W') then
      signal errOcc;
   if !.msg.class.nr.0 \= '' then
      signal errOcc;

   lineNr = 1;
   !.msg.class.nr.0 = 1;
   do i = 1 to words(txt)
      select
         when length(!.msg.class.nr.lineNr) + length(word(txt, i)),
               <= 72 then do
            !.msg.class.nr.lineNr = !.msg.class.nr.lineNr,
                                        word(txt, i);
         end
         when length(!.msg.class.nr.lineNr) + length(word(txt, i)),
               > 72 & !.msg.class.nr.lineNr \= '' then do
            lineNr = lineNr + 1;
            i = i - 1;
         end
         otherwise do
            cutPos = 1;
            do until length(substr(txt, cutPos)) <= 72
               !.msg.class.nr.lineNr = substr(txt, cutPos, 71)'-';
               lineNr = lineNr + 1;
               cutPos = cutPos + 71;
            end
            !.msg.class.nr.lineNr = substr(txt, cutPos);
         end
      end
   end
   !.msg.class.nr.0 = lineNr;
   !.msg.class.0 = !.msg.class.0 + 1;

   return;

errOcc:
   say "Error occured while creating message Nr. '"nr"'";
   return;

outputMsg: procedure expose !.;
   parse arg cls, nr;

   mfTxt.0 = arg() - 2;
   if mfTxt.0 > 0 then do
      do i = 3 to arg()       /* read message field information      */
         mfNr = i - 2;
         mfTxt.mfNr = arg(i);
         mfTxt.0 = mfTxt.0 + 1;
      end
   end

   select
      when cls = 'I' then do
         class = 'INF';
         clsName = 'INFORMATIONAL';
      end
      when cls = 'W' then do
         class = 'WRN';
         clsName = 'WARNING';
      end
      when cls = 'E' then do
         class = 'ERR';
         clsName = 'ERROR';
      end
      when cls = 'S' then do
         class = 'SEV';
         clsName = 'SEVERE ERROR';
      end
      otherwise do
         say "ERROR in classification of Error Message...";
         say "--> message output aborted!";
         return;
      end
   end
   if \dataType(nr, 'W') then
      nr = 999;
   else
      if nr < 0 | nr > 999 | !.msg.class.nr.0 = '' then
         nr = 999;

   do i = 1 to !.msg.class.nr.0
      if i = 1 then
         outputStr = clsName':' !.msg.class.nr.i;
      else
         outputStr = copies(' ', length(clsName':')) !.msg.class.nr.i;
      do until mfPos = 0
         mfPos = pos('%', outputStr)     /* look for a message field */
         if mfPos > 0 then do
            mfNrCharPos = mfPos + 1;
            mfNrChar = substr(outputStr, mfNrCharPos, 1);
            if dataType(mfNrChar, 'W') then do
               mfNr = '';
               do while dataType(mfNrChar, 'W')
                  mfNr = mfNr || mfNrChar;
                  mfNrCharPos = mfNrCharPos + 1;
                  mfNrChar = substr(outputStr, mfNrCharPos, 1);
               end
               if symbol('mfTxt.'mfNr) = 'VAR' then do
                  outputStr = delStr(outputStr, mfPos, length(mfNr)+1);
                  outputStr = insert(mfTxt.mfNr, outputStr, mfPos - 1)
               end
            end
         end
      end
      say outputStr;
   end

   return;

exitWith: procedure expose !.;
   parse arg cls, nr;

   say
   say copies('-', 72);
   call outputMsg cls, nr;
   say
   say '--> exiting with RC =' nr;
   say copies('-', 72);

   exit nr;

informationalMessages: procedure expose !.;
   call addMsg 'I',   1, "Successfully read %1 lines from file '%2'";
   call addMsg "I",   2, "File '%1' does exist.";
   call addMsg "I",   3, "File '%1' doesn't exist/couldn't be found.";
   call addMsg "I",   4, "File '%1' has been copied to file '%2'.";
   call addMsg 'I',   5, "Successfully written %1 lines to file '%2'";
   /*
    * was used for testing message output
    *
   call addMsg 'I', 100, copies('a', 71)copies('b', 71)copies('c', 71);
   call addMsg 'I', 101, copies('a', 9) copies('b', 9) copies('c', 9),
                         copies('d', 9) copies('e', 9) copies('f', 9),
                         copies('g', 9) copies('h', 9) copies('i', 9),
                         copies('j', 9) copies('k', 9) copies('l', 9);
   */
   call addMsg 'I', 999,,
      "Well...sorry the desired INFORMATIONAL message was not found!";

   return;

warningMessages: procedure expose !.;
   call addMsg 'W', 999,,
      "Well...sorry the desired WARNING message was not found!";

   return;

errorMessages: procedure expose !.;
   call addMsg 'E',   2, "Syntax error runtime condition raised.";
   call addMsg 'E',   3, "Error while reading the input files.";
   call addMsg 'E',   4, "File contained no lines.";
   call addMsg 'E', 999,,
      "Well...sorry the desired ERROR message was not found!";

   return;

severeErrorMessages: procedure expose !.;
   call addMsg 'S',   1,,
      "sorry ... your OS:" !.osDep.curOS "is not supported";
   call addMsg 'S',   2, "File '%1' could not be read!";
   call addMsg 'S',   3, "File '%1' was not copied to file '%2' !";
   call addMsg 'S',   4, "File '%1' could not be written!";
   call addMsg 'S', 999,,
      "Well...sorry the desired SEVERE ERROR message was not found!";

   return;

/* ====================================================================
 * ====================================================================
 *                              main
 * ====================================================================
 * ====================================================================
 */
main:

   /* first of all read *.RPC file_____________________________________
    */
   if \readFile(rpcFileName)              then exit 1;
   if \stemCopy('_rpc.', 'fileContent.')  then exit 1;

   /*  fetch the params from *.RPC file________________________________
    */
   rpcDef = '';
   do i = 1 to _rpc.0
      rpcDef = rpcDef _rpc.i;
   end

   parse var rpcDef 'rexxFileName' . '=' rexxFileName .
   rexxFileName = strip(rexxFileName);

   parse var rpcDef 'serverFileName' . '=' serverFileName .
   serverFileName = strip(serverFileName);

   parse var rpcDef 'clientFileName' . '=' clientFileName .
   clientFileName = strip(clientFileName);

   parse var rpcDef 'defaultServerAddr' . '=' defaultServerAddr .
   defaultServerAddr = strip(defaultServerAddr);

   parse var rpcDef 'defaultServerPort' . '=' defaultServerPort .
   defaultServerPort = strip(defaultServerPort);

   parse var rpcDef 'remoteFunctions:' remoteFunctions 'globalVars:'
   remoteFunctions = strip(remoteFunctions);

   gVCount = 0;
   globalVarsNames = ''
   do i = _rpc.0 to 1 by -1 while _rpc.i <> 'globalVars:'
      gVCount = gVCount + 1;
      globalVars.gVCount = _rpc.i;
      globalVarsNames = globalVarsNames || word(_rpc.i, 1)' ';
   end
   globalVars.0 = gVCount;

   /*  print out the whole bunch of parameters_________________________
    */
   say 'rxRPCgen v1.0';
   say '   RPC fle                 :' rpcFileName;
   say '   REXX file               :' rexxFileName;
   say '   Client Stub file        :' clientFileName;
   say '   Server Stub file        :' serverFileName;
   say '   default server address  :' defaultServerAddr;
   say '   default server port     :' defaultServerPort;
   say '   remote functions        :' remoteFunctions;
   say '   global variables        :' globalVarsNames;

   /* now read the rest of the files___________________________________
    */
   if \readFile(rexxFileName)             then exit 1;
   if \stemCopy('_rex.', 'fileContent.')  then exit 1;

   if \readFile(!.rxRPCgenDir'_c.rex')    then exit 1;
   if \stemCopy('_c.', 'fileContent.')    then exit 1;

   if \readFile(!.rxRPCgenDir'_s.rex')    then exit 1;
   if \stemCopy('_s.', 'fileContent.')    then exit 1;

   /* generate stub code for remote functions__________________________
    */
   do i = 1 to words(remoteFunctions)

       fName = word(remoteFunctions, i);  /* get name of next remote */
                                          /* function                */

       /* extract the complete function for server stub code___________
        *
        * scans until next line where the first word
        * is directly followed by ':' is found.
        *
        */
       do o = 1 to _rex.0
           if wordPos(fName':', _rex.o) = 1 then do
               mlCount = 1                  /* method-line-count     */
               serverStub.fName.1 = _rex.o  /* get method header     */
               _rex.o = '0#'                /* mark line as processed*/
               do o = o + 1 to _rex.0,      /* get method body       */
               while pos(':', word(_rex.o, 1)) = 0
                   mlCount = mlCount + 1;
                   serverStub.fName.mlCount = _rex.o;
                   _rex.o = '0#'
               end
               serverStub.fName.0 = mlCount;
           end
       end

       /* build client stub code for that function_____________________
        */
       clientStub.fName.0 = 3;
       clientStub.fName.1 = fName':';
       clientStub.fName.2 = "    cmd = '"fName"';";
       clientStub.fName.3 = "    signal prepareRemoteCall;";
   end

   /* write client stub rexx file______________________________________
    * _c.  - file content of        '_c.rex'
    * _cl  - current line number in '_c.rex'
    * cl   - current line number in client stub file that is being
    *        generated now.
    */
writeClientStub:
    cl = 0;
    _cl = 0;

    call copyUntilNextBreak_c;

    call defaultHostAndPortCode defaultServerAddr, defaultServerPort;
    do i = 1 to defaultHostPort.0         /* insert code for testing */
        cl = cl + 1;                      /* host/port variables and */
        client.cl = defaultHostPort.i;    /* setting defaults if     */
    end                                   /* necessary.              */

    call copyUntilNextBreak_c;

    do i = 1 to _rex.0
        if _rex.i \== '0#' then do
            cl = cl + 1
            client.cl = _rex.i
        end
    end

    call copyUntilNextBreak_c;

    do i = 1 to words(remoteFunctions)    /* insert code for remote  */
        fName = word(remoteFunctions, i); /* function client-stub    */
        do o = 1 to clientStub.fName.0
            cl = cl + 1
            client.cl = clientStub.fName.o;
        end
    end

    call copyUntilNextBreak_c;

    client.0 = cl;
    if \writeFile(clientFileName, client.) then
        exit 1;

    /* write server stub rexx file_____________________________________
     * _s.  - file content of        '_s.rex'
     * _sl  - current line number in '_s.rex'
     * sl   - current line number in server stub file that is being
     *        generated now.
     */
writeServerStub:
    sl = 0;
    _sl = 0;

    call copyUntilNextBreak_s;

    do i = 1 to defaultHostPort.0
        sl = sl + 1
        server.sl = defaultHostPort.i;
    end

    call copyUntilNextBreak_s;

    do i = globalVars.0 to 1 by -1
        sl = sl + 1
        server.sl = '    'globalVars.i;
    end

    call copyUntilNextBreak_s;

    do i = 1 to words(remoteFunctions)
        fName = word(remoteFunctions, i);
        do o = 1 to serverStub.fName.0
            sl = sl + 1
            server.sl = serverStub.fName.o;
        end
        sl = sl + 1;          /* one empty line after each method.   */
        server.sl = '';
    end

    call copyUntilNextBreak_s;

    server.0 = sl;
    if \writeFile(serverFileName, server.) then
        exit 1;

    say 'client and server code successfully created...';
    exit 0;

defaultHostAndPortCode:
    parse arg host, port;

    defaultHostPort.0 = 4;
    defaultHostPort.1 = "    if host = '' | symbol('host') \= 'VAR'",
                        "then /* host name or dotted IP addr. */";
    defaultHostPort.2 = "        host =" defaultServerAddr';';
    defaultHostPort.3 = "    if port = '' | symbol('port') \= 'VAR'",
                        "then /* port number of RPC service.  */";
    defaultHostPort.4 = "        port =" defaultServerPort';';
    return;

copyUntilNextBreak_c:
    do _cl = _cl + 1 to _c.0 while \abbrev(_c._cl, '@_')
        cl = cl + 1
        client.cl = _c._cl;
    end
    return;

copyUntilNextBreak_s:
    do _sl = _sl + 1 to _s.0 while \abbrev(_s._sl, '@_')
        sl = sl + 1
        server.sl = _s._sl;
    end
    return;

/* ====================================================================
 * ====================================================================
 *                         file utility functions
 * ====================================================================
 * ====================================================================
 *
 *___________________________________________________________readFile__
 * return  : 0 = failure
 *           1 = success
 */
readFile: procedure expose !. fileContent.;
    parse arg fileName;

    errorOccured = 0;
    fileContent. = 0;

    select
        when wordPos(!.osDep.curOS, 'AIX LIN WIN') > 0 then do
            if (stream(fileName, 'C', 'OPEN READ') \== 'READY:') then
                errorOccured = 1;
            else do
                do i = 1 while lines(fileName)
                    fileContent.i = linein(fileName)
                end
                fileContent.0 = i - 1;
                call stream fileName, 'C', 'CLOSE'
            end
        end
        otherwise
            call exitWith 'S', 1;               /* OS not supported  */
    end

    if errorOccured then do
        call outputMsg "S", 2, fileName;
        return 0;
    end
    call outputMsg "I", 1, fileContent.0, fileName;

    return 1;

/*__________________________________________________________writeFile__
 */
writeFile: procedure expose !.
    use arg fileName, dateiInhalt.

    errorOccured = 0

    select
        when wordPos(!.osDep.curOS, 'AIX LIN WIN') > 0 then do
            if stream(fileName, 'C', 'OPEN WRITE REPLACE') \== 'READY:' then
                errorOccured = 1;
            else do
                do i = 1 to dateiInhalt.0
                    call lineout fileName, dateiInhalt.i
                end
                call stream fileName, 'C', 'CLOSE'
            end
      end
      otherwise
         call exitWith 'S', 1;                  /* OS not supported  */
   end

    if errorOccured then do
        call outputMsg "S", 4, fileName;
        return 0;
    end
    call outputMsg "I", 5, dateiInhalt.0, fileName;

    return 1;

/*
/*_________________________________________________________deleteFile__
 * return  : 0 = failure
 *           1 = success
 */
deleteFile: procedure expose !.
    parse arg fileName

    errorOccured = 0

    select
        when wordPos(!.osDep.curOS, 'AIX LIN WIN') > 0 then do
            call SysFileDelete fileName
            if result > 0 & result \= 2 & result \= 87 then
                errorOccured = 1
        end
        otherwise
            call exitWith 'S', 1;                  /* OS not supported  */
    end

    if errorOccured then
        return 0;

    return 1;
*/

/*___________________________________________________________copyFile__
 * return  : 0 = failure
 *           1 = success
 */
copyFile: procedure expose !.;
    parse arg sourceFile destFile;

    errorOccured = 0;
    fileContent. = 0;
    select
        when wordPos(!.osDep.curOS, 'AIX LIN') > 0 then do
            'cp' sourceFile destFile;
            if RC \= 0 then
                errorOccured = 1;
        end
        when wordPos(!.osDep.curOS, 'WIN') > 0 then do
            'copy' sourceFile destFile;
            if RC \= 0 then
                errorOccured = 1;
        end
        when !.osDep.curOS = 'CMS' then do
            parse upper var sourceFile fn '.' ft;
            sourceFile = fn ft '*';
            parse upper var destFile fn '.' ft;
            destFile = fn ft 'A';
            'copyfile' sourceFile destFile;
            if RC \= 0 then
                errorOccured = 1;
        end
        otherwise
            call exitWith 'S', 1;               /* OS not supported  */
    end

    if errorOccured then do
        call outputMsg "S", 3, sourceFile, destFile;
        return 0;
    end
    call outputMsg "I", 4, sourceFile, destFile;

    return 1;

/* ====================================================================
 * ====================================================================
 *                         rexx utility functions
 * ====================================================================
 * ====================================================================
 *
 *___________________________________________________________stemCopy__
 * params   : - string naming the destination stem
 *            - string naming the source      stem
 *            (examples for such strings: 'lines.' "args." 'DATA.')
 * returns  : x  - successfully copied x entries
 *            -1 - failure
 */
stemCopy:
   parse arg dest, src;

   INTERPRET,
      'do !i! = 0 to 'src'0;',
      dest'!i! = 'src'!i!;',
      'end;';

   if !i! - 1 = value(src'0') then
      return 1

   reurn -1

/*___________________________________________________________askYesNo__
 */
askYesNo: procedure expose !.;
    parse arg question;

    question = question "[y/n] : ";
    do while length(question) > 72
        cutPos = lastPos(' ', question, 72);
        parse var question line =(cutPos) question;
        call lineout , line;
    end
    do i = 1 to length(question)
        call charout , substr(question, i, 1);
    end
    answer = linein();

    if translate(answer) = 'Y' then
        return 1;

    return 0;
