// test.h       

#include <time.h>

#define IRQ             0x0a


#ifdef _DEFINE
#define INIT(x) = x
#define EXTERN
#else 
#define EXTERN extern
#define INIT(x)
#endif


#define D7              0x80
#define D6              0x40
#define D5              0x20
#define D4              0x10
#define D3              0x08
#define D2              0x04
#define D1              0x02
#define D0              0x01

//===========================================================================
// Register definitions
//
// Naming conventions:
//
//     xxx_REG      Indirect register port - says which register on a chip is 
//                  being accessed.
//     xxx_DATA     Indirect data port
//     xxx_yyy      Indirect register number
//     xxx_yyy_zzz  Bit mask for bit within an indirect data port
//     xxx_yyy_MASK Bit mask for all bits present within an indirect data port
//
//===========================================================================

//---------------------------------------------------------------------------
// The mixer register are accessed indirectly. First, write a register # to 
// MIX_REG. Then, access the data for that register through MIX_DATA.
//
// All registers are readable and writable.
//
// Note that some registers change meaning depending on record/playback mode:
//   MMA on playback becomes LINEIN on record.
//   MIC_LINEIN on playback becomes MIC on record. 
//---------------------------------------------------------------------------

#define BASE        0x380

#define MIX_REG             (BASE+0)
#define MIX_DATA            (BASE+1) 

#define MIX_LINEOUT_L       0x04
#define MIX_LINEOUT_R       0x05 
#define MIX_LINEOUT_MASK        D1|D2|D3|D4|D5  

#define MIX_MODE            0x08
#define MIX_MODE_MASK           D1|D2|D5
#define MIX_MODE_MUTE           D5
#define MIX_MODE_SELECT_L       D2
#define MIX_MODE_SELECT_R       D1  

#define MIX_FM_L            0x09
#define MIX_FM_R            0x0a 
#define MIX_FM_MASK             D2|D3|D4|D5|D6  

#define MIX_MMA_L           0x0b
#define MIX_MMA_R           0x0c
#define MIX_MMA_MASK            D2|D3|D4|D5|D6  

#define MIX_MIC_L           MIX_MMA_L
#define MIX_MIC_R           MIX_MMA_R
#define MIX_MIC_MASK        MIX_MMA_MASK 

#define MIX_LINEIN_L        0x0d
#define MIX_LINEIN_R        0x0e 
#define MIX_LINEIN_MASK         D2|D3|D4|D5|D6   
    

#define MIX_MIC_LINEIN_L    MIX_LINEIN_L
#define MIX_MIC_LINEIN_R    MIX_LINEIN_R
#define MIX_MIC_LINEIN_MASK     MIX_MIC_MASK

#define MIX_SELECT          0x11
#define MIX_SELECT_MASK         D0|D1|D2
#define MIX_SELECT_MONO         D2
#define MIX_SELECT_RECORD_L     D1
#define MIX_SELECT_RECORD_R     D0


//---------------------------------------------------------------------------
// The FM register are accessed indirectly. First, write a register # to 
// FM_REG. Then, access the data for that register through FM_DATA. 
//
// There are two sets of registers, corresponding to channels 1 and 2   
// 
// The FM registers are all write-only.
//---------------------------------------------------------------------------
#define     FM_REG      (BASE+0x08)
#define     FM_DATA     (BASE+0x09)
#define     FM_REG2     (BASE+0x0a)
#define     FM_DATA2    (BASE+0x0b)  

#define     FM_TEST     0x01
#define     FM_TEST_WHITE_NOISE 0x05

//---------------------------------------------------------------------------
// The MMA register are accessed indirectly. First, write a register # to 
// MMA_REG. Then, access the data for that register through MMA_DATA. 
//
// There are two sets of registers, corresponding to channels 1 and 2
//
// Not all registers are readable and writable. Only the following registers
// are readable:
//                  MMA_SELT
//                  MMA_TIMER2_L
//                  MMA_TIMER2_H
//                  MMA_PCM_DATA
//                  MMA_MIDI_DATA
//
// Only the following registers are present in channel 2:
//
//                  MMA_SELT
//                  MMA_PCM_CONTROL
//                  MMA_VOLUME
//                  MMA_PCM_DATA
//                  MMA_PCM_FORMAT
//
//---------------------------------------------------------------------------

#define MMA_REG             (BASE+0x0c)
#define MMA_STATUS          MMA_REG
#define MMA_DATA            (BASE+0x0d)
#define MMA_REG2            (BASE+0x0e)
#define MMA_DATA2           (BASE+0x0f) 

#define MMA_SELT            0x00 
#define MMA_SELT_MASK               D0

#define MMA_TEST            0x01 

#define MMA_TIMER0_L        0x02
#define MMA_TIMER0_H        0x03 

#define MMA_BASE_L          0x04
#define MMA_BASE_H          0x05 

#define MMA_TIMER2_L        0x06
#define MMA_TIMER2_H        0x07 

#define MMA_TIMER_CONTROL   0x08 

#define MMA_PCM_CONTROL     0x09
#define MMA_PCM_CONTROL_ADPRST          0x80
#define MMA_PCM_CONTROL_SELECT_R        0x40
#define MMA_PCM_CONTROL_SELECT_L        0x20
#define MMA_PCM_CONTROL_SAMP_RATE_MASK  0x18 
#define MMA_PCM_CONTROL_PCM             0x04
#define MMA_PCM_CONTROL_PLAY            0x02
#define MMA_PCM_CONTROL_ADPST           0x01

#define MMA_VOLUME          0x0a 

#define MMA_PCM_DATA        0x0b 

#define MMA_PCM_FORMAT      0x0c 

#define MMA_MIDI_CONTROL    0x0d 

#define MMA_MIDI_DATA       0x0e

//---------------------------------------------------------------------------
// The TMPX and FIFO ports are direct
// TMPX is write-only
// FIFO is read/write but, reset FIFO pointers before reading!
//---------------------------------------------------------------------------

#define FIFO_REG            (BASE+2)

#define TMPX_REG            (BASE+4) 
#define TMPX_PLAY                   D0
#define TMPX_TSM_RESET              D1
#define TMPX_MRESET                 D2
#define TMPX_AGEN_RESET             D3
#define TMPX_LOW_TEST               D4
#define TMPX_HI_TEST                D5
#define TMPX_TEST_INN_OUT           D6
                               
typedef unsigned char   BYTE;   // 8-bit unsigned entity
typedef unsigned short  WORD;   // 16-bit unsigned number
typedef unsigned int    UINT;   // machine sized unsigned number (preferred)
typedef long            LONG;   // 32-bit signed number
typedef unsigned long   DWORD;  // 32-bit unsigned number
typedef int             BOOL;   // BOOLean (0 or !=0)
typedef char            CHAR;
typedef int             INT;
typedef void            VOID; 
#define FAR _far
#define NEAR _near
typedef BYTE FAR        *LPBYTE;
typedef VOID FAR        *LPVOID;
typedef BYTE huge*      PH_BYTE;

#define FALSE           0
#define TRUE            1

typedef VOID (_cdecl _interrupt _far *LPISR)(VOID);

#define LOWORD(l)           ((WORD)(l))
#define HIWORD(l)           ((WORD)(((DWORD)(l) >> 16) & 0xFFFF))
#define LOBYTE(w)           ((BYTE)(w))
#define HIBYTE(w)           ((BYTE)(((WORD)(w) >> 8) & 0xFF))                               

#define MAKELONG(low, high) ((LONG)(((WORD)(low)) | (((DWORD)((WORD)(high))) << 16)))
#define MAKELP(sel, off)    ((void FAR*)MAKELONG((off), (sel)))
#define SELECTOROF(lp)      HIWORD(lp)
#define OFFSETOF(lp)        LOWORD(lp)
#define FIELDOFFSET(type, field)    ((int)(&((type NEAR*)1)->field)-1) 


//--------------------------------------------------------------------
// Macros
//--------------------------------------------------------------------
#define DELAY {IODelay();}
        
#define MMA_OUT(_reg_, _v_) \
        OutIndirect(MMA_REG, _reg_, _v_)

#define MMA_OUT2(_reg_, _v_) \
        OutIndirect(MMA_REG2, _reg_, _v_) 

#define MMA_IN(_reg_) \
        InIndirect(MMA_REG, _reg_)

#define MIX_OUT(_reg_, _v_) \
        OutIndirect(MIX_REG, _reg_, _v_) 

//--------------------------------------------------------------------
// Globals 
//--------------------------------------------------------------------

#define SILENCE 0x80
#define FIFO_TRIGGER_LEVEL  64
#define DATASIZE        (128 - FIFO_TRIGGER_LEVEL)
#define DISKREADSIZE    (1024L * 2)
#define SAMPLE_SIZE     (1024L * 128)


WORD                    Irq INIT(IRQ);
WORD                    IntVect;
WORD                    PicImr;
WORD                    PicEoi;
WORD                    PicMask;
BOOL                    bWaveDone;
BOOL                    bufferFull INIT(FALSE);

int                     mmaStatus;
LPISR                   old_ISR INIT(0L);

DWORD                   ISR_hit INIT(0L);
DWORD                   dataSize INIT(0);
DWORD                   readPos INIT(0L);
DWORD                   writePos INIT(0L);

DWORD                   remainingChunk INIT(0L);

BYTE                    dataRate INIT(2);

PH_BYTE                 data;
BYTE                    diskread[DISKREADSIZE];
BYTE                    diskBuffer[DISKREADSIZE+16]; 


#define MIN_REGISTER        0x01
#define MAX_REGISTER        0xF5
#define ADLIB_FM_ADDRESS    0x388       /* adlib address/status register */
#define ADLIB_FM_DATA       0x389       /* adlib data register           */



//--------------------------------------------------------------------
//  FM stuff
//--------------------------------------------------------------------
/*
* FM Instrument definition for .SBI files - SoundBlaster instrument
* - these are the important parts - we will skip the header, but since
*   I am not sure where it starts and ends so I have had to guess.
*   However it SEEMS! to work. Each array has two values, one for
*   each operator.
*/
typedef struct
    {
    BYTE SoundCharacteristic[2];    /* modulator frequency multiple...  */
    BYTE Level[2];                  /* modulator frequency level...     */
    BYTE AttackDecay[2];            /* modulator attack/decay...        */
    BYTE SustainRelease[2];         /* modulator sustain/release...     */
    BYTE WaveSelect[2];             /* output waveform distortion       */
    BYTE Feedback;                  /* feedback algorithm and strength  */
    } FMInstrument;

/*
* Enumerated F-Numbers (in octave 4) for the chromatic scale.
*/

#define NOTE_D4b 0x16B
#define NOTE_D4  0x181
#define NOTE_E4b 0x198 
#define NOTE_E4  0x1B0
#define NOTE_F4  0x1CA 
#define NOTE_G4b 0x1E5 
#define NOTE_G4  0x202 
#define NOTE_4b 0x220
#define NOTE_A4  0x241 
#define NOTE_B4b 0x263  
#define NOTE_B4  0x287  
#define NOTE_C4  0x2AE


//---------------------------------------------------------------------
// Prototypes
//---------------------------------------------------------------------
VOID  OutIndirect (WORD wBase, WORD wOffset,  BYTE  bValue);
BYTE  InIndirect (WORD  wBase, WORD wOffset);
BOOL TestIndirect(WORD wBase, WORD wOffset, BYTE wMask); 
VOID OutTmpx(BYTE bValue);
BYTE InTmpx(VOID);
VOID ResetFifo(VOID);
VOID SetTmpxPlay(VOID);
VOID SetTmpxRecord(VOID);
VOID  PlayWave(CHAR *pFileName, int Rate, BOOL bEndlessPar);
VOID EndWave(VOID);
VOID  RecWave(CHAR *pFileName, int Rate);
VOID  delay (VOID);
VOID MReset(VOID);
VOID _far IODelay(VOID); 
VOID  MixPlayMMAMax(VOID); 
VOID MixPlayFMMax(VOID);
VOID ResetMMA(VOID);
VOID FMPlayScale(VOID);
void WriteFM(int reg, int value);
int  ReadFM(void);
int  AdlibExists(void);
void FMReset(void);
void FMKeyOff(int voice);
void FMKeyOn(int voice, int freq, int octave);
void FMVoiceVolume(int voice, int vol);
void FMSetVoice(int voiceNum, FMInstrument *ins);
int  LoadSBI(char filename[], FMInstrument *ins); 
VOID SetTestVoice(VOID);
void Wait(clock_t wait);
BOOL PlayFM(CHAR *pFileName);  
BOOL CheckBreak(VOID);