// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : rules.h
//
//  DESCRIPTION :
//         Classes CurSituation & NextSituation etc (Definition)
//
// ##############################################################



// ###############
#ifndef _RULES_H
#define _RULES_H

// ================
#include "symbol.h"
// ================

// =========
enum SymbolTypes
{
  UNDEF_SymbolTypes,
  // ---------
  BLANK_Symbol,
  INPUT_Symbol,
  INTERNAL_NOT_MARKER_Symbol,
  MARKER_INTERNAL_Symbol,
  // ---------
  NEVER_MIND_PseudoSymbol,
  ANY_NOT_MARKER_PseudoSymbol,
  ANY_NOT_MARKER_AND_NOT_BLANK_PseudoSymbol,
  // WRITE_NOTHING_PseudoSymbol,
  // ---------
  SIZE_OF_SymbolTypes
};



// =========
enum OusideRuleTypes
{
  DETERMINISTIC_OUTSIDE_RULE_TYPE,
  PASSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE,
  ACTIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE,
  AGGRESSIVE_NON_DETERMINISTIC_OUTSIDE_RULE_TYPE,
  // ---------
  NUMBER_OF_OUTSIDE_RULE_TYPES
};


// =========
class CurSituation
{
friend bool pre_operator (
		const CurSituation& inst1_i, const CurSituation& inst2_i, 
		bool ret_if_less_i, bool ret_if_greater_i, bool ret_if_end_i
		);
friend bool operator< (const CurSituation& inst1_i, const CurSituation& inst2_i);
friend bool operator== (const CurSituation& inst1_i, const CurSituation& inst2_i);
  private :
    // -------------------
    const state_t		cur_state_;
    state_t  			cur_substate1_;
    state_t			cur_substate2_;
    const vector<symbol_mt>	cur_symbols_;	// cur_symbols_.size() == number of tapes
    vector<SymbolTypes>		cur_symbol_mtypes_;

    const bool			controlling_flag_;

    static vector<state_t> 	allowed_states_s;
    static vector<state_t> 	allowed_neutral_cur_states_s;
    static vector<symbol_mt> 	allowed_symbols_s;


  public :
    CurSituation ();
    CurSituation (
		const state_t&		cur_state_i, 
		const state_t&		cur_substate1_i, 
		const state_t&		cur_substate2_i, 
		const vector<symbol_mt>&	cur_symbols_i,
		const vector<SymbolTypes>&	cur_symbol_mtypes_i,
		bool			controlling_flag_i
		);
    ~CurSituation ();

   state_t	get_state () const;
   state_t	get_substate1 () const;
   state_t	get_substate2 () const;
   void		set_substate1 (const state_t& state_i);
   void		set_substate2 (const state_t& state_i);
   bool		get_controlling_flag () const;
   symbol_mt	get_symbol (size_t index_i) const;
   size_t	get_total_symbols () const;
   string	getstr_symbols (size_t max_symbol_size_i = 0) const;
   string	getstr_show (const string& msg_i = string()) const;
   vector<symbol_mt>	get_symbols () const;
   static void	set_shared_data1_s (
 			const vector<symbol_mt>&	allowed_symbols_i,
			const vector<state_t>&	allowed_states_i
	       		);
   static void	set_shared_data2_s (
			const vector<state_t>& allowed_states_i,
			const vector<state_t>& allowed_neutral_cur_states_i
			);
   static void	clear_shared_data_s ();
};

// ---------

// =========
class SymbolAndShift 
{ 
friend class NextSituation;
  private :
    symbol_mt symbol_; 
    shift_t shift_; 

  public :
    SymbolAndShift (const symbol_mt& symbol_i, shift_t shift_i) : symbol_ (symbol_i), shift_ (shift_i) {}
    void	set_symbol (const symbol_mt& symbol_i);
    void	set_no_shift ();
    void	set_shift (const shift_t& shift_i);
    symbol_mt	get_symbol () const {return symbol_; }
    shift_t	get_shift () const {return shift_; }
    ~SymbolAndShift () {} 
};

// =========
class NextSituation
{
  private :
    const state_t			next_state_;
    state_t				next_substate1_;
    state_t   				next_substate2_;
    const vector<SymbolAndShift>	next_symbols_and_shifts_;	// next_symbol_and_shift_.size() == number of tapes
    const vector<SymbolTypes>		next_symbol_mtypes_;

    static vector<state_t> 	allowed_states_s;
    static vector<symbol_mt> 	allowed_symbols_s;
    static vector<symbol_mt> 	write_nothing_pseudo_symbols_s;

  public :
    NextSituation ();
    NextSituation (
		const state_t&	next_state_i, 
		const state_t&	next_substate1_i, 
		const state_t&	next_substate2_i, 
		const vector<SymbolAndShift>& next_symbols_and_shifts_i
		);
    ~NextSituation ();	

   state_t	get_state () const;
   state_t	get_substate1 () const;
   state_t	get_substate2 () const;
   void		set_substate1 (const state_t& state_i);
   void		set_substate2 (const state_t& state_i);
   symbol_mt	get_symbol (size_t index_i) const;
   void		set_symbol_mtype (size_t index_i, SymbolTypes type_i);
   shift_t	get_shift (size_t index_i) const;
   size_t	get_total_symbols () const;
   string	getstr_symbols_and_shifts(size_t max_symbol_size_i = 0) const;
   string	getstr_show (const string& msg_i = string()) const;
   vector<SymbolAndShift>	get_symbols_and_shifts () const;  

   static void	set_shared_data1_s (
			const vector<symbol_mt>&	write_nothing_pseudo_symbols_i,
			const vector<symbol_mt>&	allowed_symbols_i,
			const vector<state_t>&	allowed_states_i
			);
   static void	set_shared_data2_s (const vector<state_t>& allowed_states_i);
   static void	clear_shared_data_s ();
};




// =========
// =========
class CurOutsideSituation
{
friend bool operator< (const CurOutsideSituation& inst1_i, const CurOutsideSituation& inst2_i);
  private :
    const state_t	cur_daemon_state_;
    const state_t	cur_tester_state_;
    const state_t	cur_apparatus_state_;
    const OusideRuleTypes	non_deterministic_rule_no_;

  public :
    CurOutsideSituation (
		const state_t&	cur_daemon_state_i,
		const state_t&	cur_tester_state_i,
		const state_t&	cur_apparatus_state_i,
		OusideRuleTypes	non_deterministic_rule_no_i = DETERMINISTIC_OUTSIDE_RULE_TYPE
		);
    ~CurOutsideSituation ();

    state_t	get_cur_daemon_state () const;
    state_t	get_cur_tester_state () const;
    state_t	get_cur_apparatus_state () const;
    OusideRuleTypes	get_non_deterministic_rule_no () const;

}; // CurOutsideSituation


// =========
class NextOutsideSituation
{
  private :
    const state_t	next_daemon_state_;
    const state_t	next_tester_state_;
    const state_t	next_apparatus_state_;

    const bool		program_state_allowed_flag_;

  public :
    NextOutsideSituation (
		const state_t&	next_daemon_state_i,
		const state_t&	next_tester_state_i,
		const state_t&	next_apparatus_state_i,
		bool		program_state_allowed_flag_i
		);
    ~NextOutsideSituation ();

    state_t	get_next_daemon_state () const;
    state_t	get_next_tester_state () const;
    state_t	get_next_apparatus_state () const;
    bool	get_program_state_allowed_flag () const;


}; // NextOutsideSituation


#endif	// _RULES_H
