unit pFIBStatistic;

interface

uses
   Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
   pTestInfo,pAppStatInfoInt,FIBDatabase,pFIBProps,fib,StdFuncs,pFIBQuery,
   FIBQuery, pFIBCacheQueries
  ;

type
  TpFIBStatParam = (
    fspExecuteCount,
    fspSumTimeExecute,
    fspAvgTimeExecute,
    fspMaxTimeExecute,
    fspLastTimeExecute
  );

  TpFIBStatParams= set of TpFIBStatParam;


  TpFIBStatistic = class(TComponent)
  private
    tr:TFIBTransaction;
    FAppStatInfo:TTestInfo;
    FApplicationID :string;
    FDatabase   :TFIBDatabase;
    FLogFileName:string;
    FLogParams  :TpFIBStatParams;
    FExistStatTable :TpFIBExistObject;
    procedure SetActive(Value:boolean);
    function  GetActive:boolean;
    procedure SetLogFileName(FileName:string);
    procedure SetLogParams (Value:TpFIBStatParams);
    procedure SetDatabase(DB:TFIBDatabase);
    procedure CheckDatabase;
    procedure OnDisconnect(Sender:TObject);
  protected
    procedure   Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner:TComponent); override;
    destructor  Destroy; override;
    procedure   Clear;
    procedure   SaveLog;
    procedure   EraseLog;
    procedure   SortPrint(const VarName:string;Ascending:boolean);
    function    ExistStatTable:boolean;
    procedure   CreateStatTable;
    procedure   SaveLogToDB;
  published
    property    Active:boolean read GetActive write SetActive;
    property    LogFileName:string read FLogFileName write SetLogFileName;
    property    Database   :TFIBDatabase read FDatabase  write SetDatabase;
    property    ApplicationID :string read FApplicationID write FApplicationID;
    property    LogParams  :TpFIBStatParams read FLogParams write SetLogParams
     default
      [fspExecuteCount,fspSumTimeExecute,fspAvgTimeExecute, fspMaxTimeExecute,
        fspLastTimeExecute
      ];
  end;



implementation
uses StrUtil, FIBConsts;

{ TpFIBStatistic }

procedure TpFIBStatistic.CheckDatabase;
begin
 if not Assigned(Database) then
  FIBError(feDatabaseNotAssigned,[CmpFullName(Self)]);
 if not Database.Connected then
  FIBError(feDatabaseClosed,[CmpFullName(Self)]);
end;

procedure TpFIBStatistic.Clear;
begin
 FAppStatInfo.Clear;
end;

constructor TpFIBStatistic.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAppStatInfo:=TTestInfo.Create;
  FLogParams  :=[
    fspExecuteCount,
    fspSumTimeExecute,
    fspAvgTimeExecute,
    fspMaxTimeExecute,
    fspLastTimeExecute
  ];
  FExistStatTable:=eoUnknown;
  tr:=TFIBTransaction.Create(Self);
end;

destructor TpFIBStatistic.Destroy;
begin
  inherited;
  Active:=false;
  FAppStatInfo.Free;
end;

procedure TpFIBStatistic.EraseLog;
begin
 FAppStatInfo.EraseLog
end;

procedure   TpFIBStatistic.CreateStatTable;
var q:TpFIBQuery;
begin
 if not ExistStatTable then begin
  tr.DefaultDatabase:=Database;
  q:=TpFIBQuery.Create(Self);
  q.Database:=Database;
  q.Transaction:=tr;
  q.ParamCheck :=false;
  try
   tr.StartTransaction;
   q.SQL.Text :='CREATE GENERATOR GEN_FIB$APP_STATISIC_ID';
   q.ExecQuery;

   q.SQL.Text :=
    'CREATE TABLE FIB$APP_STATISIC ( ID INTEGER NOT NULL, '+
    'APP_ID VARCHAR(12), SQL_TEXT BLOB SUB_TYPE 2 SEGMENT SIZE 1,'+
    'EXECUTECOUNT INTEGER, SUMTIMEEXECUTE INTEGER, AVGTIMEEXECUTE INTEGER,'+
    'MAXTIMEEXECUTE INTEGER, MAXTIME_PARAMS BLOB SUB_TYPE 2 SEGMENT SIZE 1,'+
    'LASTTIMEEXECUTE INTEGER, LOG_DATE '+
     iifStr(Database.SQLDialect<2,'DATE','TIMESTAMP')+
    ' ,CMP_NAME VARCHAR(100), ATTACHMENT_ID INTEGER)';
   q.ExecQuery;
   q.SQL.Text :=
    'ALTER TABLE FIB$APP_STATISIC ADD CONSTRAINT PK_FIB$APP_STATISIC PRIMARY KEY (ID)';
   q.ExecQuery;
   q.SQL.Text :=
    'CREATE TRIGGER FIB$APP_STATISIC_BI_BI FOR FIB$APP_STATISIC '+
    'ACTIVE BEFORE INSERT POSITION 0 '+
    'AS BEGIN '+#13#10+
    'IF (NEW.ID IS NULL) THEN '+#13#10+
    'NEW.ID = GEN_ID(GEN_FIB$APP_STATISIC_ID,1);'+#13#10+
    'NEW.log_date =''NOW'';'+#13#10+'END';
   q.ExecQuery;
   tr.Commit;
   FExistStatTable:=eoYes;
  finally
   q.Free;
  end;
 end;
end;

function TpFIBStatistic.ExistStatTable: boolean;
begin
 CheckDatabase;
 case FExistStatTable of
  eoYes: Result:=True;
  eoNo : Result:=False;
 else
  Result:=
   Database.QueryValue(
    'Select Count(*) from RDB$RELATIONS Where RDB$RELATION_NAME=''FIB$APP_STATISIC''',0
   )>0;
  if Result then
   FExistStatTable:=eoYes
  else
   FExistStatTable:=eoNo;
 end;
end;

procedure TpFIBStatistic.SaveLogToDB;
var i:integer;
    q:TpFIBQuery;
    s:string;
begin
 if not ExistStatTable then raise Exception.Create(SFIBStatNoSave);
 if IsBlank(FApplicationID) then
  raise Exception.Create(SFIBStatNoSaveAppID);
 tr.DefaultDatabase:=Database;
 q:=  GetQueryForUse(tr,
   'Insert into FIB$APP_STATISIC '+
   '( APP_ID, SQL_TEXT, EXECUTECOUNT , SUMTIMEEXECUTE , AVGTIMEEXECUTE ,  MAXTIMEEXECUTE ,'+
   'LASTTIMEEXECUTE, MAXTIME_PARAMS, CMP_NAME,ATTACHMENT_ID) '+
   'Values (?AP,?S,?E,?SU,?A,?M,?L,?MP,?C,?AT)'
 );
 with FAppStatInfo do
 try
  tr.StartTransaction;
  for i :=0  to Pred(ObjCount) do
  begin
   q.ParamByName('S' ).asString :=ObjName(i);
   q.ParamByName('AP').asString :=FApplicationID;
   q.ParamByName('E').asInteger:=GetVarInt(ObjName(i),scExecuteCount);
   q.ParamByName('SU').asInteger:=GetVarInt(ObjName(i),scSumTimeExecute);
   q.ParamByName('A').asInteger:=GetVarInt(ObjName(i),scAvgTimeExecute);
   q.ParamByName('M').asInteger:=GetVarInt(ObjName(i),scMaxTimeExecute);
   q.ParamByName('L').asInteger:=GetVarInt(ObjName(i),scLastTimeExecute);
   q.ParamByName('C').asString :=GetVarStr(ObjName(i),scLastQuery);
   q.ParamByName('AT').asInteger:=Database.AttachmentID;
   s:= GetVarStrings(ObjName(i),scMaxTimeExecute).Text;
   if s<>'' then
    q.ParamByName('MP').asString:=s
   else
    q.ParamByName('MP').IsNull:=true;
   q.ExecQuery;
  end;
 finally
   FreeQueryForUse(q);
   tr.Commit;
 end
end;

function  TpFIBStatistic.GetActive: boolean;
begin
 Result:= IAppStatInfo(FAppStatInfo)=AppStatInfo;
end;

procedure TpFIBStatistic.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (Operation=opRemove) then
   if AComponent=FDatabase then FDatabase:=nil;
  inherited Notification(AComponent,Operation);
end;

procedure TpFIBStatistic.SaveLog;
begin
 FAppStatInfo.SaveLog
end;

procedure TpFIBStatistic.SetActive(Value: boolean);
begin
 if Value then
  AppStatInfo:=FAppStatInfo
 else
  AppStatInfo:=nil
end;

procedure TpFIBStatistic.SetLogFileName(FileName: string);
begin
 FLogFileName:=FileName;
 FAppStatInfo.LogFileName:=FileName
end;

procedure TpFIBStatistic.SetDatabase(DB:TFIBDatabase);
begin
 if Assigned(FDatabase) then
  FDatabase.RemoveEvent(OnDisconnect,detBeforeDisconnect);
 FDatabase:=DB;
 if Assigned(FDatabase) then
  FDatabase.AddEvent(OnDisconnect,detBeforeDisconnect);
 FExistStatTable:=eoUnknown;
end;

procedure TpFIBStatistic.OnDisconnect(Sender:TObject);
begin
 FExistStatTable:=eoUnknown;
end;

procedure TpFIBStatistic.SetLogParams (Value:TpFIBStatParams);
begin  
 with FAppStatInfo do
 begin
   SetLogParamsInc(scExecuteCount  ,  fspExecuteCount    in FLogParams );
   SetLogParamsInc(scSumTimeExecute,  fspSumTimeExecute  in FLogParams );
   SetLogParamsInc(scAvgTimeExecute,  fspAvgTimeExecute  in FLogParams );
   SetLogParamsInc(scMaxTimeExecute,  fspMaxTimeExecute  in FLogParams );
   SetLogParamsInc(scLastTimeExecute, fspLastTimeExecute in FLogParams );
 end;
end;

procedure TpFIBStatistic.SortPrint(const VarName: string;
  Ascending: boolean);
begin
 FAppStatInfo.SortPrint(VarName,Ascending)
end;

end.
