/*
 *  Reference counting classes.
 *  Written and tested with Borland C++ Builder 3.0.
 *
 *  References:
 *  Bjarne Stroustrup, The C++ Programming Language Third Edition, 27.7.
 *
 *  Author:
 *  Boris Botstein.
 *  botstein@yahoo.com
 *  www.geocities.com/botstein/
 *
 *  12 Oct 1999.
 */

#ifndef REFCNT_HPP
#define REFCNT_HPP

template < class X > class value_smart_ref {
protected:
	class holder {
	public:
		X* rep;
		int count, accessed;

		holder(X* ptr) : rep(ptr), count(1), accessed(0) {}
		~holder() { delete rep; }
	};
	holder* value;

	void unbind() { if(--value->count == 0) delete value; }

public:
	explicit value_smart_ref(X* ptr = 0) : value(new holder(ptr)) {}
	value_smart_ref(const value_smart_ref< X >& rhs) : value(rhs.value) {
		value->count++;
	}
	~value_smart_ref() { unbind(); }

	void bind(const value_smart_ref< X >& rhs) {
		if(rhs.value != value) {
			unbind();
			value = rhs.value;
			value->count++;
		}
	}
	void bind(X* ptr) {
		if(rhs.value->rep != ptr) {
			unbind();
			value = new holder(ptr);
		}
	}

	value_smart_ref< X >& operator=(const value_smart_ref< X >& rhs) {
		bind(rhs);    
		return *this;
	}
	value_smart_ref< X >& operator=(X* ptr) {
		bind(ptr);
		return *this;
	}

	X* get() const { value->accessed++; return value->rep; }

	X* operator->() { return get(); }
	X& operator*() { return *get(); }

	const X* operator->() const { return get(); }
	const X& operator*() const { return *get(); }

	int counter() const { return value->count; }
	int accessed() const { return value->accessed; }
};

template < class X > class key_smart_ref : public value_smart_ref< X > {
public:
	explicit key_smart_ref(X* ptr = 0) : value_smart_ref< X >(ptr) {}
	key_smart_ref(const key_smart_ref< X >& rhs) : value_smart_ref< X >(rhs) {}

	key_smart_ref< X >& operator=(const key_smart_ref< X >& rhs) {
		value_smart_ref< X >::operator=(rhs);
		return *this;
	}

	bool operator==(const key_smart_ref< X >& rhs) const {
		return *(value->rep) == *(rhs.value->rep);
	}
	bool operator<(const key_smart_ref< X >& rhs) const {
		return *(value->rep) < *(rhs.value->rep);
	}
};

#endif // REFCNT_HPP

