//
//
//        
//                         
//                        
//                         Ŀ Ŀ     Ŀ  Ŀ
//                                             
//                               \        
//                    Ĵ      \\       Ŀ
//                          \         \\          
//                              \         \          
//                         \         
//                   
//               
//                        Timer Library
//          
//         
//        
//
//                    Microsoft Windows 95/98/NT Version
//
//  Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Dan Higdon
//
// Chronos
//
//      Chronos is the system timer interface.  Through Chronos, the user
//  can register functions to be invoked at a particular interval.
//
//

//
//
//                                Includes
//
//

#include <assert.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <mmsystem.h>

#include "chronos.hpp"

//
//
//                               Structures
//
//

struct iChronosUserProc
{
    UINT            hTimer;
    ChronosCallback fn;
    void            *parm;
};

//
//
//                               Routines
//
//

// Imported from the Mythos class system
extern "C" int mythos_atexit (register void (*__func) (void));
extern "C" void mythos_atexit_remove (register void (*__func) (void));

extern "C" double chronos_rdtsc();

// A cleanup function
static void ichronos_cleanup ();

//
//
//                                 Data
//
//

const int MAX_CALLBACKS = 32;

static volatile ulong   system_freq;
static volatile int     instance = 0;
static iChronosUserProc timer_handles[MAX_CALLBACKS];
static bool             __chronos_use_pentium = false;
double                  __chronos_scale;

//
//
//                                 Code
//
//

//
// A simple utility to convert between "timer multipliers" and milliseconds
//
inline ulong mult2ms (ulong multiplier)
{
    assert (multiplier > 0);

    return 549 / (multiplier * 10);
}


//
// ichronos_timer_proc
//
//  This routine will call our timer proc.
//
void CALLBACK ichronos_timer_proc (UINT idTimer, UINT msg, DWORD dwUser, DWORD dw1, DWORD dw2)
{
    iChronosUserProc *pcup = (iChronosUserProc *)dwUser;

    assert (idTimer == pcup->hTimer);

    pcup->fn (pcup->parm);
}


//
// chronos_init
//
// Initialize Chronos
//
void chronos_init ()
{
    if (instance == 0)
    {
        system_freq = 0;
        chronos_interval (0);
        memset (timer_handles, 0, sizeof (timer_handles));
        mythos_atexit (ichronos_cleanup);

        // See if we can use the Pentium timer register
        SYSTEM_INFO  ProfilerSysinfo;

        GetSystemInfo (&ProfilerSysinfo);
        __chronos_use_pentium = false;

        if (ProfilerSysinfo.dwProcessorType == PROCESSOR_INTEL_PENTIUM)
        {
            int     c0,cf;  // clocks.
            double  t0,tf;  // ticks.

            int x = 1;

            c0 = clock();
            t0 = chronos_rdtsc();

            for (int i=0;i < 20000000; i++)
                x = (i & x) ? 0 : 1;

            tf = chronos_rdtsc();
            cf = clock();

            double rate = (tf-t0)/((cf-c0)*(1000000.0 / double(CLOCKS_PER_SEC)));
            __chronos_scale = 1.0 / (rate * 1000.0);

            __chronos_use_pentium = true;
        }
    }

    instance++;
}


//
// chronos_term
//
// Terminate Chronos
//
void chronos_term ()
{
    mythos_atexit_remove (ichronos_cleanup);

    if (instance > 0)
    {
        if (--instance == 0)
        {
            ichronos_cleanup();
        }
    }
}


//
// ichronos_cleanup
//
static void ichronos_cleanup ()
{
    chronos_clear_callbacks ();

    if (system_freq != 0)
    {
        timeEndPeriod (system_freq);
    }

    system_freq = 0;
}


//
// chronos_interval_mhz
//
int chronos_interval_mhz (dword mlhz)
{
    assert (mlhz <= dword (1193181000));

    // calculate the new system frequency
    return chronos_interval (word ((mlhz / 18207) >> 1));
}


//
// chronos_interval
//
//  Program the timer to the given number of multiples of its base
//  frequency (18.207 hz), based on 0  (0 == 18hz, 1 = 36hz, etc)
//  Return whether or not you succeeded
//
int chronos_interval (word multiplier)
{
    // Close out any existing periods
    if (system_freq != 0)
        timeEndPeriod (system_freq);

    // calculate the new system frequency
    system_freq = mult2ms (multiplier + 1);

    // Inform the hardware
    if (timeBeginPeriod (system_freq) == TIMERR_NOERROR)
        return 1;

    system_freq = 0;
    return 0;
}


//
// chronos_add_callback
//
// Add the function/parameter to the callback list for execution every
// 'freq' timer ticks.
//
void chronos_add_callback (ChronosCallback func, dword freq, void *parm)
{
    assert (func != 0);
    assert (freq > 0);

    for (int i = 0; i < MAX_CALLBACKS; i++)
    {

        if (timer_handles[i].hTimer == 0)
        {
            timer_handles[i].fn     = func;
            timer_handles[i].parm   = parm;
            timer_handles[i].hTimer = timeSetEvent (freq * system_freq,
                                                    system_freq,
                                                    ichronos_timer_proc,
                                                    (DWORD)&timer_handles[i],
                                                    TIME_PERIODIC);
            return;
        }
    }

    // Error condition if we get here
    assert (!"Unable to add the callback!");
}


//
// chronos_clear_callbacks
//
// Clear all callback functions.
//
void chronos_clear_callbacks ()
{
    for (int i = 0; i < MAX_CALLBACKS; i++)
    {
        if (timer_handles[i].hTimer)
        {
            timeKillEvent (timer_handles[i].hTimer);
            timer_handles[i].hTimer = 0;
        }
    }
}


//
// chronos_time_now_precise
//
// Return the current time in processor clock fractions
//
double chronos_time_now_precise ()
{
    if (__chronos_use_pentium)
        return chronos_rdtsc() * __chronos_scale;
    else
        return double (timeGetTime());
}


//
// chronos_time_now
//
// Return the current time in milliseconds.
//
clock_t chronos_time_now ()
{
    if (__chronos_use_pentium)
        return clock_t (chronos_rdtsc() * __chronos_scale);
    else
        return timeGetTime();
}

// End of module - chronos.cpp 
