/*************************************************************************
** interpcom-2.3    (command interpreter)                                **
** _GMP.c : expression evaluator based on gmp library                    **
**                                                                       **
** Copyright (C) 2001  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"
#include <gmp.h>
extern mpq_t        GMP_Zero;
extern mpq_t        GMP_one;
extern mpz_t        GMP_Zero_int;
extern mpz_t        GMP_one_int;

int
_GMP_iszero(void *z)
{
    mpq_t          *t;

    t = (mpq_t *)z;
    if (mpq_cmp(*t, GMP_Zero) == 0)
        return 1;
    else
        return 0;
}

void
_GMP_Zero(void *z)
{
    mpq_t          *t;

    t = (mpq_t *)z;
    mpq_set(*t, GMP_Zero);
}

void
_GMP_clear(void *z)
{
    mpq_t          *t;

    t = (mpq_t *) z;
    mpq_clear(*t);
    free(z);
}

void
_GMP_neg(void *z)
{
    mpq_t          *t;

    t = (mpq_t *)z;
    mpq_neg(*t, *t);
}

void
_GMP_add(void *res, void *a, void *b)
{
    mpq_t          *t,
                   *u,
                   *v;

    t = (mpq_t *)res;
    u = (mpq_t *)a;
    v = (mpq_t *)b;
    mpq_add(*t, *u, *v);
}

void
_GMP_sub(void *res, void *a, void *b)
{
    mpq_t          *t,
                   *u,
                   *v;

    t = (mpq_t *)res;
    u = (mpq_t *)a;
    v = (mpq_t *)b;
    mpq_sub(*t, *u, *v);
}

void
_GMP_mul(void *res, void *a, void *b)
{
    mpq_t          *t,
                   *u,
                   *v;

    t = (mpq_t *)res;
    u = (mpq_t *)a;
    v = (mpq_t *)b;
    mpq_mul(*t, *u, *v);
}

void
_GMP_pow(void *res, void *a, void *b)
{
    mpq_t          *t;

    t = (mpq_t *)res;
    mpq_set(*t, GMP_Zero);
}

void
_GMP_div(void *res, void *a, void *b)
{
    mpq_t          *t,
                   *u,
                   *v;

    t = (mpq_t *)res;
    u = (mpq_t *)a;
    v = (mpq_t *)b;
    mpq_div(*t, *u, *v);
}

void
_GMP_set(void *res, char *h)
{
    mpq_t          *t;
    mpz_t           den,
                    num;
    char           *hden,
                   *hnum;
    int             i,
                    j,
                    k,
                    len;

    len = strlen(h) + 1;
    hnum = (char *) malloc((size_t) len * sizeof(char));
    hden = (char *) malloc((size_t) len * sizeof(char));
    memset(hnum, 0, len);
    memset(hden, 0, len);
    t = (mpq_t *)res;
    mpz_init(den);
    mpz_init(num);
    j = -1;

    for (i = 0; i < len; i++) {
        if (h[i] == '/')
            break;
        hnum[i] = h[i];
        j = i;
    }

    if (j == -1)
        mpq_set(*t, GMP_Zero);
    else {
        k = 0;
        for (i = j + 2; i < len; j++)
            hden[k++] = h[i];
        mpz_set_str(num, hnum, 10);
        if (k > 0)
            mpz_set_str(den, hden, 10);
        else
            mpz_set(den, GMP_one_int);
        if (mpz_cmp(den, GMP_Zero_int) == 0) {
            mpq_set(*t, GMP_Zero);
        }
        else {
            mpq_set_num(*t, num);
            mpq_set_den(*t, den);
            mpq_canonicalize(*t);
        }
    }

    mpz_clear(den);
    mpz_clear(num);
    free(hden);
    free(hnum);
}

void
_GMP_Init(void **s)
{
    mpq_t          *t;

    t = (mpq_t *) malloc((size_t) sizeof(mpq_t));
    mpq_init(t[0]);
    s[0] = (void *) t;
}

char *
_GMP_print(void *c)
{
    mpq_t          *t;
    mpz_t           num,
                    den;
    char           *h,
                   *hnum,
                   *hden;
    int             len;

    t = (mpq_t *)c;
    hnum = NULL;
    hden = NULL;
    mpz_init(num);
    mpz_init(den);
    mpq_get_num(num, *t);
    mpq_get_den(den, *t);
    hnum = mpz_get_str(hnum, 10, num);
    hden = mpz_get_str(hden, 10, den);
    len = strlen(hden) + strlen(hnum) + 2;
    h = (char *) malloc((size_t)len * sizeof(char));
    memset(h, 0, len);
    strcat(h, hnum);
    strcat(h, "/");
    strcat(h, hden);
    mpz_clear(num);
    mpz_clear(den);
    free(hnum);
    free(hden);
    return h;
}

void *
_GMP_copy(void *c)
{
    mpq_t          *t,
                   *u;

    t = (mpq_t *)c;
    u = (mpq_t *) malloc(sizeof(mpq_t));
    mpq_init(u[0]);
    mpq_set(*u, *t);
    return (void *) u;
}

int
_GMP_sign(void *c)
{
    mpq_t          *t;

    t = (mpq_t *)c;
    return mpq_sgn(*t);
}

EXPREVAL_GEN        _GMP_exprev = {
    _GMP_clear,
    _GMP_Zero,
    _GMP_neg,
    _GMP_iszero,
    _GMP_add,
    _GMP_sub,
    _GMP_mul,
    _GMP_pow,
    _GMP_div,
    _GMP_set,
    _GMP_Init,
    _GMP_print,
    _GMP_copy,
    _GMP_sign,
    0,
    NULL,
    NULL,
    1,
    "expression evaluator based on the gmp library, using rational numbers"
};





void
_GMP_fabs(void **res, void **x)
{
    mpq_t          *t,
                   *u;
    mpz_t           num,
                    den;

    t = (mpq_t *) x[0];
    u = (mpq_t *) res[0];
    mpz_init(num);
    mpz_init(den);
    mpq_get_num(num, *t);
    mpq_get_den(den, *t);
    mpz_abs(num, num);
    mpz_abs(den, den);
    mpq_set_num(*u, num);
    mpq_set_den(*u, den);
    mpz_clear(num);
    mpz_clear(den);
}




FUNCTIONGen         Funcs_GMP[] =
{
   /* name, function to call */
   { "abs",     1,    _GMP_fabs },           /* 0    */
   { 0 }                                     /* 1    */
};

int                 _GMP_NBFONC0=1;

