#include <stdio.h>
#include "datety.hpp"

extern "C" {

const YearTy century = 1900;

const DayTy daysInMonth[] =
	{  0, 31, 28, 31,  30,  31,  30,  31,  31,  30,  31,  30,  31 };
	  /* Jan Feb Mar  Apr  May  Jun  Jul  Aug  Sep  Oct  Nov  Dec */

static const DayTy monthTotal[] =
	{  0,  0, 31, 59,  90, 120, 151, 181, 212, 243, 273, 304, 334, 365 };

static const YearTy y2k_border = 80;

YearTy get_full_year(YearTy year) {
	if(year < 100) {
		if(year < y2k_border) year += 100;
		year += century;
	}
	return year;
}

int LeapYear(YearTy year) {
	year = ::get_full_year(year);
	return ((year % 4) == 0) && (((year % 100) != 0) ||
		((year % 400) == 0)) ? 1 : 0;
}

JulTy DaysToYear(YearTy year) {
	year = ::get_full_year(year);
	if(!year) return (JulTy)0l;
	JulTy amount = (JulTy)(year - 1);
	return (amount*365l) - (amount/100l) + (amount/4l) + (amount/400l);
}

DayTy DaysToDate(YearTy year, MonthTy month, DayTy day) {
	year = ::get_full_year(year);
	if((month > 12) || !year || !day || !month) return (DayTy)0;
	int isLeap = LeapYear(year);
	DayTy monthDays = ((month == 2) && isLeap) ?
		(daysInMonth[month] + 1) : (daysInMonth[month]);
	if(day > monthDays) return (DayTy)0;
	else return (isLeap && (month > 2)) ?
		(monthTotal[month] + day + 1) : (monthTotal[month] + day);
}

void Encode(YearTy year, MonthTy month, DayTy day, JulTy& JulDate) {
	JulDate = (JulTy)DaysToDate(year, month, day);
	if(JulDate) JulDate += DaysToYear(year);
}

void Decode(YearTy& year, MonthTy& month, DayTy& day, const JulTy JulDate) {
	if(JulDate > 0) {
		year = (YearTy)((double)JulDate / 365.2425) + 1;
		int amount = (int)(JulDate - DaysToYear(year));
		if(amount <= 0) amount = (int)(JulDate - DaysToYear(--year));
		int total = (LeapYear(year) ? 366 : 365);
		if(amount > total) {
			year++;
			amount -= total;
		}
		day = LeapYear(year);
		if(amount <= 59) day = 0;
		for(int i = 2; i <= 13; i++)
			if(amount <= static_cast<int>(monthTotal[i] + day)) {
				month = --i;
				if(month <= 2) day = 0;
				day = amount - monthTotal[month] - day;
				return;
			}
	}
	else year = month = day = 0;
}

int DayOfWeek(YearTy year, MonthTy month, DayTy day) {
	int dw, century, yr;
	year = ::get_full_year(year);
	if(month < 3) {
		month += 10;
		year--;
	}
	else month -= 2;
	century = (year / 100);
	yr = year % 100;
	dw = (((26 * month - 2) / 10) + day + yr + (yr / 4) +
		(century / 4) - (2 * century)) % 7;
	return ((dw < 0) ? (dw + 7) : dw) + 1;
}

// Our internal formats (by Andrew Tuman):

char* d2xbase(YearTy year, MonthTy month, DayTy day, char* buffer) {
	::sprintf(buffer, "%04d%02d%02d", year, month, day);
	return buffer;
}

char* d2h(const char* value, char* buffer) {
	buffer[0] = (char)(((value[0] - '0') << 4) | (value[1] - '0'));
	buffer[1] = (char)(((value[2] - '0') << 4) | (value[3] - '0'));
	buffer[2] = (char)(((value[4] - '0') << 4) | (value[5] - '0'));
	buffer[3] = (char)(((value[6] - '0') << 4) | (value[7] - '0'));
	buffer[4] = 0;
	return buffer;
}

char* h2d(const char* value, char* buffer) {
	buffer[0] = (char)(((unsigned char)(value[0]) >> 4) + '0');
	buffer[1] = (char)(((unsigned char)(value[0]) & 0x0f) + '0');
	buffer[2] = (char)(((unsigned char)(value[1]) >> 4) + '0');
	buffer[3] = (char)(((unsigned char)(value[1]) & 0x0f) + '0');
	buffer[4] = (char)(((unsigned char)(value[2]) >> 4) + '0');
	buffer[5] = (char)(((unsigned char)(value[2]) & 0x0f) + '0');
	buffer[6] = (char)(((unsigned char)(value[3]) >> 4) + '0');
	buffer[7] = (char)(((unsigned char)(value[3]) & 0x0f) + '0');
	buffer[8] = 0;
	return buffer;
}

}

