//	2up.cpp
//	Copyright 1998 Michael A. Quinlan
//	mikeq@primenet.com
//
//	Print files in 2-up format, two pages per sheet of paper. Requires a
//	Laser Printer that supports HPCL.

#define StartPage StartPagexxx
#define EndPage EndPagexxx
#include <windows.h>
#undef StartPage
#undef EndPage

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <io.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "GetLastE.h"

#define ESC			"\x1B"				// Escape character

#define PTRRESET	ESC "E"				// Printer reset sequence

#define PTRSETUP	PTRRESET	\
					ESC "&l1O"	\
					ESC "(10U"	\
					ESC "(s0p16.67h8.5v0s0b0T"	\
					ESC "&l5C"

#define BARSETUP	ESC "&f1yxS"	\
					ESC "*px125Y"	\
					ESC "*c3300a125b%dg2P"	\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC	"*p+250Y"	\
					ESC "*c2P"		\
					ESC "&f1s1x4X"

///////////////////////////////////////////////////////////////////////
//	Options
static int nStartPage = 1;
static bool bDuplex = false;
static int nBarDensity = 0;
static int nTabSize = 4;
static char* pszPrintFile = NULL;

///////////////////////////////////////////////////////////////////////
//	Printer file
static HANDLE	hPrn;

///////////////////////////////////////////////////////////////////////
//	Page size constants
const int nMaxLineLen = 86;
const int nLinesPerPage = 68;
const int nTitleLineLen = nMaxLineLen * 2 + 3;

///////////////////////////////////////////////////////////////////////
//	Global variables
static int nFilesPrinted;
static char	achLhsPage[nLinesPerPage][nMaxLineLen];
static char	achRhsPage[nLinesPerPage][nMaxLineLen];

///////////////////////////////////////////////////////////////////////
//	Open the default printer.
static HANDLE OpenPrn(const char* pszDocName = NULL)
{
	// Get the name of the default printer from the registry
	char szPrinterInfo[256];
	GetProfileString("windows", "device", "", szPrinterInfo, sizeof(szPrinterInfo));
	strtok(szPrinterInfo, ",");

	// Open the default printer
	HANDLE	hPrn;
	if (!OpenPrinter(szPrinterInfo, &hPrn, NULL))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "OpenPrinter(\"%s\") failed: %s.\n", szPrinterInfo, szBuf);
		return INVALID_HANDLE_VALUE;
	}
	
	// Build the document info structure for the spooler.
	DOC_INFO_1 di = { "2UP Printout", NULL, "RAW" };

	if (pszDocName != NULL)
		di.pDocName = (char*)pszDocName;

	if (pszPrintFile != NULL)
		di.pOutputFile = pszPrintFile;

	// Start the print job with the spooler
	DWORD dwJob = StartDocPrinter(hPrn, 1, (unsigned char*)&di);
	if (dwJob == 0)
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "StartDocPrinter() failed: %s.\n", szBuf);
		ClosePrinter(hPrn);
		return INVALID_HANDLE_VALUE;
	}

	// Return the handle for the opened printer.
	return hPrn;
}

///////////////////////////////////////////////////////////////////////
//	Write a block of data to the printer.
static void WritePrn(HANDLE hPrn, const void* pvBuffer, int nLen)
{
	DWORD dwWritten;
	if (!WritePrinter(hPrn, (void*)pvBuffer, nLen, &dwWritten))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "WritePrinter() failed: %s.\n", szBuf);
	}
}

///////////////////////////////////////////////////////////////////////
//	Write a nul-terminated string to the printer
static void WritePrn(HANDLE hPrn, const char* pvBuffer)
{
	WritePrn(hPrn, pvBuffer, strlen(pvBuffer));
}

///////////////////////////////////////////////////////////////////////
//	Start a new page on the printer.
static void StartPage(HANDLE hPrn)
{
	if (!StartPagePrinter(hPrn))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "StartPagePrinter() failed: %s.\n", szBuf);
	}
}

///////////////////////////////////////////////////////////////////////
//	End a page on the printer.
static void EndPage(HANDLE hPrn)
{
	if (!EndPagePrinter(hPrn))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "EndPagePrinter() failed: %s.\n", szBuf);
	}
}

///////////////////////////////////////////////////////////////////////
//	Close the printer
static void ClosePrn(HANDLE hPrn)
{
	if (!EndDocPrinter(hPrn))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "EndDocPrinter() failed: %s.\n", szBuf);
	}

	if (!ClosePrinter(hPrn))
	{
		char szBuf[256];
		GetLastError(szBuf, sizeof(szBuf));
		fprintf(stderr, "ClosePrinter() failed: %s.\n", szBuf);
	}
}

///////////////////////////////////////////////////////////////////////
//	Read one line of input. Reads a maximum of nMaxLineLen bytes.
//	Assumes that the result buffer has been filled with blanks.
//	Returns the line length or EOF (-1) if eof or -2 if form feed found.
//	Deletes trailing blanks from lines, so that long blank lines don't
//	wrap.
static int ReadLine(FILE* pfIn, char* pszLine)
{
	int nLineLen = 0;		// Line length so far
	int nBlanks = 0;		// Number of blanks accumulated

	// Loop reading characters
	for (;;)
	{
		// Read the next input character
		int nChar = getc(pfIn);

		switch (nChar)
		{
		// EOF: return the line read so far. If we haven't read anything
		// so far, return EOF.
		case EOF:
			{
				if (nLineLen == 0)
					return EOF;
				else
					return nLineLen;
			}
			break;

		// New Line: Return the line read so far.
		case '\n':
			return nLineLen;

		// Carriage Return: Ignore this character.
		case '\r':
			break;

		// Tab: Add spaces up to the next tab stop.
		case '\t':
			{
				const int nLen = nLineLen + nBlanks;
				nBlanks += nTabSize - (nLen % nTabSize);
			}
			break;

		// Form Feed: If at the start of a line, swallow the FF and return -2
		// Otherwise, return the line read so far and push the FF so we will
		// see it as the start of the next line.
		case '\f':
			{
				if (nLineLen == 0)
					return -2;

				ungetc(nChar, pfIn);
				return nLineLen;
			}
			break;

		// All other characters
		default:
			{
				// Convert "other" control characters to spaces.
				if (iscntrl((unsigned char)nChar) || isspace((unsigned char)nChar))
					++nBlanks;

				else
				{
					// If we have filled up the line, return it without any
					// trailing blanks.
					const int nLen = nLineLen + nBlanks;
					if (nLen >= nMaxLineLen)
					{
						ungetc(nChar, pfIn);
						return nLineLen;
					}

					// Add the trailing blanks to the line.
					nLineLen += nBlanks;
					nBlanks = 0;

					// Add the character after the trailing blanks.
					pszLine[nLineLen++] = (char)nChar;
				}
			}
			break;
		}
	}
}

///////////////////////////////////////////////////////////////////////
//	Read one page of lines. Return the number of lines on the page
//	(i.e. the line number of the last non-blank line on the page), or
//	-1 if EOF.
static int ReadPage(FILE* pfIn, char achPage[nLinesPerPage][nMaxLineLen])
{
	// Set the page buffer to blanks.
	memset(achPage, ' ', nLinesPerPage * nMaxLineLen);

	// Skip over any blank lines and form feeds at the top of the page.
	int nLineLen;
	do {
		nLineLen = ReadLine(pfIn, achPage[0]);
		if (nLineLen == EOF)
			return -1;
	} while (nLineLen <= 0);

	int nLines = 1;		// Line number of the last non-blank line on the page

	// Read the remaining lines of the page. Stop when we hit EOF,
	// a Form Feed, or the bottom line of the page.
	for (int nLine = 1; nLine < nLinesPerPage; nLine++)
	{
		// Read one line
		nLineLen = ReadLine(pfIn, achPage[nLine]);

		// If EOF or Form Feed, end the page.
		if (nLineLen < 0)
			break;

		// Record the location of the last non-blank line.
		if (nLineLen > 0)
			nLines = nLine;
	}

	return nLines;
}

///////////////////////////////////////////////////////////////////////
//	Build the title line
static void BuildTitle(char* pszTitleLine, const char* pszFileName, const char* pszFileDate, const char* pszPageNo)
{
	// Inititalize the title line
	memset(pszTitleLine, ' ', nTitleLineLen);
	pszTitleLine[nTitleLineLen] = '\0';

	// Get the lengths of the three pieces of the title
	int nFileNameLen = strlen(pszFileName);
	if (nFileNameLen >= nTitleLineLen)
		nFileNameLen = nTitleLineLen - 1;

	int nFileDateLen = strlen(pszFileDate);
	if (nFileDateLen >= nTitleLineLen)
		nFileDateLen = nTitleLineLen - 1;

	int nPageNoLen = strlen(pszPageNo);
	if (nPageNoLen >= nTitleLineLen)
		nPageNoLen = nTitleLineLen - 1;

	// Copy the date to the title (left)
	memcpy(pszTitleLine, pszFileDate, nFileDateLen);

	// Copy the page number to the title (right)
	const int nPageNoOffset = nTitleLineLen - nPageNoLen;
	memcpy(pszTitleLine + nPageNoOffset, pszPageNo, nPageNoLen);

	// Center the file name on the page.
	int nFileNameOffset = (nTitleLineLen - nFileNameLen) / 2;
	memcpy(pszTitleLine + nFileNameOffset, pszFileName, nFileNameLen);

	// Make sure there is a blank before and after the title.
	if (nFileNameOffset > 0)
		pszTitleLine[nFileNameOffset - 1] = ' ';
	if ((nFileNameOffset + nFileNameLen) < (nTitleLineLen - 1))
		pszTitleLine[nFileNameOffset + nFileNameLen] = ' ';
}

///////////////////////////////////////////////////////////////////////
//	Print one sheet, containing two pages of text.
static void PrintSheet(const char* pszFileName, const char* pszFileDate, int nPageNo, int nPages)
{
	StartPage(hPrn);

	char szPageNo[32];
	sprintf(szPageNo, "Page %d of %d", nPageNo, nPages);

	char szTitle[nTitleLineLen + 1];
	BuildTitle(szTitle, pszFileName, pszFileDate, szPageNo);

	WritePrn(hPrn, "\r\n");
	WritePrn(hPrn, szTitle);
	WritePrn(hPrn, "\r\n\r\n\r\n");

	for (int nLine = 0; nLine < nLinesPerPage; nLine++)
	{
		WritePrn(hPrn, achLhsPage[nLine], nMaxLineLen);
		WritePrn(hPrn, " | ");

		int nLineLen = nMaxLineLen;
		while (nLineLen > 0 && achRhsPage[nLine][nLineLen - 1] == ' ')
			--nLineLen;
		if (nLineLen > 0)
			WritePrn(hPrn, achRhsPage[nLine], nLineLen);

		if (nLine != (nLinesPerPage - 1))
			WritePrn(hPrn, "\r\n");
	}

	WritePrn(hPrn, "\f");
	EndPage(hPrn);
}

///////////////////////////////////////////////////////////////////////
//	Print the sheets of a file.
static void PrintFile(FILE* pfIn, const char* pszFileName, const char* pszFileDate, fpos_t fpStart, int nPages,
					  int nModulo = 1, int nWhich = 0)
{
	// Skip to the starting page number
	clearerr(pfIn);
	fsetpos(pfIn, &fpStart);
	int nPageNo = nStartPage;

	while (!feof(pfIn) && !ferror(pfIn))
	{
		// Read the two pages
		if (ReadPage(pfIn, achLhsPage) <= 0)
			break;
		ReadPage(pfIn, achRhsPage);

		// Print the two pages on a single sheet
		if ((nPageNo % nModulo) == nWhich)
			PrintSheet(pszFileName, pszFileDate, nPageNo, nPages);

		// Incr the page number and get the next two pages
		++nPageNo;
	}
}

///////////////////////////////////////////////////////////////////////
//	Print a file in two-up format, monoplex.
static void TwoUpMonoplex(FILE* pfIn, const char* pszFileName, const char* pszFileDate, fpos_t fpStart, int nPages)
{
	// Read and print the pages.
	fprintf(stderr, "Printing...");
	PrintFile(pfIn, pszFileName, pszFileDate, fpStart, nPages, 1, 0);
}

///////////////////////////////////////////////////////////////////////
//	Print a file in two-up format, duplex.
static void TwoUpDuplex(FILE* pfIn, const char* pszFileName, const char* pszFileDate, fpos_t fpStart, int nPages)
{
	fprintf(stderr, "Printing even-numbered pages...");
	PrintFile(pfIn, pszFileName, pszFileDate, fpStart, nPages, 2, 0);

	fprintf(stderr, "Done.\n"
					"Reload pages in the bottom tray , printed side up, with right-side page\n"
					"away from the printer.\n"
					"Press ENTER when the printer is ready...");
	if (pszPrintFile == NULL)
		getchar();
	else
		fprintf(stderr, "\n");

	fprintf(stderr, "Printing odd-numbered pages...");
	PrintFile(pfIn, pszFileName, pszFileDate, fpStart, nPages, 2, 1);
}

///////////////////////////////////////////////////////////////////////
//	Print a file in two-up format.
static void TwoUp(const char* pszFileName)
{

	// Open the file for input.
	FILE* pfIn = fopen(pszFileName, "rb");
	if (pfIn == NULL)
	{
		perror(pszFileName);
		return;
	}

	// Get the file date/time
	char szFileDate[nTitleLineLen] = "";
	struct _stat s;
	if (_stat(pszFileName, &s) == 0)
	{
		struct tm* ptm = localtime(&s.st_mtime);
		strftime(szFileDate, sizeof(szFileDate), "%m/%d/%Y %I:%M:%S%p", ptm);
	}

	fprintf(stderr, "%s...", pszFileName);

	// Count the number of pages by reading a page at a time.
	fprintf(stderr, "Counting...");
	fpos_t fpStart = (fpos_t)-1;
	int nPages = 0;
	while (!feof(pfIn) && !ferror(pfIn))
	{
		// Incr the page counter
		++nPages;

		// Remember the starting print position.
		if (fpStart == (fpos_t)-1 && nPages >= nStartPage)
			fgetpos(pfIn, &fpStart);

		// Read the two pages
		if (ReadPage(pfIn, achLhsPage) <= 0)
		{
			--nPages;
			break;
		}
		ReadPage(pfIn, achRhsPage);
	}

	if (nPages <= 0)
	{
		fprintf(stderr, "Empty document skipped\n");
		return;
	}

	if (fpStart == (fpos_t)-1)
		fgetpos(pfIn, &fpStart);

	// Read and print the pages.
	if (nPages > 1 && bDuplex)
		TwoUpDuplex(pfIn, pszFileName, szFileDate, fpStart, nPages);
	else
		TwoUpMonoplex(pfIn, pszFileName, szFileDate, fpStart, nPages);

	fprintf(stderr, "Done\n");
	++nFilesPrinted;
}

///////////////////////////////////////////////////////////////////////
//	Process all files that match a path designation. Expand wild
//	cards.
static void ProcessPath(const char* pszPath)
{
	_finddata_t fd;
	long hFind = _findfirst(pszPath, &fd);
	if (hFind == -1)
	{
		fprintf(stderr, "%s: File not found.\n", pszPath);
		return;
	}

	char	szOrigDrive[_MAX_DRIVE];
	char	szOrigDir[_MAX_DIR];
	char	szOrigFName[_MAX_FNAME];
	char	szOrigExt[_MAX_EXT];
	_splitpath(pszPath, szOrigDrive, szOrigDir, szOrigFName, szOrigExt);

	const int nOrigDriveLen = strlen(szOrigDrive);
	const int nOrigDirLen = strlen(szOrigDir);
	const int nOrigFNameLen = strlen(szOrigFName);
	const int nOrigExtLen = strlen(szOrigExt);

	do {
		if ((fd.attrib & (_A_SUBDIR | _A_HIDDEN | _A_SYSTEM)) == 0)
		{
			const int nNameLen = strlen(fd.name);

			if ((nOrigDriveLen + nOrigDirLen + nNameLen) >= _MAX_PATH)
			{
				fprintf(stderr, "%s%s%s: Path name is too long.\n", szOrigDrive, szOrigDir, fd.name);
				continue;
			}

			char	szFName[_MAX_FNAME];
			char	szExt[_MAX_EXT];
			_splitpath(fd.name, NULL, NULL, szFName, szExt);

			const int nFNameLen = strlen(szFName);
			const int nExtLen = strlen(szExt);

			if (nOrigFNameLen == 0 && nFNameLen != 0)
				continue;

			if (nOrigExtLen == 1 && nExtLen > 1)
				continue;

			char	szPath[_MAX_PATH];
			memcpy(szPath, szOrigDrive, nOrigDriveLen);
			memcpy(szPath + nOrigDriveLen, szOrigDir, nOrigDirLen);
			memcpy(szPath + nOrigDriveLen + nOrigDirLen, fd.name, nNameLen + 1);

			char	szFullPath[_MAX_PATH];
			if (_fullpath(szFullPath, szPath, sizeof(szFullPath)) == NULL)
			{
				fprintf(stderr, "%s: Unable to resolve path.\n", szPath);
				continue;
			}

			TwoUp(szFullPath);
		}

	} while (_findnext(hFind, &fd) != -1);

	_findclose(hFind);
}

///////////////////////////////////////////////////////////////////////
//	Display program usage information.
static void Usage(const char* pszMsg = NULL)
{
	if (pszMsg != NULL)
		fprintf(stderr, "%s.\n", pszMsg);

	fprintf(stderr, "Usage: 2up path [/start:n] [/tab:n] [/[no]duplex] [/bars[:n]] [/file:filename]\n"
					"       path     = File name(s) to print. Wildcards are allowed.\n"
					"       start    = Starting page number.\n"
					"       tab      = Tab size.\n"
					"       duplex   = Print even pages, pause, print odd pages.\n"
					"       bars     = Print bars across the page. n is the density (10%).\n"
					"       file     = Print to a file.\n");
	exit(3);
}

///////////////////////////////////////////////////////////////////////
//	Determine if a parameter is an option or a file name
static bool isOption(const char* psz)
{
	return *psz == '/' || *psz == '-';
}

///////////////////////////////////////////////////////////////////////
//	Process an option.
static void GetOption(const char* pszOption)
{
	if (_strnicmp(pszOption + 1, "start", 5) == 0 &&
		(pszOption[6] == ':' || pszOption[6] == '='))
	{
		int n = atoi(pszOption + 7);
		if (n <= 0)
			fprintf(stderr, "Invalid starting page ignored: %s.\n", pszOption);
		else
			nStartPage = n;
	}

	else if (_strnicmp(pszOption + 1, "tab", 3) == 0 &&
		(pszOption[4] == ':' || pszOption[4] == '='))
	{
		int n = atoi(pszOption + 5);
		if (n <= 0)
			fprintf(stderr, "Invalid tab size ignored: %s.\n", pszOption);
		else
			nTabSize = n;
	}

	else if (_stricmp(pszOption + 1, "duplex") == 0)
		bDuplex = true;

	else if (_stricmp(pszOption + 1, "noduplex") == 0)
		bDuplex = false;

	else if (_stricmp(pszOption + 1, "bars") == 0)
		nBarDensity = 10;

	else if (_strnicmp(pszOption + 1, "bars", 4) == 0 &&
		(pszOption[5] == ':' || pszOption[5] == '='))
	{
		int n = atoi(pszOption + 6);
		if (n < 0 || n > 100)
			fprintf(stderr, "Invalid bar density ignored: %s.\n", pszOption);
		else
			nBarDensity = n;
	}

	else if (_stricmp(pszOption + 1, "nobars") == 0)
		nBarDensity = 0;

	else if (_strnicmp(pszOption + 1, "file", 4) == 0 &&
		(pszOption[5] == ':' || pszOption[5] == '='))
	{
		pszPrintFile = _fullpath(NULL, pszOption + 6, 0);
	}

	else
		fprintf(stderr, "Invalid option ignored: %s.\n", pszOption);
}

///////////////////////////////////////////////////////////////////////
//	Get all options.
static void GetOptions(int argc, char** argv)
{
	char* pszOptions = _strdup(getenv("2UP"));
	if (pszOptions != NULL)
	{
		fprintf(stderr, "Using options from environment: 2UP=%s.\n", pszOptions);

		char *pszStart = pszOptions + strspn(pszOptions, " ,");
		while (*pszStart != '\0')
		{
			char* pszNext;
			if (*pszStart == '"')
			{
				++pszStart;
				pszNext = pszStart;
				while (pszNext[0] != '"' || pszNext[1] == '"')
				{
					if (pszNext[0] == '\0')
					{
						fprintf(stderr, "Unclosed string; options ignored: %s.\n", pszStart - 1);
						goto out;
					}
					else if (pszNext[0] == '\\' && pszNext[1] != '\0')
					{
						int nLen = strlen(pszNext);
						memmove(pszNext, pszNext + 1, nLen);
					}
					else if (pszNext[0] == '"' && pszNext[1] == '"')
					{
						int nLen = strlen(pszNext);
						memmove(pszNext, pszNext + 1, nLen);
					}

					++pszNext;

				}
				*pszNext++ = '\0';
			}
			else
			{
				pszNext = pszStart + strcspn(pszStart, " ,");
				if (*pszNext != '\0')
					*pszNext++ = '\0';
			}

			if (!isOption(pszStart))
				fprintf(stderr, "Invalid option ignored: %s.\n", pszStart);
			else
				GetOption(pszStart);

			pszStart = pszNext;
		}
out:
		free(pszOptions);
	}

	for (int nArg = 1; nArg < argc; nArg++)
	{
		if (isOption(argv[nArg]))
			GetOption(argv[nArg]);
	}
}

///////////////////////////////////////////////////////////////////////
//	Main function. Process the parms.
int main(int argc, char** argv)
{
	int nArg;

	fprintf(stderr, "2up -- Print text files -- Compiled %s %s\n", __DATE__, __TIME__);
	fprintf(stderr, "Copyright 1998 Michael A. Quinlan\n");

	if (argc <= 1)
		Usage();

	GetOptions(argc, argv);

	if (bDuplex && ((nStartPage % 2) == 0))
		--nStartPage;

	fprintf(stderr, "Options: /START:%d /TAB:%d /%sDUPLEX /BARS:%d%s%s\n",
		nStartPage,
		nTabSize,
		bDuplex ? "" : "NO",
		nBarDensity,
		pszPrintFile == NULL ? "" : " /FILE:",
		pszPrintFile == NULL ? "" : pszPrintFile);

	int nDocNameLen = 0;
	for (nArg = 0; nArg < argc; nArg++)
		nDocNameLen += strlen(argv[nArg]) + 1;

	char* pszDocName = (char*)malloc(nDocNameLen);
	pszDocName[0] = '\0';
	for (nArg = 0; nArg < argc; nArg++)
	{
		strcat(pszDocName, argv[nArg]);
		if (nArg != (argc - 1))
			strcat(pszDocName, " ");
	}

	hPrn = OpenPrn(pszDocName);
	if (hPrn == INVALID_HANDLE_VALUE)
		Usage("Unable to access the printer");

	WritePrn(hPrn, PTRSETUP);
	if (nBarDensity > 0)
	{
		char* pszBarSetup = new char [strlen(BARSETUP) + 2];
		sprintf(pszBarSetup, BARSETUP, nBarDensity);
		WritePrn(hPrn, pszBarSetup);
		delete [] pszBarSetup;
	}

	for (nArg = 1; nArg < argc; nArg++)
	{
		if (!isOption(argv[nArg]))
			ProcessPath(argv[nArg]);
	}

	WritePrn(hPrn, PTRRESET);

	ClosePrn(hPrn);
	fprintf(stderr, "%d file%s printed.\n", nFilesPrinted, nFilesPrinted == 1 ? "" : "s");
	return 0;
}
