/* PhotoOrganizer - $RCSfile: MediaPlayerPanel.java,v $                                                         
 * Copyright (C) 2001 Dmitriy Rogatkin.  All rights reserved.                   
 * Redistribution and use in source and binary forms, with or without           
 * modification, are permitted provided that the following conditions           
 * are met:                                                                     
 * 1. Redistributions of source code must retain the above copyright            
 *    notice, this list of conditions and the following disclaimer.             
 * 2. Redistributions in binary form must reproduce the above copyright         
 *    notice, this list of conditions and the following disclaimer in the       
 *    documentation and/or other materials provided with the distribution.      
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND      
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE       
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE  
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR  
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL      
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR  
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER  
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT          
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY   
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF      
 *  SUCH DAMAGE.                                                                
 *                                                                              
 *  Visit http://drogatkin.openestate.net to get the latest infromation         
 *  about Rogatkin's products.                                                  
 *  $Id: MediaPlayerPanel.java,v 1.12 2001/08/22 07:32:04 rogatkin Exp $                       
 */                                                                             

package photoorganizer.renderer;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.MessageFormat;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.sound.sampled.AudioFormat;

import javax.sound.sampled.AudioFormat.Encoding;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import photoorganizer.*;
import photoorganizer.formats.*;

import javazoom.jl.decoder.*;
import javazoom.jl.player.*;

public class MediaPlayerPanel extends JPanel implements ActionListener, IrdControllable { 
		
	static final int MS_IN_FRAME = 26;
	protected static final MessageFormat INFO_FMT = new MessageFormat(Resources.FMT_PLAYING_INFO);
	
	// for remote options
	protected MediaPlayerPanel() throws IOException {
		this(true);
	}
	
	public MediaPlayerPanel(boolean listPlayer) throws IOException {
		setLayout(new BorderLayout());
		// normal panel
		JPanel normalPanel = new JPanel();
		normalPanel.setLayout(new BorderLayout());
		JScrollPane sp;
		normalPanel.add(sp = new JScrollPane(tp_info = new JEditorPane()), BorderLayout.NORTH);
		sp.setPreferredSize(new Dimension(200,60));
		tp_info.setContentType("text/html"); // Resources.CT_TEXT_HTML
		tp_info.setEditable(false);
		normalPanel.add(progress = new JSlider(), BorderLayout.CENTER);
		remoteCommandMap = new HashMap(4);
		add(normalPanel, BorderLayout.NORTH);
		// advanced panel
		advancedPanel = new JPanel();
		advancedPanel.setLayout(new FlowLayout());
		advancedPanel.add(new JButton("Mark"));
		// buttons panel
		JPanel buttons = new JPanel();
		buttons.setLayout(new FlowLayout());
		buttons.add(resume = new JButton(Resources.CMD_RESUME));
		resume.addActionListener(this);
		resume.setEnabled(false);
		remoteCommandMap.put(Resources.CMD_RESUME, resume);
		if (listPlayer) {
			buttons.add(skip = new JButton(Resources.CMD_SKIP));
			skip.addActionListener(this);
			skip.setEnabled(false);
			remoteCommandMap.put(Resources.CMD_SKIP, skip);
		} else
			remoteCommandMap.put(Resources.CMD_SKIP, null);
		buttons.add(stop = new JButton(Resources.CMD_STOP));
		stop.addActionListener(this);
		stop.setEnabled(false);
		remoteCommandMap.put(Resources.CMD_STOP, stop);
		buttons.add(close = new JButton(Resources.CMD_CLOSE));
		close.addActionListener(this);
		remoteCommandMap.put(Resources.CMD_CLOSE, close);
		buttons.add(advanced = new JButton(Resources.CMD_ADVANCED));
		add(buttons, BorderLayout.CENTER);
		advanced.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (getComponentCount() > 2) {
					remove(advancedPanel);
					advanced.setText(Resources.CMD_ADVANCED);
				} else {
					add(advancedPanel, BorderLayout.SOUTH);
					advanced.setText(Resources.LABEL_NORMAL);
				}
				pack();
			}
			});
		progress.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				JSlider source = (JSlider)e.getSource();
				if (source.getValueIsAdjusting()) {
					int fps = (int)source.getValue();
					stop();
					//synchronized(MediaPlayerPanel.this) {
					mark = fps * 1000 / MS_IN_FRAME ;
					//}
					//resume();
				}    
			}
		});
	}
	
	public MediaPlayerPanel(AbstractFormat media, Window window, int introFrames) throws IOException {
		this(false);
		this.window = window;
		if (window != null)
			window.addWindowListener(new java.awt.event.WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					//System.err.println("Closing win called");
					MediaPlayerPanel.this.window.dispose();
				}
				public void windowClosed(WindowEvent e) {
					//System.err.println("Closed win called");
					close();
				}
				}									 
			);
		replay(media, introFrames);
		autoClose = true;
	}
	
	public static MediaPlayerPanel getIrdControllable() {
		try {
			return new MediaPlayerPanel();
		} catch(IOException ioe) {
			return null;
		}
	}
	
	public synchronized void waitForCompletion() {
		if (!play)
			return;
		try {
			wait();
		} catch(InterruptedException ie) {
		}
	}
	
	public void replay(AbstractFormat media) throws IOException {
		replay(media, 0);
	}
	
	public synchronized WindowListener getWindowListener() {
		if (windowListener == null)
			windowListener = new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					close();
				}
				};
		return windowListener;
	}
	
	public void replay(final AbstractFormat media, final int intro) throws IOException {
		
		if (!closed)
			close();
		decoder = new Decoder();
		try {
			audio = FactoryRegistry.systemRegistry().createAudioDevice();
		} catch(Exception e) {
			throw new IOException ("createAudioDevice: "+e);
		}
		//audio.open(decoder);
		progress.setMaximum((int)media.getInfo().getLongAttribute(AbstractInfo.LENGTH));
		resume();
		bitstream = new Bitstream(media.getAsStream());
		synchronized(this) {
			closed = false;
		}
		updateTitle(Resources.TITLE_NOW_PLAYING+media.toString());
		tp_info.setText(INFO_FMT.format(media.getInfo().getFiveMajorAttributes()));
		
		new Thread(new Runnable() {
			public void run() {
				try {
					long ts = System.currentTimeMillis();
					int fr = 0;
					for(int f=0; (intro==0 || f<intro) && ! closed; f++) {
						if (!play)
							synchronized(audio) {
								resume.setEnabled(true);
								try {
									audio.wait();
								} catch(InterruptedException ie) {
								}
							}
						Header h = bitstream.readFrame();
						if (h == null)
							break;
						// sample buffer set when decoder constructed
						SampleBuffer output = (SampleBuffer)decoder.decodeFrame(h, bitstream);
						if (mark < f) {
						synchronized(MediaPlayerPanel.this) {
							if (!audio.isOpen() && decoder != null) {
								AudioFormat af;
								((JavaSoundAudioDevice)audio).open(af = new AudioFormat(decoder.getOutputFrequency(),
																						16,
																						decoder.getOutputChannels(),
																						true,
																						false));
								//System.err.println("Opened audio as "+af+'\n'+h);
								System.err.println("Playing "+media+" intro "+intro+" header "+h);
							}
						}						
						//synchronized(MediaPlayerPanel.this) {						
							audio.write(output.getBuffer(), 0, output.getBufferLength());
							progress.setValue(audio.getPosition()/1000);
						}
						//}
						bitstream.closeFrame();
						fr++;
					}
					if(!closed)
						audio.flush();
					System.err.println("Frames "+fr+" time "+((System.currentTimeMillis()-ts)/1000)+" audio time "+audio.getPosition()/1000);
					play = false;
					synchronized(MediaPlayerPanel.this) {
						MediaPlayerPanel.this.notify();
					}
					// update buttons
					progress.setValue(0);
					if (window != null && autoClose)
						window.dispose();
					else {
						if (skip != null)
							skip.setEnabled(false);
						stop.setEnabled(false);
						resume.setEnabled(false);
					}
				} catch(Throwable t) {					
					if (t instanceof JavaLayerException && ((JavaLayerException)t).getException() != null)
						((JavaLayerException)t).getException().printStackTrace();
					else
						t.printStackTrace();
					if (t instanceof ThreadDeath)
						throw (ThreadDeath)t;
				}
			}
		}, "MediaPlayer "+media).start();
	}
	
	public boolean isClosed() {
		return closed;
	}
	
	public void actionPerformed(ActionEvent e) {
		Object src = e.getSource();		
		if (src == resume) 
			resume();
		else if (src == stop)
			stop();
		else if (src == close) {
			if (window != null) {
				window.dispose();
				window = null;
			} else {
				close();
				updateTitle(Resources.TITLE_CLOSED);
				tp_info.setText("");
			}
			synchronized(this) {
				notify();
			}
		} else if (src == skip) {
			stop();
			synchronized(this) {
				notify();
			}
		}
	}
	
	public void stop() {
		if (audio != null && play) {
			play = false;
			stop.setEnabled(false);
			if (skip != null)
				skip.setEnabled(false);			
		}
	}
	
	public void resume() {
		if (audio != null && !play) {			
			synchronized(audio) {
				play = true;
				audio.notify();
			}
			stop.setEnabled(true);
			close.setEnabled(true);
			if (skip != null)
				skip.setEnabled(true);
			resume.setEnabled(false);
		}
	}

	synchronized public void close() {
		closed = true;
		progress.setValue(0);
		play = false;
		if (audio != null) {	
			audio.close();
		}
		decoder = null;
		if (bitstream != null)
			try {
				bitstream.close();
			} catch (BitstreamException ex) { 
			}
		if (inputStream != null)
			try {
				inputStream.close();
			} catch(IOException ioe) {
			}
		stop.setEnabled(false);
		if (skip != null)
			skip.setEnabled(false);
		resume.setEnabled(false);
		close.setEnabled(false);
		notify();
	}
	
	// remote controllable
	public String getName() {
		return Resources.COMP_MEDIA_PAYER;
	}

	public String toString() {
		return getName();
	}
	
	public Iterator	getKeyMnemonics() {
		return remoteCommandMap.keySet().iterator();
	}
	
	public boolean doAction(String keyCode) {
		System.err.println("MP:Doing action "+keyCode);
		if (remoteCommandMap.get(keyCode) == null)
			return false;
		actionPerformed(new ActionEvent(remoteCommandMap.get(keyCode), keyCode.hashCode(), keyCode));
		return true;
	}
	
	public void bringOnTop() {
		requestFocus();
	}
	
	protected void finalize() throws Throwable {
		super.finalize();
		if (!closed)
			close();
	}		
	
	public void updateTitle(String title) {
		Object f = getTopLevelAncestor();
		if (f != null && f instanceof Frame)
			((Frame)f).setTitle(title);
	}

	protected void pack() {		
		Object f = getTopLevelAncestor();
		if (f != null && f instanceof Frame)
			((Frame)f).pack();
		else
			doLayout();
	}

	/**
	 * The MPEG audio bitstream. 
	 */
	private Bitstream bitstream;
	
	/**
	 * The MPEG audio decoder. 
	 */
	private Decoder	decoder; 
	
	/**
	 * The AudioDevice the audio samples are written to. 
	 */
	private AudioDevice	audio;
	
	/**
	 * Has the player been closed?
	 */
	private volatile boolean closed = false;
	
	private InputStream inputStream;
	
	protected boolean autoClose;
	
	protected boolean play;
	protected JSlider progress;
	protected JButton stop, resume, close, skip, advanced;
	protected JEditorPane tp_info;
	protected Window window;
	protected WindowListener windowListener;
    protected JPanel advancedPanel;	
	protected Map remoteCommandMap;
	protected int mark;
}
