
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import java.util.*;
import org.webmacro.util.*;
import java.io.*;


/**
  * A term is a string or macro that can be used as an argument to a 
  * function, the return value of a function, an argument to a directive,
  * or as a term in a conditional expression. It can have any of the 
  * following types: String (possibly composed from a list) or List.
  * It can get that value as the result of a variable, a variable 
  * concatenated with a string, the return value of a function, etc.
  * <p>
  * The term class is somewhat illusory--term objects don't exist. Instead
  * the Term parser will create an object of the appropriate type and 
  * return that instead. 
  */
abstract public class Term implements Macro 
{

   /**
     * Parse a term. A term is a word or a quoted string, or 
     * a variable.  
     * <p>
     * @returns a String, Variable, or QuotedString 
     * @exception ParseException if the sytax was invalid and we could not recover
     * @exception IOException if we could not successfullly read the parseTool
     */
   final public static Object parse(ParseTool in) 
      throws ParseException, IOException
   {

      Object term = null;;
      switch(in.ttype) {
         case '$': // variable
            term = Variable.parse(in);
            break;

         case '[': // list
            term = List.parse(in);
            break;

         case '\"': case '\'':
            term = ParsedString.parse(in);
            break;

         case in.TT_WORD: // it's a term if it starts with a letter or digit
            String word = in.sval;
            if (Character.isLetterOrDigit(word.charAt(0))) {
               boolean isLong = false;
               boolean isInt = true;
               LOOP: for (int i=0; i < word.length(); i++) {
                  switch(word.charAt(i)) {
                     case '0': case '1': case '2': case '3': case '4':
                     case '5': case '6': case '7': case '8': case '9':
                     case '-':
                               break;
                     case 'l': case 'L':
                               if (i == (word.length() - 1)) {
                                  isLong = true;
                               } else {
                                  isInt = false;
                                  break LOOP;
                               }
                               break;
                     default:
                               isInt = false;
                               break LOOP; 
                  }
               }

               if (isInt) {
                  if (isLong) {
                     term = new Long(word.substring(0,word.length() - 1));
                  } else {
                     term = new Integer(word);
                  }
               } else if (in.sval.equalsIgnoreCase("false")) {
                  term = Boolean.FALSE;
               } else if (in.sval.equalsIgnoreCase("true")) {
                  term = Boolean.TRUE;
               } else {
                  term = in.sval;
               }
               in.nextToken(); // we've eaten it
            } 
            break;

         default: // we can't handle it
            term = null;
      }
      return term;      
   }
}



