import java.awt.*;
import java.io.*;
import java.util.*;
import java.net.*;



public class Webserver
{
	public static void main(String[] args)
	{
		
		if (args.length > 0 )
		{
			System.out.println("Ingoring arguments.");
		}

		MainFrame f = new MainFrame();
		ConnectionAcceptor c = new ConnectionAcceptor(f);

		f.setBounds(0,0,400,200);	
		f.show(c);

		c.start();
	}

}




/****************** MainFrame ********************/

/*MainFrame updates the user of events. Right now the 
text area is editable.  I'd like to change that someday.
Perhaps by using the new event model.*/

class MainFrame extends Frame
{
	MainFrame()
	{
		setTitle("Java Personal Webserver 0.9");
		setLayout( new BorderLayout());
		add("Center", ta);
		ta.setEditable(false);
	}


	public void show(ConnectionAcceptor c)
	{
		connectionAcceptor = c;
		super.show();
	}

	public boolean handleEvent(Event evt)
	{
		if (evt.id == Event.WINDOW_DESTROY) 
		{
			if (!done) 
			{
				done = true;
				if (connectionAcceptor!=null)
					connectionAcceptor.makeDone();
				System.exit(0);
			}
		}
		else return super.handleEvent(evt);
		return true;
	}


	public void println(String str)
	{
		ta.append(str + "\n");
	}

	private TextArea ta = new TextArea();
	private boolean done = false;
	private ConnectionAcceptor connectionAcceptor;

}

/*************** ConnectionAcceptor ******************/



/*
ConnectionAcceptor
written by Clay Lenhart

When creating the object it must have a MainFrame class to print updates to.

ConnectionAcceptor accepts connections on port 80.  It then creates 
a ConnectionThread and passes the input and output responiblities to 
it.  ConnectionAcceptor then waits for another connection.

ConnectionAcceptor creates objects of type ConnectionThread, ConnectionManager, and 
Preferences.
*/

class ConnectionAcceptor extends Thread
{
	ConnectionAcceptor(MainFrame f)
	{
		statusFrame = f;
		setPriority(Thread.MAX_PRIORITY);  /*this thread should go as quickly as possible*/
	}


	public void run()
	{
		/*load prefrences*/
		prefs.LoadPreferences(statusFrame);
		

		/*load fileTypes*/
		LoadFileTypes(statusFrame);
		
		/*the ConnectionManager is essentially a GroupThread class. When
		a Thread is created, the Thread adds itself to connectionList.
		connectionList is used to find out how many connections are still
		open, and possibly be used later to kill threads by an administrator.*/
		connectionList = new ConnectionManager();
		
		
		try
		{ 
			s = new ServerSocket(80,30);  /*port 80 is the http port*/
				/*only allow 30 connections to build up in the queue*/
		}
		catch(Exception e)
		{
			statusFrame.println("Unable to open port.  Please exit webserver.");
			stop();
		}
		try
		{
			statusFrame.println("Ready, serving from " + InetAddress.getLocalHost().getHostName());
		}
		catch(Exception e)
		{
			statusFrame.println("Difficulty in acquiring host name.  Network may be down.");
		}
			int connectionNumber = 0;	/*connection ID number for ConnectionThreads*/
		try
		{
			while (!done)
			{
				
				Socket connection = s.accept();  /*wait for next connection*/

				ConnectionThread t = new ConnectionThread(
					connectionList,statusFrame,prefs,
					mimeTypes,connection,"Connection number: " + connectionNumber);
					/*create a thread and give all the information.*/
				t.start();  /*Start the thread*/
				connectionNumber++;/*increment the thread ID number*/
				yield(); /*yield to other threads for the non-preemptive 
						multitasking operating systems.*/
			}
		}
		catch(Exception e) 
		{
			statusFrame.println("An error has occured while making a connection.");
			statusFrame.println("No more connections will be made until you exit and restart application.");
			stop();
		}  
			
	}
	
	/*makeDone attempts to finish all "ConnectionThreads" */
	public void makeDone()
	{
		done=true;

		if ((connectionList != null) && (connectionList.activeCount()>0))
		{

			statusFrame.println(connectionList.activeCount() + " Connections active");
			statusFrame.println("Waiting for connections to finish.");
			statusFrame.println("Personal Webserver will automatically exit in 1 minute.");
		

			Calendar in1Minute  = Calendar.getInstance();
			in1Minute.add(Calendar.MINUTE,1);



			
			
			
			boolean timedOut=false;
			while (!timedOut && (connectionList.activeCount() > 0))
			{
				yield();
				
				timedOut = in1Minute.before(Calendar.getInstance());
			}	

			/*kill all threads in this group (all "ConnectionThreads")*/
			connectionList.stop(); 
		}
		
	}


	/*LoadFileTypes reads all the mime file types from Webserver.mim file.*/
	private void LoadFileTypes(MainFrame f)
	{
		File fileTypesFile = new File("Webserver.mim");
		if (!fileTypesFile.isFile())
		{
			f.println("Could not find file: " + fileTypesFile);
			return;
		}
			
		String line=null,mime="", ext="";
		try
		{
			FileInputStream fileTypesFileInputStream = new FileInputStream(fileTypesFile);
			DataInputStream fileTypesInputStream = new DataInputStream(
				fileTypesFileInputStream);
			int l = (int)fileTypesFile.length();
			byte[] buffer = new byte[l];
			fileTypesInputStream.read(buffer);
			
			StringTokenizer bufferST = new StringTokenizer(new String(buffer),"\r\n");

			
			if (bufferST.hasMoreTokens())
				line = (String)bufferST.nextToken();
			else
			{
				f.println(fileTypesFile +" is not a Webserver fileType File. NowEntries read.");
				return;
			}

			if( !(line.equals("Webserver/0.9/fileTypes")))
			/*Is the file a Webserver mime type file*/
			{
				f.println(fileTypesFile + " is not a Webserver fileType File. No entries read.");
				return;
			}
	
			StringTokenizer st = null;
	
			statusFrame.println("Loading mime file types");

			while(bufferST.hasMoreTokens() && !((line=(String) bufferST.nextElement()).startsWith("%end%")))
			{
				
				/*line = (String) bufferST.nextElement();*/
				
				st = new StringTokenizer(line," \t");  /*blank spaces separate the mime 
									type with the file extension*/
				if (st.countTokens()==2)
				{
					mime = st.nextToken();   /*mime types come first*/
					ext = st.nextToken();	/*then the extension*/
					mimeTypes.put(ext.toLowerCase(),mime);  /*put each in the hash table*/
					statusFrame.println(mime + " " + ext);
				}
				
			}
			fileTypesInputStream.close();  
		}
		catch(Exception e)
		{
			statusFrame.println("An error occured while reading the mime types. No more mime types will be read.");
			
		}
	}

	private boolean done=false;
	private String rootDirectory="", index="";
        private ConnectionManager connectionList;
	private Hashtable mimeTypes = new Hashtable();
	private Preferences prefs = new Preferences();
	private MainFrame statusFrame;
	private ServerSocket s;
}


/*************** ConnectionManager *******************/

/*This is a ThreadGroup, but could be expanded.  This might be used to kill threads or 
connections by an administrator.*/

class ConnectionManager extends ThreadGroup
{
	ConnectionManager()
	{
		super("Connection Manager");
	}



}


/************** ConnectionThread *********************/


/*
ConnectionThread
written by Clay Lenhart

Once a connection has been made, the ConnectionThread handles the
dialog and closes the connection.  

ConnectionThread creates no new objects.

*/


class ConnectionThread extends Thread
{
	ConnectionThread(ThreadGroup tg, MainFrame mf, Preferences p, Hashtable mT, Socket c, String str)
	{
		super(tg, str); 	/*add this Thread to the ThreadGroup tg */

		statusFrame = mf;	/*updates the administrator of connections*/
		mimeTypes = mT;		/*the list of mime types in a Hashtable*/
		prefs = p;		/*the Preferences object*/
		connection = c;		/*The Socket to send and receive data*/
		setPriority(Thread.MAX_PRIORITY-2);  /*We want a responsive connection.*/

	}
	


	public void run()
	{
		try
		{
			String date = (new Date()).toString(); /*grab the date when the thread started*/

			/*initilize input and output streams*/
			DataInputStream in = new DataInputStream(connection.getInputStream());
			DataOutputStream out = new DataOutputStream(connection.getOutputStream());
		


			String[] commands = readCommands(in);
			

			/*decifer request*/
	
			StringTokenizer st = new StringTokenizer(commands[0]," ");
			
			/*find the http command (cmd) and the filename (file) in the header*/
			if (!(st.countTokens()>=2)) 
			{
				sendNotImplemented(out);
			}
			else
			{
				String cmd="",file="",extension="";
				cmd = st.nextToken();
				file = st.nextToken();
			
				/*Inform administrator of the connection.*/
				statusFrame.println(date + " " + connection.getInetAddress().getHostAddress() +" "+ file);
	
				if (cmd.equals("GET"))  /*If the command is a GET*/
				{

					/*open file to read and send*/
					File inputFile = new File(prefs.getRootDir() + file);
					if (inputFile.isDirectory()) /*Is web surfer requesting the index file?*/
						inputFile = new File(prefs.getRootDir()+ file + prefs.getIndex());
					inputFile = new File(inputFile.getCanonicalPath());

					/*acquire file extension for the mime type later.*/
					st = new StringTokenizer(new String(inputFile.toString()),".");
					while(st.hasMoreTokens())
						extension = st.nextToken();
			
					/*As a security precaution, check to make sure file is not outside of the root directory.*/
					if (!withinSecurityLimits(inputFile,prefs))
						sendForbidden(out);
					else
					{
						
						if (!(inputFile.canRead()))   /*Check to make sure file exists and is readable.*/ 
							sendNotFound(out, inputFile);
						else 
						{
							
							String mime = (String)mimeTypes.get(extension.toLowerCase());  /*Get the mime type from the mime hash table.*/
							if(mime==null)  /*extension is not in hash table*/
								sendNotImplemented(out);
							else sendFile(inputFile, out, mime);
						}
					}
				}
				else sendNotImplemented(out);
			
			}		
			connection.close();
		}
		catch(Exception e)
		{ 			
			try
			{
				connection.close();
				stop();  /*kill the thread*/
			}
			catch(Exception ex)
			{


				stop();
			} 
		}

		
	}

	/*sendFile sends everything needed to the client, including the header.*/
	private void sendFile(File fileObject, DataOutputStream out, String mime)
	{
		try
		{
			out.writeBytes("HTTP/0.9 200 OK\r\n");
			out.writeBytes("Server: Java Personal Webserver 0.9\r\n");
			out.writeBytes("MIME-Version: 1.0\r\n");
			out.writeBytes("Content-type: " + mime + "\r\n");
			out.writeBytes("\r\n");

			FileInputStream inFileStream = new FileInputStream(fileObject);
			long l = fileObject.length();
			byte[] b = new byte[(int)l];
			
			/*I found that reading to a buffer was much quicker than anything else*/
			inFileStream.read(b); 
 
			out.write(b);


		}
		catch(Exception e)
		{
			stop(); /*kill this thread, connection lost.*/
		}
	}

	private void sendNotFound(DataOutputStream out, File f)
	{
		sendMessage(out,"HTTP/0.9 404 Object Not Found");
	}

	private void sendNotImplemented(DataOutputStream out)
	{
		sendMessage(out,"HTTP/0.9 501 Not Implemented");
	}

	private void sendForbidden(DataOutputStream out)
	{
		sendMessage(out,"HTTP/0.9 403 Forbidden");
	}

	private void sendMessage(DataOutputStream out, String message)
	{
		try
		{
			out.writeBytes(message + "\r\n\r\n");
			out.writeBytes("<html><body><h1>" + message +"</h1></body></html>\r\n");
		}
		catch(Exception e)
		{
			stop() /*connection lost, kill the thread.*/;
		}
	}

	private boolean withinSecurityLimits(File f, Preferences prefs)
	/*return true if Webserver is allowed to serve file.*/
	{
		try
		{
			String fString = f.getCanonicalPath();
			
	
			/*Check to make sure file is within root directory*/
			File root = new File (prefs.getRootDir());
			
	
			if (fString.startsWith(root.getCanonicalPath()))  
				return true;	
			return false;
				
		}
		catch(Exception e)
		{
			return false;
		}
	}

	private String[] readCommands(InputStream in)
	{
		try
		{
			Vector lines= new Vector();
				
			Vector vector = new Vector(100);
			int dataLength = 0;
			boolean done = false;
			Character CR = new Character('\r');
			Character LF = new Character('\n');

			int c, oldc=0;
			while (!done)
			{
				Thread.yield();
				c = in.read();
				
				if (c == -1)
					stop(); /*kill the thread: connection lost.*/
				else	
				{
					vector.addElement(new Character((char)c));
					dataLength++;
					if (dataLength>=4)
						done = (((Character)vector.elementAt(dataLength-4)).equals(CR)) &&
							(((Character)vector.elementAt(dataLength-3)).equals(LF))  &&
							(((Character)vector.elementAt(dataLength-2)).equals(CR))  &&
							(((Character)vector.elementAt(dataLength-1)).equals(LF)) ;
				}
				
					
				
					
			}

			char[] charArray = new char[dataLength];
			for (int n=dataLength-1; n>=0;n--)
				charArray[n] = ((Character)vector.elementAt(n)).charValue();
			StringTokenizer st = new StringTokenizer(new String(charArray),"\r\n");
			int numOfLines = st.countTokens();
			String[] commands = new String[numOfLines];
			for (int n=numOfLines;n>0;n--)
				commands[numOfLines-n] = st.nextToken();
			return commands;

		}
		catch(Exception e)
		{
			/*connection lost:  kill this thread*/
			stop();
			String[] result = {""};
			return result;
		}
		
	}

	private Socket connection;
	private MainFrame statusFrame;
	private Hashtable mimeTypes;
	private Preferences prefs;

			
}


/*************** Preferences *********************/



/*Preferences loads and holds the preferences.  Later verions I'll include saving the prefs.*/
class Preferences
{
	Preferences()
	{
		/*set Defaults*/
		index = "index.html";
		rootDirectory = "c:\\";
		File rootDirFile = new File("Webserver.class");
		try
		{
			rootDirFile = new File(rootDirFile.getCanonicalPath());
		}
		catch(Exception e) {return;}
		rootDirectory = rootDirFile.getParent();
	}

	public String getIndex()
	{
		return index;
	}
	
	public void setIndex(String i)
	{
		index = i;
	}
	
	public String getRootDir()
	{
		return rootDirectory;
	}

	public void setRootDir(String d)
	{
		rootDirectory = d;
	}

	public void LoadPreferences(MainFrame statusFrame)
	{



		/*check to make sure Webserver.prf exists*/
		
		if (!prefFile.isFile())
		{
			statusFrame.println("I could not find file: " + prefFile);
			return;
		}
			

		/*read Webserver.prf*/
		try
		{
			FileInputStream prefInputStream = new FileInputStream(prefFile);

			StringTokenizer st = null;
			String line = null;

			int l = (int) prefFile.length();
			byte[] buffer = new byte[l];
			prefInputStream.read(buffer);
			StringTokenizer bufferST = new StringTokenizer(new String(buffer),"\r\n");
			int numOfLines = bufferST.countTokens();
			if (numOfLines>0)
				line = bufferST.nextToken();

			if (!(line.equals("Webserver/0.9/prefs")))
			/*The first line of Webserver.prf should be "Webserver/0.9/prefs"*/
			{
				statusFrame.println(prefFile + " is not a Webserver preference file.  Using default settings.");
				
				return;
			}



			
			for (int n = 1;(n<numOfLines)&&!((line=bufferST.nextToken()).startsWith("%end%"));n++)
			/*The last line of Webserver.prf should be "end"*/	
			{
				
				/* varible=value */
				st = new StringTokenizer(line,"=");
				if(st.countTokens()!=2)
				{
					statusFrame.println("Incorrect Webserver Preference File format.");
					statusFrame.println("Please Exit.");
					return;
				}

				String varible = st.nextToken();
				String value = st.nextToken();

				/*case statement*/
				if (varible.equals("rootDir"))
				{
					
					File dir = new File(value);
					dir = new File(dir.getCanonicalPath());
					if (!dir.isDirectory())
						statusFrame.println(dir + " is not a valid directory.  Please edit the line rootDir= in the file Webserver.prf.");
					rootDirectory = dir.getPath();
					statusFrame.println("rootDir=" + rootDirectory);
					
				}
				else if (varible.equals("index"))
				{
					index = value;
					statusFrame.println("index=" + index);
				}
				/*end case statement*/
				
			}
			prefInputStream.close();
			
		}
		catch(Exception e)
		{
			statusFrame.println("An error occured while loading the preferences.  Default values may be used.");
			return;
		}
	}

	/*public void SavePreferences()
	{
	}
	*/

	private String rootDirectory, index="";
	private File prefFile = new File("Webserver.prf");
}




