/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.ui;

import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.Rect;
import com.jniwrapper.win32.gdi.Icon;
import com.jniwrapper.win32.gdi.Region;

/**
 * This class provides ability to change or query native window attributes.
 *
 * @author Serge Piletsky
 */
public class WindowDecorator
{
    public static final int OPAQUE = 255;
    public static final int TRANSPARENT = 0;

    private Wnd _windowHandle;
    private boolean _transparent = false;
    private byte _transparency = (byte) OPAQUE;
    private boolean _palleteWindow = false;
    private boolean _topmost = false;
    private boolean _rounded = false;
    private boolean _captionVisible = true;
    private boolean _borderVisible = true;

    public WindowDecorator()
    {
    }

    public WindowDecorator(Wnd wnd)
    {
        setWindowHandle(wnd);
    }

    /**
     * @return native window handle
     */
    public Wnd getWindowHandle()
    {
        if (_windowHandle == null)
        {
            throw new IllegalStateException();
        }

        return _windowHandle;
    }

    /**
     * Sets native window handle
     * @param windowHandle
     */
    public void setWindowHandle(Wnd windowHandle)
    {
        if (windowHandle == null)
        {
            throw new IllegalArgumentException();
        }

        _windowHandle = windowHandle;
    }

    /**
     * @return true if window is transparent otherwise returns false
     */
    public boolean isTransparent()
    {
        return _transparent;
    }

    /**
     * Sets or removes window transparancy
     * @param transparent
     */
    public void setTransparent(boolean transparent)
    {
        if (_transparent == transparent)
        {
            return;
        }

        _transparent = transparent;
        Wnd windowHandle = getWindowHandle();
        long oldWindowStyle = getWindowExStyle();
        long newWindowStyle;
        if (transparent)
        {
            newWindowStyle = oldWindowStyle | Wnd.WS_EX_LAYERED;
            setWindowExStyle(newWindowStyle);
            getWindowHandle().setLayeredWindowAttributes(0, _transparency, Wnd.LWA_ALPHA);
        }
        else
        {
            newWindowStyle = oldWindowStyle & ~Wnd.WS_EX_LAYERED;
            setWindowExStyle(newWindowStyle);
            // Ask the window and its children to repaint
            Handle NULL = new Handle();
            windowHandle.redraw(NULL, NULL,
                    Wnd.RDW_ERASE |
                    Wnd.RDW_INVALIDATE |
                    Wnd.RDW_FRAME |
                    Wnd.RDW_ALLCHILDREN);
        }
    }

    /**
     * @return window transparency value
     */
    public byte getTransparency()
    {
        return _transparency;
    }

    /**
     * Sets window transparency
     * @param transparency
     * 0 makes the window completely transparent
     * 255 makes it completely opaque
     */
    public void setTransparency(byte transparency)
    {
        if (_transparency == transparency)
        {
            return;
        }

        _transparency = transparency;

        if (!isTransparent())
        {
            setTransparent(true);
        }

        getWindowHandle().setLayeredWindowAttributes(0, transparency, Wnd.LWA_ALPHA);
    }

    /**
     * @return true if window has pallete style otherwise returns false
     */
    public boolean isPalleteWindow()
    {
        return _palleteWindow;
    }

    /**
     * Sets pallete window style
     * @param palleteWindow
     */
    public void setPalleteWindow(boolean palleteWindow)
    {
        if (_palleteWindow == palleteWindow)
        {
            return;
        }

        _palleteWindow = palleteWindow;
        long oldWindowStyle = getWindowExStyle();
        long newWindowStyle;
        if (_palleteWindow)
        {
            newWindowStyle = oldWindowStyle | Wnd.WS_EX_PALETTEWINDOW;
        }
        else
        {
            newWindowStyle = oldWindowStyle & ~Wnd.WS_EX_PALETTEWINDOW;
        }

        setWindowExStyle(newWindowStyle);
        Handle NULL = new Handle();
        getWindowHandle().redraw(NULL, NULL,
                Wnd.RDW_FRAME |
                Wnd.RDW_INVALIDATE);
    }

    public boolean isTopmost()
    {
        return _topmost;
    }

    /**
     * Makes window topmost
     * @param topmost
     */
    public void setTopmost(boolean topmost)
    {
        if (_topmost == topmost)
        {
            return;
        }

        _topmost = topmost;
        int zOrder = Wnd.HWND_NOTOPMOST;
        if (_topmost)
        {
            zOrder = Wnd.HWND_TOPMOST;
        }
        Wnd.setWindowPos(getWindowHandle(), new Wnd(zOrder), 0, 0, 0, 0, Wnd.SWP_NOSIZE | Wnd.SWP_NOMOVE);
    }

    /**
     * Sets extended widow style
     * @param flags
     */
    public void setWindowExStyle(long flags)
    {
        final Wnd windowHandle = getWindowHandle();
        windowHandle.setWindowLong(Wnd.GWL_EXSTYLE, flags);
        Wnd.setWindowPos(windowHandle, windowHandle, 0, 0, 0, 0, Wnd.SWP_NOSIZE | Wnd.SWP_NOMOVE | Wnd.SWP_NOZORDER | Wnd.SWP_FRAMECHANGED);
    }

    /**
     * @return extended window style flags
     */
    public long getWindowExStyle()
    {
        final Wnd windowHandle = getWindowHandle();
        return windowHandle.getWindowLong(Wnd.GWL_EXSTYLE);
    }

    /**
     * Sets window style
     * @param flags
     */
    public void setWindowStyle(long flags)
    {
        final Wnd windowHandle = getWindowHandle();
        windowHandle.setWindowLong(Wnd.GWL_STYLE, flags);
        Wnd.setWindowPos(windowHandle, windowHandle, 0, 0, 0, 0, Wnd.SWP_NOSIZE | Wnd.SWP_NOMOVE | Wnd.SWP_NOZORDER | Wnd.SWP_FRAMECHANGED);
    }

    /**
     * @return window style flags
     */
    public long getWindowStyle()
    {
        final Wnd windowHandle = getWindowHandle();
        return windowHandle.getWindowLong(Wnd.GWL_STYLE);
    }

    /**
     * Checks if window is rounded.
     * @return true if window is rounded
     */
    public boolean isRounded()
    {
        return _rounded;
    }

    /**
     * Makes window rounded (with rounded corners).
     * @param rounded
     * @param ellipseWidth
     * @param ellipseHeight
     */
    public void setRounded(boolean rounded, int ellipseWidth, int ellipseHeight)
    {
        if (rounded == _rounded)
            return;
        _rounded = rounded;
        final Wnd windowHandle = getWindowHandle();
        Region region = null;
        if (rounded)
        {
            Rect windowRect = new Rect();
            Wnd.getWindowRect(windowHandle, windowRect);
            // normalize rectangle coordinates
            windowRect.moveTo(0, 0);
            region = Region.createRoundRectRegion(
                    (int) windowRect.getLeft(),
                    (int) windowRect.getTop(),
                    (int) windowRect.getRight(),
                    (int) windowRect.getBottom(),
                    ellipseWidth,
                    ellipseHeight);
        }
        Wnd.setWindowRgn(windowHandle, region, true);
    }

    static final int DEFAULT_ELLIPSE_HEIGHT = 30;
    static final int DEFAULT_ELLIPSE_WIDTH = 30;

    /**
     * Makes window rounded (with rounded corners).
     * @param rounded
     */
    public void setRounded(boolean rounded)
    {
        setRounded(rounded, DEFAULT_ELLIPSE_WIDTH, DEFAULT_ELLIPSE_HEIGHT);
    }

    /**
     * Assigns specified icon to window.
     * @param icon
     */
    public void setWindowIcon(Icon icon)
    {
        final Wnd windowHandle = getWindowHandle();
        windowHandle.sendMessage(Msg.WM_SETICON, icon.getType(), (int)icon.getValue());
    }

    /**
     * Shows or hides window.
     * @param visible
     */
    public void setVisible(boolean visible)
    {
        long windowStyle = getWindowStyle();
        if (!visible)
            windowStyle &= ~Wnd.WS_VISIBLE;
        else
            windowStyle |= Wnd.WS_VISIBLE;

        setWindowStyle(windowStyle);
    }

    /**
     * Shows or hides window caption.
     * @param value
     */
    public void setCaptionVisible(boolean value)
    {
        if (_captionVisible == value)
            return;
        _captionVisible = value;
        long oldWindowStyle = getWindowStyle();
        long newWindowStyle;
        if (value)
        {
            newWindowStyle = oldWindowStyle | Wnd.WS_CAPTION;
        }
        else
        {
            newWindowStyle = oldWindowStyle & ~Wnd.WS_CAPTION;
        }
        setWindowStyle(newWindowStyle);
        Handle NULL = new Handle();
        getWindowHandle().redraw(NULL, NULL,
                Wnd.RDW_ERASE |
                Wnd.RDW_INTERNALPAINT);
    }

    /**
     * Checks if window caption is visible
     * @return true if window caption is visible
     */
    public boolean isCaptionVisible()
    {
        return _captionVisible;
    }

    /**
     * Shows or hides window border.
     * @param value
     */
    public void setBorderVisible(boolean value)
    {
        if (_borderVisible == value)
            return;
        _borderVisible = value;
        long oldWindowStyle = getWindowStyle();
        long newWindowStyle;
        if (value)
        {
            newWindowStyle = oldWindowStyle | Wnd.WS_BORDER;
        }
        else
        {
            newWindowStyle = oldWindowStyle & ~Wnd.WS_THICKFRAME;
        }
        setWindowStyle(newWindowStyle);
    }

    /**
     * Checks if window border is visible
     * @return true if window border is visible
     */
    public boolean isBorderVisible()
    {
        return _borderVisible;
    }

    /**
     * Sets custom window region.
     * Returns previously assigned region in order to restore it later.
     * @param region
     * @return region previously assigned to window
     */
    public Region setRegion(Region region)
    {
        Wnd windowHandle = getWindowHandle();
        Region prevRegion = Wnd.getWindowRgn(windowHandle);
        Wnd.setWindowRgn(windowHandle, region, true);
        return prevRegion;
    }
}
