/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.io;

import com.jniwrapper.util.FlagSet;
import com.jniwrapper.win32.VersionInfo;

import java.io.FileFilter;
import java.lang.reflect.Constructor;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * This class listens to the file system change notifications and raises
 * events when a directory, or a file in the directory changes.
 * The class uses different watching mechanisms depending on the
 * current OS version.
 *
 * @author Serge Piletsky
 */
public class FileSystemWatcher
{
    private String _path;
    private FileFilter _fileFilter;
    private WatcherOptions _notifyOptions = new WatcherOptions();
    private boolean _watchSubree = false;
    private List _fileSystemListeners = new LinkedList();
    private WatcherStrategy _watcherStrategy;

    /**
     * Creates new instance to watch for modifications in the given path.
     * Sub-directories of the passed path are not monitored.
     * @param path a path to watch to.
     */
    public FileSystemWatcher(String path)
    {
        _path = path;

        VersionInfo versionInfo = new VersionInfo();
        if (versionInfo.isNT())
        {
            _watcherStrategy = createStrategy(WinNTWatcherStrategy.class);
        }
        else
        {
            _watcherStrategy = createStrategy(Win9xWatcherStrategy.class);
        }
    }

    /**
     * Creates new instance to watch for modifications in the given path.
     * @param path a path to watch
     * @param watchSubree if true, sub-directories will be also monitored.
     */
    public FileSystemWatcher(String path, boolean watchSubree)
    {
        this(path);
        _watchSubree = watchSubree;
    }

    /**
     * Sets custom watching strategies.
     * @param strategyClass a custom strategy implementation class.
     */
    public void setStrategy(Class strategyClass)
    {
        if (_watcherStrategy.getClass().equals(strategyClass))
            return;
        if (_watcherStrategy.isWatching())
            throw new IllegalStateException("Unable to change stragetgy while watcher is running.");
        _watcherStrategy = createStrategy(strategyClass);
    }

    protected WatcherStrategy createStrategy(Class strategyClass)
    {
        WatcherStrategy result = null;
        try
        {
            final Constructor constructor = strategyClass.getConstructor(new Class[] {FileSystemWatcher.class});
            result = (WatcherStrategy) constructor.newInstance(new Object[] {this});
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * Creates FileSystemWatcher
     * @param path is a path to watch
     * @param fileFilter only files with specified extension will be watched
     * @param watchSubree
     */
    public FileSystemWatcher(String path, FileFilter fileFilter, boolean watchSubree)
    {
        this(path, watchSubree);
        _fileFilter = fileFilter;
    }

    /**
     * Creates FileSystemWatcher
     * @param path is a path to watch
     * @param fileFilter only files with specified extension will be watched. For exapmle .log|.
     */
    public FileSystemWatcher(String path, FileFilter fileFilter)
    {
        this(path);
        _fileFilter = fileFilter;
    }

    public String getPath()
    {
        return _path;
    }

    public FileFilter getFileFilter()
    {
        return _fileFilter;
    }

    public boolean isWatchSubree()
    {
        return _watchSubree;
    }

    /**
     * @return {@link WatcherOptions} that contains set of notify filters
     */
    public WatcherOptions getOptions()
    {
        return _notifyOptions;
    }

    /**
     * Adds {@link FileSystemEventListener}
     * @param listener
     */
    public void addFileSystemListener(FileSystemEventListener listener)
    {
        if (!_fileSystemListeners.contains(listener))
            _fileSystemListeners.add(listener);
    }

    /**
     * Removes {@link FileSystemEventListener}
     * @param listener
     */
    public void removeFileSystemListener(FileSystemEventListener listener)
    {
        _fileSystemListeners.remove(listener);
    }

    protected void fireFileSystemEvent(FileSystemEvent event)
    {
        List listeners = null;
        synchronized(this)
        {
            listeners = new LinkedList(_fileSystemListeners);
            for (Iterator i = listeners.iterator(); i.hasNext();)
            {
                FileSystemEventListener listener = (FileSystemEventListener) i.next();
                listener.handle(event);
            }
        }
    }

    /**
     * Starts watching
     * @throws FileSystemException
     */
    public void start() throws FileSystemException
    {
        _watcherStrategy.start();
    }

    /**
     * Stops watching
     * @throws FileSystemException
     */
    public void stop() throws FileSystemException
    {
        _watcherStrategy.stop();
    }

    public boolean isWatching()
    {
        return _watcherStrategy.isWatching();
    }

    public class WatcherOptions extends FlagSet
    {
        public static final int NOTIFY_CHANGE_FILE_NAME   = 0x00000001;
        public static final int NOTIFY_CHANGE_DIR_NAME    = 0x00000002;
        public static final int NOTIFY_CHANGE_ATTRIBUTES  = 0x00000004;
        public static final int NOTIFY_CHANGE_SIZE        = 0x00000008;
        public static final int NOTIFY_CHANGE_LAST_WRITE  = 0x00000010;
        public static final int NOTIFY_CHANGE_LAST_ACCESS = 0x00000020;
        public static final int NOTIFY_CHANGE_CREATION    = 0x00000040;
        public static final int NOTIFY_CHANGE_SECURITY    = 0x00000100;

        public WatcherOptions()
        {
            super();
            reset();
        }

        public void reset()
        {
            clear();
            add(NOTIFY_CHANGE_FILE_NAME | NOTIFY_CHANGE_DIR_NAME | NOTIFY_CHANGE_ATTRIBUTES | NOTIFY_CHANGE_SIZE | NOTIFY_CHANGE_LAST_WRITE);
        }

        public boolean isNotifyChangeFileName()
        {
            return contains(NOTIFY_CHANGE_FILE_NAME);
        }

        public void setNotifyChangeFileName(boolean value)
        {
            if (value)
                add(NOTIFY_CHANGE_FILE_NAME);
            else
                remove(NOTIFY_CHANGE_FILE_NAME);
        }

        public boolean isNotifyChangeDirName()
        {
            return contains(NOTIFY_CHANGE_DIR_NAME);
        }

        public void setNotifyChangeDirName(boolean value)
        {
            if (value)
                add(NOTIFY_CHANGE_DIR_NAME);
            else
                remove(NOTIFY_CHANGE_DIR_NAME);
        }

        public boolean isNotifyChangeAttributes()
        {
            return contains(NOTIFY_CHANGE_ATTRIBUTES);
        }

        public void setNotifyChangeAttributes(boolean value)
        {
            if (value)
                add(NOTIFY_CHANGE_ATTRIBUTES);
            else
                remove(NOTIFY_CHANGE_ATTRIBUTES);
        }

        public boolean isNotifyChangeSize()
        {
            return contains(NOTIFY_CHANGE_SIZE);
        }

        public void setNotifyChangeSize(boolean value)
        {
            if (value)
                add(NOTIFY_CHANGE_SIZE);
            else
                remove(NOTIFY_CHANGE_SIZE);
        }

        public boolean isNotifyLastModified()
        {
            return contains(NOTIFY_CHANGE_LAST_WRITE);
        }

        public void setNotifyLastModified(boolean value)
        {
            if (value)
                add(NOTIFY_CHANGE_LAST_WRITE);
            else
                remove(NOTIFY_CHANGE_LAST_WRITE);
        }
    }
}
