/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.gdi;

import com.jniwrapper.*;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.LastErrorException;
import com.jniwrapper.win32.shell.SHFileInfo;
import com.jniwrapper.win32.shell.Shell32;
import com.jniwrapper.win32.system.Module;
import com.jniwrapper.win32.ui.User32;

import java.awt.image.BufferedImage;

/**
 * This class represents generic methods to work with Icon.
 *
 * @author Serge Piletsky
 */
public class Icon extends GdiObject
 {
    static final FunctionName FUNCTION_LOAD_ICON = new FunctionName("LoadIcon");
    static final FunctionName FUNCTION_LOAD_IMAGE = new FunctionName("LoadImage");
    static final FunctionName FUNCTION_EXTRACT_ICON = new FunctionName("ExtractIcon");
    static final FunctionName FUNCTION_EXTRACT_ASSOCIATED_ICON = new FunctionName("ExtractAssociatedIcon");
    static final String FUNCTION_GET_ICON_INFO = "GetIconInfo";

    public static final int ICON_SMALL = 0;
    public static final int ICON_BIG = 1;

    public static final int ICON_SMALL_WIDTH = 16;
    public static final int ICON_SMALL_HEIGHT = 16;
    public static final int ICON_BIG_WIDTH = 32;
    public static final int ICON_BIG_HEIGHT = 32;

    private int _type = ICON_SMALL;

    public Icon()
    {
        super();
    }

    public Icon(int type)
    {
        setType(type);
    }

    public Icon(long value)
    {
        super(value);
    }

    public Icon(long value, int type)
    {
        super(value);
        setType(type);
    }

    public int getType()
    {
        return _type;
    }

    public void setType(int type)
    {
        _type = type;
    }

    static Function getFunction(Object functionName)
    {
        return User32.getInstance().getFunction(functionName.toString());
    }

    public void loadIcon(Pointer hInstance, String fileName)
    {
        final Function function = getFunction(FUNCTION_LOAD_ICON);
        function.invoke(this, new Parameter[]
        {
            hInstance,
            User32.getInstance().stringParam(fileName),
        });
    }

    public int getWidth(int type)
    {
        int width = type==ICON_BIG?ICON_BIG_WIDTH:ICON_SMALL_WIDTH;
        return width;
    }

    public int getHeight(int type)
    {
        int height = type==ICON_BIG?ICON_BIG_HEIGHT:ICON_SMALL_HEIGHT;
        return height;
    }

    public int getWidth()
    {
        return getWidth(getType());
    }

    public int getHeight()
    {
        return getHeight(getType());
    }

    public void loadFromFile(String fileName)
    {
        loadFromFile(fileName, getWidth(), getHeight());
    }

    public void loadFromFile(String fileName, int width, int height)
    {
        Handle handle = User32.loadResourceFromFile(fileName, ImageType.ICON.getValue(), width, height);
        this.setValue(handle.getValue());
    }

    public void loadStandardIcon(int id)
    {
        loadStandardIcon(id, getWidth(), getHeight());
    }

    public void loadStandardIcon(int id, int width, int height)
    {
        final Function function = getFunction(FUNCTION_LOAD_IMAGE);
        function.invoke(this, new Parameter[]
        {
            new Handle(),
            new Int32(id),
            new UInt32(ImageType.ICON.getValue()),
            new Int(width),
            new Int(height),
            new UInt32(ImageLoadParameters.SHARED)
        });
        if (getValue() == 0) {
            throw new LastErrorException("Cannot load icon for " + id + ".");
        }
    }

    public static Icon loadStandardIconByID(int id)
    {
        Icon result = new Icon();
        result.loadStandardIcon(id);
        return result;
    }

    /**
     * Retrieves an icon from the specified file.
     * @param fileName is a file name to extract icon from
     * @param iconType type of icon to extract (ICON_SMALL or ICON_BIG)
     * @param iconIndex zero-based index of the icon to retrieve
     * @return extracted Icon
     */
    public static Icon extractFromFile(String fileName, int iconType, int iconIndex)
    {
        Icon result = new Icon(iconType);
        final Function extractIconFunction = Shell32.getInstance().getFunction(FUNCTION_EXTRACT_ICON.toString());
        extractIconFunction.invoke(result,
            Module.getCurrent(),
            Shell32.getInstance().stringParam(fileName),
            new UInt(iconIndex));
        return result;
    }

    /**
     * Extracts small associated icon.
     * @param iconPath full path that contains the icon
     * @return extracted Icon
     */
    public static Icon extractSmallAssociatedIcon(String iconPath)
    {
        SHFileInfo fileInfo = SHFileInfo.getFileInfo(iconPath, SHFileInfo.SHGFI_SMALLICON | SHFileInfo.SHGFI_ICON);
        return fileInfo.getIcon();
    }

    /**
     * Extracts big associated icon.
     * @param iconPath full path that contains the icon
     * @return extracted Icon
     */
    public static Icon extractBigAssociatedIcon(String iconPath)
    {
        Icon result = new Icon();
        final Function extractIconFunction = Shell32.getInstance().getFunction(FUNCTION_EXTRACT_ASSOCIATED_ICON.toString());
        extractIconFunction.invoke(result,
            Module.getCurrent(),
            Shell32.getInstance().stringParam(iconPath),
            new Pointer(new UInt16()));
        return result;
    }

    public IconInfo getIconInfo()
    {
        IconInfo result = new IconInfo();
        final Function function = User32.getInstance().getFunction(FUNCTION_GET_ICON_INFO);
        Bool returnValue = new Bool();
        function.invoke(returnValue, this, new Pointer(result));
        if (!returnValue.getValue())
            throw new LastErrorException("Failed to retrieve IconInfo");
        return result;
    }

    /**
     * Converts windows Bitmap to {@link java.awt.image.BufferedImage}.
     *
     * @return image Image instance.
     */
    public BufferedImage toImage()
    {
        final IconInfo iconInfo = getIconInfo();
        Bitmap mask = iconInfo.getMaskBitmap();
        Bitmap color = iconInfo.getColorBitmap();
        WindowDC dc = new WindowDC(null);
        BitmapStructure bitmapStructure = new BitmapStructure();
        getObject(color, bitmapStructure);
        final int bitCount = (int)bitmapStructure.getBitsPixel();
        BitmapInfo bitmapInfo = bitCount >= 16?new BitmapInfo():new BitmapInfo(bitCount);
        BitmapInfoHeader bitmapInfoHeader = bitmapInfo.getBitmapInfoHeader();
        final int width = (int)bitmapStructure.getBitmapWidth();
        final int height = (int)bitmapStructure.getBitmapHeight();
        // fill bitmap info
        bitmapInfoHeader.setWidth(width);
        bitmapInfoHeader.setHeight(height);
        bitmapInfoHeader.setPlanes(1);
        bitmapInfoHeader.setBitCount(bitCount);
        bitmapInfoHeader.setCompression(Bitmap.Compression.RGB);
        if (bitCount == 4 || bitCount == 8)
        {
            final int clrUsed = 1 << bitCount;
            bitmapInfoHeader.setClrUsed(clrUsed);
            bitmapInfoHeader.setClrImportant(clrUsed);
        }
        // retrieve bytes count
        final Pointer NULL = new Pointer(null, true);
        Bitmap.getDIBits(dc, color, 0, height, NULL, bitmapInfo, Bitmap.DIB_RGB_COLORS);
        // retrieve bytes from bitmap
        int bytesCount = (int)bitmapInfoHeader.getSizeImage();
        PrimitiveArray colorBits = new PrimitiveArray(UInt8.class, bytesCount);
        Bitmap.getDIBits(dc, color, 0, height, new Pointer(colorBits), bitmapInfo, Bitmap.DIB_RGB_COLORS);

        PrimitiveArray maskBits = new PrimitiveArray(UInt8.class, bytesCount);
        Bitmap.getDIBits(dc, mask, 0, height, new Pointer(maskBits), bitmapInfo, Bitmap.DIB_RGB_COLORS);

        dc.release();
        // transform data
        PrimitiveArray colors = bitmapInfo.getColors();
        BufferedImage result = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
        int index1 = 0;
        int index2 = 0;
        for (int y = height - 1; y >= 0; y--)
        {
            for (int x = 0; x < width; x++)
            {
                int rgb1 = 0;
                int rgb2 = 0;
                switch (bitCount)
                {
                    case 8: // 256 colors, used pallete
                    {
                        RGBQuad pixelColor1 = (RGBQuad)colors.getElement(index1++);
                        int r1 = (int)pixelColor1.getRed();
                        int g1 = (int)pixelColor1.getGreen();
                        int b1 = (int)pixelColor1.getBlue();
                        rgb1 = 0xFF000000 | b1 | g1 | r1;
                        RGBQuad pixelColor2 = (RGBQuad)colors.getElement(index2++);
                        int r2 = (int)pixelColor2.getRed();
                        int g2 = (int)pixelColor2.getGreen();
                        int b2 = (int)pixelColor2.getBlue();
                        rgb2 = b2 | g2 | r2;
                        break;
                    }
                    case 16:// 2 bytes per pixel, 5-5-5 bits per pixel
                    {
                        int byte1 = readByte(colorBits, index1++);
                        int byte2 = readByte(colorBits, index1++);
                        int word = byte2 << 8 | byte1;
                        int b1 = (word & 0x1F) << 3;
                        int g1 = (word & 0x3E0) << 6;
                        int r1 = (word & 0x7C00) << 9;
                        rgb1 = 0xFF000000 | b1 | g1 | r1;
                        byte1 = readByte(maskBits, index2++);
                        byte2 = readByte(maskBits, index2++);
                        word = byte2 << 8 | byte1;
                        int b2 = (word & 0x1F) << 3;
                        int g2 = (word & 0x3E0) << 6;
                        int r2 = (word & 0x7C00) << 9;
                        rgb2 = b2 | g2 | r2;
                        break;
                    }
                    case 24:// 3 bytes per pixel
                    case 32:// 4 bytes per pixel, high byte used as alpha
                    {
                        int b1 = readByte(colorBits, index1++);
                        int g1 = readByte(colorBits, index1++);
                        int r1 = readByte(colorBits, index1++);
                        rgb1 = b1 | g1 << 8 | r1 << 16;
                        if (bitCount == 32)
                        {
                            int alpha = readByte(colorBits, index1++);
                            // XP returns valid alpha-channel, but other OS'es return 0 always
                            if (alpha == 0)
                            {
                                alpha = 0xFF;
                            }
                            rgb1 |= alpha << 24;
                        }
                        else
                        {
                            rgb1 |= 0xFF000000;
                        }
                        int b2 = readByte(maskBits, index2++);
                        int g2 = readByte(maskBits, index2++);
                        int r2 = readByte(maskBits, index2++);
                        rgb2 = b2 | g2 << 8 | r2 << 16;
                        if (bitCount == 32)
                        {
                            index2++;
                        }
                    }
                }
                if (rgb2 != 0)
                {
                    result.setRGB(x, y, rgb2);
                }
                else
                {
                    result.setRGB(x, y,  rgb1);
                }
            }
        }
        return result;
    }

    private int readByte(PrimitiveArray bits, int offset)
    {
        final UInt8 element = (UInt8) bits.getElement(offset);
        return (int)element.getValue();
    }

    /*
     * SystemIcon class represents EnumItemeration of standard system icons.
     */
    public static class SystemIcon extends EnumItem
    {
        public static final SystemIcon SAMPLE = new SystemIcon(32512);
        public static final SystemIcon HAND = new SystemIcon(32513);
        public static final SystemIcon QUES = new SystemIcon(32514);
        public static final SystemIcon BANG = new SystemIcon(32515);
        public static final SystemIcon NOTE = new SystemIcon(32516);
        public static final SystemIcon WINLOGO = new SystemIcon(32517);

        private Icon _icon;

        public SystemIcon(int value)
        {
            super(value);
        }

        public Icon getIcon()
        {
            if (_icon == null)
            {
                _icon = loadStandardIconByID(getValue());
            }
            return _icon;
        }
    }
}
