/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.gdi;

import com.jniwrapper.*;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.ui.User32;

import java.awt.image.BufferedImage;

/**
 * This class provides functionality for working with Windows bitmap images.
 * <br>
 * <br>
 * <b>NOTE:</b> For painting operations, a Bitmap should be selected to the
 * current device context using {@link com.jniwrapper.win32.gdi.DC#selectObject(Bitmap)}.
 *
 * @author Serge Piletsky
 */
public class Bitmap extends GdiObject
{
    private static final String FUNCTION_CREATE_BITMAP = "CreateBitmap";
    private static final String FUNCTION_CREATE_BITMAP_INDIRECT = "CreateBitmapIndirect";
    private static final String FUNCTION_CREATE_COMPATIBLE_BITMAP = "CreateCompatibleBitmap";
    private static final String FUNCTION_LOAD_BITMAP = "LoadBitmap";
    private static final String FUNCTION_GET_DIBITS = "GetDIBits";
    private static final String FUNCTION_CREATE_DIBITMAP = "CreateDIBitmap";

    public Bitmap()
    {
    }

    public Bitmap(long value)
    {
        super(value);
    }

    /**
     * Creates a bitmap with the specified width, height, and color format.
     * Analogue of <code>CreateBitmap</code> Windows API function.
     *
     * @param width the bitmap width in pixels.
     * @param height the bitmap height in pixels.
     * @param planes the number of color planes used by the device.
     * @param bitsPerPel the number of bits required to identify the color of a single pixel.
     * @param colorData an array of color data used to set the colors in a rectangle of pixels.
     * Scan lines of the image data must be word-aligned.
     * Scan lines that are not word-aligned must be padded with zeros).
     * If the parameter is <b>null</b>, the contents of the new bitmap is undefined.
     */
    public Bitmap(long width, long height, int planes, int bitsPerPel, UInt32[] colorData)
    {
        Function function = Gdi32.get(FUNCTION_CREATE_BITMAP);
        Parameter[] param = new Parameter[5];
        param[0] = new Int(width);
        param[1] = new Int(height);
        param[2] = new UInt(planes);
        param[3] = new UInt(bitsPerPel);
        if (colorData == null)
        {
            param[4] = new Handle();
        }
        else
        {
            ComplexArray arr = new ComplexArray(colorData);
            param[4] = new Pointer(arr);
        }
        function.invoke(this, param);
    }

    /**
     * Creates a bitmap with the specified width, height, and color format.
     * Analogue of <code>CreateBitmapIndirect</code> Windows API function.
     *
     * @param width the width of bitmap in pixels
     * @param height the height of bitmap in pixels
     * @param widthByte number of bytes in each scan line
     * @param planes count of color planes
     * @param bitsPixel number of bits to indicate pixel's color.
     */
    public Bitmap(long width, long height, long widthByte, long planes, long bitsPixel)
    {
        BitmapStructure bitmapStruct = new BitmapStructure(width, height, widthByte, planes, bitsPixel);
        Function function = Gdi32.get(FUNCTION_CREATE_BITMAP_INDIRECT);
        function.invoke(this, new Pointer(bitmapStruct));
    }

    /**
     * Creates a bitmap compatible with specified device context.
     * Analogue of <code>CreateCompatibleBitmap</code> Windows API function.
     *
     * @param hDC the current device context.
     * @param width the bitmap width in pixels.
     * @param height the bitmap height in pixels.
     */
    public Bitmap(DC hDC, int width, int height)
    {
        Function function = Gdi32.get(FUNCTION_CREATE_COMPATIBLE_BITMAP);
        function.invoke(this, hDC, new Int(width), new Int(height));
    }


    /**
     * Class PredefinedBitmap enumeration represents predefined windows bitmaps.
     */
    public static class PredefinedBitmap extends EnumItem
    {
        public static final PredefinedBitmap BTNCORNERS = new PredefinedBitmap(32758);
        public static final PredefinedBitmap BTSIZE = new PredefinedBitmap(32761);
        public static final PredefinedBitmap CHECK = new PredefinedBitmap(32760);
        public static final PredefinedBitmap CHECKBOXES = new PredefinedBitmap(32759);
        public static final PredefinedBitmap CLOSE = new PredefinedBitmap(32754);
        public static final PredefinedBitmap COMBO = new PredefinedBitmap(32738);
        public static final PredefinedBitmap DNARROW = new PredefinedBitmap(32752);
        public static final PredefinedBitmap DNARROWD = new PredefinedBitmap(32742);
        public static final PredefinedBitmap DNARROWI = new PredefinedBitmap(32736);
        public static final PredefinedBitmap LFARROW = new PredefinedBitmap(32750);
        public static final PredefinedBitmap LFARROWD = new PredefinedBitmap(32740);
        public static final PredefinedBitmap LFARROWI = new PredefinedBitmap(32734);
        public static final PredefinedBitmap MNARROW = new PredefinedBitmap(32739);
        public static final PredefinedBitmap REDUCE = new PredefinedBitmap(32749);
        public static final PredefinedBitmap REDUCED = new PredefinedBitmap(32746);
        public static final PredefinedBitmap RESTORE = new PredefinedBitmap(32747);
        public static final PredefinedBitmap RESTORED = new PredefinedBitmap(32744);
        public static final PredefinedBitmap RGARROW = new PredefinedBitmap(32751);
        public static final PredefinedBitmap RGARROWD = new PredefinedBitmap(32741);
        public static final PredefinedBitmap RGARROWI = new PredefinedBitmap(32735);
        public static final PredefinedBitmap SIZE = new PredefinedBitmap(32766);
        public static final PredefinedBitmap UPARROW = new PredefinedBitmap(32753);
        public static final PredefinedBitmap UPARROWD = new PredefinedBitmap(32743);
        public static final PredefinedBitmap UPARROWI = new PredefinedBitmap(32737);
        public static final PredefinedBitmap ZOOM = new PredefinedBitmap(32748);
        public static final PredefinedBitmap ZOOMD = new PredefinedBitmap(32745);

        private PredefinedBitmap(int value)
        {
            super(value);
        }
    }

    /**
     * Creates predefined bitmap.
     * Analogue of <code>LoadBitmap</code> Windows API function.
     * <br>
     * <br>
     * <b>NOTE:</b> Use this constructor to load predefined bitmap(s).
     * To load bitmaps(s) from a file use <code>Bitmap(String fileName)</code>.
     *
     * @param predefinedBitmap the name of predefined bitmap.
     */
    public Bitmap(PredefinedBitmap predefinedBitmap)
    {
        Function function = Gdi32.get(FUNCTION_LOAD_BITMAP);
        function.invoke(this, new Handle(), new UInt32(predefinedBitmap.getValue()));
    }

    /**
     * Creates bitmap from external BMP file.
     *
     * @param fileName the path and file name to a BMP file.
     */
    public Bitmap(String fileName)
    {
        loadFromFile(fileName);
    }

    public void loadFromFile(String fileName)
    {
        this.setValue(User32.loadResourceFromFile(fileName, ImageType.BITMAP.getValue()).getValue());
    }

    public static final int DIB_RGB_COLORS = 0; // color table in RGBs
    public static final int DIB_PAL_COLORS = 1; // color table in palette indices

    public static long getDIBits(DC dc, Bitmap bitmap, int startScan, int scanLines,
                                 Pointer bits, BitmapInfo bitmapInfo, int usage)
    {
        Int result = new Int();
        Function function = Gdi32.get(FUNCTION_GET_DIBITS);
        function.invoke(result, new Parameter[]
        {
            dc,
            bitmap,
            new UInt(startScan),
            new UInt(scanLines),
            bits,
            new Pointer(bitmapInfo),
            new UInt(usage)
        });
        return result.getValue();
    }

    public long getDIBits(DC dc, int startScan, int scanLines, Pointer bits, BitmapInfo bitmapInfo, int usage)
    {
        return getDIBits(dc, this, startScan, scanLines, bits, bitmapInfo, usage);
    }

    public static final int CBM_INIT = 0x04;   // initialize bitmap

    public static Bitmap createDIBitmap(DC dc, BitmapInfoHeader bitmapInfoHeader,
                                        int init, Pointer data, BitmapInfo bitmapInfo, int usage)
    {
        Bitmap result = new Bitmap();
        Function function = Gdi32.get(FUNCTION_CREATE_DIBITMAP);
        function.invoke(result, new Parameter[]
        {
            dc,
            new Pointer(bitmapInfoHeader),
            new UInt16(init),
            data,
            new Pointer(bitmapInfo),
            new UInt(usage),
        });
        return result;
    }

    public static class Compression extends EnumItem
    {
        public static final Compression RGB = new Compression(0);
        public static final Compression RLE8 = new Compression(1);
        public static final Compression RLE4 = new Compression(2);
        public static final Compression BITFIELDS = new Compression(3);
        public static final Compression JPEG = new Compression(4);
        public static final Compression PNG = new Compression(5);

        protected Compression(int value)
        {
            super(value);
        }
    }

    /**
     * Converts windows Bitmap to {@link java.awt.image.BufferedImage}.
     *
     * @return image Image instance.
     */
    public BufferedImage toImage()
    {
        BitmapStructure bitmap = new BitmapStructure();
        getObject(this, bitmap);
        WindowDC dc = new WindowDC(null);
        final int bitCount = (int)bitmap.getBitsPixel();
        BitmapInfo bitmapInfo = bitCount >= 16 ? new BitmapInfo() : new BitmapInfo(bitCount);
        BitmapInfoHeader bitmapInfoHeader = bitmapInfo.getBitmapInfoHeader();
        final int width = (int)bitmap.getBitmapWidth();
        final int height = (int)bitmap.getBitmapHeight();
        // fill bitmap info
        bitmapInfoHeader.setWidth(width);
        bitmapInfoHeader.setHeight(height);
        bitmapInfoHeader.setPlanes(1);
        bitmapInfoHeader.setBitCount(bitCount);
        bitmapInfoHeader.setCompression(Compression.RGB);
        if (bitCount == 4 || bitCount == 8)
        {
            final int clrUsed = 1 << bitCount;
            bitmapInfoHeader.setClrUsed(clrUsed);
            bitmapInfoHeader.setClrImportant(clrUsed);
        }
        // retrieve bytes count
        final Pointer NULL = new Pointer(null, true);
        getDIBits(dc, 0, height, NULL, bitmapInfo, DIB_RGB_COLORS);
        // retrieve bytes from bitmap
        int bytesCount = (int)bitmapInfoHeader.getSizeImage();
        PrimitiveArray bits = new PrimitiveArray(UInt8.class, bytesCount);
        getDIBits(dc, 0, height, new Pointer(bits), bitmapInfo, DIB_RGB_COLORS);

        dc.release();
        // transform data
        PrimitiveArray colors = bitmapInfo.getColors();
        BufferedImage result = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        int index = 0;
        for (int y = height - 1; y >= 0; y--)
        {
            for (int x = 0; x < width; x++)
            {
                int rgb = 0;
                switch (bitCount)
                {
                    case 8: // 256 colors, used pallete
                        {
                            RGBQuad color = (RGBQuad)colors.getElement(index++);
                            rgb = color.getRGB();
                            break;
                        }
                    case 16:// 2 bytes per pixel, 5-5-5 bits per pixel
                        {
                            int byte1 = readByte(bits, index++);
                            int byte2 = readByte(bits, index++);
                            int word = byte2 << 8 | byte1;
                            int b = (word & 0x1F) << 3;
                            int g = (word & 0x3E0) << 6;
                            int r = (word & 0x7C00) << 9;
                            rgb = b | g | r;
                            break;
                        }
                    case 24:// 3 bytes per pixel
                    case 32:// 4 bytes per pixel, high byte used as alpha
                        {
                            int b = readByte(bits, index++);
                            int g = readByte(bits, index++);
                            int r = readByte(bits, index++);
                            rgb = b | g << 8 | r << 16;
                            if (bitCount == 32)
                            {
                                int alpha = readByte(bits, index++);
                                // XP returns valid alpha-channel, but other OS'es return 0 always
                                if (alpha == 0)
                                {
                                    alpha = 0xFF;
                                }
                                rgb |= alpha << 24;
                            }
                        }
                }
                result.setRGB(x, y, rgb);
            }
        }
        return result;
    }

    private int readByte(PrimitiveArray bits, int offset)
    {
        final UInt8 element = (UInt8)bits.getElement(offset);
        return (int)element.getValue();
    }
}
