/** \file 2D_Math.h
    Generic 2D Vector math.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_2D_MATH
#define H_2D_MATH

//#include <math.h>    // sqrt()
#include <CON_Utils.h>


struct fVector2D;


/** A 2 Dimensional vector (x,y) of integers. 
    Provides the useful vector math routines.
*/
struct Vector2D
{
   int x,y;

   int getX() const { return x; }
   int getY() const { return y; }

// =====================================
// Constructors
// =====================================

   Vector2D(const Vector2D& v)         { x = v.x; y = v.y; }
   Vector2D(const fVector2D& v);
   Vector2D(const int i)               { x = y = i; }
   Vector2D(const int _x=0, const int _y=0) { x = _x; y = _y; }
   Vector2D(const int i[2])            { x = i[0]; y = i[1]; }

#ifdef _WINDEF_
   Vector2D(const POINT& p)            { x = p.x; y = p.y; }
   POINT toPOINT() const               { POINT p = {x, y}; return p; }
#endif

// =====================================
// Access grants
// =====================================

   const int& operator[](const int i) const  { return (&x)[i]; }
   int& operator[](const int i)              { return (&x)[i]; }

// =====================================
// Assignment operators
// =====================================

   Vector2D& operator+=(const Vector2D& v)   { x += v.x; y += v.y; return *this; }
   Vector2D& operator-=(const Vector2D& v)   { x -= v.x; y -= v.y; return *this; }
   Vector2D& operator*=(const Vector2D& v)   { x *= v.x; y *= v.y; return *this; }
   Vector2D& operator*=(const int s)         { x *= s;   y *= s;   return *this; }
   Vector2D& operator*=(const float s)       { x = (int) (x * s);  y = (int) (y * s);   return *this; }
   Vector2D& operator/=(const Vector2D& v)   { x /= v.x; y /= v.y; return *this; }
   Vector2D& operator/=(const int s)         { x /= s;   y /= s;   return *this; }
   Vector2D& operator/=(const float s)       { x = (int) (x / s);  y = (int) (y / s);   return *this; }

// =====================================
// Unary operators
// =====================================

   const Vector2D& operator+() const         { return *this; }
   Vector2D operator-() const                { return Vector2D(-x, -y); }

// =====================================
// Binary operators
// =====================================

// Addition and subtraction
   friend Vector2D operator+(const Vector2D&, const Vector2D&);
   friend Vector2D operator-(const Vector2D&, const Vector2D&);

// Scalar multiplication and division
   friend Vector2D operator*(const Vector2D&, const int);
   friend Vector2D operator*(const Vector2D&, const float);
   friend Vector2D operator*(const int,   const Vector2D&);
   friend Vector2D operator*(const float, const Vector2D&);
   friend Vector2D operator/(const Vector2D&, const int);
   friend Vector2D operator/(const Vector2D&, const float);

// Equality
   friend int operator==(const Vector2D&, const Vector2D&);
   friend int operator!=(const Vector2D&, const Vector2D&);

// Vector dominance
   friend int operator< (const Vector2D&, const Vector2D&);
   friend int operator<=(const Vector2D&, const Vector2D&);
   friend int operator> (const Vector2D&, const Vector2D&);
   friend int operator>=(const Vector2D&, const Vector2D&);

// Dot Product
   friend int operator*(const Vector2D&, const Vector2D&);

// Length-related functions
   int SquareMagnitude() const               { return x*x + y*y; }
   float Magnitude() const                   { return (float) sqrt((double)SquareMagnitude()); }

// Returns vector with same direction and unit length
   Vector2D Normalized() const               { return (*this / Magnitude()); }
   Vector2D& Normalize()                     { return *this /= Magnitude(); }

// Return min/max component of the input vector
   int Min() const                           { return ::Min(x, y); }
   int Max() const                           { return ::Max(x, y); }

// Return memberwise min/max of input vectors
   friend Vector2D Minimize(const Vector2D&, const Vector2D&);
   friend Vector2D Maximize(const Vector2D&, const Vector2D&);
};


/** A 2 Dimensional vector (x,y) of floating point values. 
    Provides the useful vector math routines.
*/
struct fVector2D
{
   float x,y;

   float getX() const { return x; }
   float getY() const { return y; }

// =====================================
// Constructors
// =====================================

   fVector2D(const fVector2D& v)              { x = v.x; y = v.y; }
   fVector2D(const Vector2D& v)               { x = (float) v.x; y = (float) v.y; }
   fVector2D(const float f)                   { x = y = f; }
   fVector2D(const float _x=0.0f, const float _y=0.0f) { x = _x; y = _y; }
   fVector2D(const float f[2])                { x = f[0]; y = f[1]; }

#ifdef _WINDEF_
   POINT toPOINT() const                      { POINT p = {(int) x, (int) y}; return p; }
#endif

// =====================================
// Access grants
// =====================================

   const float& operator[](const int i) const { return (&x)[i]; }
   float& operator[](const int i)             { return (&x)[i]; }

// =====================================
// Assignment operators
// =====================================

   fVector2D& operator+=(const fVector2D& v)  { x += v.x; y += v.y; return *this; }
   fVector2D& operator-=(const fVector2D& v)  { x -= v.x; y -= v.y; return *this; }
   fVector2D& operator*=(const fVector2D& v)  { x *= v.x; y *= v.y; return *this; }
   fVector2D& operator*=(const float s)       { x *= s;   y *= s;   return *this; }
   fVector2D& operator/=(const fVector2D& v)  { x /= v.x; y /= v.y; return *this; }
   fVector2D& operator/=(const float s)       { x /= s;   y /= s;   return *this; }

// =====================================
// Unary operators
// =====================================

   const fVector2D& operator+() const         { return *this; }
   fVector2D operator-() const                { return fVector2D(-x, -y); }

// =====================================
// Binary operators
// =====================================

// Addition and subtraction
   friend fVector2D operator+(const fVector2D&, const fVector2D&);
   friend fVector2D operator-(const fVector2D&, const fVector2D&);

// Scalar multiplication and division
   friend fVector2D operator*(const fVector2D&, const float);
   friend fVector2D operator*(const float, const fVector2D&);
   friend fVector2D operator/(const fVector2D&, const float);

// Equality
   friend int operator==(const fVector2D&, const fVector2D&);
   friend int operator!=(const fVector2D&, const fVector2D&);

// Vector dominance
   friend int operator< (const fVector2D&, const fVector2D&);
   friend int operator<=(const fVector2D&, const fVector2D&);
   friend int operator> (const fVector2D&, const fVector2D&);
   friend int operator>=(const fVector2D&, const fVector2D&);

// Dot Product
   friend float operator*(const fVector2D&, const fVector2D&);

// Length-related functions
   float SquareMagnitude() const              { return x*x + y*y; }
   float Magnitude() const                    { return (float) sqrt(SquareMagnitude()); }

// Returns vector with same direction and unit length
   fVector2D Normalized() const               { return (*this / Magnitude()); }
   fVector2D& Normalize()                     { return *this /= Magnitude(); }

// Return min/max component of the input vector
   float Min() const                          { return ::Min(x, y); }
   float Max() const                          { return ::Max(x, y); }

// Return memberwise min/max of input vectors
   friend fVector2D Minimize(const fVector2D&, const fVector2D&);
   friend fVector2D Maximize(const fVector2D&, const fVector2D&);
};


class Rect2D
{
public:
   Rect2D() : tl(0,0), br(-1,-1) {}
   Rect2D(const Vector2D& topleft, const Vector2D& bottomright) : tl(topleft), br(bottomright) {}
   Rect2D(const int x1, const int y1, const int x2, const int y2) : tl(x1,y1), br(x2,y2) {}
   Rect2D(const Rect2D& r) : tl(r.tl), br(r.br) {}

#ifdef _WINDEF_
   Rect2D(const RECT& r) : tl(r.left, r.top), br(r.right-1, r.bottom-1) {}
   RECT  toRECT() const     { RECT r = {tl.x, tl.y, br.x+1, br.y+1}; return r; }
#endif

   int getWidth()  const    { return (br.x - tl.x + 1); }
   int getHeight() const    { return (br.y - tl.y + 1); }

   Vector2D getSize() const { return Vector2D(getWidth(), getHeight()); }

   Rect2D& operator+=(const Vector2D& v)  { this->tl += v; this->br += v; return *this; }
   Rect2D& operator-=(const Vector2D& v)  { this->tl -= v; this->br -= v; return *this; }

   friend Rect2D operator+(const Rect2D&, const Vector2D&);
   friend Rect2D operator-(const Rect2D&, const Vector2D&);

   friend int operator==(const Rect2D&, const Rect2D&);
   friend int operator!=(const Rect2D&, const Rect2D&);

// intersection
   friend Rect2D operator*(const Rect2D&, Rect2D&); 
   Rect2D& operator*=(const Rect2D&);

   int inside(const Vector2D& v) const
   {
      return (v.x >= tl.x && v.x <= br.x && v.y >= tl.y && v.y <= br.y);
   }

   int overlapping(const Rect2D& r) const
   {
      return (tl.x <= r.br.x && tl.y <= r.br.y && br.x >= r.tl.x && br.y >= r.tl.y);
   }

   Vector2D tl,br;
};


// *************************************
// inline Implemetations
// *************************************

inline Vector2D::Vector2D(const fVector2D& v)
{
   x = (int) v.x; y = (int) v.y;
}


inline Vector2D operator+(const Vector2D& u, const Vector2D& v)
{
   return Vector2D(u.x+v.x, u.y+v.y);
}


inline Vector2D operator-(const Vector2D& u, const Vector2D& v)
{
   return Vector2D(u.x-v.x, u.y-v.y); 
}


inline Vector2D operator*(const Vector2D& v, const int s)
{
   return Vector2D(v.x*s, v.y*s); 
}


inline Vector2D operator*(const Vector2D& v, const float s)
{
   return Vector2D((int)(v.x*s), (int)(v.y*s));
}


inline Vector2D operator*(const int s, const Vector2D& v)
{
   return Vector2D(s*v.x, s*v.y);
}


inline Vector2D operator*(const float s, const Vector2D& v)
{
   return Vector2D((int)(s*v.x), (int)(s*v.y));
}


inline Vector2D operator/(const Vector2D& v, const int s)
{
   return Vector2D(v.x/s, v.y/s); 
}


inline Vector2D operator/(const Vector2D& v, const float s)
{
   return Vector2D((int) (v.x/s), (int)(v.y/s)); 
}


inline int operator==(const Vector2D& u, const Vector2D& v)
{
  return (u.x == v.x && u.y == v.y);
}


inline int operator!= (const Vector2D& u, const Vector2D& v)
{
  return (!(u==v));
}


inline int operator<(const Vector2D& u, const Vector2D& v)
{
   return (u.x < v.x && u.y < v.y);
}


inline int operator<=(const Vector2D& u, const Vector2D& v)
{
   return (u.x <= v.x && u.y <= v.y);
}


inline int operator>(const Vector2D& u, const Vector2D& v)
{
  return (!(u<=v));
}


inline int operator>=(const Vector2D& u, const Vector2D& v)
{
  return (!(u<v));
}


inline int operator*(const Vector2D& u, const Vector2D& v)
{
   return u.x*v.x + u.y*v.y; 
}


inline Vector2D Minimize(const Vector2D& u, const Vector2D& v)
{
  return Vector2D(Min(u.x,v.x),Min(u.y,v.y));
}

inline Vector2D Maximize(const Vector2D& u, const Vector2D& v)
{
  return Vector2D(Max(u.x,v.x),Max(u.y,v.y));
}


inline fVector2D operator+(const fVector2D& u, const fVector2D& v)
{
   return fVector2D(u.x+v.x, u.y+v.y);
}


inline fVector2D operator-(const fVector2D& u, const fVector2D& v)
{
   return fVector2D(u.x-v.x, u.y-v.y); 
}


inline fVector2D operator*(const fVector2D& v, const float s)
{
   return fVector2D(v.x*s, v.y*s);
}


inline fVector2D operator*(const float s, const fVector2D& v)
{
   return fVector2D(s*v.x, s*v.y);
}


inline fVector2D operator/(const fVector2D& v, const float s)
{
   return fVector2D(v.x/s, v.y/s); 
}


inline int operator==(const fVector2D& u, const fVector2D& v)
{
  return (ISEQ(u.x,v.x) && ISEQ(u.y,v.y));
}


inline int operator!= (const fVector2D& u, const fVector2D& v)
{
  return (!(u==v));
}


inline int operator<(const fVector2D& u, const fVector2D& v)
{
   return (u.x < v.x && u.y < v.y);
}


inline int operator<=(const fVector2D& u, const fVector2D& v)
{
   return (u.x <= v.x && u.y <= v.y);
}


inline int operator>(const fVector2D& u, const fVector2D& v)
{
  return (!(u<=v));
}


inline int operator>=(const fVector2D& u, const fVector2D& v)
{
  return (!(u<v));
}


inline float operator*(const fVector2D& u, const fVector2D& v)
{
   return u.x*v.x + u.y*v.y; 
}


inline fVector2D Minimize(const fVector2D& u, const fVector2D& v)
{
  return fVector2D(Min(u.x,v.x),Min(u.x,v.x));
}


inline fVector2D Maximize(const fVector2D& u, const fVector2D& v)
{
  return fVector2D(Max(u.x,v.x),Max(u.x,v.x));
}


inline Rect2D operator+(const Rect2D& r, const Vector2D& v)
{
   return (Rect2D(r.tl + v, r.br + v));
}

inline Rect2D operator-(const Rect2D& r, const Vector2D& v)
{
   return (Rect2D(r.tl - v, r.br - v));
}


inline int operator==(const Rect2D& q, const Rect2D& r)
{
   return q.tl == r.tl && q.br == r.br;
}


inline int operator!=(const Rect2D& q, const Rect2D& r)
{
   return !(q == r);
}


inline Rect2D operator*(const Rect2D& q, const Rect2D& r)
{
   Rect2D t(Maximize(q.tl, r.tl), Minimize(q.br, r.br));

   if (t.tl > t.br)
      return Rect2D();  // no intersection

   return t;
}


inline Rect2D& Rect2D::operator*=(const Rect2D& r)
{
   return *this = *this * r; 
}


#endif // H_2D_MATH