// HexEdit.cpp : Defines the class behaviors for the application.
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <locale.h>

#include <afxadv.h>     // for CRecentFileList
#include <io.h>         // for _access()

// #include <afxhtml.h>    // for CHtmlView

#define COMPILE_MULTIMON_STUBS 1
#include <MultiMon.h>   // for multiple monitor support
#undef COMPILE_MULTIMON_STUBS
#include "HexEdit.h"
#include "HexFileList.h"
#include "boyer.h"

#include "MainFrm.h"
#include "ChildFrm.h"
#include "HexEditDoc.h"
#include "HexEditView.h"
#include "Dialog.h"
#include "About.h"
#include "Misc.h"

// The following is not in a public header
extern BOOL AFXAPI AfxFullPath(LPTSTR lpszPathOut, LPCTSTR lpszFileIn);

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// The following is the subkey of HKEY_CLASSES_ROOT that is used to enable the
// "Open with HexEdit" shell shortcut menu option.
static const char *HEXEDIT_SUBKEY = "*\\shell\\HexEdit";
static const char *HEXEDIT_SUBSUBKEY  = "*\\shell\\HexEdit\\command";

/////////////////////////////////////////////////////////////////////////////
// CHexEditApp
const char *CHexEditApp::szHexEditClassName = "HexEditMDIFrame";

BEGIN_MESSAGE_MAP(CHexEditApp, CWinApp)
        ON_COMMAND(CG_IDS_TIPOFTHEDAY, ShowTipOfTheDay)
        //{{AFX_MSG_MAP(CHexEditApp)
        ON_COMMAND(ID_APP_ABOUT, OnAppAbout)
        ON_COMMAND(ID_OPTIONS, OnOptions)
        ON_COMMAND(ID_OPTIONS2, OnOptions2)
        ON_COMMAND(ID_RECORD, OnMacroRecord)
        ON_COMMAND(ID_PLAY, OnMacroPlay)
        ON_UPDATE_COMMAND_UI(ID_PLAY, OnUpdateMacroPlay)
        ON_UPDATE_COMMAND_UI(ID_RECORD, OnUpdateMacroRecord)
        ON_COMMAND(ID_PROPERTIES, OnProperties)
        ON_COMMAND(ID_OPTIONS3, OnOptions3)
        ON_COMMAND(ID_MULTI_PLAY, OnMultiPlay)
        ON_COMMAND(ID_HELP_EMAIL, OnHelpEmail)
        ON_UPDATE_COMMAND_UI(ID_HELP_EMAIL, OnUpdateHelpEmail)
	ON_COMMAND(ID_TEST, OnTest)
	ON_COMMAND(ID_HELP_WEB, OnHelpWeb)
	ON_UPDATE_COMMAND_UI(ID_HELP_WEB, OnUpdateHelpWeb)
        ON_UPDATE_COMMAND_UI(ID_MULTI_PLAY, OnUpdateMacroPlay)
	ON_COMMAND(ID_WEB_PAGE, OnWebPage)
	//}}AFX_MSG_MAP
        // Standard file based document commands
        ON_COMMAND(ID_FILE_NEW, OnFileNew)
        ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
        ON_COMMAND_EX_RANGE(ID_FILE_MRU_FILE1, ID_FILE_MRU_FILE16, OnOpenRecentFile)
        ON_COMMAND(ID_APP_EXIT, OnAppExit)
        // Standard print setup command
        ON_COMMAND(ID_FILE_PRINT_SETUP, OnFilePrintSetup)
        // Message sent to mainthread
        ON_THREAD_MESSAGE(WM_USER+12, OnBGSearchFinished)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CHexEditApp construction

#ifdef _DEBUG
// Memory allocation hook function used for debugging memory allocation exceptions
#if _MSC_VER <= 1100
    static int alloc_hook( int allocType, void *userData, size_t size, int blockType, 
        long requestNumber, const char *filename, int lineNumber)
#else
    // For some reason this declaration has changed slightly (filename is now uchar)
    static int alloc_hook( int allocType, void *userData, size_t size, int blockType, 
        long requestNumber, const unsigned char *filename, int lineNumber)
#endif
{
    BOOL retval = TRUE;         // TRUE = proceed normally, FALSE = alloc error
    // Change retval to 0 (in debugger) before returning to test error handling
    switch (allocType)
    {
    case _HOOK_ALLOC:           // malloc/calloc <= C++ new
        return retval;
    case _HOOK_REALLOC:         // realloc
        return retval;
    case _HOOK_FREE:            // free <= C++ delete & delete[]
        return retval;
    }
    ASSERT(0);
    return TRUE;
}
#endif

const short CHexEditApp::version_ = 111;

CHexEditApp::CHexEditApp()
{
//    version_ = 101;

    // Add a memory allocation hook for debugging purposes
    // (Does nothing in release version.)
    _CrtSetAllocHook(&alloc_hook);
    recording_ = FALSE;
    playing_ = 0;
    pview_ = NULL;
    refresh_off_ = false;
    open_plf_ = open_oem_plf_ = NULL;
    last_cb_size_ = 0;
#ifndef NDEBUG
    // Make default capacity for mac_ vector small to force reallocation sooner.
    // This increases likelihood of catching bugs related to reallocation.
    mac_.reserve(2);
#else
    // Pre-allocate room for 128 elts for initial speed
    mac_.reserve(128);
#endif

    // Set up the default colour scheme
    default_partitions_.clear();
    default_partitions_.push_back(partn("ASCII text", RGB(0,0,0), "32:126"));
    default_partitions_.push_back(partn("Special (TAB,LF,CR)", RGB(0,128,0), "9,10,13"));
    default_partitions_.push_back(partn("ANSI text", RGB(0,0,128), "130:140,145:156,159:255"));
    default_partitions_.push_back(partn("Other", RGB(128,0,0), "0:255"));

    pboyer_ = NULL;
}

/////////////////////////////////////////////////////////////////////////////
// The one and only CHexEditApp object

CHexEditApp theApp;

UINT CHexEditApp::wm_hexedit = ::RegisterWindowMessage("HexEditOpenMessage");

/////////////////////////////////////////////////////////////////////////////
// CHexEditApp initialization

BOOL CHexEditApp::InitInstance()
{
        // Test is we are running under NT
        OSVERSIONINFO osvi;
        osvi.dwOSVersionInfoSize = sizeof(osvi);
        GetVersionEx(&osvi);
        is_nt_ = osvi.dwPlatformId == VER_PLATFORM_WIN32_NT;

        // Determine if multiple monitor supported (Win 98 or NT >= 5.0)
        // Note that Windows 95 is 4.00 and Windows 98 is 4.10
        mult_monitor_ = 
            (osvi.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS &&
                 osvi.dwMajorVersion == 4 &&
                 osvi.dwMinorVersion >= 10  ||
                 osvi.dwMajorVersion >= 5) && ::GetSystemMetrics(SM_CMONITORS) > 1;
//        mult_monitor_ = osvi.dwMajorVersion >= 5;

        // Work out if there is a previous instance running
        hwnd_1st_ = ::FindWindow(szHexEditClassName, NULL);

        InitEbcdic();                   // Read any EBCDIC conversion table and validate conversions

        // Set the locale to the native environment -- hopefully the MSC run-time
        // code does something suitable.  (This is currently just for thousands sep.)
        setlocale(LC_ALL, "");      // Set to native locale

        // Work out if we appear to be in US for spelling changes
        is_us_ = strnicmp("English_United States", ::setlocale(LC_COLLATE, NULL), 20) == 0;

        // Work out if there is a HEXEDIT.INI file in the windows directory
        char ininame[MAX_PATH + 13];
        if (GetWindowsDirectory(ininame, MAX_PATH) != 0)
        {
            size_t len = strlen(ininame);
            if (ininame[len-1] != '\\')
                strcat(ininame, "\\");
            strcat(ininame, "HexEdit.INI");
        }
        else
            ininame[0] = '\0';

        if (_access(ininame, 0) != 0)
            SetRegistryKey("ECSoftware");      // INI file not found so use registry

        LoadOptions();

        // Construct property pages for later use in options dialog
        // Note: Had to put this here since before the constructor (being
        // called in the app constructor) was trying to reference resources
        // before that was possible.
        opt_general.Construct(CGeneralPage::IDD);
        opt_sysdisplay.Construct(CSysDisplayPage::IDD);
        opt_macro.Construct(CMacroPage::IDD);
        opt_display.Construct(CDisplayPage::IDD);
        opt_defaults.Construct(CDefaultsPage::IDD);
        opt_partitions.Construct(CPartitionsPage::IDD, is_us_ ? IDS_COLORS : 0);

        // Enable help button for all pages (must be done after construction)
        opt_general.m_psp.dwFlags |= PSP_HASHELP;
        opt_sysdisplay.m_psp.dwFlags |= PSP_HASHELP;
        opt_macro.m_psp.dwFlags |= PSP_HASHELP;
        opt_display.m_psp.dwFlags |= PSP_HASHELP;
        opt_defaults.m_psp.dwFlags |= PSP_HASHELP;
        opt_partitions.m_psp.dwFlags |= PSP_HASHELP;

#ifdef _AFXDLL
        Enable3dControls();                     // Call this when using MFC in a shared DLL
#else
        Enable3dControlsStatic();       // Call this when linking to MFC statically
#endif

        // xxx add options for number of MRU files
        LoadStdProfileSettings(10);  // Load standard INI file options (including MRU)

        // Register the application's document templates.  Document templates
        //  serve as the connection between documents, frame windows and views.

//        CMultiDocTemplate* pDocTemplate;
        // Made pDocTemplate a member variable so I can use it to get all documents of the app.
        // A better way may have been to get the CDocTemplate from m_pDocManger->m_templateList?
        m_pDocTemplate = new CMultiDocTemplate(
                IDR_HEXEDTYPE,
                RUNTIME_CLASS(CHexEditDoc),
                RUNTIME_CLASS(CChildFrame), // custom MDI child frame
                RUNTIME_CLASS(CHexEditView));
        AddDocTemplate(m_pDocTemplate);

        // create main MDI Frame window
        CMainFrame* pMainFrame = new CMainFrame;
        if (!pMainFrame->LoadFrame(IDR_MAINFRAME))
                return FALSE;
        m_pMainWnd = pMainFrame;
        m_pMainWnd->DragAcceptFiles();

        // Make sure previous instance is no minimised before opening files in it
        if (hwnd_1st_ != 0 && one_only_)
        {
            ::BringWindowToTop(hwnd_1st_);
            WINDOWPLACEMENT wp;
            wp.length = sizeof(wp);
            if (::GetWindowPlacement(hwnd_1st_, &wp) &&
                (wp.showCmd == SW_MINIMIZE || wp.showCmd == SW_SHOWMINIMIZED))
            {
                ::ShowWindow(hwnd_1st_, SW_RESTORE);
            }
        }

        // This used to be after the command line parsing but was moved here so that
        // when files are opened the size of the main window is known so that they
        // are opened in sensible sizes and positions.
        pMainFrame->ShowWindow(m_nCmdShow);
//          pMainFrame->ShowWindow(SW_SHOWMAXIMIZED);

        // Parse command line for standard shell commands, DDE, file open
        // Note: CCommandLineInfo class was overridden with the CCommandLineParser
        // class - this can open as many files as specified on the command line
        // but does this in CCommandLineParse::ParseParam rather than storing
        // the file name for CWinApp::ProcessShellCommands (and sets
        // m_nShellCommand to FileNothing).
        CCommandLineParser cmdInfo;
        ParseCommandLine(cmdInfo);

        // Don't create empty document by default
        if (cmdInfo.m_nShellCommand == CCommandLineInfo::FileNew)
            cmdInfo.m_nShellCommand = CCommandLineInfo::FileNothing;

        // Dispatch commands specified on the command line
        if (!ProcessShellCommand(cmdInfo))
                return FALSE;

        // If only one copy allowed terminate now
        if (hwnd_1st_ != 0 && one_only_)
            return FALSE;

        pMainFrame->UpdateWindow();

        // CG: This line inserted by 'Tip of the Day' component.
        ShowTipAtStartup();

        if (GetProfileInt("MainFrame", "ShowPropDlg", 0) == 1)
        {
            ASSERT(pMainFrame->pprop_ == NULL);
            pMainFrame->pprop_ = new CPropSheet(prop_page_);

            // Set focus back to the active view (if there is a view open)
            CHexEditView *pview = GetView();    // The active view (or NULL if none)
            if (pview != NULL)
                pview->SetFocus();
        }
        if (GetProfileInt("MainFrame", "ShowCalcDlg", 0) == 1)
        {
            ASSERT(pMainFrame->pcalc_ != NULL);
            ASSERT(pMainFrame->pcalc_->m_hWnd == 0);
            pMainFrame->pcalc_->Create();

            pMainFrame->pcalc_->ShowWindow(SW_RESTORE);
            pMainFrame->pcalc_->visible_ = TRUE;
            pMainFrame->pcalc_->edit_.Put();
            pMainFrame->pcalc_->FixFileButtons();
        }

        RunAutoExec();

        return TRUE;
}

void CHexEditApp::OnFileNew() 
{
    CWinApp::OnFileNew();
    if (recording_ && mac_.size() > 0 && (mac_.back()).ktype == km_focus)
    {
        // We don't want focus change recorded (see CHexEditView::OnSetFocus)
        mac_.pop_back();
    }
    SaveToMacro(km_new);
}

void CHexEditApp::OnFileOpen() 
{
    CFileDialog dlgFile(TRUE);

    // Set up buffer where selected file names are stored & default to none
    char all_files[8192];
    all_files[0] = '\0';
    dlgFile.m_ofn.lpstrFile = all_files;
    dlgFile.m_ofn.nMaxFile = sizeof(all_files)-1;

    // Set up the title of the dialog
    CString title;
    VERIFY(title.LoadString(AFX_IDS_OPENFILE));
    dlgFile.m_ofn.lpstrTitle = title;

    // Get the name of the all files filter from resource file
    CString allFilter;
    VERIFY(allFilter.LoadString(AFX_IDS_ALLFILTER));

    // Set up a single file for "All files"
    CString strFilter;
    // xxx add more filters here?
    strFilter += allFilter;
    strFilter += (TCHAR)'\0';   // next string please
    strFilter += _T("*.*");
    strFilter += (TCHAR)'\0';   // last string
    dlgFile.m_ofn.lpstrFilter = strFilter;
//    dlgFile.m_ofn.lpstrFilter = NULL;

    if (!dir_open_.IsEmpty())
        dlgFile.m_ofn.lpstrInitialDir = dir_open_;

    // Set flags: file must exist to be returned, Read only check box enabled
    dlgFile.m_ofn.Flags |= (OFN_ALLOWMULTISELECT | OFN_FILEMUSTEXIST | OFN_SHOWHELP);
    dlgFile.m_ofn.Flags &= ~OFN_HIDEREADONLY;

    if (dlgFile.DoModal() != IDOK)
    {
        mac_error_ = 2;
        return;
    }

    // For some absolutely ridiculous reason if the user only selects one file
    // the full filename is just returned rather than as specified for more than
    // one file with OFN_ALLOWMULTISELECT.  So we need special code to handle this.
    if (dlgFile.m_ofn.nFileOffset < strlen(all_files))
    {
        ASSERT(all_files[dlgFile.m_ofn.nFileOffset-1] == '\\');
        all_files[strlen(all_files)+1] = '\0';      // Ensure double null ended
        all_files[dlgFile.m_ofn.nFileOffset-1] = '\0';
    }

    // Get directory name as first part of files buffer
    CString dir_name(all_files);
    dir_open_ = dir_name;

    // Get file names separated by nul char ('\0') stop on 2 nul chars
    for (const char *pp = all_files + strlen(all_files) + 1; *pp != '\0';
                     pp = pp + strlen(pp) + 1)
    {
        CString filename;
        if (dir_name[dir_name.GetLength()-1] == '\\')
            filename = dir_name + pp;
        else
            filename= dir_name + "\\" + pp;
        CHexEditDoc *pdoc;
        if ((pdoc = (CHexEditDoc*)(AfxGetApp()->OpenDocumentFile(filename))) != NULL)
        {
            if ((dlgFile.m_ofn.Flags & OFN_READONLY) != 0)
                pdoc->readonly_ = TRUE;

            // Completed OK so store in macro if recording
            if (recording_ && mac_.size() > 0 && (mac_.back()).ktype == km_focus)
            {
                // We don't want focus change recorded (see CHexEditView::OnSetFocus)
                mac_.pop_back();
            }
            SaveToMacro(km_open, filename);
        }
        else
            mac_error_ = 20;
    }

    // BG search finished message may be lost if modeless dlg running
    CheckBGSearchFinished();
}

BOOL CHexEditApp::OnOpenRecentFile(UINT nID)
{
    // I had to completely override OnOpenRecentFile (copying most of the code from
    // CWinApp::OnOpenRecentFile) since it does not return FALSE if the file is not
    // found -- in fact it always returns TRUE.  I'd say this is an MFC bug.
    ASSERT(m_pRecentFileList != NULL);

    ASSERT(nID >= ID_FILE_MRU_FILE1);
    ASSERT(nID < ID_FILE_MRU_FILE1 + (UINT)m_pRecentFileList->GetSize());
    int nIndex = nID - ID_FILE_MRU_FILE1;
    ASSERT((*m_pRecentFileList)[nIndex].GetLength() != 0);

    TRACE2("MRU: open file (%d) '%s'.\n", (nIndex) + 1,
                    (LPCTSTR)(*m_pRecentFileList)[nIndex]);

    // Save file name now since its index will be zero after it's opened
    CString file_name = (*m_pRecentFileList)[nIndex];

    if (OpenDocumentFile((*m_pRecentFileList)[nIndex]) == NULL)
    {
        m_pRecentFileList->Remove(nIndex);
        mac_error_ = 10;                        // User has been told that file could not be found
        return FALSE;
    }
    else
    {
        // Completed OK so store in macro if recording
        if (recording_ && mac_.size() > 0 && (mac_.back()).ktype == km_focus)
        {
            // We don't want focus change recorded (see CHexEditView::OnSetFocus)
            mac_.pop_back();
        }
        SaveToMacro(km_open, file_name);
        return TRUE;
    }
    ASSERT(0);                                  // We shouldn't get here
}

void CHexEditApp::OnFilePrintSetup()
{
    CPrintDialog pd(TRUE);
    DoPrintDialog(&pd);
    SaveToMacro(km_print_setup);

    // BG search finished message may be lost if modeless dlg running
    CheckBGSearchFinished();
}

void CHexEditApp::OnBGSearchFinished(WPARAM wParam, LPARAM lParam)
{
    ASSERT(bg_search_);

    // Make sure doc is valid (may have been closed before we got the message)
    POSITION posn = m_pDocTemplate->GetFirstDocPosition();
    while (posn != NULL)
    {
        CHexEditDoc *pdoc = dynamic_cast<CHexEditDoc *>(m_pDocTemplate->GetNextDoc(posn));
        ASSERT(pdoc != NULL);
        if (pdoc == (void *)lParam)
        {
            pdoc->BGSearchFinished();
            return;
        }
    }
}

// Call BGSearchFinished for all documents to make sure all views have been updated
void CHexEditApp::CheckBGSearchFinished()
{
    if (!bg_search_) return;

    POSITION posn = m_pDocTemplate->GetFirstDocPosition();
    while (posn != NULL)
    {
        CHexEditDoc *pdoc = dynamic_cast<CHexEditDoc *>(m_pDocTemplate->GetNextDoc(posn));
        ASSERT(pdoc != NULL);
        pdoc->BGSearchFinished();
    }
}

void CHexEditApp::OnMacroRecord() 
{
    recording_ = !recording_;
    // Don't clear the last macro until we get the first key of the next
    if (recording_) no_keys_ = TRUE;            // Flag to say new macro started

#ifdef _DEBUG
    // Some commands call other commands as part of there task.  This can acc-
    // identally result in extra commands being unintentionally recorded. To
    // help detect this we display a trace message (in debug version) if there
    // is more than one entry in the macro vector - to detect problems we must
    // record a macro with just one command & check the debug window when run.
    if (!recording_ && mac_.size() > 1)
        TRACE1("Macro size is %ld\n", (long)mac_.size());
#endif
    // If invoked from toolbar make sure focus returns to view
    CHexEditView *pview = GetView();    // The active view (or NULL if none)
    if (pview != NULL && pview != pview->GetFocus())
        pview->SetFocus();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    ASSERT(mm != NULL);
    mm->bb_edit_rec_.LoadBitmaps(recording_ ? IDB_EDIT_RECS : IDB_EDIT_RECU,
                                    IDB_EDIT_RECD,0,IDB_EDIT_RECX);
    mm->bb_edit_rec_.Invalidate();
}

void CHexEditApp::OnUpdateMacroRecord(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(recording_);
}

void CHexEditApp::OnMacroPlay() 
{
    ASSERT(!recording_);
    macro_play();

    // Set focus to currently active view
    CHexEditView *pview = GetView();    // The active view (or NULL if none)
    if (pview != NULL && pview != pview->GetFocus())
        pview->SetFocus();
}

void CHexEditApp::OnMultiPlay() 
{
    ASSERT(!recording_);

    CMultiplay dlg;
    dlg.plays_ = plays_;

    if (dlg.DoModal() == IDOK)
    {
        plays_ = dlg.plays_;
        macro_play(plays_);
    }

    // BG search finished message may be lost if modeless dlg running
    CheckBGSearchFinished();

    // Set focus to currently active view
    CHexEditView *pview = GetView();    // The active view (or NULL if none)
    if (pview != NULL && pview != pview->GetFocus())
        pview->SetFocus();
}

void CHexEditApp::OnUpdateMacroPlay(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(!recording_ && mac_.size() > 0);
}

void CHexEditApp::RunAutoExec()
{
    ASSERT(mac_dir_.Right(1) == "\\");
    CString filename = mac_dir_ + "autoexec.hem"; 
    std::vector<key_macro> mac;
    CString comment;
    int halt_lev;
    long plays;

    if (::_access(filename, 0) == 0 &&
        macro_load(filename, &mac, comment, halt_lev, plays))
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText(comment);
        macro_play(plays, &mac, halt_lev);

        // Set focus to currently active view
        CHexEditView *pview = GetView();    // The active view (or NULL if none)
        if (pview != NULL && pview != pview->GetFocus())
            pview->SetFocus();
    }
}

void CHexEditApp::OnAppExit()
{
    SaveToMacro(km_exit);
    CWinApp::OnAppExit();
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditApp commands

int CHexEditApp::ExitInstance()
{
    // Save "save on exit" option if it has changed
    if (save_exit_ != orig_save_exit_)
        WriteProfileInt("Options", "SaveExit", save_exit_ ? 1 : 0);

    // Save other options if saving on exit
    if (save_exit_)
        SaveOptions();

    // If we wrote something big to the clipboard ask the user if they want to delete it
    if (last_cb_size_ > 100000 && ::OpenClipboard(HWND(0)))
    {
        HANDLE hh;                              // Handle to clipboard memory
        char *pp;                               // Pointer to clipboard text
        size_t len;                             // Size of text on clipboard

        if ((hh = ::GetClipboardData(CF_TEXT)) != NULL &&
            (pp = reinterpret_cast<char *>(::GlobalLock(hh))) != NULL &&
            (len = strlen(pp)) == last_cb_size_)
        {
            if (AfxMessageBox("Leave data on clipboard?", MB_YESNO) != IDYES)
                ::EmptyClipboard();
        }

        ::CloseClipboard();
    }

    if (pboyer_ != NULL)
        delete pboyer_;

    return CWinApp::ExitInstance();
}

BOOL CHexEditApp::PreTranslateMessage(MSG* pMsg) 
{
    return CWinApp::PreTranslateMessage(pMsg);
}


BOOL CHexEditApp::OnIdle(LONG lCount) 
{
	return CWinApp::OnIdle(lCount);
}

// Starts bg searches in all documents except the active doc (passed in pp)
void CHexEditApp::StartSearches(CHexEditDoc *pp)
{
    POSITION posn = m_pDocTemplate->GetFirstDocPosition();
    while (posn != NULL)
    {
        CHexEditDoc *pdoc = dynamic_cast<CHexEditDoc *>(m_pDocTemplate->GetNextDoc(posn));
        ASSERT(pdoc != NULL);
        if (pdoc != pp)
            pdoc->StartSearch();
    }
}

void CHexEditApp::NewSearch(const unsigned char *pat, size_t len, BOOL icase, int tt)
{
    CSingleLock s2(&appdata_, TRUE);

    if (pboyer_ != NULL) delete pboyer_;
    pboyer_ = new boyer(pat, len);

    icase_ = icase;
    text_type_ = tt;
}

void CHexEditApp::LoadStdProfileSettings(UINT nMaxMRU)
{
    // Call the base clas in case it does anything we haven't though of
    CWinApp::LoadStdProfileSettings(nMaxMRU);

    if (m_pRecentFileList != NULL)
    {
        delete m_pRecentFileList;
        m_pRecentFileList = NULL;
    }
    if (nMaxMRU != 0)
    {
        // create file MRU since nMaxMRU not zero
        m_pRecentFileList = new CHexFileList(0, "RecentFiles", "File%d", nMaxMRU);
        m_pRecentFileList->ReadList();
    }
}

// Retrieve options from .INI file/registry
void CHexEditApp::LoadOptions()
{
    switch(GetProfileInt("Options", "SaveExit", 2))
    {
    case 0:
        orig_save_exit_ = save_exit_ = FALSE;
        break;
    case 2:
        // Option not found so make sure it is saved next time
        orig_save_exit_ = !(save_exit_ = TRUE);
        break;
    default:
        orig_save_exit_ = save_exit_ = TRUE;
    }
    open_restore_ = GetProfileInt("MainFrame", "Restore", 1) ? TRUE : FALSE;
    hex_ucase_ = GetProfileInt("Options", "UpperCaseHex", 1) ? TRUE : FALSE;
    nice_addr_ = GetProfileInt("Options", "NiceAddresses", 1) ? TRUE : FALSE;
    backup_ = GetProfileInt("Options", "CreateBackup", 0) ? TRUE : FALSE;
    bg_search_ = GetProfileInt("Options", "BackgroundSearch", 1) ? TRUE : FALSE;
    one_only_ = GetProfileInt("Options", "OneInstanceOnly", 1) ? TRUE : FALSE;
    large_cursor_ = GetProfileInt("Options", "LargeCursor", 0) ? TRUE : FALSE;

    mac_dir_ = GetProfileString("MacroOptions", "Folder");
    if (mac_dir_.IsEmpty())
    {
        char fullname[_MAX_PATH];           // Full name of exe directory
        char *end;                          // End of path of help file

        strncpy(fullname, m_pszHelpFilePath, sizeof(fullname)-1);
        fullname[sizeof(fullname)-1] = '\0';
        if ((end = strrchr(fullname, '\\')) == NULL && (end = strrchr(fullname, ':')) == NULL)
            end = fullname;
        else
            ++end;
        *end = '\0';
        mac_dir_ = fullname;
    }
    ASSERT(mac_dir_.Right(1) == "\\");
    refresh_ = GetProfileInt("MacroOptions", "Refresh", 1);
    num_secs_ = GetProfileInt("MacroOptions", "Seconds", 5);
    num_keys_ = GetProfileInt("MacroOptions", "Keys", 1);
    num_plays_ = GetProfileInt("MacroOptions", "Plays", 1);
    refresh_bars_ = GetProfileInt("MacroOptions", "StatusBarUpdate", 1) ? TRUE : FALSE;
    refresh_props_ = GetProfileInt("MacroOptions", "PropertiesUpdate", 0) ? TRUE : FALSE;
    halt_level_ = GetProfileInt("MacroOptions", "ErrorHaltLevel", 1);
    plays_ = GetProfileInt("MacroOptions", "NumPlays", 1);

    open_max_ = GetProfileInt("Options", "OpenMax", 0) ? TRUE : FALSE;
    open_autofit_ = GetProfileInt("Options", "OpenAutoFit", 0) ? TRUE : FALSE;
    open_dec_addr_ = GetProfileInt("Options", "OpenDecimalAddresses", 0) ? TRUE : FALSE;
    open_display_char_ = GetProfileInt("Options", "OpenDisplayChar", 1) ? TRUE : FALSE;
    open_ebcdic_ = GetProfileInt("Options", "OpenEBCDIC", 0) ? TRUE : FALSE;
    open_control_ = GetProfileInt("Options", "OpenControlChars", 0) ? TRUE : FALSE;
    open_graphic_ = GetProfileInt("Options", "OpenGraphicChars", 0) ? TRUE : FALSE;
    open_oem_ = GetProfileInt("Options", "OpenOemChars", 0) ? TRUE : FALSE;

    CString strFont = GetProfileString("Options", "OpenFont");      // Font info string (fields are comma sep.)
    CString strFace;                                            // Font FaceName from string
    CString strHeight;                                          // Font height as string
    AfxExtractSubString(strFace, strFont, 0, ',');
    AfxExtractSubString(strHeight, strFont, 1, ',');
    if (!strFace.IsEmpty())
    {
        open_plf_ = new LOGFONT;
        memset((void *)open_plf_, '\0', sizeof(*open_plf_));
        strncpy(open_plf_->lfFaceName, strFace, LF_FACESIZE-1);
        open_plf_->lfFaceName[LF_FACESIZE-1] = '\0';
        open_plf_->lfHeight = atol(strHeight);
        if (open_plf_->lfHeight < 2 || open_plf_->lfHeight > 100)
			open_plf_->lfHeight = 16;
        open_plf_->lfCharSet = ANSI_CHARSET;           // Only allow ANSI character set fonts
    }
    else
        open_plf_ = NULL;

    strFont = GetProfileString("Options", "OpenOemFont");     // Font info for oem font
    AfxExtractSubString(strFace, strFont, 0, ',');
    AfxExtractSubString(strHeight, strFont, 1, ',');
    if (!strFace.IsEmpty())
    {
        open_oem_plf_ = new LOGFONT;
        memset((void *)open_oem_plf_, '\0', sizeof(*open_oem_plf_));
        strncpy(open_oem_plf_->lfFaceName, strFace, LF_FACESIZE-1);
        open_oem_plf_->lfFaceName[LF_FACESIZE-1] = '\0';
        open_oem_plf_->lfHeight = atol(strHeight);
        if (open_oem_plf_->lfHeight < 2 || open_oem_plf_->lfHeight > 100)
			open_oem_plf_->lfHeight = 18;
        open_oem_plf_->lfCharSet = OEM_CHARSET;            // Only allow OEM/IBM character set fonts
    }
    else
        open_oem_plf_ = NULL;

    // Get the default colours
    open_partitions_.clear();
    for (int ii = 1;  ; ++ii)
    {
        partn pp;
        CString f1, f2, f3;
        f1.Format("Name%d", ii);
        f2.Format("Colour%d", ii);
        f3.Format("Range%d", ii);

        if (!GetColours("DefaultColours", f1, f2, f3, pp))
            break;

        // Add the record (name, colour, range) to end of vector
        open_partitions_.push_back(pp);
    }
    // If none found set up defaults
    if (ii == 1)
        open_partitions_ = default_partitions_;

    open_allow_mods_ = GetProfileInt("Options", "OpenAllowMods", 0) ? TRUE : FALSE;
    open_insert_ = GetProfileInt("Options", "OpenInsert", 0) ? TRUE : FALSE;

    open_rowsize_ = GetProfileInt("Options", "OpenColumns", 16);
    if (open_rowsize_ < 4 || open_rowsize_ > CHexEditView::max_buf) open_rowsize_ = 4;
    open_group_by_ = GetProfileInt("Options", "OpenGrouping", 4);
    if (open_group_by_ < 2) open_group_by_ = 2;
    open_offset_ = GetProfileInt("Options", "OpenOffset", 0);
    if (open_offset_ < 0 || open_offset_ >= open_rowsize_) open_offset_ = 0;

    // Always default back to little-endian for decimal & fp pages
    prop_dec_endian_ = FALSE;
    prop_fp_endian_ = FALSE;
    prop_ibmfp_endian_ = TRUE;

    // Restore more subtle things such as window placement etc
    find_x_ = GetProfileInt("Window-Settings", "FindX", -30000);
    find_y_ = GetProfileInt("Window-Settings", "FindY", -30000);
    calc_x_ = GetProfileInt("Window-Settings", "CalcX", -30000);
    calc_y_ = GetProfileInt("Window-Settings", "CalcY", -30000);

    // Last settings for property sheet
    prop_x_ = GetProfileInt("Window-Settings", "PropX", -30000);
    prop_y_ = GetProfileInt("Window-Settings", "PropY", -30000);
    prop_page_ = GetProfileInt("Property-Settings", "PropPage", 0);
    prop_dec_signed_ = GetProfileInt("Property-Settings", "DecFormat", 1);
    prop_fp_format_  = GetProfileInt("Property-Settings", "FPFormat", 1);
    prop_ibmfp_format_  = GetProfileInt("Property-Settings", "IBMFPFormat", 1);

    // Restore default file dialog directories
    dir_open_ = GetProfileString("DefaultDirectories", "Open");
    dir_read_ = GetProfileString("DefaultDirectories", "Read");
    dir_write_ = GetProfileString("DefaultDirectories", "Write");
}

bool CHexEditApp::GetColours(const char *section, const char *key1, const char *key2,
                const char *key3, partn &retval)
{
    CString name = GetProfileString(section, key1, "");
    if (name.IsEmpty()) return false;

    COLORREF colour = (COLORREF)GetProfileInt(section, key2, 0);
    CString range = GetProfileString(section, key3, "0:255");

    retval = partn(name, colour, range);
    return true;
}

void CHexEditApp::SetColours(const char *section, const char *key1, const char *key2,
                const char *key3, const partn &v)
{
    // Write the range name and RGB value
    WriteProfileString(section, key1, v.name);
    WriteProfileInt(section, key2, v.col);

    // Convert the range itself to a string and write it
    std::ostringstream strstr;
    strstr << v.range;
    WriteProfileString(section, key3, strstr.str().c_str());
}

// Save global options to .INI file/registry
void CHexEditApp::SaveOptions()
{
    // Save general options
    WriteProfileInt("Options", "SaveExit", save_exit_ ? 1 : 0);
    WriteProfileInt("Options", "UpperCaseHex", hex_ucase_ ? 1 : 0);
    WriteProfileInt("Options", "NiceAddresses", nice_addr_ ? 1 : 0);
    WriteProfileInt("Options", "CreateBackup", backup_ ? 1 : 0);
    WriteProfileInt("Options", "BackgroundSearch", bg_search_ ? 1 : 0);
    WriteProfileInt("Options", "OneInstanceOnly", one_only_ ? 1 : 0);
    WriteProfileInt("Options", "LargeCursor", large_cursor_ ? 1 : 0);

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (mm != NULL)
        mm->SaveFrameOptions();

    // Save macro options
    ASSERT(mac_dir_.Right(1) == "\\");
    WriteProfileString("MacroOptions", "Folder", mac_dir_);
    WriteProfileInt("MacroOptions", "Refresh", refresh_);
    WriteProfileInt("MacroOptions", "Seconds", num_secs_);
    WriteProfileInt("MacroOptions", "Keys", num_keys_);
    WriteProfileInt("MacroOptions", "Plays", num_plays_);
    WriteProfileInt("MacroOptions", "StatusBarUpdate", refresh_bars_ ? 1 : 0);
    WriteProfileInt("MacroOptions", "PropertiesUpdate", refresh_props_ ? 1 : 0);
    WriteProfileInt("MacroOptions", "ErrorHaltLevel", halt_level_);
    WriteProfileInt("MacroOptions", "NumPlays", plays_);

    // Save default window options
    WriteProfileInt("Options", "OpenMax", open_max_ ? 1 : 0);
    WriteProfileInt("Options", "OpenAutoFit", open_autofit_ ? 1 : 0);
    WriteProfileInt("Options", "OpenDecimalAddresses", open_dec_addr_ ? 1 : 0);
    WriteProfileInt("Options", "OpenDisplayChar", open_display_char_ ? 1 : 0);
    WriteProfileInt("Options", "OpenEBCDIC", open_ebcdic_ ? 1 : 0);
    WriteProfileInt("Options", "OpenControlChars", open_control_ ? 1 : 0);
    WriteProfileInt("Options", "OpenGraphicChars", open_graphic_ ? 1 : 0);
    WriteProfileInt("Options", "OpenOemChars", open_oem_ ? 1 : 0);

    CString strFont;
    if (open_plf_ != NULL)
    {
        strFont.Format("%s,%ld", open_plf_->lfFaceName, open_plf_->lfHeight);
        WriteProfileString("Options", "OpenFont", strFont);
    }
    if (open_oem_plf_ != NULL)
    {
        strFont.Format("%s,%ld", open_oem_plf_->lfFaceName, open_oem_plf_->lfHeight);
        WriteProfileString("Options", "OpenOemFont", strFont);
    }

    WriteProfileInt("Options", "OpenAllowMods", open_allow_mods_ ? 1 : 0);
    WriteProfileInt("Options", "OpenInsert", open_insert_ ? 1 : 0);

    WriteProfileInt("Options", "OpenColumns", open_rowsize_);
    WriteProfileInt("Options", "OpenGrouping", open_group_by_);
    WriteProfileInt("Options", "OpenOffset", open_offset_);

    // Save info about modeless dialogs (find and properties)
    if (find_y_ != -30000)
    {
        WriteProfileInt("Window-Settings", "FindX", find_x_);
        WriteProfileInt("Window-Settings", "FindY", find_y_);
    }
    if (calc_y_ != -30000)
    {
        WriteProfileInt("Window-Settings", "CalcX", calc_x_);
        WriteProfileInt("Window-Settings", "CalcY", calc_y_);
    }
    if (prop_y_ != -30000)
    {
        WriteProfileInt("Window-Settings", "PropX", prop_x_);
        WriteProfileInt("Window-Settings", "PropY", prop_y_);
    }
    WriteProfileInt("Property-Settings", "PropPage", prop_page_);
    WriteProfileInt("Property-Settings", "DecFormat", prop_dec_signed_);
    WriteProfileInt("Property-Settings", "FPFormat", prop_fp_format_);
    WriteProfileInt("Property-Settings", "IBMFPFormat", prop_ibmfp_format_);

    // Save directories for file dialogs
    WriteProfileString("DefaultDirectories", "Open", dir_open_);
    WriteProfileString("DefaultDirectories", "Read", dir_read_);
    WriteProfileString("DefaultDirectories", "Write", dir_write_);
}

void CHexEditApp::OnProperties() 
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // Save this so that we get it before the page activation "keystroke"
    // (km_prop_file, km_prop_char, etc)
    SaveToMacro(km_prop);
    if (mm->pprop_ == NULL)
        mm->pprop_ = new CPropSheet(prop_page_);
    else
        mm->pprop_->SetFocus();
    mm->pprop_->UpdateWindow(); // Needed for when prop dlg opened in a macro
}

void CHexEditApp::OnOptions() 
{
    display_options(&opt_general);
}

void CHexEditApp::OnOptions2()
{
    display_options(&opt_display);
}

void CHexEditApp::OnOptions3() 
{
    display_options(&opt_macro);
}

void CHexEditApp::display_options(CPropertyPage *display_page /*=NULL*/)
{
    // Construct property sheet + its pages
    COptSheet options_sheet(_T("HexEdit Options"));

    get_options();

    options_sheet.AddPage(&opt_general);
    options_sheet.AddPage(&opt_sysdisplay);
    options_sheet.AddPage(&opt_macro);
    if (GetView() != NULL)
    {
        options_sheet.AddPage(&opt_partitions);
        options_sheet.AddPage(&opt_display);
    }
    options_sheet.AddPage(&opt_defaults);
    if (display_page != NULL)
        options_sheet.SetActivePage(display_page);

    options_sheet.m_psh.dwFlags &= ~(PSH_HASHELP);      // Turn off help button

    options_sheet.DoModal();
//    if (options_sheet.DoModal() == IDOK)
//      SaveToMacro(km_options, (long)display_page);
//    else
//      mac_error_ = 2;

    // BG search finished message may be lost if modeless dlg running
    CheckBGSearchFinished();
}

// Get current options into member variables for pages
void CHexEditApp::get_options()
{
    CHexEditView *pview;                // The active view (or NULL if none)
    char buf[_MAX_PATH + 3];            // Stores value of key (not used)
    long buf_size = sizeof(buf);        // Size of buffer and returned key value

    // Init dialog fields with option values
    opt_general.shell_open_ = 
        RegQueryValue(HKEY_CLASSES_ROOT, HEXEDIT_SUBSUBKEY, buf, &buf_size) == ERROR_SUCCESS;
    opt_general.one_only_ = one_only_;
    opt_general.bg_search_ = bg_search_;
    opt_general.save_exit_ = save_exit_;
    opt_general.backup_ = backup_;

    opt_sysdisplay.open_restore_ = open_restore_;
    opt_sysdisplay.hex_ucase_ = hex_ucase_;
    opt_sysdisplay.nice_addr_ = nice_addr_;
    opt_sysdisplay.large_cursor_ = large_cursor_;

    opt_macro.refresh_ = refresh_;
    opt_macro.num_secs_ = num_secs_;
    opt_macro.num_keys_ = num_keys_;
    opt_macro.num_plays_ = num_plays_;
    opt_macro.refresh_bars_ = refresh_bars_;
    opt_macro.refresh_props_ = refresh_props_;
    opt_macro.halt_level_ = halt_level_;

    if ((pview = GetView()) != NULL)
    {
        // Set up the Window Colours page
        opt_partitions.partitions_ = pview->partitions_;

        // Set up the Window Display page
        // Get the name of the active view window and save in window_name_
        ((CMDIChildWnd *)((CMainFrame *)AfxGetMainWnd())->MDIGetActive())->
            GetWindowText(opt_display.window_name_);

        // Init page variables based on current view values
        opt_display.autofit_ = pview->autofit_;
        opt_display.char_ = pview->display_char_;
        opt_display.cols_ = pview->rowsize_;
        opt_display.grouping_ = pview->group_by_;
//      opt_display.offset_ = pview->offset_;
        opt_display.offset_ = (pview->rowsize_ - pview->offset_)%pview->rowsize_;

        // Save view caret/selection position to restore later
        end_base_ = pview->GetSelAddr(start_addr_, end_addr_);
    }

    opt_defaults.open_max_ = open_max_;
    opt_defaults.open_autofit_ = open_autofit_;
    opt_defaults.open_dec_addr_ = open_dec_addr_;
    opt_defaults.open_display_char_ = open_display_char_;
    opt_defaults.open_ebcdic_ = open_ebcdic_;
    opt_defaults.open_control_ = open_control_;
    opt_defaults.open_graphic_ = open_graphic_;
    opt_defaults.open_oem_ = open_oem_;

    opt_defaults.open_plf_ = open_plf_;
    opt_defaults.open_oem_plf_ = open_oem_plf_;

    opt_defaults.open_allow_mods_ = open_allow_mods_;
    opt_defaults.open_insert_ = open_insert_;
}

void CHexEditApp::set_general()
{
    char buf[_MAX_PATH + 3];            // Stores value of key (not used)
    long buf_size = sizeof(buf);        // Size of buffer and returned key value

    save_exit_ = opt_general.save_exit_;

    if (opt_general.shell_open_ != 
        (RegQueryValue(HKEY_CLASSES_ROOT, HEXEDIT_SUBSUBKEY, buf, &buf_size) == ERROR_SUCCESS))
    {
        // Option has been changed (turned on or off)
        if (opt_general.shell_open_)
        {
            // Create the registry entries that allow "Open with HexEdit" on shortcut menus
            const char *s1 = "Open with HexEdit";
//            const char *s2 = "HexEdit.exe %1";
            char s2[_MAX_PATH + 3];         // Full name of exe file
            char *end;                      // End of path of help file

            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
            ASSERT(aa->m_pszHelpFilePath != NULL);
            strncpy(s2, aa->m_pszHelpFilePath, sizeof(s2)-15);
            s2[sizeof(s2)-15] = '\0';
            if ((end = strrchr(s2, '\\')) == NULL && (end = strrchr(s2, ':')) == NULL)
                end = s2;
            else
                ++end;
            strcpy(end, "HexEdit.exe %1");

            RegSetValue(HKEY_CLASSES_ROOT, HEXEDIT_SUBKEY, REG_SZ, s1, strlen(s1));
            RegSetValue(HKEY_CLASSES_ROOT, HEXEDIT_SUBSUBKEY, REG_SZ, s2, strlen(s2));
        }
        else
        {
            RegDeleteKey(HKEY_CLASSES_ROOT, HEXEDIT_SUBSUBKEY); // Delete subkey first (for NT)
            RegDeleteKey(HKEY_CLASSES_ROOT, HEXEDIT_SUBKEY);    // Delete registry entries
        }
    }
    if (bg_search_ != opt_general.bg_search_)
    {
        if (opt_general.bg_search_)
        {
            bg_search_ = TRUE;

            // Create bg search thread for all docs
            POSITION posn = m_pDocTemplate->GetFirstDocPosition();
            while (posn != NULL)
            {
                CHexEditDoc *pdoc = dynamic_cast<CHexEditDoc *>(m_pDocTemplate->GetNextDoc(posn));
                ASSERT(pdoc != NULL);
                pdoc->CreateThread();
                if (pboyer_ != NULL)        // If a search has already been done do bg search this file
                    pdoc->StartSearch();
            }

            // We don't need change thread priority for active doc here since
            // when active view regains focus this is done automatically.
        }
        else
        {
            // Kill bg searches for all docs
            POSITION posn = m_pDocTemplate->GetFirstDocPosition();
            while (posn != NULL)
            {
                CHexEditDoc *pdoc = dynamic_cast<CHexEditDoc *>(m_pDocTemplate->GetNextDoc(posn));
                ASSERT(pdoc != NULL);

                // Kill bg thread (and clean up data structures)
                pdoc->KillThread();

                // Signal all views to remove display of search strings
                CBGSearchHint bgsh(FALSE);
                pdoc->UpdateAllViews(NULL, 0, &bgsh);
            }

            // Change this after killing threads to avoid assertions
            bg_search_ = FALSE;
        }
    }

    one_only_ = opt_general.one_only_;
    backup_ = opt_general.backup_;
}

void CHexEditApp::set_sysdisplay()
{
    open_restore_ = opt_sysdisplay.open_restore_;

    if (large_cursor_ != opt_sysdisplay.large_cursor_)
    {
        large_cursor_ = opt_sysdisplay.large_cursor_;

        // Change caret of all views
        CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
        CMDIChildWnd *nextc;    // Loops through all MDI child frames

        // Find all view windows
        for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
             nextc != NULL;
             nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
        {
            CHexEditView *pview = dynamic_cast<CHexEditView *>(nextc->GetActiveView());
            // Note pview may be NULL if in print preview
            if (pview != NULL && pview->IsKindOf(RUNTIME_CLASS(CHexEditView)))
            {
                if (large_cursor_)
                    pview->BlockCaret();
                else
                    pview->LineCaret();
            }
        }
    }

    if (hex_ucase_ != opt_sysdisplay.hex_ucase_ || nice_addr_ != opt_sysdisplay.nice_addr_)
    {
        hex_ucase_ = opt_sysdisplay.hex_ucase_;
        nice_addr_ = opt_sysdisplay.nice_addr_;

        // Invalidate all views to change display of hex digits
        CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
        CMDIChildWnd *nextc;    // Loops through all MDI child frames

        // Invalidate all view windows
        for (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
             nextc != NULL;
             nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
        {
            CHexEditView *pview = dynamic_cast<CHexEditView *>(nextc->GetActiveView());
            // Note pview may be NULL if in print preview
            if (pview != NULL && pview->IsKindOf(RUNTIME_CLASS(CHexEditView)))
            {
                pview->recalc_display(); // xxx nec.?
                pview->DoInvalidate();
            }
        }

        // Make sure text in search control on edit bar has correct case
        if ((mm->m_wndEditBar.GetStyle() & WS_VISIBLE) != 0)
            mm->sec_search_.Redisplay();
    }
}

// Change macro options according to Macro Options page
void CHexEditApp::set_macro()
{
    // Set global options from the macro page values (possibly) changed
    // xxx save any of these option changes to recorded macro?
    refresh_ = opt_macro.refresh_;
    num_secs_ = opt_macro.num_secs_;
    num_keys_ = opt_macro.num_keys_;
    num_plays_ = opt_macro.num_plays_;
    refresh_bars_ = opt_macro.refresh_bars_;
    refresh_props_ = opt_macro.refresh_props_;
    halt_level_ = opt_macro.halt_level_;
}

bool CHexEditApp::set_partitions()
{
    // Update the colours for the current view
    CHexEditView *pview = GetView();    // The active view
    ASSERT(pview != NULL);
    if (pview != NULL)
    {
        pview->set_colours(opt_partitions.partitions_);
        pview->DoInvalidate();
    }
    return true;
}

// Changes the display according to the options changed.
// Returns true if there were any changes or false if none
bool CHexEditApp::set_display()
{
    bool one_done = false;              // Have we already made a change (saving undo info)
    // one_done is used to keep track of the changes made to the view so that
    // "previous_too" flag is set for all changes made at once except for the
    // first.  This allows all the changes made at the same time to be undone
    // with one undo operation, which will make more sense to the user.

    CHexEditView *pview = GetView();    // The active view
    ASSERT(pview != NULL);
    if (pview != NULL)
    {
        if (pview->autofit_ != opt_display.autofit_)
        {
            pview->do_autofit(opt_display.autofit_ ? 1 : 0);
            one_done = true;
        }
        if (pview->display_char_ != opt_display.char_)
        {
            pview->do_chartoggle(opt_display.char_ ? 1 : 0);
            if (one_done) pview->undo_.back().previous_too = true;
            one_done = true;
        }
        if (!opt_display.autofit_ && pview->rowsize_ != opt_display.cols_)
        {
            pview->change_rowsize(opt_display.cols_);
            if (one_done) pview->undo_.back().previous_too = true;
            one_done = true;
        }
        if (pview->group_by_ != opt_display.grouping_)
        {
            pview->change_group_by(opt_display.grouping_);
            if (one_done) pview->undo_.back().previous_too = true;
            one_done = true;
        }
//      if (pview->real_offset_ != opt_display.offset_)
//      {
//          pview->change_offset(opt_display.offset_);
        if (pview->real_offset_ != (opt_display.cols_ - opt_display.offset_)%opt_display.cols_)
        {
            pview->change_offset((opt_display.cols_ - opt_display.offset_)%opt_display.cols_);
            if (one_done) pview->undo_.back().previous_too = true;
            one_done = true;
        }

        if (one_done)
        {
            if (end_base_)
                pview->SetSel(pview->addr2pos(end_addr_), pview->addr2pos(start_addr_), true);
            else
                pview->SetSel(pview->addr2pos(start_addr_), pview->addr2pos(end_addr_));
            pview->recalc_display(); // xxx nec.?
            pview->DoInvalidate();
        }
    }
    return one_done;
}

void CHexEditApp::set_defaults()
{
    // xxx save these option changes to recorded macro?
    open_max_ = opt_defaults.open_max_;
    open_autofit_ = opt_defaults.open_autofit_;
    open_dec_addr_ = opt_defaults.open_dec_addr_;
    open_display_char_ = opt_defaults.open_display_char_;
    open_ebcdic_ = opt_defaults.open_ebcdic_;
    open_control_ = opt_defaults.open_control_;
    open_graphic_ = opt_defaults.open_graphic_;
    open_oem_ = opt_defaults.open_oem_;

    open_plf_ = opt_defaults.open_plf_;
    open_oem_plf_ = opt_defaults.open_oem_plf_;

    open_allow_mods_ = opt_defaults.open_allow_mods_;
    open_insert_ = opt_defaults.open_insert_;
}

void CHexEditApp::ShowTipAtStartup(void)
{
        // CG: This function added by 'Tip of the Day' component.

        CCommandLineInfo cmdInfo;
        ParseCommandLine(cmdInfo);
        if (cmdInfo.m_bShowSplash)
        {
                CTipDlg dlg;
                if (dlg.m_bStartup)
                        dlg.DoModal();
        }

}

void CHexEditApp::ShowTipOfTheDay(void)
{
        // CG: This function added by 'Tip of the Day' component.

        CTipDlg dlg;
        dlg.DoModal();

        // BG search finished message may be lost if modeless dlg running
        CheckBGSearchFinished();

        // This bit not added by the automatic "Tip of the Day" component
        SaveToMacro(km_tips);
}

void CHexEditApp::OnHelpEmail() 
{
    SendEmail();
}

void CHexEditApp::OnUpdateHelpEmail(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(::GetProfileInt("MAIL", "MAPI", 0) == 1);
}

void CHexEditApp::OnWebPage() 
{
    // Go to hexedit web site
    ::BrowseWeb(IDS_WEB_ADDRESS);
}

#if 0 // This dummy doc no longer needed since we no lomger use CHtmlView
// This is all needed I think just to get a dummy document class
class CDummyDoc : public CDocument
{
protected: // create from serialization only
    CDummyDoc() {}
	DECLARE_DYNCREATE(CDummyDoc)
public:
    virtual ~CDummyDoc() {}
protected:
	DECLARE_MESSAGE_MAP()
};

IMPLEMENT_DYNCREATE(CDummyDoc, CDocument)

BEGIN_MESSAGE_MAP(CDummyDoc, CDocument)
END_MESSAGE_MAP()
#endif

void CHexEditApp::OnHelpWeb()
{
#if 0  // Don't use HtmlView, just fire up browser instead
    static CMultiDocTemplate *pDocTemplate = NULL;
    if (pDocTemplate == NULL)
    {
        pDocTemplate = new CMultiDocTemplate(
                IDR_HEXEDTYPE,
                RUNTIME_CLASS(CDummyDoc),
                RUNTIME_CLASS(CChildFrame), // custom MDI child frame
                RUNTIME_CLASS(CHtmlView));
    }

    CDocument *pDoc = pDocTemplate->OpenDocumentFile(NULL);

    ASSERT(pDoc != NULL);
    if (pDoc == NULL) return;

    POSITION pos = pDoc->GetFirstViewPosition();
    if (pos != NULL)
    {
        CHtmlView *pView = (CHtmlView *)pDoc->GetNextView(pos);
        ASSERT_KINDOF(CHtmlView, pView);

        // Go to hexedit web site
	    CString str;
	    VERIFY(str.LoadString(IDS_WEB_HELP));
	    pView->Navigate2(str);
    }
#else
    ::BrowseWeb(IDS_WEB_HELP);
#endif
}

void CHexEditApp::OnUpdateHelpWeb(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(TRUE);
}

// Just for testing things (invoked with ^T)
void CHexEditApp::OnTest() 
{
}

#if 0
/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
        CAboutDlg();

// Dialog Data
        //{{AFX_DATA(CAboutDlg)
        enum { IDD = IDD_ABOUTBOX };
        CString version_;
        //}}AFX_DATA

        // ClassWizard generated virtual function overrides
        //{{AFX_VIRTUAL(CAboutDlg)
        protected:
        virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support
        //}}AFX_VIRTUAL

// Implementation
protected:
        //{{AFX_MSG(CAboutDlg)
                // No message handlers
        //}}AFX_MSG
        DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
        //{{AFX_DATA_INIT(CAboutDlg)
        version_ = _T("");
        //}}AFX_DATA_INIT
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    version_.Format("%g", double(aa->version_)/100.0);
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
        CDialog::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CAboutDlg)
        DDX_Text(pDX, IDC_VERSION, version_);
        //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
        //{{AFX_MSG_MAP(CAboutDlg)
                // No message handlers
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()
#endif

// App command to run the dialog
void CHexEditApp::OnAppAbout()
{
        CAbout aboutDlg;
        aboutDlg.line1_.Format("HexEdit version %g", double(version_)/100.0);
        aboutDlg.line4_ = "Free for non-commercial use.";

        aboutDlg.DoModal();

        // BG search finished message may be lost if modeless dlg running
        CheckBGSearchFinished();
}

/////////////////////////////////////////////////////////////////////////////
// CCommandLineParser member functions

void CCommandLineParser::ParseParam(const TCHAR* pszParam, BOOL bFlag, BOOL bLast)
{
    if (non_std)
    {
        if (bFlag)
        {
            // If there are any flags set then assume this is a standard command line
            non_std = FALSE;
        }
        else
        {
            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

            if (aa->hwnd_1st_ != 0 && aa->one_only_)
            {
                // Get full path name as other instance may have different current directory
                char fullname[_MAX_PATH];
                AfxFullPath(fullname, pszParam);

                ATOM atom = ::GlobalAddAtom(fullname);
                ASSERT(atom != 0);
                if (atom == 0) return;
                DWORD dw = 0;
                ::SendMessageTimeout(aa->hwnd_1st_, aa->wm_hexedit, 0, (LPARAM)atom,
                                     SMTO_ABORTIFHUNG, 2000, &dw);
                ::GlobalDeleteAtom(atom);
                return;
            }
            else if (aa->OpenDocumentFile(pszParam))
            {
                m_nShellCommand = FileNothing;
                return;
            }
        }
    }

    // Do standard command line processing using base class
    CCommandLineInfo::ParseParam(pszParam, bFlag, bLast);
}

/////////////////////////////////////////////////////////////////////////////
// Global functions

// Return a ptr to the active view or NULL if none
CHexEditView *GetView()
{
    // Get view and toggle read only
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CMDIChildWnd *pwind = mm->MDIGetActive();

    // Ignore if there are no views open
    if (pwind != NULL)
    {
        CHexEditView *pv = dynamic_cast<CHexEditView *>(pwind->GetActiveView());
        if (pv != NULL)                         // May be NULL if print preview
        {
            ASSERT_KINDOF(CHexEditView, pv);    // Is this always true (print preview?)
            if (pv->IsKindOf(RUNTIME_CLASS(CHexEditView)))
                return pv;
        }
    }
    return NULL;
}

int HMessageBox(LPCTSTR lpszText, UINT nType /*=MB_OK*/, UINT nIDHelp /*=0*/)
{
    int retval = AfxMessageBox(lpszText, nType, nIDHelp);

    // BG search finished message may be lost while modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();

    return retval;
}

int HMessageBox(UINT nIDPrompt, UINT nType /*=MB_OK*/, UINT nIDHelp /*=-1*/)
{
    int retval = AfxMessageBox(nIDPrompt, nType, nIDHelp);

    // BG search finished message may be lost while modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();

    return retval;
}

