unit Slockfrm;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, registry, RegFrm, Slock, ExtCtrls, Menus;

type
  TfrmMain = class(TForm)
    btnRegister: TButton;
    Slock1: TSlock;
    grpStatus: TGroupBox;
    lblStatus: TLabel;
    lblStatusDisp: TLabel;
    lblDaysLeft: TLabel;
    lblDaysLeftDisp: TLabel;
    lblStartsLeftDisp: TLabel;
    lblExpiryDateDisp: TLabel;
    lblProtType: TLabel;
    lblProtTypeDisp: TLabel;
    lblStartsLeft: TLabel;
    lblExpiryDate: TLabel;
    lblRegisteredTo: TLabel;
    lblRegisteredToDisp: TLabel;
    lblChallengeString: TLabel;
    lblChallengeStringDisp: TLabel;
    bvlStatus: TBevel;
    lblGraceStartsLeft: TLabel;
    lblGraceStartsLeftDisp: TLabel;
    mnuMainMenu: TMainMenu;
    mnmFile: TMenuItem;
    mnpExit: TMenuItem;
    mnmRegister: TMenuItem;
    mnpRegNow: TMenuItem;
    mnmFunction: TMenuItem;
    mnpNormFunc: TMenuItem;
    btnExit: TButton;
    procedure Slock1FirstStart(Sender: TObject);
    procedure Slock1ClockMovedBack(Sender: TObject);
    procedure Slock1RegistryTamper(Sender: TObject);
    procedure Slock1Reminder(Sender: TObject);
    procedure Slock1WrongUnlockCode(Sender: TObject);
    procedure Slock1Expired(Sender: TObject);
    procedure btnRegisterClick(Sender: TObject);
    procedure Slock1Register(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Slock1RegInfoChanged(Sender: TObject);
    procedure Slock1InGracePeriod(Sender: TObject);
    procedure Slock1RegWriteErr(Sender: TObject);
    procedure mnpNormFuncClick(Sender: TObject);
    procedure mnpExitClick(Sender: TObject);
    procedure mnpRegNowClick(Sender: TObject);
    procedure btnExitClick(Sender: TObject);
    procedure Slock1Extend(Sender: TObject);
    procedure Slock1TimedOut(Sender: TObject);
  private
    procedure UpdateStatus;
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.DFM}

{*******************************************************************************
* Procedure : btnRegisterClick                                                 *
********************************************************************************
* Purpose   : Opens up the window in which the registration information can    *
*             be entered into                                                  *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.btnRegisterClick(Sender: TObject);
var
     frmRegister: TfrmRegister;
begin
     // create the registration window
     frmRegister := TfrmRegister.Create(self);

     // set the caption and show the window
     frmRegister.Caption := 'Register ' + Caption;
     frmRegister.ShowModal;

     // release the window
     frmRegister.Free;

     // Update the status
     UpdateStatus;

end; {btnRegisterClick}

{*******************************************************************************
* Procedure : FormCreate                                                       *
********************************************************************************
* Purpose   : We check the current status of the protection in the form create *
*             of the main form. (Slock1.CheckProtection) The internal state of *
*             Slock is set, and we then use 'UpdateStatus' to read the status  *
*             of Slock out into the status display                             *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.FormCreate(Sender: TObject);
begin
     // check the protection of the Slock component
     SLock1.CheckProtection;

     // Update the status display and the state of the Buttons/Menupoints
     UpdateStatus;
end; {FormCreate}

{*******************************************************************************
* Procedure : UpdateStatus                                                     *
********************************************************************************
* Purpose   : Loads the internal status of the Slock component into the status *
*             display. Also sets the state of the menus and buttons            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.UpdateStatus;
begin
     // load the status display
     case Slock1.Status of
     stNotExpired:
          lblStatusDisp.Caption := 'Not Expired';
     stExpired:
          lblStatusDisp.Caption := 'Expired';
     stGrace:
          lblStatusDisp.Caption := 'InGrace Period';
     stRegistered:
          lblStatusDisp.Caption := 'Registered';
     end;

     // load the days/starts left display
     case Slock1.ProtectionType of
     ptStartCount:
          begin
               lblProtTypeDisp.Caption := 'Start Count';
               lblDaysLeftDisp.Caption := 'N/A';
               lblStartsLeftDisp.Caption := IntToStr(Slock1.AllowedStarts);
               lblGraceStartsLeftDisp.Caption := IntToStr(Slock1.GracePeriod);
               lblExpiryDateDisp.Caption := 'N/A';
          end;
     ptDayCount:
          begin
               lblProtTypeDisp.Caption := 'Day Count';
               lblDaysLeftDisp.Caption := IntToStr(Slock1.AllowedDays);
               lblStartsLeftDisp.Caption := 'N/A';
               lblGraceStartsLeftDisp.Caption := IntToStr(Slock1.GracePeriod);
               lblExpiryDateDisp.Caption := DateToStr(Slock1.ExpiryDate);
          end;
     ptExpiryDate:
          begin
               lblProtTypeDisp.Caption := 'Expiry Date';
               lblDaysLeftDisp.Caption := 'N/A';
               lblStartsLeftDisp.Caption := 'N/A';
               lblGraceStartsLeftDisp.Caption := IntToStr(Slock1.GracePeriod);
               lblExpiryDateDisp.Caption := DateToStr(Slock1.ExpiryDate);
          end;
     ptTimed:
          begin
               lblProtTypeDisp.Caption := 'Timed';
               lblDaysLeftDisp.Caption := 'N/A';
               lblStartsLeftDisp.Caption := 'N/A';
               lblGraceStartsLeftDisp.Caption := 'N/A';
               lblExpiryDateDisp.Caption := 'N/A';
          end;
     end;

     // load the registration information
     if Slock1.Registered then
     begin
          lblRegisteredToDisp.Caption := Slock1.RegName;
          lblChallengeStringDisp.Caption := Slock1.ChallengeString;
     end
     else
     begin
          lblRegisteredToDisp.Caption := 'N/A';
          lblChallengeStringDisp.Caption := 'N/A';
     end;

     // set the state of the buttons/menus
     btnRegister.Enabled := not Slock1.Registered;
     mnmRegister.Enabled := not Slock1.Registered;
     mnmFunction.Enabled := (Slock1.Status <> stExpired);
end; {UpdateStatus}

{*******************************************************************************
* Procedure : Slock1Reminder                                                   *
********************************************************************************
* Purpose   : This is the normal reminder event which is triggered when Slock  *
*             has not been registered and has not expired. You can use this as *
*             a reminder that the user should register.                        *
*             The procedure shown here is more complcated than normally,       *
*             because we want to be able to demonstrate all the different forms*
*             of protection.                                                   *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1Reminder(Sender: TObject);
begin
     if Slock1.ProtectionType = ptStartCount then
          ShowMessage('This is the normal reminder: ' + IntToStr(Slock1.AllowedStarts) +
                      ' starts left in the evaluation period.');
     if Slock1.ProtectionType = ptDayCount then
          ShowMessage('This is the normal reminder: ' + IntToStr(Trunc(Slock1.ExpiryDate - Now)) +
                      ' days left in the evaluation period.');
     if Slock1.ProtectionType = ptExpiryDate then
          ShowMessage('This is the normal reminder: ' + IntToStr(Trunc(Slock1.ExpiryDate - Now)) +
                      ' days left in the evaluation period.');
end; {Slock1Reminder}

{*******************************************************************************
* Procedure : Slock1RegInfoChanged                                             *
********************************************************************************
* Purpose   : This is triggered when the registration information stored in    *
*             Windows registry is checked, and it is found that the name or    *
*             machine has been changed. The input of a new unlock string is    *
*             necessary before the component will be marked as 'registered'    *
*             again.                                                           *
*             For the machine specific registration to be active, the property *
*             'SpecificMachine' must be set to true. Use this with care, as it *
*             can lead to 'false positives' if the user changes the Windows    *
*             registration information.                                        *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1RegInfoChanged(Sender: TObject);
begin
     ShowMessage('The registration info was changed.');
end; {Slock1RegInfoChanged}

{*******************************************************************************
* Procedure : Slock1InGracePeriod                                              *
********************************************************************************
* Purpose   : This is triggered when the program has expired, but is still in  *
*             the grace period. This gives you the opportunity to display a    *
*             more urgent message to promt the user that his software will soon*
*             expire.                                                          *
*             The procedure shown here is more complcated than normally,       *
*             because we want to be able to demonstrate all the different forms*
*             of protection.                                                   *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1InGracePeriod(Sender: TObject);
begin
     if Slock1.ProtectionType = ptStartCount then
          ShowMessage('This is the GRACE PERIOD: ' + IntToStr(Slock1.GracePeriod) +
                      ' starts left in the evaluation period.');
     if Slock1.ProtectionType = ptDayCount then
          ShowMessage('This is the GRACE PERIOD: ' + IntToStr(Slock1.GracePeriod)
                       + ' days left in the evaluation period.');
     if Slock1.ProtectionType = ptExpiryDate then
          ShowMessage('This is the GRACE PERIOD: ' + IntToStr(Slock1.GracePeriod)
                       + ' days left in the evaluation period.');
end; {Slock1InGracePeriod}

{*******************************************************************************
* Procedure : Slock1Register                                                   *
********************************************************************************
* Purpose   : Is called when the OnRegister event in Slock is triggered. In    *
*             this example we just show a short message, but we could also     *
*             use this event to show a thankyou screen, or even enable some    *
*             additional functions.                                            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1Register(Sender: TObject);
begin
     ShowMessage('The program has been registered. Thank you for registering!');
     UpdateStatus;
end; {Slock1Register}

{*******************************************************************************
* Procedure : Slock1RegWriteErr                                                *
********************************************************************************
* Purpose   : Is called when the read after write check of the Windows registry*
*             shows that the information was not correctly written.            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1RegWriteErr(Sender: TObject);
begin
     ShowMessage('There was an error writing to the registry. The program will terminate.');
     Halt;
end; {Slock1RegWriteErr}

{*******************************************************************************
* Procedure : mnpNormFuncClick                                                 *
********************************************************************************
* Purpose   : This is an example of using Slock to control access to functions *
*             in the 'host' program. The menu point 'Function' is enabled      *
*             or disabled according to the status of Slock.Status.             *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.mnpNormFuncClick(Sender: TObject);
begin
     ShowMessage('This is a normal function - it will be disabled after the program expired.');
end; {mnpNormFuncClick}

{*******************************************************************************
* Procedure : mnpRegNowClick                                                   *
********************************************************************************
* Purpose   : This calls event 'btnRegisterClick' to register Slock            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.mnpRegNowClick(Sender: TObject);
begin
     btnRegisterClick(self);
end; {mnpRegNowClick}

{*******************************************************************************
* Procedure : Slock1FirstStart                                                 *
********************************************************************************
* Purpose   : This is called the first time that the Slock.CheckProtection     *
*             Method is called (for the current registry key settings). You    *
*             can use this to display a welcome message or Licensing terms and *
*             conditions or even an advert for your other software.            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1FirstStart(Sender: TObject);
begin
     ShowMessage('This is the first start! Welcome to the Slock demonstration program by TSM Inc.');
end; {Slock1FirstStart}

{*******************************************************************************
* Procedure : Slock1ClockMovedBack                                             *
********************************************************************************
* Purpose   : People often try to get around a fixed expiry date by setting    *
*             the clock back. This event is triggered when the system clock    *
*             been set back to a date that is before the Slock was even        *
*             created. Normally you would end the program under such a         *
*             a condition.                                                     *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1ClockMovedBack(Sender: TObject);
begin
     ShowMessage('The clock was moved back!');
     Halt;
end;

{*******************************************************************************
* Procedure : Slock1RegistryTamper                                             *
********************************************************************************
* Purpose   : Thsi is triggered if the registry has been tampered with (e.g.   *
*             one of the two copies of the registry information has been       *
*             changed or deleted. If this is the case, the user will have to   *
*             enter his Registration code again.                               *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1RegistryTamper(Sender: TObject);
begin
     ShowMessage('The registry has been tampered with.');
end; {Slock1RegistryTamper}

{*******************************************************************************
* Procedure : Slock1WrongUnlockCode                                            *
********************************************************************************
* Purpose   : This event is triggered when the user enters the wrong unlock    *
*             code. To discourage random attempts at breaking the code with    *
*             brute force, we end the program when the user enters the wrong   *
*             code.                                                            *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1WrongUnlockCode(Sender: TObject);
begin
     ShowMessage('The unlock code was wrong. Program was not registered and will end.');
     Halt;
end; {Slock1WrongUnlockCode}

{*******************************************************************************
* Procedure : Slock1Expired                                                    *
********************************************************************************
* Purpose   : This event is triggered when the 'CheckProtection' method is     *
*             called and the component has expired (and there is no grace      *
*             period left).                                                    *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1Expired(Sender: TObject);
begin
     ShowMessage('The program has expired. Some Functions have been disabled. Register to re-enable them.');
end; {Slock1Expired}

{*******************************************************************************
* Procedure : Slock1Extend                                                     *
********************************************************************************
* Purpose   : This event is triggered when the 'RegisterNow' method is called  *
*             with an extension code. The Extension code increments the        *
*             grace period.                                                    *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.Slock1Extend(Sender: TObject);
begin
     ShowMessage('The component has been extended!');
end;

{*******************************************************************************
* Procedure : btnExitClick                                                     *
********************************************************************************
* Purpose   : Closes the program.                                              *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.btnExitClick(Sender: TObject);
begin
     Close;
end; {btnExitClick}

{*******************************************************************************
* Procedure : mnpExitClick                                                     *
********************************************************************************
* Purpose   : Closes the program.                                              *
********************************************************************************
* Paramters : None                                                             *
********************************************************************************
* Returns   : None                                                             *
*******************************************************************************}
procedure TfrmMain.mnpExitClick(Sender: TObject);
begin
     Close;
end; {mnpExitClick}

procedure TfrmMain.Slock1TimedOut(Sender: TObject);
begin
     UpdateStatus;
     ShowMessage('The component has timed out!');
end;

end.
