{$INCLUDE ..\cDefines.inc}
unit cTypes;

{                                                                              }
{                            Type base class v3.04                             }
{                                                                              }
{      This unit is copyright  1999-2002 by David Butler (david@e.co.za)      }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    Its original file name is cTypes.pas                      }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   [ cTypes ]                                                                 }
{   1999/11/12  0.01  Split cTypes from cDataStruct and cHolder.               }
{                      Default implementations for Assign, IsEqual             }
{   2001/07/30  1.02  Removed interfaces in AType (each interface adds four    }
{                      bytes to the instance size).                            }
{   [ cDataStructs ]                                                           }
{   2001/08/20  2.03  Merged cTypes and cDataStructs to allow object           }
{                      interface implementation in base classes.               }
{   [ cTypes ]                                                                 }
{   2002/05/15  3.04  Created cTypes from cDataStructs.                        }
{                                                                              }

interface

uses
  // Delphi
  SysUtils,

  // Fundamentals
  cUtils;



{                                                                              }
{ Note on class naming convention:                                             }
{   Classes with the A-prefix are abstract base classes. They define the       }
{   interface for the type and must never be instanciated.                     }
{                                                                              }



{                                                                              }
{ AType                                                                        }
{   Abstract base class for data structures.                                   }
{                                                                              }
{   Provides an interface for commonly used data operations such as            }
{   assigning, comparing and duplicating.                                      }
{                                                                              }
{   Duplicate creates a new instance of the object and copies the content.     }
{   Clear sets an instance's content (value) to an empty/zero state.           }
{   IsEqual compares content of an instances.                                  }
{   Compare is the ranking function used by sorting and searching.             }
{   Assign's default implementation calls the protected AssignTo.              }
{                                                                              }
type
  AType = class
    protected
    Procedure TypeError (const Msg : String; const Error : Exception = nil;
              const ErrorClass : ExceptClass = nil);
    Procedure MethodNotImplementedError (const Method : String);

    Procedure Init; virtual;
    Procedure AssignTo (const Dest : TObject); virtual;

    Function  GetAsString : String; virtual;
    Procedure SetAsString (const S : String); virtual;

    public
    Constructor Create;
    class Function CreateInstance : AType; virtual;

    Function  Duplicate : TObject; virtual;
    Procedure Assign (const Source : TObject); virtual;
    Procedure Clear; virtual;
    Function  IsEmpty : Boolean; virtual;
    Function  IsEqual (const V : TObject) : Boolean; virtual;
    Function  Compare (const V : TObject) : TCompareResult; virtual;
    Function  HashValue : LongWord; virtual;
    Property  AsString : String read GetAsString write SetAsString;
  end;
  EType = class (Exception);
  TypeClass = class of AType;
  ATypeArray = Array of AType;
  TypeClassArray = Array of TypeClass;



{                                                                              }
{ AType helper functions                                                       }
{                                                                              }
Function  TypeDuplicate (const V : TObject) : TObject;
Procedure TypeAssign (const A, B : TObject);
Function  TypeIsEqual (const A, B : TObject) : Boolean;
Function  TypeCompare (const A, B : TObject) : TCompareResult;
Function  TypeGetAsString (const V : TObject) : String;
Procedure TypeSetAsString (const V : TObject; const S : String);
Function  TypeHashValue (const A : TObject) : LongWord;



implementation



{                                                                              }
{ AType                                                                        }
{                                                                              }
Constructor AType.Create;
  Begin
    inherited Create;
    Init;
  End;

Procedure AType.Init;
  Begin
  End;

Procedure AType.TypeError (const Msg : String; const Error : Exception; const ErrorClass : ExceptClass);
var S : String;
  Begin
    S := {$IFDEF DEBUG}ObjectClassName (self) + ': ' + {$ENDIF}
         Msg;
    if Assigned (Error) then
      S := S + ': ' + Error.Message;
    if Assigned (ErrorClass) then
      raise ErrorClass.Create (S) else
      raise EType.Create (S);
  End;

Procedure AType.MethodNotImplementedError (const Method : String);
  Begin
    TypeError ('Method ' + ObjectClassName (self) + '.' + Method + ' not implemented');
  End;

class Function AType.CreateInstance : AType;
  Begin
    raise EType.Create ('Method ' + ClassClassName (self) + '.CreateInstance not implemented');
  End;

Procedure AType.Clear;
  Begin
    MethodNotImplementedError ('Clear');
  End;

{$WARNINGS OFF}
Function AType.IsEmpty : Boolean;
  Begin
    MethodNotImplementedError ('IsEmpty');
  End;
{$WARNINGS ON}

Function AType.Duplicate : TObject;
  Begin
    try
      Result := CreateInstance;
      try
        AType (Result).Assign (self);
      except
        FreeAndNil (Result);
        raise;
      end;
    except
      on E : Exception do
        TypeError ('Duplicate failed', E);
    end;
  End;

Procedure AType.Assign (const Source : TObject);
var R : Boolean;
  Begin
    if Source is AType then
      try
        AType (Source).AssignTo (self);
        R := True;
      except
        R := False;
      end else
      R := False;
    if not R then
      TypeError (ObjectClassName (self) + ' can not assign from ' + ObjectClassName (Source));
  End;

Procedure AType.AssignTo (const Dest : TObject);
  Begin
    TypeError (ObjectClassName (self) + ' can not assign to ' + ObjectClassName (Dest));
  End;

{$WARNINGS OFF}
Function AType.IsEqual (const V : TObject) : Boolean;
  Begin
    TypeError (ObjectClassName (self) + ' can not compare with ' + ObjectClassName (V));
  End;

Function AType.Compare (const V : TObject) : TCompareResult;
  Begin
    TypeError (ObjectClassName (self) + ' can not compare with ' + ObjectClassName (V));
  End;

Function AType.HashValue : LongWord;
  Begin
    try
      Result := HashStr (GetAsString, MaxLongWord, True);
    except
      on E : Exception do
        TypeError ('Hash error', E);
    end;
  End;
{$WARNINGS ON}

Function AType.GetAsString : String;
  Begin
    MethodNotImplementedError ('GetAsString');
  End;

Procedure AType.SetAsString (const S : String);
  Begin
    MethodNotImplementedError ('SetAsString');
  End;



{                                                                              }
{ AType helper functions                                                       }
{                                                                              }
Function TypeGetAsString (const V : TObject) : String;
  Begin
    if V is AType then
      Result := AType (V).GetAsString else
      raise EType.Create (ObjectClassName (V) + ' can not convert to string');
  End;

Procedure TypeSetAsString (const V : TObject; const S : String);
  Begin
    if V is AType then
      AType (V).SetAsString (S) else
      raise EType.Create (ObjectClassName (V) + ' can not set as string');
  End;

Function TypeDuplicate (const V : TObject) : TObject;
  Begin
    if V is AType then
      Result := AType (V).Duplicate else
    if not Assigned (V) then
      Result := nil else
      raise EType.Create (ObjectClassName (V) + ' can not duplicate');
  End;

Function TypeIsEqual (const A, B : TObject) : Boolean;
  Begin
    if A = B then
      Result := True else
    if A is AType then
      Result := AType (A).IsEqual (B) else
    if B is AType then
      Result := AType (B).IsEqual (A) else
      raise EType.Create (ObjectClassName (A) + ' and ' + ObjectClassName (B) + ' can not compare');
  End;

Function TypeCompare (const A, B : TObject) : TCompareResult;
  Begin
    if A = B then
      Result := crEqual else
    if A is AType then
      Result := AType (A).Compare (B) else
    if B is AType then
      Result := NegatedCompareResult (AType (B).Compare (A)) else
      Result := crUndefined;
  End;

Procedure TypeAssign (const A, B : TObject);
  Begin
    if A = B then
      exit else
    if A is AType then
      AType (A).Assign (B) else
    if B is AType then
      AType (B).AssignTo (A) else
      raise EType.Create (ObjectClassName (B) + ' can not assign to ' + ObjectClassName (A));
  End;

{$WARNINGS OFF}
Function TypeHashValue (const A : TObject) : LongWord;
  Begin
    if not Assigned (A) then
      Result := 0 else
    if A is AType then
      Result := AType (A).HashValue else
      raise EType.Create (ObjectClassName (A) + ' can not calculate hash value');
  End;
{$WARNINGS ON}



end.
