{$INCLUDE ..\cDefines.inc}
unit cArrays;

{                                                                              }
{                        Data structures: Arrays v3.13                         }
{                                                                              }
{      This unit is copyright  1999-2002 by David Butler (david@e.co.za)      }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    Its original file name is cArrays.pas                     }
{                     It was generated 29 Oct 2002 02:22.                      }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   [ cDataStructs ]                                                           }
{   1999/11/12  0.01  Split cTypes from cDataStruct and cHolder.               }
{   1999/11/14  0.02  Added AListType.                                         }
{   2000/02/08  1.03  Initial version. AArray, TArray and TStreamArray.        }
{   2000/06/07  1.04  Base classes (AIntegerArray, ASet).                      }
{   2000/06/08  1.05  Added AObjectArray.                                      }
{   2000/06/03  1.06  Added AArray, AIntegerArray, AExtendedArray,             }
{                     AStringArray and ABitArray (formerly ASet) with some     }
{                     implementations.                                         }
{   2000/06/06  1.07  TFlatBitArray implementation.                            }
{                     Added AInt64Array.                                       }
{   2000/06/08  1.08  Added TObjectArray.                                      }
{   2000/06/14  1.09  Converted cDataStructs to template.                      }
{   2001/07/15  1.10  Changed memory arrays to pre-allocate when growing.      }
{   2001/08/20  2.11  Merged cTypes and cDataStructs to allow object           }
{                     interface implementation in base classes.                }
{   [ cArrays ]                                                                }
{   2002/05/15  3.12  Created cArrays unit from cDataStructs.                  }
{                     Refactored for Fundamentals 3.                           }
{   2002/09/30  3.13  Moved stream array classes to unit cStreamArrays.        }
{                                                                              }

interface

uses
  // Delphi
  SysUtils,

  // Fundamentals
  cUtils,
  cTypes;

const
  UnitName      = 'cArrays';
  UnitVersion   = '3.13';
  UnitDesc      = 'Data structures: Arrays';
  UnitCopyright = '(c) 1999-2002 David Butler';



{                                                                              }
{ ARRAY BASE CLASSES                                                           }
{   Classes with the A-prefix are abstract base classes. They define the       }
{   interface for the type and must never be instanciated.                     }
{   Instead, create one of the implementation classes (T-prefix).              }
{                                                                              }



{                                                                              }
{ AArray                                                                       }
{                                                                              }
type
  AArray = class (AType)
    protected
    Procedure IndexError (const Idx : Integer); virtual;

    Function  GetCount : Integer; virtual; abstract;
    Procedure SetCount (const NewCount : Integer); virtual; abstract;

    public
    Procedure Clear; override;
    Property  Count : Integer read GetCount write SetCount;

    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; virtual; abstract;
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); virtual; abstract;
    Procedure Sort; virtual;
    Procedure ReverseOrder; virtual;

    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; virtual; abstract;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); virtual; abstract;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); virtual; abstract;
    Function  AddArray (const V : AArray) : Integer; virtual; abstract;
  end;
  EArray = class (Exception);
  ArrayClass = class of AArray;



{                                                                              }
{ ALongIntArray                                                                }
{   Base class for an array of LongInt's.                                      }
{                                                                              }
type
  ALongIntArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : LongInt); virtual; abstract;
    Function  GetItem (const Idx : Integer) : LongInt; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : LongIntArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : LongIntArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ALongIntArray interface                                                      }
    Property  Item [const Idx : Integer] : LongInt read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : LongIntArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : LongInt); virtual;
    Function  AddItem (const Value : LongInt) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : LongIntArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : LongInt; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  ELongIntArray = class (EArray);



{                                                                              }
{ AIntegerArray                                                                }
{                                                                              }
type
  AIntegerArray = ALongIntArray;
  EIntegerArray = ELongIntArray;



{                                                                              }
{ ALongWordArray                                                               }
{   Base class for an array of LongWord's.                                     }
{                                                                              }
type
  ALongWordArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : LongWord); virtual; abstract;
    Function  GetItem (const Idx : Integer) : LongWord; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : LongWordArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : LongWordArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ALongWordArray interface                                                     }
    Property  Item [const Idx : Integer] : LongWord read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : LongWordArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : LongWord); virtual;
    Function  AddItem (const Value : LongWord) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : LongWordArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : LongWord; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  ELongWordArray = class (EArray);



{                                                                              }
{ ACardinalArray                                                               }
{                                                                              }
type
  ACardinalArray = ALongWordArray;
  ECardinalArray = ELongWordArray;



{                                                                              }
{ AInt64Array                                                                  }
{   Base class for an array of Int64's.                                        }
{                                                                              }
type
  AInt64Array = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Int64); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Int64; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : Int64Array; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AInt64Array interface                                                        }
    Property  Item [const Idx : Integer] : Int64 read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : Int64Array read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Int64); virtual;
    Function  AddItem (const Value : Int64) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : Int64Array) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Int64; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  EInt64Array = class (EArray);



{                                                                              }
{ ASingleArray                                                                 }
{   Base class for an array of Single's.                                       }
{                                                                              }
type
  ASingleArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Single); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Single; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : SingleArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ASingleArray interface                                                       }
    Property  Item [const Idx : Integer] : Single read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : SingleArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Single); virtual;
    Function  AddItem (const Value : Single) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : SingleArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Single; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  ESingleArray = class (EArray);



{                                                                              }
{ ADoubleArray                                                                 }
{   Base class for an array of Double's.                                       }
{                                                                              }
type
  ADoubleArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Double); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Double; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : DoubleArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ADoubleArray interface                                                       }
    Property  Item [const Idx : Integer] : Double read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : DoubleArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Double); virtual;
    Function  AddItem (const Value : Double) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : DoubleArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Double; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  EDoubleArray = class (EArray);



{                                                                              }
{ AExtendedArray                                                               }
{   Base class for an array of Extended's.                                     }
{                                                                              }
type
  AExtendedArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Extended); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Extended; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AExtendedArray interface                                                     }
    Property  Item [const Idx : Integer] : Extended read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : ExtendedArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Extended); virtual;
    Function  AddItem (const Value : Extended) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : ExtendedArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Extended; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;
  EExtendedArray = class (EArray);



{                                                                              }
{ APointerArray                                                                }
{   Base class for an array of Pointer's.                                      }
{                                                                              }
type
  APointerArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Pointer); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Pointer; virtual; abstract;

    Function  GetRange (const LoIdx, HiIdx : Integer) : PointerArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : PointerArray); virtual;

    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { APointerArray interface                                                      }
    Property  Item [const Idx : Integer] : Pointer read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : PointerArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Pointer); virtual;
    Function  AddItem (const Value : Pointer) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : PointerArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Pointer; const PrevPos : Integer = -1) : Integer; virtual;
  end;
  EPointerArray = class (EArray);



{                                                                              }
{ AStringArray                                                                 }
{   Base class for an array of Strings.                                        }
{                                                                              }
type
  EStringArray = class (EArray);
  AStringArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : String); virtual; abstract;
    Function  GetItem (const Idx : Integer) : String; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : StringArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : StringArray); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AStringArray interface                                                   }
    Property  Item [const Idx : Integer] : String read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : StringArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : String = ''); virtual;
    Function  AddItem (const Value : String) : Integer; reintroduce; overload; virtual;
    Function  AddArray (const V : StringArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : String; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ AObjectArray                                                                 }
{   Base class for an array of objects.                                        }
{                                                                              }
type
  EObjectArray = class (EArray);
  AObjectArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : TObject); virtual; abstract;
    Function  GetItem (const Idx : Integer) : TObject; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : ObjectArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : ObjectArray); virtual;
    Function  GetAsString : String; override;
    Function  GetIsItemOwner : Boolean; virtual; abstract;
    Procedure SetIsItemOwner (const IsItemOwner : Boolean); virtual; abstract;

    public
    { AType implementations                                                    }
    Procedure Clear; override;
    Procedure Assign (const Source : TObject); override;
    Function  IsEqual (const V : TObject) : Boolean; override;
    Function  Compare (const V : TObject) : TCompareResult; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  AddArray (const V : AArray) : Integer; reintroduce; overload; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;

    { AObjectArray interface                                                   }
    Property  Item [const Idx : Integer] : TObject read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : ObjectArray read GetRange write SetRange;
    Function  AddItem (const Value : TObject) : Integer; virtual;
    Function  AddArray (const V : ObjectArray) : Integer; reintroduce; overload; virtual;

    Function  PosNext (const Find : TObject; const PrevPos : Integer) : Integer; overload; virtual;
    Function  PosNext (var Item : TObject; const ClassType : TClass; const PrevPos : Integer = -1) : Integer; overload; virtual;
    Function  PosNext (var Item : TObject; const ClassName : String; const PrevPos : Integer = -1) : Integer; overload; virtual;
    Function  Find (const ClassType : TClass; const Count : Integer = 1) : TObject; overload; virtual;
    Function  Find (const ClassName : String; const Count : Integer = 1) : TObject; overload; virtual;
    Function  FindAll (const ClassType : TClass) : ObjectArray; overload;
    Function  FindAll (const ClassName : String) : ObjectArray; overload;
    Function  CountItems (const ClassType : TClass) : Integer; overload; virtual;
    Function  CountItems (const ClassName : String) : Integer; overload; virtual;

    Property  IsItemOwner : Boolean read GetIsItemOwner write SetIsItemOwner;
    Procedure ReleaseItems; virtual; abstract;
    Procedure FreeItems; virtual; abstract;
    Function  ReleaseItem (const Idx : Integer) : TObject; virtual; abstract;
  end;



{                                                                              }
{ ABitArray                                                                    }
{   Base class for a bit array (a set) implementations.                        }
{   Bits are defined as False at initialization.                               }
{   FindRange finds Count consequetive bits set to Value, starting at Start.   }
{     It returns the index of the leftmost bit or -1 if not found.             }
{                                                                              }
type
  EBitArray = class (EArray);
  ABitArray = class (AArray)
    protected
    Function  GetBit (const Idx : Integer) : Boolean; virtual; abstract;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); virtual; abstract;
    Function  GetRange (const Idx : Integer) : LongWord; virtual;
    Procedure SetRange (const Idx : Integer; const Value : LongWord); virtual;

    public
    { AType implementations                                                    }
    Procedure Assign (const Source : TObject); override;

    { AArray implementations                                                   }
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;
    Function  AddArray (const V : AArray) : Integer; override;
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;

    { ABitArray interface                                                      }
    Property  Bit [const Idx : Integer] : Boolean read GetBit write SetBit; default;
    Property  Range [const Idx : Integer] : LongWord read GetRange write SetRange;
    Function  CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean; virtual;
    Procedure Fill (const Idx, Count : Integer; const Value : Boolean); virtual;
    Function  IsEqual (const V : TObject) : Boolean; override;
    Function  AddItem (const Value : Boolean) : Integer; reintroduce; overload; virtual;
    Procedure Invert; virtual;
    Function  Find (const Value : Boolean = False; const Start : Integer = 0;
              const FindForward : Boolean = True) : Integer; virtual;
    Function  FindRange (const Value : Boolean = False; const Start : Integer = 0;
              const Count : Integer = 1; const FindForward : Boolean = True) : Integer; virtual;
  end;



{                                                                              }
{ ARRAY IMPLEMENTATIONS                                                        }
{                                                                              }



{                                                                              }
{ TLongIntArray                                                                }
{   ALongIntArray implemented using a dynamic array.                           }
{                                                                              }
type
  TLongIntArray = class (ALongIntArray)
    protected
    FData  : LongIntArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { ALongIntArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : LongInt); override;
    Function  GetItem (const Idx : Integer) : LongInt; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : LongIntArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : LongIntArray); override;
    Procedure SetData (const Data : LongIntArray);

    public
    Constructor Create (const V : LongIntArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ALongIntArray implementations                                            }
    Procedure Assign (const V : LongIntArray); reintroduce; overload;
    Procedure Assign (const V : array of LongInt); reintroduce; overload;
    Property  Data : LongIntArray read FData write SetData;
  end;



{                                                                              }
{ TIntegerArray                                                                }
{                                                                              }
type
  TIntegerArray = TLongIntArray;



{                                                                              }
{ TLongWordArray                                                               }
{   ALongWordArray implemented using a dynamic array.                          }
{                                                                              }
type
  TLongWordArray = class (ALongWordArray)
    protected
    FData  : LongWordArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { ALongWordArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : LongWord); override;
    Function  GetItem (const Idx : Integer) : LongWord; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : LongWordArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : LongWordArray); override;
    Procedure SetData (const Data : LongWordArray);

    public
    Constructor Create (const V : LongWordArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ALongWordArray implementations                                            }
    Procedure Assign (const V : LongWordArray); reintroduce; overload;
    Procedure Assign (const V : array of LongWord); reintroduce; overload;
    Property  Data : LongWordArray read FData write SetData;
  end;



{                                                                              }
{ TCardinalArray                                                               }
{                                                                              }
type
  TCardinalArray = TLongWordArray;



{                                                                              }
{ TInt64Array                                                                  }
{   AInt64Array implemented using a dynamic array.                             }
{                                                                              }
type
  TInt64Array = class (AInt64Array)
    protected
    FData  : Int64Array;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { AInt64Array implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : Int64); override;
    Function  GetItem (const Idx : Integer) : Int64; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : Int64Array; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array); override;
    Procedure SetData (const Data : Int64Array);

    public
    Constructor Create (const V : Int64Array = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AInt64Array implementations                                            }
    Procedure Assign (const V : Int64Array); reintroduce; overload;
    Procedure Assign (const V : array of Int64); reintroduce; overload;
    Property  Data : Int64Array read FData write SetData;
  end;



{                                                                              }
{ TSingleArray                                                                 }
{   ASingleArray implemented using a dynamic array.                            }
{                                                                              }
type
  TSingleArray = class (ASingleArray)
    protected
    FData  : SingleArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { ASingleArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : Single); override;
    Function  GetItem (const Idx : Integer) : Single; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : SingleArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray); override;
    Procedure SetData (const Data : SingleArray);

    public
    Constructor Create (const V : SingleArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ASingleArray implementations                                            }
    Procedure Assign (const V : SingleArray); reintroduce; overload;
    Procedure Assign (const V : array of Single); reintroduce; overload;
    Property  Data : SingleArray read FData write SetData;
  end;



{                                                                              }
{ TDoubleArray                                                                 }
{   ADoubleArray implemented using a dynamic array.                            }
{                                                                              }
type
  TDoubleArray = class (ADoubleArray)
    protected
    FData  : DoubleArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { ADoubleArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : Double); override;
    Function  GetItem (const Idx : Integer) : Double; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : DoubleArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray); override;
    Procedure SetData (const Data : DoubleArray);

    public
    Constructor Create (const V : DoubleArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { ADoubleArray implementations                                            }
    Procedure Assign (const V : DoubleArray); reintroduce; overload;
    Procedure Assign (const V : array of Double); reintroduce; overload;
    Property  Data : DoubleArray read FData write SetData;
  end;



{                                                                              }
{ TExtendedArray                                                               }
{   AExtendedArray implemented using a dynamic array.                          }
{                                                                              }
type
  TExtendedArray = class (AExtendedArray)
    protected
    FData  : ExtendedArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { AExtendedArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : Extended); override;
    Function  GetItem (const Idx : Integer) : Extended; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray); override;
    Procedure SetData (const Data : ExtendedArray);

    public
    Constructor Create (const V : ExtendedArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AExtendedArray implementations                                            }
    Procedure Assign (const V : ExtendedArray); reintroduce; overload;
    Procedure Assign (const V : array of Extended); reintroduce; overload;
    Property  Data : ExtendedArray read FData write SetData;
  end;



{                                                                              }
{ TStringArray                                                                 }
{   AStringArray implemented using a dynamic array.                            }
{                                                                              }
type
  TStringArray = class (AStringArray)
    protected
    FData  : StringArray;
    FCount : Integer;

    { ACollection implementations                                              }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { AStringArray implementations                                            }
    Procedure SetItem (const Idx : Integer; const Value : String); override;
    Function  GetItem (const Idx : Integer) : String; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : StringArray; override;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : StringArray); override;
    Procedure SetData (const Data : StringArray);

    public
    Constructor Create (const V : StringArray = nil);

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;
    Procedure Assign (const Source : TObject); reintroduce; overload; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;

    { AStringArray implementations                                            }
    Procedure Assign (const V : StringArray); reintroduce; overload;
    Procedure Assign (const V : array of String); reintroduce; overload;
    Property  Data : StringArray read FData write SetData;
  end;



{                                                                              }
{ TObjectArray                                                                 }
{   AObjectArray implemented using a dynamic array.                            }
{                                                                              }
type
  TObjectArray = class (AObjectArray)
    protected
    FData        : ObjectArray;
    FCount       : Integer;
    FIsItemOwner : Boolean;

    Procedure SetData (const Data : ObjectArray); virtual;
    Function  GetIsItemOwner : Boolean; override;
    Procedure SetIsItemOwner (const IsItemOwner : Boolean); override;

    { AArray implementations                                                   }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { AObjectArray implementations                                             }
    Procedure SetItem (const Idx : Integer; const Value : TObject); override;
    Function  GetItem (const Idx : Integer) : TObject; override;
    Function  GetRange (const LoIdx, HiIdx : Integer) : ObjectArray; override;

    public
    { TObjectArray interface                                                   }
    Constructor Create (const V : ObjectArray = nil; const IsItemOwner : Boolean = False);
    Destructor Destroy; override;

    Property  Data : ObjectArray read FData write SetData;
    Property  IsItemOwner : Boolean read FIsItemOwner write FIsItemOwner;
    Procedure FreeItems; override;
    Procedure ReleaseItems; override;
    Function  ReleaseItem (const Idx : Integer) : TObject; override;

    { AType implementations                                                    }
    class Function CreateInstance : AType; override;

    { AArray implementations                                                   }
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; override;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); override;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); override;
  end;



{                                                                              }
{ TFlatBitArray                                                                }
{   ABitArray stored as an array of LongWords.                                 }
{   GetBit and SetBit is very fast.                                            }
{                                                                              }
type
  TFlatBitArray = class (ABitArray)
    protected
    FData  : LongWordArray;
    FCount : Integer;

    { AArray implementations                                                   }
    Function  GetCount : Integer; override;
    Procedure SetCount (const NewCount : Integer); override;

    { ABitArray implementations                                                }
    Function  GetBit (const Idx : Integer) : Boolean; override;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); override;
    Function  GetRange (const Idx : Integer) : LongWord; override;
    Procedure SetRange (const Idx : Integer; const Value : LongWord); override;

    public
    { AType implementations                                                    }
    class Function CreateInstance : AType; override;

    { ABitArray implementations                                                }
    Procedure Fill (const LoIdx, HiIdx : Integer; const Value : Boolean); override;
    Function  CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean; override;
  end;
  TBitArray = TFlatBitArray;



{                                                                              }
{ Self testing code                                                            }
{                                                                              }
Procedure SelfTest;



implementation

uses
  // Fundamentals
  cStrings;



{                                                                              }
{                                                                              }
{ TYPE BASE CLASSES                                                            }
{                                                                              }
{                                                                              }



{                                                                              }
{ AArray                                                                       }
{                                                                              }
Procedure AArray.IndexError (const Idx : Integer);
  Begin
    raise EArray.Create ({$IFDEF DEBUG}ObjectClassName (self) + ': ' + {$ENDIF}
          'Array index out of bounds'
          {$IFDEF DEBUG} + ': ' + IntToStr (Idx) + '/' + IntToStr (GetCount){$ENDIF});
  End;

Procedure AArray.Clear;
  Begin
    Count := 0;
  End;

Procedure AArray.Sort;

  Procedure QuickSort (L, R : Integer);
  var I, J : Integer;
      M    : Integer;
    Begin
      Repeat
        I := L;
        J := R;
        M := (L + R) shr 1;
        Repeat
          While CompareItems (I, M) = crLess do
            Inc (I);
          While CompareItems (J, M) = crGreater do
            Dec (J);
          if I <= J then
            begin
              ExchangeItems (I, J);
              Inc (I);
              Dec (J);
            end;
        Until I > J;
        if L < J then
          QuickSort (L, J);
        L := I;
      Until I >= R;
    End;

var I : Integer;
  Begin
    I := Count;
    if I > 0 then
      QuickSort (0, I - 1);
  End;

Procedure AArray.ReverseOrder;
var I, L : Integer;
  Begin
    L := Count;
    For I := 1 to L div 2 do
      ExchangeItems (I - 1, L - I);
  End;



{                                                                              }
{ ALongIntArray                                                                }
{                                                                              }
Procedure ALongIntArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : LongInt;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ALongIntArray.AddItem (const Value : LongInt) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ALongIntArray.GetRange (const LoIdx, HiIdx : Integer) : LongIntArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function ALongIntArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := ALongIntArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    ALongIntArray (Result).Count := C;
    For I := 0 to C - 1 do
      ALongIntArray (Result) [I] := Item [L + I];
  End;

Procedure ALongIntArray.SetRange (const LoIdx, HiIdx : Integer; const V : LongIntArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure ALongIntArray.Fill (const Idx, Count : Integer; const Value : LongInt);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function ALongIntArray.AddArray (const V : LongIntArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ALongIntArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : LongInt;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function ALongIntArray.PosNext (const Find : LongInt; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : LongInt;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ALongIntArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := IntToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + IntToStr (Item [I]);
    Result := Result;
  End;

Procedure ALongIntArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0 else
          Item [L - 1] := StrToInt (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure ALongIntArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is ALongIntArray then
      begin
        L := ALongIntArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := ALongIntArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function ALongIntArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is ALongIntArray then
      begin
        L := ALongIntArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> ALongIntArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function ALongIntArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is ALongIntArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := ALongIntArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure ALongIntArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure ALongIntArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ ALongWordArray                                                               }
{                                                                              }
Procedure ALongWordArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : LongWord;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ALongWordArray.AddItem (const Value : LongWord) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ALongWordArray.GetRange (const LoIdx, HiIdx : Integer) : LongWordArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function ALongWordArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := ALongWordArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    ALongWordArray (Result).Count := C;
    For I := 0 to C - 1 do
      ALongWordArray (Result) [I] := Item [L + I];
  End;

Procedure ALongWordArray.SetRange (const LoIdx, HiIdx : Integer; const V : LongWordArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure ALongWordArray.Fill (const Idx, Count : Integer; const Value : LongWord);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function ALongWordArray.AddArray (const V : LongWordArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ALongWordArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : LongWord;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function ALongWordArray.PosNext (const Find : LongWord; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : LongWord;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ALongWordArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := IntToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + IntToStr (Item [I]);
    Result := Result;
  End;

Procedure ALongWordArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0 else
          Item [L - 1] := StrToInt (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure ALongWordArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is ALongWordArray then
      begin
        L := ALongWordArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := ALongWordArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function ALongWordArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is ALongWordArray then
      begin
        L := ALongWordArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> ALongWordArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function ALongWordArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is ALongWordArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := ALongWordArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure ALongWordArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure ALongWordArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ AInt64Array                                                                  }
{                                                                              }
Procedure AInt64Array.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Int64;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AInt64Array.AddItem (const Value : Int64) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AInt64Array.GetRange (const LoIdx, HiIdx : Integer) : Int64Array;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function AInt64Array.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := AInt64Array (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    AInt64Array (Result).Count := C;
    For I := 0 to C - 1 do
      AInt64Array (Result) [I] := Item [L + I];
  End;

Procedure AInt64Array.SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure AInt64Array.Fill (const Idx, Count : Integer; const Value : Int64);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function AInt64Array.AddArray (const V : Int64Array) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AInt64Array.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Int64;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function AInt64Array.PosNext (const Find : Int64; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Int64;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AInt64Array.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := IntToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + IntToStr (Item [I]);
    Result := Result;
  End;

Procedure AInt64Array.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0 else
          Item [L - 1] := StrToInt (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure AInt64Array.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is AInt64Array then
      begin
        L := AInt64Array (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := AInt64Array (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function AInt64Array.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is AInt64Array then
      begin
        L := AInt64Array (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> AInt64Array (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function AInt64Array.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is AInt64Array then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := AInt64Array (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure AInt64Array.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure AInt64Array.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ ASingleArray                                                                 }
{                                                                              }
Procedure ASingleArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Single;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ASingleArray.AddItem (const Value : Single) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ASingleArray.GetRange (const LoIdx, HiIdx : Integer) : SingleArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function ASingleArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := ASingleArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    ASingleArray (Result).Count := C;
    For I := 0 to C - 1 do
      ASingleArray (Result) [I] := Item [L + I];
  End;

Procedure ASingleArray.SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure ASingleArray.Fill (const Idx, Count : Integer; const Value : Single);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function ASingleArray.AddArray (const V : SingleArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ASingleArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Single;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function ASingleArray.PosNext (const Find : Single; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Single;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ASingleArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := FloatToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + FloatToStr (Item [I]);
    Result := Result;
  End;

Procedure ASingleArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure ASingleArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is ASingleArray then
      begin
        L := ASingleArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := ASingleArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function ASingleArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is ASingleArray then
      begin
        L := ASingleArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> ASingleArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function ASingleArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is ASingleArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := ASingleArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure ASingleArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure ASingleArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ ADoubleArray                                                                 }
{                                                                              }
Procedure ADoubleArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Double;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ADoubleArray.AddItem (const Value : Double) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ADoubleArray.GetRange (const LoIdx, HiIdx : Integer) : DoubleArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function ADoubleArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := ADoubleArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    ADoubleArray (Result).Count := C;
    For I := 0 to C - 1 do
      ADoubleArray (Result) [I] := Item [L + I];
  End;

Procedure ADoubleArray.SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure ADoubleArray.Fill (const Idx, Count : Integer; const Value : Double);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function ADoubleArray.AddArray (const V : DoubleArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ADoubleArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Double;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function ADoubleArray.PosNext (const Find : Double; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Double;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ADoubleArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := FloatToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + FloatToStr (Item [I]);
    Result := Result;
  End;

Procedure ADoubleArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure ADoubleArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is ADoubleArray then
      begin
        L := ADoubleArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := ADoubleArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function ADoubleArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is ADoubleArray then
      begin
        L := ADoubleArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> ADoubleArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function ADoubleArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is ADoubleArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := ADoubleArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure ADoubleArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure ADoubleArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ AExtendedArray                                                               }
{                                                                              }
Procedure AExtendedArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Extended;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AExtendedArray.AddItem (const Value : Extended) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AExtendedArray.GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function AExtendedArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := AExtendedArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    AExtendedArray (Result).Count := C;
    For I := 0 to C - 1 do
      AExtendedArray (Result) [I] := Item [L + I];
  End;

Procedure AExtendedArray.SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure AExtendedArray.Fill (const Idx, Count : Integer; const Value : Extended);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function AExtendedArray.AddArray (const V : ExtendedArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AExtendedArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Extended;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function AExtendedArray.PosNext (const Find : Extended; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Extended;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AExtendedArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := FloatToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + FloatToStr (Item [I]);
    Result := Result;
  End;

Procedure AExtendedArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure AExtendedArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is AExtendedArray then
      begin
        L := AExtendedArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := AExtendedArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function AExtendedArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is AExtendedArray then
      begin
        L := AExtendedArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> AExtendedArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function AExtendedArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is AExtendedArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := AExtendedArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure AExtendedArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure AExtendedArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ APointerArray                                                                }
{                                                                              }
Procedure APointerArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Pointer;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function APointerArray.AddItem (const Value : Pointer) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function APointerArray.GetRange (const LoIdx, HiIdx : Integer) : PointerArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function APointerArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := APointerArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    APointerArray (Result).Count := C;
    For I := 0 to C - 1 do
      APointerArray (Result) [I] := Item [L + I];
  End;

Procedure APointerArray.SetRange (const LoIdx, HiIdx : Integer; const V : PointerArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure APointerArray.Fill (const Idx, Count : Integer; const Value : Pointer);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function APointerArray.AddArray (const V : PointerArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function APointerArray.PosNext (const Find : Pointer; const PrevPos : Integer) : Integer;
var I : Integer;
  Begin
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
  End;

Function APointerArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := PointerToStr (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + PointerToStr (Item [I]);
    Result := Result;
  End;

Procedure APointerArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := nil else
          Item [L - 1] := StrToPointer (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure APointerArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is APointerArray then
      begin
        L := APointerArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := APointerArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function APointerArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is APointerArray then
      begin
        L := APointerArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> APointerArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function APointerArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is APointerArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := APointerArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure APointerArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure APointerArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;



{                                                                              }
{ AStringArray                                                                 }
{                                                                              }
Procedure AStringArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : String;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AStringArray.AddItem (const Value : String) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AStringArray.GetRange (const LoIdx, HiIdx : Integer) : StringArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [I] := Item [L + I];
  End;

Function AStringArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, L, H, C : Integer;
  Begin
    Result := AStringArray (CreateInstance);
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L + 1;
    AStringArray (Result).Count := C;
    For I := 0 to C - 1 do
      AStringArray (Result) [I] := Item [L + I];
  End;

Procedure AStringArray.SetRange (const LoIdx, HiIdx : Integer; const V : StringArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Procedure AStringArray.Fill (const Idx, Count : Integer; const Value : String);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Function AStringArray.AddArray (const V : StringArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AStringArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : String;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Function AStringArray.PosNext (const Find : String; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : String;
  Begin
    if IsSortedAscending then // binary search
      begin
        if MaxI (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
  
        For I := MaxI (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AStringArray.GetAsString : String;
var I, L : Integer;
  Begin
    L := Count;
    if L = 0 then
      begin
        Result := '';
        exit;
      end;
    Result := UnquoteText (Item [0]);
    For I := 1 to L - 1 do
      Result := Result + ',' + UnquoteText (Item [I]);
    Result := Result;
  End;

Procedure AStringArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if L = 0 then
      begin
        Count := 0;
        exit;
      end;
    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := '' else
          Item [L - 1] := QuoteText (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Procedure AStringArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is AStringArray then
      begin
        L := AStringArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Item [I] := AStringArray (Source).Item [I];
      end else
      inherited Assign (Source);
  End;
  
Function AStringArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is AStringArray then
      begin
        L := AStringArray (V).Count;
        Result := L = Count;
        if not Result then
          exit;
        For I := 0 to L - 1 do
          if Item [I] <> AStringArray (V).Item [I] then
            begin
              Result := False;
              exit;
            end;
      end else
      Result := inherited IsEqual (V);
  End;

Function AStringArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is AStringArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := AStringArray (V) [I];
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;

Procedure AStringArray.Delete (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (I, GetItem (I + Count));
        SetCount (C - L);
      end;
  End;

Procedure AStringArray.Insert (const Idx : Integer; const Count : Integer);
var I, C, J, L : Integer;
  Begin
    if Count <= 0 then
      exit;
    C := GetCount;
    SetCount (C + Count);
    J := MinI (MaxI (Idx, 0), C);
    L := C - J;
    For I := C - 1 downto C - L do
      SetItem (I + Count, GetItem (I));
  End;





{                                                                              }
{ ABitArray                                                                    }
{                                                                              }
Function ABitArray.GetRange (const Idx : Integer) : LongWord;
var I : Integer;
  Begin
    Result := 0;
    For I := 0 to BitsPerLongWord - 1 do
      if Bit [Idx + I] then
        Result := Result or BitMaskTable [I];
  End;

Procedure ABitArray.SetRange (const Idx : Integer; const Value : LongWord);
var I : Integer;
    C : LongWord;
  Begin
    C := 1;
    For I := Idx to Idx + BitsPerLongWord - 1 do
      begin
        Bit [I] := Value and C <> 0;
        C := C shl 1;
      end;
  End;

Procedure ABitArray.Fill (const Idx, Count : Integer; const Value : Boolean);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Bit [I] := Value;
  End;

Function ABitArray.CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean;
var I : Integer;
  Begin
    For I := LoIdx to HiIdx do
      if Bit [I] <> Value then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure ABitArray.Assign (const Source : TObject);
var I, L : Integer;
  Begin
    if Source is ABitArray then
      begin
        L := AArray (Source).Count;
        Count := L;
        For I := 0 to L - 1 do
          Bit [I] := ABitArray (Source) [I];
      end else
      inherited Assign (Source);
  End;

Function ABitArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
  Begin
    if V is ABitArray then
      begin
        L := AArray (V).Count;
        if Count <> L then
          begin
            Result := False;
            exit;
          end;
        For I := 0 to L - 1 do
          if Bit [I] <> ABitArray (V) [I] then
            begin
              Result := False;
              exit;
            end;
        Result := True;
      end else
      Result := inherited IsEqual (V);
  End;

Procedure ABitArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Boolean;
  Begin
    I := Bit [Idx1];
    Bit [Idx1] := Bit [Idx2];
    Bit [Idx2] := I;
  End;

Function ABitArray.AddItem (const Value : Boolean) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Bit [Result] := Value;
  End;

Function ABitArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
  Begin
    Result := cUtils.Compare (Bit [Idx1], Bit [Idx2]);
  End;

Procedure ABitArray.Invert;
var I : Integer;
  Begin
    For I := 0 to Count - 1 do
      Bit [I] := not Bit [I];
  End;

Function ABitArray.Find (const Value : Boolean; const Start : Integer; const FindForward : Boolean) : Integer;
var I, C : Integer;
  Begin
    I := Start;
    C := Count;
    if FindForward then
      begin
        While (I < C) and (Bit [I] <> Value) do
          Inc (I);
        if I < C then
          Result := I else
          Result := -1;
      end else
      begin
        I := MinI (I, C);
        While (I >= 0) and (Bit [I] <> Value) do
          Dec (I);
        Result := I;
      end;
  End;

Function ABitArray.FindRange (const Value : Boolean; const Start : Integer; const Count : Integer; const FindForward : Boolean) : Integer;
var I, C, F : Integer;
  Begin
    I := Start;
    if FindForward then
      begin
        C := self.Count;
        F := 0;
        While (I + F < C) and (F < Count) do
          if Bit [I + F] = Value then
            Inc (F) else
            begin
              Inc (I, F + 1);
              F := 0;
            end;
        if F < Count then
          Result := -1 else
          Result := I;
      end else
      begin
        F := 0;
        While (I - F >= 0) and (F < Count) do
          if Bit [I - F] = Value then
            Inc (F) else
            begin
              Dec (I, F + 1);
              F := 0;
            end;
        if F < Count then
          Result := -1 else
          Result := I - F + 1;
      end;
  End;

Procedure ABitArray.Delete (const Idx : Integer; const Count : Integer);
var I, C : Integer;
  Begin
    C := GetCount;
    {$IFOPT R+}
    if (Idx < 0) or (Idx + Count > C) then
      IndexError (Idx);
    {$ENDIF}
    For I := Idx + Count to C - 1 do
      SetBit (I - Count, GetBit (I));
    SetCount (C - Count);
  End;

Procedure ABitArray.Insert (const Idx : Integer; const Count : Integer);
var I, C : Integer;
  Begin
    C := GetCount;
    {$IFOPT R+}
    if (Idx < 0) or (Idx > C) then
      IndexError (Idx);
    {$ENDIF}
    SetCount (C + Count);
    For I := Idx to C - 1 do
      SetBit (I + Count, GetBit (I));
    Fill (Idx, Idx + Count - 1, False);
  End;

Function ABitArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I, C : Integer;
  Begin
    C := GetCount;
    {$IFOPT R+}
    if (LoIdx < 0) or (LoIdx > HiIdx) or (HiIdx >= C) then
      IndexError (HiIdx);
    {$ENDIF}
    Result := ABitArray (CreateInstance);
    C := HiIdx - LoIdx + 1;
    Result.Count := C;
    For I := 0 to C - 1 do
      ABitArray (Result) [I] := GetBit (LoIdx + I);
  End;

Function ABitArray.AddArray (const V : AArray) : Integer;
var I, C : Integer;
  Begin
    if V is ABitArray then
      begin
        Result := Count;
        C := ABitArray (V).Count;
        if C = 0 then
          exit;
        SetCount (Result + C);
        For I := 0 to C - 1 do
          SetBit (Result + I, ABitArray (V).GetBit (I));
      end else
      begin
        TypeError (ObjectClassName (self) + ' can not add array ' + ObjectClassName (V));
        Result := -1;
      end;
  End;



{                                                                              }
{ AObjectArray                                                                 }
{                                                                              }
Procedure AObjectArray.Clear;
  Begin
    if IsItemOwner then
      FreeItems else
      ReleaseItems;
  End;

Procedure AObjectArray.Assign (const Source : TObject);
var I, L : Integer;
    V, S : TObject;
  Begin
    S := Source;
    if S is AObjectArray then
      begin
        FreeItems;
        IsItemOwner := AObjectArray (S).IsItemOwner;
        L := AArray (S).Count;
        Count := L;
        if GetIsItemOwner then
          For I := 0 to L - 1 do
            begin
              V := AObjectArray (S) [I];
              if V is AArray then
                Item [I] := AArray (V).Duplicate else
                Item [I] := V;
            end
        else
          For I := 0 to L - 1 do
            Item [I] := AObjectArray (S) [I];
      end else
      inherited Assign (Source);
  End;

Function AObjectArray.IsEqual (const V : TObject) : Boolean;
var I, L : Integer;
    A, B : TObject;
  Begin
    if V is AObjectArray then
      begin
        L := AArray (V).Count;
        if Count <> L then
          begin
            Result := False;
            exit;
          end;
        For I := 0 to L - 1 do
          begin
            A := Item [I];
            B := AObjectArray (V) [I];
            Result := A = B;
            if not Result then
              exit;
            end;
        Result := True;
      end else
      Result := inherited IsEqual (V);
  End;

Function AObjectArray.Compare (const V : TObject) : TCompareResult;
var I, C1, C2 : Integer;
    A, B : TObject;
  Begin
    if V is AObjectArray then
      begin
        C1 := GetCount;
        C2 := AObjectArray (V).GetCount;
        if C1 < C2 then
          Result := crLess else
        if C1 > C2 then
          Result := crGreater else
          begin
            Result := crEqual;
            For I := 0 to GetCount - 1 do
              begin
                A := Item [I];
                B := AObjectArray (V) [I];
                if A <> B then
                  begin
                    Result := crUndefined;
                    exit;
                  end;
              end;
          end;
      end else
      Result := inherited Compare (V);
  End;

Procedure AObjectArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : TObject;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AObjectArray.AddItem (const Value : TObject) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AObjectArray.AddArray (const V : ObjectArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AObjectArray.AddArray (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if V is AObjectArray then
      begin
        Result := Count;
        L := V.Count;
        Count := Result + L;
        For I := 0 to L - 1 do
          Item [Result + I] := AObjectArray (V) [I];
      end else
      Result := inherited AddArray (V);
  End;

Procedure AObjectArray.Delete (const Idx, Count : Integer);
var I, C, J, L : Integer;
  Begin
    J := MaxI (Idx, 0);
    C := GetCount;
    L := MinI (Count, C - J);
    if L > 0 then
      begin
        For I := J to J + C - 1 do
          SetItem (Idx + I, GetItem (Idx + Count + I));
        SetCount (C - L);
      end;
  End;

Function AObjectArray.PosNext (const Find : TObject; const PrevPos : Integer) : Integer;
var I : Integer;
  Begin
    For I := MaxI (PrevPos + 1, 0) to Count - 1 do
      if Find = Item [I] then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function AObjectArray.PosNext (var Item : TObject; const ClassType : TClass; const PrevPos : Integer) : Integer;
var I : Integer;
  Begin
    For I := MaxI (PrevPos + 1, 0) to Count - 1 do
      begin
        Item := GetItem (I);
        if Item.InheritsFrom (ClassType) then
          begin
            Result := I;
            exit;
          end;
      end;
    Item := nil;
    Result := -1;
  End;

Function AObjectArray.PosNext (var Item : TObject; const ClassName : String; const PrevPos : Integer) : Integer;
var I : Integer;
  Begin
    For I := MaxI (PrevPos + 1, 0) to Count - 1 do
      begin
        Item := GetItem (I);
        if Assigned (Item) and Item.ClassNameIs (ClassName) then
          begin
            Result := I;
            exit;
          end;
      end;
    Item := nil;
    Result := -1;
  End;

Function AObjectArray.Find (const ClassType : TClass; const Count : Integer) : TObject;
var I, J : Integer;
  Begin
    I := -1;
    For J := 1 to Count do
      begin
        I := PosNext (Result, ClassType, I);
        if I = -1 then
          break;
      end;
    if I = -1 then
      Result := nil;
  End;

Function AObjectArray.Find (const ClassName : String; const Count : Integer) : TObject;
var I, J : Integer;
  Begin
    I := -1;
    For J := 1 to Count do
      begin
        I := PosNext (Result, ClassName, I);
        if I = -1 then
          break;
      end;
    if I = -1 then
      Result := nil;
  End;

Function AObjectArray.FindAll (const ClassType : TClass) : ObjectArray;
var I : Integer;
    V : TObject;
  Begin
    SetLength (Result, 0);
    I := PosNext (V, ClassType);
    While I >= 0 do
      begin
        Append (Result, V);
        I := PosNext (V, ClassType, I);
      end;
  End;

Function AObjectArray.FindAll (const ClassName : String) : ObjectArray;
var I : Integer;
    V : TObject;
  Begin
    SetLength (Result, 0);
    I := PosNext (V, ClassName);
    While I >= 0 do
      begin
        Append (Result, V);
        I := PosNext (V, ClassName, I);
      end;
  End;

Function AObjectArray.CountItems (const ClassType : TClass) : Integer;
var I : Integer;
    V : TObject;
  Begin
    Result := 0;
    I := PosNext (V, ClassType);
    While I >= 0 do
      begin
        Inc (Result);
        I := PosNext (V, ClassType, I);
      end;
  End;

Function AObjectArray.CountItems (const ClassName : String) : Integer;
var I : Integer;
    V : TObject;
  Begin
    Result := 0;
    I := PosNext (V, ClassName);
    While I >= 0 do
      begin
        Inc (Result);
        I := PosNext (V, ClassName, I);
      end;
  End;

Function AObjectArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var A, B : TObject;
  Begin
    A := Item [Idx1];
    B := Item [Idx2];
    if A = B then
      Result := crEqual else
      Result := crUndefined;
  End;

Function AObjectArray.GetRange (const LoIdx, HiIdx : Integer) : ObjectArray;
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AObjectArray.SetRange (const LoIdx, HiIdx : Integer; const V : ObjectArray);
var I, L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (Count - 1, HiIdx);
    C := MinI (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function AObjectArray.GetAsString : String;
var I, L : Integer;
    V : TObject;
  Begin
    Result := '';
    L := Count;
    For I := 0 to L - 1 do
      begin
        V := Item [I];
        Result := Result + PointerToStr (V);
        if I < L - 1 then
          Result := Result + ',';
      end;
  End;



{                                                                              }
{ ARRAY IMPLEMENTATIONS                                                        }
{                                                                              }



{                                                                              }
{ Allocation strategy used in array implementations to reduce memory copies:   }
{   i)   For Count <= 16 or for the first allocation: allocate the exact size. }
{   ii)  For growing blocks: pre-allocate 1/8th of NewCount.                   }
{   iii) For shrinking blocks: shrink actual allocation when Count is less     }
{        than half of the allocated size.                                      }
{                                                                              }

{                                                                              }
{ TIntegerArray                                                                }
{                                                                              }
Procedure TIntegerArray.SetItem (const Idx : Integer; const Value : Integer);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TIntegerArray.GetItem (const Idx : Integer) : Integer;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TIntegerArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TIntegerArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TIntegerArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TIntegerArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TIntegerArray.GetRange (const LoIdx, HiIdx : Integer) : IntegerArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TIntegerArray.SetRange (const LoIdx, HiIdx : Integer; const V : IntegerArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Integer));
  End;

Constructor TIntegerArray.Create (const V : IntegerArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TIntegerArray.CreateInstance : AType;
  Begin
    Result := TIntegerArray.Create;
  End;

Procedure TIntegerArray.SetData (const Data : IntegerArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TIntegerArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TIntegerArray;
    TIntegerArray (Result).FCount := C;
    if C > 0 then
      TIntegerArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TIntegerArray.Assign (const V : IntegerArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TIntegerArray.Assign (const V : array of Integer);
  Begin
    FData := AsIntegerArray (V);
    FCount := Length (FData);
  End;

Procedure TIntegerArray.Assign (const Source : TObject);
  Begin
    if Source is TIntegerArray then
      begin
        FCount := TIntegerArray (Source).FCount;
        FData := Copy (TIntegerArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TCardinalArray                                                               }
{                                                                              }
Procedure TCardinalArray.SetItem (const Idx : Integer; const Value : Cardinal);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TCardinalArray.GetItem (const Idx : Integer) : Cardinal;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TCardinalArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TCardinalArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TCardinalArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TCardinalArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TCardinalArray.GetRange (const LoIdx, HiIdx : Integer) : CardinalArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TCardinalArray.SetRange (const LoIdx, HiIdx : Integer; const V : CardinalArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Cardinal));
  End;

Constructor TCardinalArray.Create (const V : CardinalArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TCardinalArray.CreateInstance : AType;
  Begin
    Result := TCardinalArray.Create;
  End;

Procedure TCardinalArray.SetData (const Data : CardinalArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TCardinalArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TCardinalArray;
    TCardinalArray (Result).FCount := C;
    if C > 0 then
      TCardinalArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TCardinalArray.Assign (const V : CardinalArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TCardinalArray.Assign (const V : array of Cardinal);
  Begin
    FData := AsCardinalArray (V);
    FCount := Length (FData);
  End;

Procedure TCardinalArray.Assign (const Source : TObject);
  Begin
    if Source is TCardinalArray then
      begin
        FCount := TCardinalArray (Source).FCount;
        FData := Copy (TCardinalArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TInt64Array                                                                  }
{                                                                              }
Procedure TInt64Array.SetItem (const Idx : Integer; const Value : Int64);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TInt64Array.GetItem (const Idx : Integer) : Int64;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TInt64Array.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TInt64Array.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TInt64Array.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TInt64Array.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TInt64Array.GetRange (const LoIdx, HiIdx : Integer) : Int64Array;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TInt64Array.SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Int64));
  End;

Constructor TInt64Array.Create (const V : Int64Array);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TInt64Array.CreateInstance : AType;
  Begin
    Result := TInt64Array.Create;
  End;

Procedure TInt64Array.SetData (const Data : Int64Array);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TInt64Array.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TInt64Array;
    TInt64Array (Result).FCount := C;
    if C > 0 then
      TInt64Array (Result).FData := Copy (FData, L, C);
  End;

Procedure TInt64Array.Assign (const V : Int64Array);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TInt64Array.Assign (const V : array of Int64);
  Begin
    FData := AsInt64Array (V);
    FCount := Length (FData);
  End;

Procedure TInt64Array.Assign (const Source : TObject);
  Begin
    if Source is TInt64Array then
      begin
        FCount := TInt64Array (Source).FCount;
        FData := Copy (TInt64Array (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TSingleArray                                                                 }
{                                                                              }
Procedure TSingleArray.SetItem (const Idx : Integer; const Value : Single);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TSingleArray.GetItem (const Idx : Integer) : Single;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TSingleArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TSingleArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TSingleArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TSingleArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TSingleArray.GetRange (const LoIdx, HiIdx : Integer) : SingleArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TSingleArray.SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Single));
  End;

Constructor TSingleArray.Create (const V : SingleArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TSingleArray.CreateInstance : AType;
  Begin
    Result := TSingleArray.Create;
  End;

Procedure TSingleArray.SetData (const Data : SingleArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TSingleArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TSingleArray;
    TSingleArray (Result).FCount := C;
    if C > 0 then
      TSingleArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TSingleArray.Assign (const V : SingleArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TSingleArray.Assign (const V : array of Single);
  Begin
    FData := AsSingleArray (V);
    FCount := Length (FData);
  End;

Procedure TSingleArray.Assign (const Source : TObject);
  Begin
    if Source is TSingleArray then
      begin
        FCount := TSingleArray (Source).FCount;
        FData := Copy (TSingleArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TDoubleArray                                                                 }
{                                                                              }
Procedure TDoubleArray.SetItem (const Idx : Integer; const Value : Double);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TDoubleArray.GetItem (const Idx : Integer) : Double;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TDoubleArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TDoubleArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TDoubleArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TDoubleArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TDoubleArray.GetRange (const LoIdx, HiIdx : Integer) : DoubleArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TDoubleArray.SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Double));
  End;

Constructor TDoubleArray.Create (const V : DoubleArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TDoubleArray.CreateInstance : AType;
  Begin
    Result := TDoubleArray.Create;
  End;

Procedure TDoubleArray.SetData (const Data : DoubleArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TDoubleArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TDoubleArray;
    TDoubleArray (Result).FCount := C;
    if C > 0 then
      TDoubleArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TDoubleArray.Assign (const V : DoubleArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TDoubleArray.Assign (const V : array of Double);
  Begin
    FData := AsDoubleArray (V);
    FCount := Length (FData);
  End;

Procedure TDoubleArray.Assign (const Source : TObject);
  Begin
    if Source is TDoubleArray then
      begin
        FCount := TDoubleArray (Source).FCount;
        FData := Copy (TDoubleArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TExtendedArray                                                               }
{                                                                              }
Procedure TExtendedArray.SetItem (const Idx : Integer; const Value : Extended);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TExtendedArray.GetItem (const Idx : Integer) : Extended;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TExtendedArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TExtendedArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Procedure TExtendedArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TExtendedArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TExtendedArray.GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TExtendedArray.SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (Extended));
  End;

Constructor TExtendedArray.Create (const V : ExtendedArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TExtendedArray.CreateInstance : AType;
  Begin
    Result := TExtendedArray.Create;
  End;

Procedure TExtendedArray.SetData (const Data : ExtendedArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TExtendedArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TExtendedArray;
    TExtendedArray (Result).FCount := C;
    if C > 0 then
      TExtendedArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TExtendedArray.Assign (const V : ExtendedArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TExtendedArray.Assign (const V : array of Extended);
  Begin
    FData := AsExtendedArray (V);
    FCount := Length (FData);
  End;

Procedure TExtendedArray.Assign (const Source : TObject);
  Begin
    if Source is TExtendedArray then
      begin
        FCount := TExtendedArray (Source).FCount;
        FData := Copy (TExtendedArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;


{                                                                              }
{ TStringArray                                                                 }
{                                                                              }
Procedure TStringArray.SetItem (const Idx : Integer; const Value : String);
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    FData [Idx] := Value;
  End;

Function TStringArray.GetItem (const Idx : Integer) : String;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TStringArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TStringArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if FCount = NewCount then
      exit;
    FCount := NewCount;

    N := NewCount;
    L := Length (FData);
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLength (FData, N);
  End;

Procedure TStringArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TStringArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;

Function TStringArray.GetRange (const LoIdx, HiIdx : Integer) : StringArray;
var L, H : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    if H >= L then
      Result := Copy (FData, L, H - L + 1) else
      Result := nil;
  End;

Procedure TStringArray.SetRange (const LoIdx, HiIdx : Integer; const V : StringArray);
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (MinI (Length (V), H - L + 1), 0);
    if C > 0 then
      Move (V [0], FData [L], C * Sizeof (String));
  End;

Constructor TStringArray.Create (const V : StringArray);
  Begin
    inherited Create;
    SetData (V);
  End;

class Function TStringArray.CreateInstance : AType;
  Begin
    Result := TStringArray.Create;
  End;

Procedure TStringArray.SetData (const Data : StringArray);
  Begin
    FData := Data;
    FCount := Length (FData);
  End;

Function TStringArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var L, H, C : Integer;
  Begin
    L := MaxI (0, LoIdx);
    H := MinI (HiIdx, FCount);
    C := MaxI (0, H - L + 1);
    Result := CreateInstance as TStringArray;
    TStringArray (Result).FCount := C;
    if C > 0 then
      TStringArray (Result).FData := Copy (FData, L, C);
  End;

Procedure TStringArray.Assign (const V : StringArray);
  Begin
    FData := Copy (V);
    FCount := Length (FData);
  End;

Procedure TStringArray.Assign (const V : array of String);
  Begin
    FData := AsStringArray (V);
    FCount := Length (FData);
  End;

Procedure TStringArray.Assign (const Source : TObject);
  Begin
    if Source is TStringArray then
      begin
        FCount := TStringArray (Source).FCount;
        FData := Copy (TStringArray (Source).FData, 0, FCount);
      end else
      inherited Assign (Source);
  End;




{                                                                              }
{ TObjectArray                                                                 }
{                                                                              }
Procedure TObjectArray.FreeItems;
  Begin
    FreeObjectArray (FData);
    FData := nil;
    FCount := 0;
  End;

Procedure TObjectArray.ReleaseItems;
  Begin
    FData := nil;
    FCount := 0;
  End;

Function TObjectArray.GetIsItemOwner : Boolean;
  Begin
    Result := FIsItemOwner;
  End;

Procedure TObjectArray.SetIsItemOwner (const IsItemOwner : Boolean);
  Begin
    FIsItemOwner := IsItemOwner;
  End;

Procedure TObjectArray.SetItem (const Idx : Integer; const Value : TObject);
var V : TObject;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    if FIsItemOwner then
      begin
        V := FData [Idx];
        if V = Value then
          exit;
        FreeAndNil (V);
      end;
    FData [Idx] := Value;
  End;

Function TObjectArray.GetItem (const Idx : Integer) : TObject;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
  End;

Function TObjectArray.ReleaseItem (const Idx : Integer) : TObject;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := FData [Idx];
    if Assigned (Result) and FIsItemOwner then
      FData [Idx] := nil;
  End;

Function TObjectArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TObjectArray.SetCount (const NewCount : Integer);
var L, N : Integer;
  Begin
    if NewCount = FCount then
      exit;

    if (NewCount < FCount) and FIsItemOwner then
      FreeObjectArray (FData, NewCount, FCount - 1);
    FCount := NewCount;

    L := Length (FData);
    N := NewCount;
    if (N > 16) and (L > 0) then
      if N > L then
        N := N + N shr 3 else
      if N > L shr 1 then
        exit;

    if N <> L then
      SetLengthAndZero (FData, N);
  End;

Function TObjectArray.GetRange (const LoIdx, HiIdx : Integer) : ObjectArray;
  Begin
    Result := Copy (FData, LoIdx, MinI (HiIdx, FCount - 1) - LoIdx + 1);
  End;

Procedure TObjectArray.SetData (const Data : ObjectArray);
  Begin
    if FIsItemOwner then
      FreeItems;
    FData := Data;
    FCount := Length (FData);
  End;

Constructor TObjectArray.Create (const V : ObjectArray; const IsItemOwner : Boolean);
  Begin
    inherited Create;
    FData := V;
    FIsItemOwner := IsItemOwner;
    FCount := Length (FData);
  End;

class Function TObjectArray.CreateInstance : AType;
  Begin
    Result := TObjectArray.Create;
  End;

Destructor TObjectArray.Destroy;
  Begin
    if FIsItemOwner then
      FreeItems;
    inherited Destroy;
  End;

Function TObjectArray.DuplicateRange (const LoIdx, HiIdx : Integer) : AArray;
var I : Integer;
    V : TObject;
  Begin
    Result := CreateInstance as TObjectArray;
    For I := LoIdx to MinI (HiIdx, FCount - 1) do
      begin
        V := FData [I];
        if V is AType then
          V := AType (V).Duplicate;
        TObjectArray (Result).AddItem (V);
      end;
  End;

Procedure TObjectArray.Delete (const Idx : Integer; const Count : Integer = 1);
  Begin
    FCount := MaxI (FCount - Remove (FData, Idx, Count, FIsItemOwner), 0);
    if FCount = 0 then
      FData := nil;
  End;

Procedure TObjectArray.Insert (const Idx : Integer; const Count : Integer = 1);
var I : Integer;
  Begin
    I := ArrayInsert (FData, Idx, Count);
    if I >= 0 then
      Inc (FCount, Count);
  End;



{                                                                              }
{ TFlatBitArray                                                                }
{                                                                              }
const
  TrueLongWord  : LongWord = $FFFFFFFF;
  FalseLongWord : LongWord = $00000000;

class Function TFlatBitArray.CreateInstance : AType;
  Begin
    Result := TFlatBitArray.Create;
  End;

Function TFlatBitArray.GetBit (const Idx : Integer) : Boolean;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    Result := cUtils.IsBitSet (FData [Idx shr 5], Idx and 31);
  End;

Procedure TFlatBitArray.SetBit (const Idx : Integer; const Value : Boolean);
var L : ^LongWord;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    L := @FData [Idx shr 5];
    if Value then
      L^ := cUtils.SetBit (L^, Idx and 31) else
      L^ := cUtils.ClearBit (L^, Idx and 31);
  End;

Function TFlatBitArray.GetCount : Integer;
  Begin
    Result := FCount;
  End;

Procedure TFlatBitArray.SetCount (const NewCount : Integer);
  Begin
    if NewCount = FCount then
      exit;
    SetLengthAndZero (FData, (NewCount + BitsPerLongWord - 1) div BitsPerLongWord);
    FCount := NewCount;
  End;

Function TFlatBitArray.GetRange (const Idx : Integer) : LongWord;
var F : Byte;
    I : Integer;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    F := Idx and 31;
    I := Idx shr 5;
    if F = 0 then
      Result := FData [I] else
      begin
        Result := FData [I] shr F;
        if I + 1 < Length (FData) then
          Result := Result or (FData [I + 1] shl (BitsPerLongWord - F));
      end;
  End;

Procedure TFlatBitArray.SetRange (const Idx : Integer; const Value : LongWord);
var F : Byte;
    I : Integer;
  Begin
    {$IFOPT R+}
    if (Idx < 0) or (Idx >= FCount) then
      IndexError (Idx);
    {$ENDIF}
    F := Idx and 31;
    I := Idx shr 5;
    if F = 0 then
      FData [I] := Value else
      begin
        FData [I] := (FData [I] and LowBitMask (F))
                  or (Value shl F);
        if I + 1 < Length (FData) then
          FData [I + 1] := (FData [I + 1] and HighBitMask (F))
                        or (Value shr (BitsPerLongWord - F));
      end;
  End;

Function TFlatBitArray.CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean;
var B, I   : LongWord;
    IL, IH : Integer;
  Begin
    {$IFOPT R+}
    if (LoIdx < 0) or (LoIdx > HiIdx) or (HiIdx >= FCount) then
      IndexError (HiIdx);
    {$ENDIF}

    // Check bits in FData [IL]
    IL := LoIdx shr 5;
    IH := HiIdx shr 5;
    B := HighBitMask (LoIdx and 31);
    I := FData [IL];
    if Value then
      Result := I or B = I else
      Result := I and not B = I;
    if not Result or (IL = IH) then
      exit;

    // Check bits in FData [IH]
    B := LowBitMask (HiIdx and 31);
    I := FData [IH];
    if Value then
      Result := I or B = I else
      Result := I and not B = I;
    if not Result or (IH = IL + 1) then
      exit;

    // Check bits in FStore [IL + 1..IR - 1]
    For I := IL + 1 to IH - 1 do
      if (Value and (FData [I] <> TrueLongWord)) or
         (not Value and (FData [I] <> FalseLongWord)) then
        begin
          Result := False;
          exit;
        end;

    Result := True;
  End;

Procedure TFlatBitArray.Fill (const LoIdx, HiIdx : Integer; const Value : Boolean);
var B, I   : LongWord;
    IL, IH : Integer;
  Begin
    {$IFOPT R+}
    if (LoIdx < 0) or (LoIdx > HiIdx) or (HiIdx >= FCount) then
      IndexError (HiIdx);
    {$ENDIF}

    IL := LoIdx shr 5;
    IH := HiIdx shr 5;

    // Set bits in FData [IL]
    if IH = IL then
      B := RangeBitMask (LoIdx and 31, HiIdx and 31) else
      B := HighBitMask (LoIdx and 31);
    I := FData [IL];
    if Value then
      FData [IL] := I or B else
      FData [IL] := I and not B;
    if IH = IL then
      exit;

    // Set bits in FData [IH]
    B := LowBitMask (HiIdx and 31);
    I := FData [IH];
    if Value then
      FData [IH] := I or B else
      FData [IH] := I and not B;
    if IH = IL + 1 then
      exit;

    // Set bits in FData [IL + 1..IR - 1]
    For I := IL + 1 to IH - 1 do
      if Value then
        FData [I] := TrueLongWord else
        FData [I] := FalseLongWord;
  End;


  
{                                                                              }
{ Self testing code                                                            }
{                                                                              }
Procedure SelfTest;
var I : Integer;
    F : TIntegerArray;
  Begin
    // TIntegerArray
    F := TIntegerArray.Create;
    For I := 0 to 16384 do
      Assert (F.AddItem (I) = I, 'Array.Add');
    Assert (F.Count = 16385, 'Array.Count');
    For I := 0 to 16384 do
      Assert (F [I] = I,     'Array.GetItem');
    For I := 0 to 16384 do
      F [I] := I + 1;
    For I := 0 to 16384 do
      Assert (F [I] = I + 1, 'Array.SetItem');
    F.Delete (0, 1);
    Assert (F.Count = 16384, 'Array.Delete');
    For I := 0 to 16383 do
      Assert (F [I] = I + 2, 'Array.Delete');
    F.Insert (0, 2);
    F [0] := 0;
    F [1] := 1;
    For I := 0 to 16384 do
      Assert (F [I] = I,     'Array.Insert');
    F.Count := 4;
    Assert (F.Count = 4,     'Array.SetCount');
    F [0] := 9;
    F [1] := -2;
    F [2] := 3;
    F [3] := 4;
    F.Sort;
    Assert (F [0] = -2,      'Array.Sort');
    Assert (F [1] = 3,       'Array.Sort');
    Assert (F [2] = 4,       'Array.Sort');
    Assert (F [3] = 9,       'Array.Sort');
    F.Clear;
    Assert (F.Count = 0,     'Array.Clear');
    F.Free;
  End;



end.

