// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: btnode.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 08/22/2000 
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The BtreeNode class is used by the gxBtree class to insert, delete, 
and search for database keys in a node. Database keys are derived 
from the DatabaseKeyB class and are kept in sorted order within 
each B-tree node.
*/
// ----------------------------------------------------------- // 
#include <string.h>
#include "btnode.h"

BtreeNode::BtreeNode(BtreeSize_t dbkey_size, BtreeNodeOrder_t order)
{
  // Each node must contain 5 or more keys
  if(order < 5) order = 5;
  
  key_size = dbkey_size;
  node_order = order-1;
  key_entries = new char[(key_size * node_order)];
  key_count = (BtreeKeyCount_t)0;
  left_child = (FAU)0;
  node_address = (FAU)-1;
}

BtreeNode::~BtreeNode()
{
  if(key_entries) delete[] key_entries;
}

FAU BtreeNode::GetBranch(BtreeKeyLocation_t key_location)
// Returns the node's left or right child based on the specified
// key location.
{
  // The -1th branch represents the nodes left child. 
  if(key_location < (BtreeKeyLocation_t)0) return left_child;


  // The right branch of this key will be returned starting
  // from the 0th branch on.
  char *ptr = key_entries + (key_size * key_location);
  FAU right_child;
  memmove(&right_child, ptr, sizeof(FAU));
  return right_child;
}

int BtreeNode::FindKeyLocation(const DatabaseKeyB &key,
			       DatabaseKeyB &compare_key,
			       BtreeKeyLocation_t &key_location)
// Find the key location. Returns 0 if an entry for the key
// already exists, 1 if the specified key is greater then
// all the entries in this node, or -1 if the specified key
// is less the all the entries in this node. The key class
// derived from the abstract DatabaseKeyB base class defines
// the key methods used by this function to compare the
// keys. The key location or the branch to keep searching
// down the tree is passed back in the "key_location"
// variable.
{
  key_location = (BtreeKeyLocation_t)key_count-1;

  while(key_location >= (BtreeKeyLocation_t)0) {
    char *ptr = key_entries + (key_size * key_location);
    memmove(&compare_key.right_child, ptr, sizeof(FAU));
    memmove(compare_key.db_key, ptr+sizeof(FAU), compare_key.KeySize());
    if(key > compare_key) return 1;
    if(key == compare_key) return 0;
    key_location--;
  }

  return -1;
}

void BtreeNode::LoadKey(DatabaseKeyB &key, BtreeKeyLocation_t key_location)
// Copies the key at the specified location into the "key" variable.
{
  char *ptr = key_entries + (key_size * key_location);
  memmove(&key.right_child, (void *)ptr, sizeof(FAU));
  memmove(key.db_key, ptr+sizeof(FAU), key.KeySize());
}

void BtreeNode::SplitNode(BtreeNode &node, BtreeKeyLocation_t split_location)
// Splits this node by moving the keys, starting at the "split_location",
// into specified node.
{
  node.key_count = key_count - (BtreeKeyCount_t)split_location; 
  char *ptr = key_entries + (split_location * key_size);
  memmove(node.key_entries, ptr, (node.key_count * key_size));
  key_count = split_location;
}

void BtreeNode::InsertDatabaseKey(DatabaseKeyB &key,
				  BtreeKeyLocation_t key_location)
// Insert the database key into the node at the specified location.
{
  memmove((key_entries + ((key_location+1) * key_size)),
	  (key_entries + (key_location * key_size)),
  	  (key_size * (key_count - (BtreeKeyCount_t)key_location)));
	  char *ptr = key_entries + (key_size * key_location);
  memmove(ptr, &key.right_child, sizeof(FAU));
  memmove(ptr+sizeof(FAU), key.db_key, key.KeySize());
  key_count++;
}

void BtreeNode::OverWriteDatabaseKey(DatabaseKeyB &key,
				     BtreeKeyLocation_t key_location)
// Overwrite the key at the specified location.
{
  char *ptr = key_entries + (key_size * key_location);
  memmove(ptr, &key.right_child, sizeof(FAU));
  memmove(ptr+sizeof(FAU), key.db_key, key.KeySize());
}

void BtreeNode::MergeNode(BtreeNode &node)
// Merge this node with the specified node.
{
  memmove((key_entries + (key_count * key_size)), node.key_entries,
	  (node.key_count * key_size));
  key_count += node.key_count;
}

void BtreeNode::DeleteDatabaseKey(BtreeKeyLocation_t key_location)
// Delete the database key at specified location.
{
  key_count--;
  memmove((key_entries + (key_location * key_size)),
	  (key_entries + ((key_location+1) * key_size)),
	  ((key_count - key_location) * key_size));
}

size_t BtreeNode::SizeOfBtreeNode()
// Calculates the total size of this Btree node.
{
  size_t node_size = (key_size * node_order);
  node_size += sizeof(key_count);
  node_size += sizeof(left_child);
  return node_size;
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
