/*===========================================================================
SOLAR v0.95.2 slrmail :: Module ..\slrmail\slrmail.c
Original Author: Kevin Houle <kjhoule@iowegia.dsm.ia.us>

This software module has been placed in the public domain.
===========================================================================*/

/* Include Files */
#include <stdio.h>
#include <dir.h>
#include <string.h>
#include "..\common\version.h"
#include "codedate.h"
#include "..\common\yesno.h"
#include "slrmail.h"

/*
 * Function: long main(int argc, char *argv[])
 * Purpose : Write user's mailbox to SOUP *.MSG file in temp_path
 * Return  : Number of mail messages written, -1 on error and set _slrerr.
*/

int main(int argc, char *argv[])
{
  extern    char    *optarg;
  unsigned  char    x;
  const     char    *optionS = "u:U:vVnNfF";
  char              orig_path[MAXPATH];
  char              username[10];

  /* Local functions */
  FILE *open_mail_file(char *mail_path);
  FILE *open_areas_file();
  FILE *open_msg_file();
  FILE *open_btemp_file();
  int  write_bmsg(FILE *btemp_file, FILE *msg_file);

  FILE *mail_file   = NULL;
  FILE *areas_file  = NULL;
  FILE *msg_file    = NULL;
  FILE *btemp_file  = NULL;

  char temp[10];
  char temp_mail_path[MAXPATH];
  int  index1         = 0;
  int  index2         = 0;
  int  y              = 0;
  int  done           = 0;
  int  exit_code      = x_success;
  int  do_mail_update = NO;
  long mail_count     = 0L;
  struct ffblk ffblk;

  /* Store original directory */

  getcwd(orig_path, MAXPATH);

  /* Set default values for globals */

  strcpy(temp_path,"NONE");
  strcpy(user_path,"NONE");
  strcpy(config_path,"NONE");
  strcpy(username,"NONE");
  strcpy(waffle_version,"NONE");

  use_fossil    = NO;  /* default to using no FOSSIL */
  mail_type     = b;   /* default to 8-bit binary format */
  mail_index    = n;   /* default to no IDX files */
  delete_mail   = NO;  /* default to not delete mail after batching */
  verbose       = NO;  /* default to non-verbose  */
  port          = 99;  /* Default for no port       */

  /* Parse command line. Be sure to compile with
     unsigned int enabled for the \xFF to work. */

  while ((x = getopt(argc, argv, optionS)) != '\xFF')
  {
    switch (toupper(x)) {     /* force parameter to upper case */
      case '?'  : usage();
                  exit_code = x_usage;
                  goto ExitMain;
      case 'U'  : if ((strcmp(username,"NONE")) == 0)
                  {
                    strcpy(username,optarg);
                    break;
                  }
                  else
                  {
                    strcpy(_slrerr,"cannot specify username twice on command line");
                    exit_code = x_fatal;
                    goto ExitMain;
                  }
      case 'V'  : verbose = YES;
                  break;
      case 'N'  : do_mail_update = YES;
                  break;
      case 'F'  : use_fossil = YES;
									break;
    }
  }

  /* Check to see if username was set. If not, try
     to set from PCBDOOR.SYS or DOORS.SYS file. */

  if (strcmp(username,"NONE") == 0)
  {
    if (read_doors() != 0)
    {
      usage();        /* no username can be determined, exit with usage */
      exit_code = x_usage;
      goto ExitMain;
    }
  }

  if (verbose == YES)
  {
    RPrintf("\nSOLAR v%s <> slrmail (%s)\n",VERSION,CODE_DATE);
    RPrintf("Processing mail for user: %s\n\n",username);
  }

  /* Load configuration files. */

  if ((load_static(username) != 0)    || \
      (load_config() != 0)            || \
      (user_defaults() != 0)          || \
      (set_directory(temp_path) != 0))
  {
    exit_code = x_fatal;
    goto ExitMain;
  }

  if (verbose == YES)
    RPrintf("Default directory set to %s\n",temp_path);

  /* Check version of Waffle */

  if (strcmp(waffle_version,"1.64") == 0)
  {
    /* Single mail file per message named username.# */

    strcpy(temp_mail_path,user_path);
    strcat(temp_mail_path,"\\");
    strcat(temp_mail_path,username);
    strcat(temp_mail_path,".*");
  }
  else
  {
    /* Single quasi-MMDF mail file for all messages */

    strcpy(temp_mail_path,user_path);
    strcat(temp_mail_path,"\\");
    strcat(temp_mail_path,MAILBOX_NAME);
  }

  if (verbose == YES)
    RPrintf("Searching for %s\n",temp_mail_path);

  /* Check for mail file */

  done = findfirst(temp_mail_path,&ffblk,0);
  if (done != 0)
  {
    RPrintf("No messages in mailbox\n");
    exit_code = x_success;
    goto ExitMain;
  }
  else
  {
    buf[0] = '\0';
    ltoa(mail_count,temp,10);
    if (verbose == NO)
      RPrintf("Packing mailbox :  ");
  }

  /* Process all mail */

  while (!done)
  {
    if (verbose == YES)
      RPrintf("\nFound mailbox file %s\n",ffblk.ff_name);

    if ((mail_file = open_mail_file(ffblk.ff_name)) != NULL)
    {
      while ((fgets(buf,1024,mail_file)) != NULL)
      {
        /* Open MSG file if needed. */
        if (!msg_file)
        {
          if (get_batch_prefix() != 0)
          {
            exit_code = x_fatal;
            goto ExitMain;
          }
          if (verbose == YES)
            RPrintf("Creating batch file. Prefix: %s\n",batch_prefix);
          if ((msg_file = open_msg_file()) == NULL)
          {
            exit_code = x_fatal;
            goto ExitMain;
          }

          /* If under 1.64, need to write initial CTRL-A string
             to MSG file for MMDF format only */

          if (strcmp(waffle_version,"1.64") == 0)
          {
            switch (mail_type) {
                case M  : fprintf(msg_file,"\x01\x01\x01\x01");
                          if (verbose == YES)
                            RPrintf("MMDF message header written\n");
                          break;
                default : break;
            } /* switch */
          } /* if */
        } /* Open message file */
       
        /* Check for other than 1.64 format mailbox. */

        if (strcmp(waffle_version, "1.64") != 0)
        {
          /* Check for leading CTRL-A string */

          if ((strncmp(buf,"\x01\x01\x01\x01",4)) == 0)
          {
            /* Check to see if we are at EOF */
          
            if (strlen(buf) == 4)
            {
              if (verbose == YES) RPrintf("End of mailbox file\n");
              break; /* Exit loop. Drop remainder on the floor. */
            }

            /* Beginning a new message. Count it. */
      
            mail_count++;
            if (verbose == YES)
            {
              RPrintf("Processing message number %lu\n",mail_count);
            }
            else
            {
              for (y = 0; y < strlen(temp); y++) RPrintf("\b");
              ltoa(mail_count,temp,10);
              RPrintf("%s",temp);
            }

            /* If not writing MMDF format, strip MMDF crap */

            if (mail_type != M)
            {
              index1 = 0;
              while (buf[index1] == '\x01') index1++;
              index2 = 0;
              while (buf[index1] != '\n') buf[index2++] = buf[index1++];
              buf[index2++] = '\n';
              buf[index2] = '\0';
              if (verbose == YES) RPrintf("MMDF formatting removed\n"); /* yea! */
            } /* if */
 
            /* If using binary message type, we need to handle a temp file. */

            if (mail_type == b)
            {
              if (btemp_file)
              {
                write_bmsg(btemp_file, msg_file);
                fclose(btemp_file);
              }
              btemp_file = open_btemp_file();
            } /* binary message file */
          } /* new message */
        } /* non-1.64 format */

        /* Write to MSG file, or btemp file */

        switch (mail_type) {
          case m  : fprintf(msg_file,"%s",buf);   break;
          case M  : fprintf(msg_file,"%s",buf);   break;
          case b  : fprintf(btemp_file,"%s",buf); break;
        }

      } /* Done reading */

      /* Close mailbox file */
      
      fclose(mail_file);
      if (verbose == YES)
        RPrintf("Finished processing mailbox file\n");

      /* If binary message type, we still have a temp file to handle */

      if (mail_type == b)
      {
        if (btemp_file)
        {
          write_bmsg(btemp_file, msg_file);
          fclose(btemp_file);
        }
      }
      
      /* Check for 1.64 again */

      if (strcmp(waffle_version,"1.64") == 0)
      {
        /* One 1.64 mailbox file has been processed. */

        mail_count++;
        if (verbose == YES)
        {
          RPrintf("Mail message number %lu completed\n",mail_count);
        }
        else
        {
          for (y = 0; y < strlen(temp); y++) RPrintf("\b");
          ltoa(mail_count,temp,10);
          RPrintf("%s",temp);
        } /* count message */
      } /* 1.64 processing */
    } /* process mail file */

    done = findnext(&ffblk);   /* Get the next mailbox file, if any. */
  } /* while */

  /* All done writing MSG file */

  fclose(msg_file);

  if (verbose == YES)
    RPrintf("All mailboxes processed. %lu messages written\n\n",mail_count);
  else
    RPrintf(" mail messages packed\n");

  /* Write info to AREAS file if messages were found. Always write a 
     mail index type of 'n', as no header files are produced for mail, 
     and an area kind of 'm' for private mail. */

  if (mail_count > 0)
  {
    if (verbose == YES) RPrintf("Updating AREAS file\n");
    if ((areas_file = open_areas_file()) == NULL)
    {
      exit_code = x_fatal;
      goto ExitMain;
    }
    fprintf(areas_file,"%s\tmailbox\t",batch_prefix);
    switch (mail_type) {
      case m  : fprintf(areas_file, "m"); break;
      case M  : fprintf(areas_file, "M"); break;
      case b  : fprintf(areas_file, "b"); break;
      }
    switch (mail_index) {
      default : fprintf(areas_file, "n"); break;
    }
    fprintf(areas_file,"m\t%s\t%lu\n",MAILBOX_DESC,mail_count);
    fclose(areas_file);
  } /* Update AREAS file */

ExitMain:
  set_directory(orig_path);
  switch (exit_code) {
    case x_success  : sprintf(logbuf,"%s %lu messages ",username,mail_count);
                      switch (mail_type) {
                        case m  : strcat(logbuf,"UNIX format");
                                  break;
                        case M  : strcat(logbuf,"MMDF format");
                                  break;
                        case b  : strcat(logbuf,"8-bit binary format");
                                  break;
                      }
                      logit("slrmail");
                      if ((mail_count > 0L) && (do_mail_update == YES) && (delete_mail == YES))
                        mail_update(username);
                      break;
    case x_usage    : break;
    case x_fatal    : if (mail_file) fclose(mail_file);
                      if (msg_file) fclose(msg_file);
                      if (areas_file) fclose(areas_file);
                      fprintf(stderr,"\n%s\n",_slrerr);
                      sprintf(logbuf,"%s",_slrerr);
                      logit("slrmail");
                      break;
  }
  return exit_code;
}

/*
 * Function: open_mail_file(char *mail_path)
 * Purpose : Open mail file in read/text mode.
 * Return  : NULL on error, pointer to file on success.
*/

FILE *open_mail_file(char *mail_path)
{
  FILE *file;

  char path[MAXPATH];

  strcpy(path,user_path);
  strcat(path,"\\");
  strcat(path,mail_path);

  file = fopen(path,"rt");
  return file;
}

/*
 * Function: open_areas_file()
 * Purpose : Open SOUP AREAS file for append in binary mode.
 * Return  : A pointer to the index file, NULL on error and set _slrerr.
*/

FILE *open_areas_file()
{
  FILE *areas_file;

  char path[MAXPATH];

  strcpy(path,temp_path);   /* build path to AREAS file */
  strcat(path,"\\");
  strcat(path,AREAS_NAME);

  if ((areas_file = fopen(path,"ab")) == NULL)  /* open failed */
    sprintf(_slrerr,"open_areas_file(): error opening %s file %s",AREAS_NAME,path);

  return areas_file;
}

/*
 * Function: open_msg_file()
 * Purpose : Open a MSG file in binary mode in temp directory.
 * Return  : Pointer to MSG file, NULL on error.
*/

FILE *open_msg_file()
{
  FILE  *msg_file   = NULL;

  char  batch_path[MAXPATH];

  strcpy(batch_path, temp_path);  /* build path to file */
  strcat(batch_path, "\\");
  strcat(batch_path, batch_prefix);  /* batch_prefix is unique by user */
  strcat(batch_path, ".");
  strcat(batch_path, BATCH_EXT);

  if ((msg_file = fopen(batch_path, "wb")) == NULL)  /* open failed */
    sprintf(_slrerr,"open_msg_file(): error opening %s for writing.", batch_path);

  return msg_file;
}

/*
 * Function: FILE *open_btemp_file()
 * Purpose : Open temporary file in binary mode.
 * Return  : Pointer to temp file, NULL on error.
*/

FILE *open_btemp_file()
{
  FILE  *btemp_file   = NULL;

  if ((btemp_file = tmpfile()) == NULL)
    sprintf(_slrerr,"open_btemp_file(): error opening tmpfile for update.");

  return btemp_file;
}

/*
 * Function: int write_bmsg(FILE *btemp_file, FILE *msg_file)
 * Purpose : Re-write temp file into MSG file.
 * Return  : Zero on success, non-zero on error.
*/

int write_bmsg(FILE *btemp_file, FILE *msg_file)
{
  extern long filesize(FILE *filename);
  extern char *long_to_endian(long long_value);

  long bsize  = 0L;
  size_t count;
  char buf[1];

  bsize = filesize(btemp_file);

  fwrite(long_to_endian(bsize),4,1,msg_file);
  fseek(btemp_file, SEEK_SET, 0);

  count = fread(buf, 1, 1, btemp_file);
  while (count != 0)
  {
    fwrite(buf, 1, 1, msg_file);
    count = fread(buf, 1, 1, btemp_file);
  }

  return 0;
}

/*
 * Function: void usage()
 * Purpose : Display command line usage.
 * Return  : N/A
*/

void usage()
{
  RPrintf("\nSOLAR v%s <> slrmail (%s)\n",VERSION,CODE_DATE);
  RPrintf("\nUsage: slrmail -u username [-v] [-n] [-f]\n");
  return;
}
