{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move down.
    This procedure moves the cursor down to the
    next row and adjusts the screen row if it's
    necessary.

*************************************************)

    Procedure Move_Down( Var All: All_Type );
      Begin
        If ( All.Data.Cursor.Row < All.Text.FileSize )
          then
            Begin
              Inc( All.Data.Cursor.Row );
              If ( All.Data.Screen_Row < Wave.Window_Bottom )
                then
                  Inc( All.Data.Screen_Row );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move up.
    This procedure moves the cursor up to the
    next row and adjusts the screen row if it's
    necessary.

*************************************************)

    Procedure Move_Up( Var Data: Data_Type );
      Begin
        If ( Data.Cursor.Row > 1 )
          then
            Begin
              Dec( Data.Cursor.Row );
              If ( Data.Screen_Row > 1 )
                then
                  Dec( Data.Screen_Row );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move page down.
    This procedure moves the cursor down to the
    bottom or to the next page and adjusts the
    screen row if it's necessary.

*************************************************)

    Procedure Move_Page_Down( Var All: All_Type );
      Var
        Count: Byte;
      Begin
        If ( All.Data.Screen_Row < Wave.Window_Bottom )
          then
            While ( All.Data.Screen_Row < Wave.Window_Bottom ) and ( All.Data.Cursor.Row < All.Text.FileSize ) do
              Begin
                Inc( All.Data.Cursor.Row );
                Inc( All.Data.Screen_Row );
              End
          else
            For Count := 1 to Pred( Wave.Window_Bottom ) do
              If ( All.Data.Cursor.Row < All.Text.FileSize )
                then
                  Inc( All.Data.Cursor.Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move page up.
    This procedure moves the cursor up to the top
    or to the previous page and adjusts the screen
    row if it's necessary.

*************************************************)

    Procedure Move_Page_Up( Var Data: Data_Type );
      Var
        Count: Byte;
      Begin
        If ( Data.Screen_Row > 1 )
          then
            While ( ( Data.Screen_Row > 1 ) and ( Data.Cursor.Row > 1 ) ) do
              Begin
                Dec( Data.Cursor.Row );
                Dec( Data.Screen_Row );
              End
          else
            For Count := 1 to Pred( Wave.Window_Bottom ) do
              If ( Data.Cursor.Row > 1 )
                then
                  Dec( Data.Cursor.Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move Where.
    This procedure moves the cursor up to the
    given location in the text and adjusts the
    screen row if it's necessary.

*************************************************)

    Procedure Move_Where( Var Data: Data_Type; Row: LongInt );
      Begin
        Write_Wait;
        Data.Cursor.Row := Row;
        If ( Row > Wave.Window_Bottom )
          then
            Data.Screen_Row := Wave.Window_Bottom
          else
            Data.Screen_Row := Row;
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move Where short.
    This procedure moves the cursor up to the
    given location in the text and adjusts the
    screen row if it's necessary.

*************************************************)

    Procedure Move_Where_Short( Var Data: Data_Type; Row: LongInt; New_Screen_Row: Byte );
      Begin
        If ( Row > 1 )
          then
            Data.Cursor.Row := Row
          else
            Data.Cursor.Row := 1;
        If ( New_Screen_Row > Wave.Window_Bottom )
          then
            Data.Screen_Row := Wave.Window_Bottom
          else
            If ( New_Screen_Row < 1 )
              then
                Data.Screen_Row := 1
              else
                Data.Screen_Row := New_Screen_Row;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll up.
    Somewhat different from most methods, this
    procedure moves the row up one line, but
    leaves the screen cursor where it is.

*************************************************)

    Procedure Scroll_Up( Var Data: Data_Type );
      Begin
        If ( Data.Cursor.Row > 1 )
          then
            Dec( Data.Cursor.Row );
        If ( Data.Screen_Row > Data.Cursor.Row )
          then
            Dec( Data.Screen_Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scroll down.
    Like Scroll up, this procedure moves the row
    down one line, but leaves the screen cursor
    where it is.

*************************************************)

    Procedure Scroll_Down( Var All: All_Type );
      Begin
        If ( All.Data.Cursor.Row < All.Text.FileSize )
          then
            Inc( All.Data.Cursor.Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move top of window.
    This procedure moves the cursor to the top
    of the current window.

*************************************************)

    Procedure Move_Top_of_Window( Var Data: Data_Type );
      Var
        Count: Byte;
      Begin
        For Count := 2 to Data.Screen_Row do
          Move_Up( Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move bottom of window.
    This procedure moves the cursor to the bottom
    of the current window.

*************************************************)

    Procedure Move_Bottom_of_Window( Var All: All_Type );
      Var
        Count: Byte;
      Begin
        For Count := All.Data.Screen_Row to Pred( Wave.Window_Bottom ) do
          Move_Down( All );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Change block.
    This function returns true if the cursor is
    less than the block in the file.

*************************************************)

    Function Change_Block( Var Block, Cursor: Point_Type ): Boolean;
      Begin
        Change_Block := ( Block.Row > Cursor.Row ) or ( ( Block.Row = Cursor.Row ) and ( Block.Column > Cursor.Column ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust delete one.
    This procedure adjusts the block locations in
    a special way after a line is deleted.

*************************************************)

    Procedure Adjust_Delete1( Var Start, Finish, Cursor: Point_Type );
      Begin
        If Change_Block( Start, Cursor )
          then
            Begin
              Dec( Start.Row );
              If ( Start.Row = Cursor.Row )
                then
                  Start.Column := ( Start.Column + Cursor.Column );
            End;
        If Change_Block( Finish, Cursor )
          then
            Begin
              Dec( Finish.Row );
              If ( Finish.Row = Cursor.Row )
                then
                  Finish.Column := ( Finish.Column + Cursor.Column );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust delete two.
    This procedure adjusts the block locations in
    a special way after a line is deleted.

*************************************************)

    Procedure Adjust_Delete2( Var Start, Finish, Cursor: Point_Type );
      Begin
        If Change_Block( Start, Cursor )
          then
            Begin
              If ( Start.Row = Cursor.Row )
                then
                  Start.Column := ( Start.Column + Cursor.Column );
              Dec( Start.Row );
            End;
        If Change_Block( Finish, Cursor )
          then
            Begin
              If ( Finish.Row = Cursor.Row )
                then
                  Finish.Column := ( Finish.Column + Cursor.Column );
              Dec( Finish.Row );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust delete three.
    This procedure adjusts the block locations in
    a special way after a line is deleted.

*************************************************)

    Procedure Adjust_Delete3( Var Start, Finish, Cursor: Point_Type );
      Begin
        If Change_Block( Start, Cursor )
          then
            If ( Start.Row = Cursor.Row )
              then
                Start.Column := 1
              else
                Dec( Start.Row );
        If Change_Block( Finish, Cursor )
          then
            If ( Finish.Row = Cursor.Row )
              then
                Finish.Column := 1
              else
                Dec( Finish.Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Adjust insert.
    This procedure adjusts the block locations in
    a special way after a line is added.

*************************************************)

    Procedure Adjust_Insert( Var Start, Finish, Cursor: Point_Type );
      Begin
        If Change_Block( Start, Cursor )
          then
            Begin
              If Alter_Block( Start, Cursor )
                then
                  Start.Column := Succ( Start.Column - Cursor.Column );
              Inc( Start.Row );
            End;
        If Change_Block( Finish, Cursor )
          then
            Begin
              If Alter_Block( Finish, Cursor )
                then
                  Finish.Column := Succ( Finish.Column - Cursor.Column );
              Inc( Finish.Row );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete line.
    This procedure deletes a line from the text
    and adjusts the block markers.

*************************************************)

    Procedure Delete_Line( Var All: All_Type; Var Changed: Boolean );
      Begin
        If Delete_Text_Line( All.Text, All.Data.Cursor.Row )
          then
            Begin
              Changed := True;
              Adjust_Delete3( Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor );
            End
          else
            Write_Error( 401 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Insert new line.
    This function inserts a new line into the text
    when allowed to do so, then adjusts the block
    markers.

*************************************************)

    Function Insert_New_Line( Var All: All_Type; Var Buffer: Line_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        If ( Return_Split_Line and Insert_Mode )
          then
            Begin
              Okay := Split_Text( All.Text, Buffer, All.Data.Cursor, True );
              If Okay
                then
                  Adjust_Insert( Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor );
              Insert_New_Line := Okay;
            End
          else
            Insert_New_Line := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move return.
    This procedure either inserts a new line or
    moves the cursor down, then resets the column.

*************************************************)

    Procedure Move_Return( Var All: All_Type; Var Buffer: Line_Type; Var Changed: Boolean );
      Begin
        If Insert_New_Line( All, Buffer )
          then
            Begin
              If ( All.Data.Cursor.Row < All.Text.FileSize )
                then
                  Begin
                    Inc( All.Data.Cursor.Row );
                    If ( All.Data.Screen_Row < Wave.Window_Bottom )
                      then
                        Inc( All.Data.Screen_Row );
                    All.Data.Cursor.Column := 1;
                  End;
              Changed := True;
            End
          else
            Write_Error( 402 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Combine.
    This Procedure combines two lines together
    when allowed to do so.

*************************************************)

    Procedure Combine( Var All: All_Type; Var Buffer: Line_Type; Var Changed: Boolean );
      Begin
        If Delete_Combine_Line and ( All.Data.Cursor.Row < All.Text.FileSize )
          then
            If Remove_Return( All.Text, Buffer, All.Data.Cursor )
              then
                Begin
                  Changed := True;
                  Adjust_Delete1( Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor )
                End
              else
                Write_Error( 403 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Back combine.
    This procedure combines two lines together
    when allowed to do so.

*************************************************)

    Procedure Back_Combine( Var All: All_Type; Var Buffer: Line_Type; Var Changed: Boolean );
      Begin
        If Backspace_Combine_Line and ( All.Data.Cursor.Row > 1 )
          then
            Begin
              Move_Up( All.Data );
              Get_Text_Line( All.Text, All.Data.Cursor.Row, Buffer );
              All.Data.Cursor.Column := Succ( Buffer.Size );
              If Remove_Return( All.Text, Buffer, All.Data.Cursor )
                then
                  Begin
                    Changed := True;
                    Adjust_Delete2( Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor )
                  End
                else
                  Write_Error( 403 );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move to block.
    This procedure moves the cursor to the
    specified block marker.

*************************************************)

    Procedure Move_To_Block( Var Data: Data_Type; Where: Point_Type );
      Begin
        Write_Wait;
        Data.Cursor := Where;
        If ( Where.Row > Wave.Window_Bottom )
          then
            Data.Screen_Row := Wave.Window_Bottom
          else
            Data.Screen_Row := Where.Row;
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Clean file name.
    This procedure attempts to clean the file name
    string so that it contains a valid file name.

*************************************************)

    Procedure Clean_File_Name( Var Data: String );
      Var
        Point,
        Dot_Count: Byte;
      Begin
        Point := 0;
        Dot_Count := 0;
        While ( Point < Length( Data ) ) do
          Begin
            Inc( Point );
            Case Data[ Point ] of
              'a'..'z': Data[ Point ] := UpCase( Data[ Point ] );
              '.': Begin
                     Inc( Dot_Count );
                     If ( Dot_Count > 1 )
                       then
                         Begin
                           Delete( Data, Point, 1 );
                           Dec( Point );
                         End;
                   End;
              'A'..'Z', '0'..'9', '$', '&', '%', '`', '''', '(', ')', '{', '}', '-', '@', '^',
                                  '~', '!', '#', '/', '\', ':':  ;
              else Begin
                     Delete( Data, Point, 1 );
                     Dec( Point );
                   End;
            End; { Case }
          End;
        Point := Pos( '.', Data );
        If ( Point <> 0 )
          then
            Begin
              Point := ( Length( Data ) - Point );
              If ( Point > 3 )
                then
                  Begin
                    Point := Point - 3;
                    Delete( Data, Succ( Length( Data ) - Point ), Point );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Valid.
    This function analyzes the name for an input/
    output operation and returns true only if the
    name if valid.  Reading is set to true if the
    file is to be read, otherwise false indicates
    writing.

*************************************************)

    Function Valid( Var Data: String; Reading: Boolean ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( Data <> '' );
        If Okay
          then
            Begin
              Clean_File_Name( Data );
              If Reading
                then
                  Okay := ( Data[ 1 ] <> '.' ) and ( Data <> 'PRN' ) and ( Data <> 'AUX' ) and ( Data <> 'CON' ) and
                          ( Data <> 'NUL' )
                else
                  Okay := ( Data[ 1 ] <> '.' ) and ( Data <> 'AUX' ) and ( Data <> 'CON' ) and ( Data <> 'NUL' );
            End;
        Valid := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read the block.
    This procedure attempts to read in a block
    from a file.

*************************************************)

    Procedure Read_The_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Block_Start, Block_Finish, Cursor: Point_Type;
                              Var Changed: Boolean );
      Var
        Name: String;
        Error_Code: Word;
        InFile: System.Text;
      Begin
        Finish_Screen;
        Get_Read_File_Name( Name );
        If Valid( Name, True )
          then
            Begin
              Write_Wait;
              Assign( InFile, Name );
             {$I-}
              Reset( InFile );
              Error_Code := IoResult;
             {$I+}
              If ( Error_Code = 0 )
                then
                  Begin
                    If Read_Block( Text, Buffer, Block_Start, Block_Finish, Cursor, InFile )
                      then
                        Changed := True
                      else
                        Write_Error( 404 );
                    Close( InFile );
                  End
                else
                  Write_Error( Error_Code );
              Write_Complete;
            End
          else
            Write_Error( 405 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write the block.
    This procedure writes the marked block in
    a file.

*************************************************)

    Procedure Write_The_Block( Var Text: Text_Type; Var Block_Start, Block_Finish: Point_Type );
      Var
        Name: String;
        Error_Code: Word;
        OutFile: System.Text;
      Begin
        Finish_Screen;
        Get_Write_File_Name( Name );
        If Valid( Name, False )
          then
            Begin
              Write_Wait;
              Assign( OutFile, Name );
             {$I-}
              Rewrite( OutFile );
              Error_Code := IoResult;
             {$I+}
              If ( Error_Code = 0 )
                then
                  Begin
                    If not Write_Block( Text, Block_Start, Block_Finish, OutFile )
                      then
                        Write_Error( 406 );
                    Close( OutFile );
                  End
                else
                  Write_Error( Error_Code );
              Write_Complete;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print the block.
    This procedure writes the marked block to
    the default printer device.

*************************************************)

    Procedure Print_The_Block( Var Text: Text_Type; Var Block_Start, Block_Finish: Point_Type );
      Var
        OutFile: System.Text;
      Begin
        Write_Wait;
        Assign( OutFile, 'Prn' );
        Rewrite( OutFile );
        If not Write_Block( Text, Block_Start, Block_Finish, OutFile )
          then
            Write_Error( 411 );
        Close( OutFile );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print the text.
    This procedure writes the entire text to the
    default printer device.

*************************************************)

    Procedure Print_Text( Var Text: Text_Type );
      Var
        Start,
        Finish: Point_Type;
        OutFile: System.Text;
      Begin
        Write_Wait;
        Start.Row := 1;
        Start.Column := 1;
        Finish.Row := Text.FileSize;
        Finish.Column := Line_Limit;
        Assign( OutFile, 'Prn' );
        Rewrite( OutFile );
        If not Write_Block( Text, Start, Finish, OutFile )
          then
            Write_Error( 411 );
        Close( OutFile );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search text.
    This procedure prompts the user for the
    necessary information, then searches the text
    for the match.

*************************************************)

    Procedure Search_Text( Var All: All_Type; Var Info: Search_Data_Type );
      Begin
        Finish_Screen;
        Get_Search_Data( Info.Find_String );
        If ( Info.Find_String <> '' )
          then
            Begin
              Set_Start( Info.Find_Position, All.Data.Cursor );
              Search_Engine( All, Info );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Search and replace text.
    This procedure prompts the user for the
    necessary information, then searches the text
    for the matchings.

*************************************************)

    Procedure Search_And_Replace_Text( Var All: All_Type; Var Info: Search_Data_Type; Var Changed: Boolean );
      Begin
        Finish_Screen;
        Get_Search_And_Replace_Data( Info.Find_String, Info.Replace_String );
        If ( Info.Find_String <> '' )
          then
            Begin
              Set_Start( Info.Find_Position, All.Data.Cursor );
              Replace_Engine( All, Info, Changed );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Continue search.
    This procedure continues the last search, or
    search and replace operation from where it
    left off.

*************************************************)

    Procedure Continue_Search( Var All: All_Type; Var Info: Search_Data_Type; Var Changed: Boolean );
      Begin
        If ( Info.Find_String <> '' )
          then
            Begin
              If ( Info.Last_Operation = Find )
                then
                  Begin
                    Finish_Screen;
                    Info.Find_Position := All.Data.Cursor;
                    Search_Engine( All, Info );
                  End
                else
                  Begin
                    Finish_Screen;
                    Info.Find_Position := All.Data.Cursor;
                    Replace_Engine( All, Info, Changed );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert control n.
    This procedure attempts to insert a new line
    at the current cursor location.

*************************************************)

    Procedure Insert_Control_N( Var All: All_Type; Var Buffer: Line_Type; Var Changed: Boolean );
      Begin
        If Insert_New_Line( All, Buffer )
          then
            Changed := True
          else
            Write_Error( 402 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Copy the block
    This procedure tries to copy the block of
    text.  If it fails, it displays the error
    message.

*************************************************)

    Procedure Copy_The_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                              Var Changed: Boolean );
      Begin
        Write_Wait;
        If ICopy_Block( Text, Buffer, Start, Finish, Cursor )
          then
            Changed := True
          else
            Write_Error( 407 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete the block
    This procedure tries to remove the block of
    text.  If it fails, it displays the error
    message.

*************************************************)

    Procedure Delete_The_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                                Var Changed: Boolean );
      Begin
        Write_Wait;
        If Delete_Block( Text, Buffer, Start, Finish, Cursor )
          then
            Changed := True
          else
            Write_Error( 408 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move the block
    This procedure tries to move the block of
    text.  If it fails, it displays the error
    message.

*************************************************)

    Procedure Move_The_Block( Var Text: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type;
                              Var Changed: Boolean );
      Begin
        Write_Wait;
        If Move_Block( Text, Buffer, Start, Finish, Cursor )
          then
            Changed := True
          else
            Write_Error( 409 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Show where.
    This procedure is designed to display on
    screen the current location of the cursor and
    possibly other information as well.

*************************************************)

    Procedure Show_Where( Var All: All_Type );
      Begin
        Finish_Screen;
        Display_Information( All );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move Line to.
    This procedure is designed to move the cursor
    to the line entered in.

*************************************************)

    Procedure Move_Line_To( Var All: All_Type );
      Var
        New_Line: LongInt;
      Begin
        Finish_Screen;
        Get_New_Line( New_Line, All.Text.FileSize );
        If ( ( New_Line >= 1 ) and ( New_Line <= All.Text.FileSize ) )
          then
            Begin
              All.Data.Cursor.Row := New_Line;
              All.Data.Cursor.Column := 1;
              If ( New_Line > Wave.Window_Bottom )
                then
                  All.Data.Screen_Row := Wave.Window_Bottom
                else
                  All.Data.Screen_Row := New_Line;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Separator.
    This procedure helps write out parts of the
    help screen quickly.

*************************************************)

    Procedure Separator( Data: String; Width: Byte );
      Begin
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, Center( Data, Width, ' ' ) );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
      End;
