{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Find look value.
    This procedure uses the Soundex system to
    manipulate the given word string to create the
    dictionary look up key value.

*************************************************)

    Procedure Find_Look_Value( Word: String; Var Look: Look_Type );
      Var
        Count: Byte;
      Begin
        While ( Pos( ' ', Word ) = 1 ) do
          Delete( Word, 1, 1 );
        For Count := 2 to Length( Word ) do
          Case UpCase( Word[ Count ] ) of
            'A', 'E', 'H', 'I', 'O', 'U', 'W', 'Y',
            #131, #132, #133, #160, #134, #142, #143,
            #136, #137, #138, #130, #144,
            #140, #139, #141, #161,
            #147, #148, #149, #162, #153,
            #150, #129, #151, #163, #154,
            #152, #145, #146, #166, #167: Word[ Count ] := #0;
            'B', 'F', 'P', 'V': Word[ Count ] := #1;
            'C', 'G', 'J', 'K', 'Q', 'S', 'X', 'Z': Word[ Count ] := #2;
            'D', 'T': Word[ Count ] := #3;
            'L': Word[ Count ] := #4;
            #164, #165,
            'M', 'N': Word[ Count ] := #5;
            'R': Word[ Count ] := #6;
            else Word[ Count ] := #0;
          End; { Case }
        For Count := Length( Word ) downto 2 do
          If ( Word[ Count ] = Word[ Pred( Count ) ] )
            then
              Delete( Word, Count, 1 );
        For Count := Length( Word ) downto 1 do
          If ( Word[ Count ] = #0 )
            then
              Delete( Word, Count, 1 );
        FillChar( Look, SizeOf( Look_Type ), #0 );
        For Count := 1 to SizeOf( Look_Type ) do
          If ( Count <= Length( Word ) )
            then
              Look[ Count ] := Word[ Count ];
        Look[ 1 ] := UpCase( Look[ 1 ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Make word.
    This procedure takes the given value and
    eliminates the preceding and succeeding
    spaces so that it can be used as a
    replacement value.

*************************************************)

    Procedure Make_Word( Var Line: String );
      Var
        Index: Byte;
      Begin
        { Remove the preceding blanks. }
        Index := Length( Line );
        While ( ( Index > 0 ) and ( Line[ 1 ] in [ ' ' ] ) ) do
          Begin
            Delete( Line, 1, 1 );
            Dec( Index );
          End;
        { Remove the trailing blanks. }
        While ( ( Index > 0 ) and ( Line[ Index ] in [ ' ' ] ) ) do
          Begin
            Delete( Line, Index, 1 );
            Dec( Index );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Convert to string.
    This procedure takes the given word from the
    word data base and converts it into a string
    so that it can be sent back as a replacement.

*************************************************)

    Procedure Convert_To_String( Var Replacement: Word_String_Type; Var New_Word: String; Capitalized: Boolean );
      Var
        Count,
        Index: Byte;
      Begin
        Index := Word_Data_Length;
        While ( Index > 0 ) and ( Replacement[ Index ] = ' ' ) do
          Dec( Index );
        New_Word := '';
        For Count := 1 to Index do
          New_Word := New_Word + Replacement[ Count ];
        If Capitalized and ( Index > 0 )
          then
            New_Word[ 1 ] := UpCase( New_Word[ 1 ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Convert from string.
    The purpose of this routine is to convert the
    given string into a Word_String_Type

*************************************************)

    Procedure Convert_From_String( Original_Word: String; Var Result: Word_String_Type; Capitalize: Boolean );
      Var
        Index,
        Word_Length: Byte;
      Begin
        FillChar( Result, SizeOf( Word_String_Type ), ' ' );
        Word_Length := Length( Original_Word );
        If ( Word_Length > Word_Data_Length )
          then
            Word_Length := Word_Data_Length;
        For Index := 1 to Word_Length do
          Result[ Index ] := Original_Word[ Index ];
        If Capitalize
          then
            Result[ 1 ] := UpCase( Result[ 1 ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Make word lowercase.
    The purpose of this routine is to remove all
    the capital letters in the given word and
    replace them with lower-case ones.
    Capitalized is true if the word started with
    a capital letter.

*************************************************)

    Procedure Make_Word_LowerCase( Original_Word: String; Var Result: Word_String_Type; Var Capitalized: Boolean );
      Const
        Capital_Letters = [ 'A' .. 'Z' ];
        LowerCase_Letters: array [ 'A' .. 'Z' ] of Char = 'abcdefghijklmnopqrstuvwxyz';
      Var
        Index,
        Word_Length: Byte;
      Begin
        Capitalized := ( Original_Word[ 1 ] in Capital_Letters );
        FillChar( Result, SizeOf( Word_String_Type ), ' ' );
        Word_Length := Length( Original_Word );
        If ( Word_Length > Word_Data_Length )
          then
            Word_Length := Word_Data_Length;
        For Index := 1 to Word_Length do
          If ( Original_Word[ Index ] in Capital_Letters )
            then
              Result[ Index ] := LowerCase_Letters[ Original_Word[ Index ] ]
            else
              Result[ Index ] := Original_Word[ Index ];
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Clear the list.
    This procedure clears the word list.

*************************************************)

    Procedure Clear_List( Var List: Word_List_Type );
      Begin
        List.Amount := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to list.
    This procedure adds the given word to the
    word list if there is any room for it.

*************************************************)

    Procedure Add_To_List( Var List: Word_List_Type; Var Word: Change_Data_Type );
      Begin
        If ( List.Amount < Maximum_Word_List_Amount )
          then
            Begin
              Inc( List.Amount );
              If ( Capital in Word.Info )
                then
                  Word.Word[ 1 ] := UpCase( Word.Word[ 1 ] );
              List.Word[ List.Amount ] := Word.Word;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: In dictionary.
    This function returns false if the given word
    is not in the dictionary.

*************************************************)

    Function In_Dictionary( Dictionary: Tree_Type; Var Word: Change_Data_Type; Var Capitalize: Boolean ): Boolean;
      Var
        Hold: Change_Data_Type;
        Found: Boolean;
      Begin
        Hold := Word;
        If Find_In_Tree( Dictionary, Word )
          then
            If ( Word.Word = Hold.Word )
              then
                Found := True
              else
                Begin
                  Repeat
                    Found := Find_Next_In_Tree( Dictionary, Word )
                  Until ( not Found ) or ( Word.Word = Hold.Word ) or ( Word.Look <> Hold.Look );
                  Found := ( Word.Word = Hold.Word );
                End
          else
            Found := False;
        If Found
          then
            Capitalize := ( Capital in Word.Info );
        In_Dictionary := Found;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: In dictionary list.
    This function returns false if the given word
    is not in the dictionary.  It also fills the
    list as it works.

*************************************************)

    Function In_Dictionary_List( Var List: Word_List_Type; Var Dictionary: Tree_Type; Var Word: Change_Data_Type;
                                 Var Capitalize: Boolean ): Boolean;
      Var
        Hold: Change_Data_Type;
        Found: Boolean;
      Begin
        Hold := Word;
        If Find_In_Tree( Dictionary, Word )
          then
            If ( Word.Word = Hold.Word )
              then
                Found := True
              else
                Begin
                  Repeat
                    Add_To_List( List, Word );
                    Found := Find_Next_In_Tree( Dictionary, Word )
                  Until ( not Found ) or ( Word.Word = Hold.Word ) or ( Word.Look <> Hold.Look );
                  Found := ( Word.Word = Hold.Word );
                End
          else
            Found := False;
        If Found
          then
            Capitalize := ( Capital in Word.Info );
        In_Dictionary_List := Found;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scan dictionaries.
    This function scans a few of the various
    dictionaries to find the word.  It returns
    found as false if it isn't in the normal
    dictionaries.

*************************************************)

    Procedure Scan_Dictionaries( Word: Change_Data_Type; Var Found: Boolean; Size: Byte );
      Var
        Okay,
        Capitalize: Boolean;
        Look_Word: Change_Data_Type;
      Begin
        Look_Word := Word;
        Case Size of
          1 .. Word1_Maximum : Okay := In_Dictionary( Main_Dictionaries[ 1 ], Look_Word, Capitalize );
          Succ( Word1_Maximum ) .. Word2_Maximum: Okay := In_Dictionary( Main_Dictionaries[ 2 ], Look_Word, Capitalize );
          Succ( Word2_Maximum ) .. Word3_Maximum: Okay := In_Dictionary( Main_Dictionaries[ 3 ], Look_Word, Capitalize );
          Succ( Word3_Maximum ) .. Word4_Maximum: Okay := In_Dictionary( Main_Dictionaries[ 4 ], Look_Word, Capitalize );
          else Okay := In_Dictionary( Main_Dictionaries[ 5 ], Look_Word, Capitalize );
        End; { Case }
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary( Personal_Dictionary, Look_Word, Capitalize );
            End;
        Found := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scan dictionaries not list.
    This function scans the various dictionaries
    to find the word.

*************************************************)

    Procedure Scan_Dictionaries_No_List( Word: Change_Data_Type; Var Found, Auto_Change, Capitalize: Boolean;
                                         Var Replacement: Word_String_Type; Size: Byte );
      Var
        Okay: Boolean;
        Look_Word: Change_Data_Type;
      Begin
        Auto_Change := False;
        Look_Word := Word;
        Case Size of
          1 .. Word1_Maximum:
            Okay := In_Dictionary( Main_Dictionaries[ 1 ], Look_Word, Capitalize );
          Succ( Word1_Maximum ) .. Word2_Maximum:
            Okay := In_Dictionary( Main_Dictionaries[ 2 ], Look_Word, Capitalize );
          Succ( Word2_Maximum ) .. Word3_Maximum:
            Okay := In_Dictionary( Main_Dictionaries[ 3 ], Look_Word, Capitalize );
          Succ( Word3_Maximum ) .. Word4_Maximum:
            Okay := In_Dictionary( Main_Dictionaries[ 4 ], Look_Word, Capitalize );
          else
            Okay := In_Dictionary( Main_Dictionaries[ 5 ], Look_Word, Capitalize );
        End; { Case }
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary( Auto_Dictionary, Look_Word, Capitalize );
              Auto_Change := Okay and ( Look_Word.Word <> Look_Word.Word2 );
              If Auto_Change
                then
                  Replacement := Look_Word.Word2;
            End;
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary( Personal_Dictionary, Look_Word, Capitalize );
            End;
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary( Correct_Dictionary, Look_Word, Capitalize );
              Auto_Change := Okay and ( Look_Word.Word <> Look_Word.Word2 );
              If Auto_Change
                then
                  Replacement := Look_Word.Word2;
            End;
        Found := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Compile the list.
    This procedure is called to compile the rest
    of the suggestion list from the other
    dictionaries after the word isn't found.

*************************************************)

    Procedure Compile_List( Var List: Word_List_Type; Var Dictionaries: Dictionary_Array_Type; Var Look: Change_Data_Type;
                            Size: Byte );
      Var
        Okay,
        Capitalize: Boolean;
      Begin
        Look.Word := '                              ';
        If not ( Size in [ 1 .. Word1_Maximum ] )
          then
            Okay := In_Dictionary_List( List, Dictionaries[ 1 ], Look, Capitalize );
        Look.Word := '                              ';
        If not ( Size in [ Succ( Word1_Maximum ) .. Word2_Maximum ] )
          then
            Okay := In_Dictionary_List( List, Dictionaries[ 2 ], Look, Capitalize );
        Look.Word := '                              ';
        If not ( Size in [ Succ( Word2_Maximum ) .. Word3_Maximum ] )
          then
            Okay := In_Dictionary_List( List, Dictionaries[ 3 ], Look, Capitalize );
        Look.Word := '                              ';
        If not ( Size in [ Succ( Word3_Maximum ) .. Word4_Maximum ] )
          then
            Okay := In_Dictionary_List( List, Dictionaries[ 4 ], Look, Capitalize );
        Look.Word := '                              ';
        If not ( Size > Succ( Word4_Maximum ) )
          then
            Okay := In_Dictionary_List( List, Dictionaries[ 5 ], Look, Capitalize );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Scan dictionaries list.
    This function scans the various dictionaries
    to find the word.  It also adds the words to
    the list as it scans the lists.

*************************************************)

    Procedure Scan_Dictionaries_List( Word: Change_Data_Type; Var Found, Auto_Change: Boolean;
                                      Var Replacement: Word_String_Type; Var List: Word_List_Type; Size: Byte;
                                      Var Capitalize: Boolean );
      Var
        Okay: Boolean;
        Look_Word: Change_Data_Type;
      Begin
        Clear_List( List );
        Auto_Change := False;
        Look_Word := Word;
        Case Size of
          1 .. Word1_Maximum:
            Okay := In_Dictionary_List( List, Main_Dictionaries[ 1 ], Look_Word, Capitalize );
          Succ( Word1_Maximum ) .. Word2_Maximum:
            Okay := In_Dictionary_List( List, Main_Dictionaries[ 2 ], Look_Word, Capitalize );
          Succ( Word2_Maximum ) .. Word3_Maximum:
            Okay := In_Dictionary_List( List, Main_Dictionaries[ 3 ], Look_Word, Capitalize );
          Succ( Word3_Maximum ) .. Word4_Maximum:
            Okay := In_Dictionary_List( List, Main_Dictionaries[ 4 ], Look_Word, Capitalize );
          else
            Okay := In_Dictionary_List( List, Main_Dictionaries[ 5 ], Look_Word, Capitalize );
        End; { Case }
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary_List( List, Auto_Dictionary, Look_Word, Capitalize );
              Auto_Change := Okay and ( Look_Word.Word <> Look_Word.Word2 );
              If Auto_Change
                then
                  Replacement := Look_Word.Word2;
            End;
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary_List( List, Personal_Dictionary, Look_Word, Capitalize );
            End;
        If ( not Okay )
          then
            Begin
              Look_Word := Word;
              Okay := In_Dictionary_List( List, Correct_Dictionary, Look_Word, Capitalize );
              Auto_Change := Okay and ( Look_Word.Word <> Look_Word.Word2 );
              If Auto_Change
                then
                  Replacement := Look_Word.Word2;
            End;
        If ( not Okay )
          then
            Compile_List( List, Main_Dictionaries, Look_Word, Size );
        Found := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to dictionary.
    This procedure adds the given word to the
    appropriate dictionary.  If it fails, then a
    error is generated which halts the program.

*************************************************)

    Procedure Add_To_Dictionary( Var Dictionary: Tree_Type; Var Word: Change_Data_Type );
      Begin
        If not Insert_In_Tree( Dictionary, Word )
          then
            Core.Write_Error( 201, 'Add_To_Dictionary: Invalid insert' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to dictionaries.
    This procedure adds the given word to the
    appropriate dictionary.  If it fails, then a
    error is generated which halts the program.

*************************************************)

    Procedure Add_To_Dictionaries( Var Word: Change_Data_Type; Size: Byte );
      Begin
        Case Size of
          1 .. Word1_Maximum: Add_To_Dictionary( Main_Dictionaries[ 1 ], Word );
          Succ( Word1_Maximum ) .. Word2_Maximum: Add_To_Dictionary( Main_Dictionaries[ 2 ], Word );
          Succ( Word2_Maximum ) .. Word3_Maximum: Add_To_Dictionary( Main_Dictionaries[ 3 ], Word );
          Succ( Word3_Maximum ) .. Word4_Maximum: Add_To_Dictionary( Main_Dictionaries[ 4 ], Word );
          else  Add_To_Dictionary( Main_Dictionaries[ 5 ], Word );
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add word to dictionaries.
    This function adds the given word to the
    appropriate dictionary if it isn't in it yet.

*************************************************)

    Procedure Add_Word_To_Dictionaries( Word: String );
      Var
        Size: Byte;
        Find,
        Escape,
        Capitalized: Boolean;
        Working_Word: Change_Data_Type;
      Begin
        If Correct_Ready
          then
            Begin
              Size := Length( Word );
              Find_Look_Value( Word, Working_Word.Look );
              Make_Word_LowerCase( Word, Working_Word.Word, Capitalized );
              Working_Word.Info := [];
              Scan_Dictionaries( Working_Word, Find, Size );
              If ( not Find )
                then
                  Begin
                    If Capitalized
                      then
                        Working_Word.Info := [ Capital ];
                    Add_To_Dictionaries( Working_Word, Size );
                  End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dump the dictionary.
    The purpose of this routine is to dump the
    dictionary to a text file.

*************************************************)

    Function Dump_The_Dictionary( Var Dictionary: Tree_Type; Var OutFile: Text ): Boolean;
      Var
        Okay: Boolean;
        Counter: LongInt;
        Data: Change_Data_Type;
        Result: String;
      Begin
        Okay := True;
        Counter := 0;
        Data.Word := '                              ';
        If Find_First_In_Tree( Dictionary, Data )
          then
            Repeat
              Convert_To_String( Data.Word, Result, ( Capital in Data.Info ) );
             {$I-}
              WriteLn( OutFile, Result );
              If ( IoResult <> 0 )
                then
                  Okay := False;
             {$I+}
              Inc( Counter );
              GotoXY( 1, WhereY );
              Write( Screen, 'Writing dictionary word', Counter:8 );
            Until ( not Find_Next_In_Tree( Dictionary, Data ) ) or ( not Okay );
        Dump_The_Dictionary := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Dump the dictionaries.
    The purpose of this routine is to dump the
    dictionaries to a text file.  It returns false
    if it fails.

*************************************************)

    Function Dump_The_Dictionaries( Var OutFile: Text ): Boolean;
      Var
        Okay: Boolean;
      Begin
        If Correct_Ready
          then
            Begin
              Okay := Dump_The_Dictionary( Main_Dictionaries[ 1 ], OutFile );
              If Okay
                then
                  Okay := Dump_The_Dictionary( Main_Dictionaries[ 2 ], OutFile );
              If Okay
                then
                  Okay := Dump_The_Dictionary( Main_Dictionaries[ 3 ], OutFile );
              If Okay
                then
                  Okay := Dump_The_Dictionary( Main_Dictionaries[ 4 ], OutFile );
              If Okay
                then
                  Okay := Dump_The_Dictionary( Main_Dictionaries[ 5 ], OutFile );
              Dump_The_Dictionaries := Okay;
            End
          else
            Dump_The_Dictionaries := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find suggestion.
    This function returns the foremost suggestion
    for the given word using the given key.

*************************************************)

    Function Find_Suggestion( Var List: Word_List_Type; Var Word: Word_String_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( List.Amount > 0 );
        If Okay
          then
            Begin
              List.Look_Point := 1;
              Word := List.Word[ List.Look_Point ];
            End;
        Find_Suggestion := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find next suggestion.
    This function tries to find the next
    suggestion after the given word.  It returns
    false if there are no more suggestions.

*************************************************)

    Function Find_Next_Suggestion( Var List: Word_List_Type; Var Word: Word_String_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( List.Amount > List.Look_Point );
        If Okay
          then
            Begin
              Inc( List.Look_Point );
              Word := List.Word[ List.Look_Point ];
            End;
        Find_Next_Suggestion := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Find previous suggestion.
    This function tries to find the previous
    suggestion before the given word.  It returns
    false if there are no more suggestions.

*************************************************)

    Function Find_Previous_Suggestion( Var List: Word_List_Type; Var Word: Word_String_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := ( List.Look_Point > 1 );
        If Okay
          then
            Begin
              Dec( List.Look_Point );
              Word := List.Word[ List.Look_Point ];
            End;
        Find_Previous_Suggestion := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Find replacement..
    This procedure allows the user to enter a
    replacement from the keyboard.  Then it tries
    to look it up in the dictionaries.

*************************************************)

  Function Find_Replacement( Var Word: String; Var Replacement, Suggested_Word: Word_String_Type; Var List: Word_List_Type;
                             Var Okay: Boolean ): Boolean;
    Var
      Size: Byte;
      Find,
      Capitalize,
      Capitalized,
      Auto_Change: Boolean;
      Working_Word: Change_Data_Type;
    Begin
      Get_Suggestion( Word );
      Make_Word( Word );
      Size := Length( Word );
      Find_Look_Value( Word, Working_Word.Look );
      Make_Word_LowerCase( Word, Working_Word.Word, Capitalized );
      Working_Word.Info := [];
      Scan_Dictionaries_List( Working_Word, Find, Auto_Change, Replacement, List, Size, Capitalize );
      If Find
        then
          Convert_From_String( Word, Replacement, ( Capitalize or Capitalized ) )
        else
          Begin
            Display_Word( Word );
            Okay := Find_Suggestion( List, Suggested_Word );
            If Okay
              then
                Replacement := Suggested_Word
              else
                Replacement := Working_Word.Word;
          End;
      Find_Replacement := Find;
    End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Present Options.
    This procedure presents the options and takes
    part of the appropriate actions depending on
    the option selected.

*************************************************)

    Procedure Present_Options( Var List: Word_List_Type; Word: String; Var Result: Char; Var Replacement: Word_String_Type );
      Var
        Done,
        Okay,
        Escape,
        Entered,
        Capitalized: Boolean;
        Suggested_Word: Word_String_Type;
      Begin
        Prepare_Screen;
        Display_Word( Word );
        Done := False;
        Entered := False;
        Okay := Find_Suggestion( List, Suggested_Word );
        Repeat
          If Okay
            then
              Begin
                Display_Suggestion( Suggested_Word );
                Offer_Options( Result );
                Case Result of
                  'A',
                  'B',
                  'D',
                  'I',
                  'Z': Done := True;
                  'C': Begin
                         Done := True;
                         Replacement := Suggested_Word;
                       End;
                  'E': Begin
                         Entered := True;
                         Done := Find_Replacement( Word, Replacement, Suggested_Word, List, Okay );
                       End;
                  'G',
                  'L': Begin
                         Done := True;
                         Replacement := Suggested_Word;
                       End;
                  'N': If not Find_Next_Suggestion( List, Suggested_Word )
                         then
                           Out_Of_Suggestions;
                  'P': If not Find_Previous_Suggestion( List, Suggested_Word )
                         then
                           Out_Of_Suggestions;
                End; { Case }
              End
            else
              Begin
                Display_Suggestion( '{ Suggestion not found }' );
                Offer_Options2( Result );
                Case Result of
                  'A',
                  'B',
                  'D',
                  'I',
                  'Z': Done := True;
                  'E': Begin
                         Entered := True;
                         Done := Find_Replacement( Word, Replacement, Suggested_Word, List, Okay );
                       End;
                End; { Case }
              End;
        Until Done;
        If ( Entered and ( not ( Result in [ 'Z', 'A', 'D' ] ) ) )
          then
            Case Result of
              'G': Result := 'E';
              'B',
              'I': Begin
                     Make_Word_LowerCase( Word, Replacement, Capitalized );
                     If Ask_Global_Replacement( Escape )
                       then
                         Result := 'E'
                       else
                         Result := 'C';
                     If Escape
                       then
                         Result := 'Z';
                   End;
              else
                Begin
                  If Ask_Global_Replacement( Escape )
                    then
                      Result := 'E'
                    else
                      Result := 'C';
                  If Escape
                    then
                      Result := 'Z';
                End;
            End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Analyze the word.
    This function takes the given word and tries
    to find it in the dictionaries.  If it finds
    the word, the result returned is A_File, if it
    finds a suitable replacement, it returns
    A_Alter, and if it doesn't find a suitable
    replacement, it returns A_Not_Found.  This
    function may safely be called in the
    background.

*************************************************)

    Function Analyze_The_Word( Word: String; Var New_Word: String ): Result_Type;
      Var
        Size: Byte;
        Find,
        Auto_Change,
        Capitalize,
        Capitalized: Boolean;
        Replacement: Word_String_Type;
        Working_Word: Change_Data_Type;
      Begin
        If Correct_Ready
          then
            Begin
              Size := Length( Word );
              Analyze_The_Word := A_Not_Found;
              Find_Look_Value( Word, Working_Word.Look );
              Make_Word_LowerCase( Word, Working_Word.Word, Capitalized );
              Working_Word.Info := [];
              Scan_Dictionaries_List( Working_Word, Find, Auto_Change, Replacement, Word_List^, Size, Capitalize );
              If Find
                then
                  If ( Capitalize and ( not Capitalized ) )
                    then
                      Analyze_The_Word := A_Capitalize
                    else
                      Analyze_The_Word := A_Fine;
              If Auto_Change
                then
                  Begin
                    Convert_To_String( Replacement, New_Word, Capitalized );
                    If ( Word = New_Word )
                      then
                        Analyze_the_Word := A_Fine
                      else
                        Analyze_The_Word := A_Alter;
                  End;
            End
          else
            Analyze_The_Word := A_Error;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Analyze the word single.
    This function takes the given word and tries
    to find it in the dictionaries.  If it finds
    the word, the result returned is A_File, if it
    finds a suitable replacement, it returns
    A_Alter, and if it doesn't find a suitable
    replacement, it returns A_Not_Found.  This
    function may safely be called in the
    background.

*************************************************)

    Function Analyze_The_Word_Single( Word: String; Var New_Word: String ): Result_Type;
      Var
        Size: Byte;
        Find,
        Auto_Change,
        Capitalize,
        Capitalized: Boolean;
        Replacement: Word_String_Type;
        Working_Word: Change_Data_Type;
      Begin
        If Correct_Ready
          then
            Begin
              Size := Length( Word );
              Analyze_The_Word_Single := A_Not_Found;
              Find_Look_Value( Word, Working_Word.Look );
              Make_Word_LowerCase( Word, Working_Word.Word, Capitalized );
              Working_Word.Info := [];
              Scan_Dictionaries_No_List( Working_Word, Find, Auto_Change, Capitalize, Replacement, Size );
              If Find
                then
                  If ( Capitalize and ( not Capitalized ) )
                    then
                      Analyze_The_Word_Single := A_Capitalize
                    else
                      Analyze_The_Word_Single := A_Fine;
              If Auto_Change
                then
                  Begin
                    Convert_To_String( Replacement, New_Word, Capitalized );
                    If ( Word = New_Word )
                      then
                        Analyze_the_Word_Single := A_Fine
                      else
                        Analyze_The_Word_Single := A_Alter;
                  End;
            End
          else
            Analyze_The_Word_Single := A_Error;
      End;

