/* -mt -f -A -K -G -O -w */
/* Compile with Turbo-C 2.0 */
/*
  This program generates METAFONT code from a Bitmaps file JIS24

  Author: Francois Jalbert
              '
  Date: November 1990

  Version: 1.0

  Date: April 1991

  Version: 2.00

  Modifications: - Added four kanjis.
                 - Fixed incorrect VGA resolution.
                 - Command line parameter now supported.
                 - Added automatic mode.
                 - Added batch mode.
                 - Updated and improved run-time messages.
                 - Long triangles added by Mr. Masatoshi Watanabe. Fantastic!
                 - Fixed and proportional parameters added.
                 - Standard and dictionary parameters added.
                 - JIS24 now accessed through low-level I/O channel for speed.

  Error Levels: 0 - Normal termination.
                1 - Error.
                2 - All fonts generated (batch).
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#ifdef __TURBOC__
#include <process.h>
#endif

#define True  1
#define False 0
/* Number of Bitmaps in JIS24 */
#define BitmapMax 7806
/* Size of each square Bitmap */
#define SizeMax  24
#define SizeMax1 25
/* DOS file name length */
#define FileNameDOS  250 /* includes path */
#define TotalNameDOS 254
/* DOS Record Size */
#define RecSize 72 /* SizeMax*SizeMax/8 */
/* Parameter flag */
#define Flag1 '/' /* DOS style */
#define Flag2 '-' /* UNIX style */
/* Parameter keywords */
#define ParamMax 20
char FixedX1[]="FIXEDWIDTH";
char FixedX2[]="FIXEDX";
char FixedX3[]="NOPROPORTIONALWIDTH";
char FixedX4[]="NOPROPORTIONALX";
char NoFixedX1[]="NOFIXEDWIDTH";
char NoFixedX2[]="NOFIXEDX";
char NoFixedX3[]="PROPORTIONALWIDTH";
char NoFixedX4[]="PROPORTIONALX";
char FixedY1[]="FIXEDHEIGHT";
char FixedY2[]="FIXEDY";
char FixedY3[]="NOPROPORTIONALHEIGHT";
char FixedY4[]="NOPROPORTIONALY";
char NoFixedY1[]="NOFIXEDHEIGHT";
char NoFixedY2[]="NOFIXEDY";
char NoFixedY3[]="PROPORTIONALHEIGHT";
char NoFixedY4[]="PROPORTIONALY";
char Standard1[]="STANDARD";
char NoStandard1[]="DICTIONARY";
char Batch1[]="BATCH";
/* Answer maximum length */
#define AnswerMax 254

typedef char FileNameType[FileNameDOS+1];
typedef char TotalNameType[TotalNameDOS+1];
typedef char ParamType[ParamMax+1];
typedef char AnswerType[AnswerMax+1];
/* Buffer for the Bitmap Data */
struct ColumnType {
  unsigned char Data1,Data2,Data3;
};
typedef struct ColumnType BufferType[SizeMax]; /* start at 0 because of fread */
/* The Bitmap array is defined larger to simplify the forthcoming code */
typedef int BitmapType[SizeMax1+1][SizeMax1+1];
struct BitmapsType {
  BitmapType Bitmap;
  int XMin,XMax,YMin,YMax;
};
/* Run time parameters */
struct RunTimeType {
  FileNameType FileName;
  /* Batch mode */
  int Batch;
  /* Automatic mode for JemTeX fonts only */
  int Automatic;
  /* Fixed or proportional fonts */
  int FixedX,FixedY;
  /* Standard or dictionary fonts */
  int Standard;
};

void Delete(char *String, int Length)
/* Delete the first Length characters of String */
{
  int Index;

  Index=0;
  do String[Index]=String[Index+Length];
  while (String[++Index]!='\0');
}

/*------------------------------- GetParameters -----------------------------*/

void SimpleQuery(char Title[], char ChoiceA[], char ChoiceB[], int *Answer)
{
  char JChar[2];
  int Valid;

  do {
    Valid=True;
    printf("%s:\n",Title);
    printf("   a)  %s\n",ChoiceA);
    printf("   b)  %s\n",ChoiceB);
    printf("Your choice? ");
    if (ferror(stdout)) exit(1);
    if (gets(JChar)==NULL) exit(1);
    if (strlen(JChar)>1) exit(1);
    JChar[0]=toupper(JChar[0]);
    if (JChar[0]=='A') *Answer=True;
    else
      if (JChar[0]=='B') *Answer=False;
      else { 
        Valid=False; 
        if (putchar('\7')==EOF) exit(1); 
    }
  } while (!Valid);
  printf("\n");
  if (ferror(stdout)) exit(1);
}

void GetMode(struct RunTimeType *RunTime)
/* Determines if the desired font is a JemTeX font */
{
  RunTime->Automatic=False;
  if (toupper(RunTime->FileName[0])=='K')
  if (toupper(RunTime->FileName[1])=='A')
  if (toupper(RunTime->FileName[2])=='N')
  if (toupper(RunTime->FileName[3])=='J')
  if (toupper(RunTime->FileName[4])=='I')
  if(('A'<=toupper(RunTime->FileName[5]))&&(toupper(RunTime->FileName[5])<='H'))
  if(('A'<=toupper(RunTime->FileName[6]))&&(toupper(RunTime->FileName[6])<='H'))
  if (strlen(RunTime->FileName)==7)
  if (toupper(RunTime->FileName[5])<='G') RunTime->Automatic=True;
  else
  if (toupper(RunTime->FileName[6])<='E') RunTime->Automatic=True;
}

void EchoParameters(struct RunTimeType *RunTime)
/* Echoes the current parameters */
{
  printf("Font=%s",RunTime->FileName);
  if (RunTime->FixedX) printf("  Fixed Width");
  else printf("  Prop. Width");
  if (RunTime->FixedY) printf("  Fixed Height");
  else printf("  Prop. Height");
  if (RunTime->Standard) printf("  Standard");
  else printf("  Dictionary");
  if (RunTime->Automatic) printf("  Automatic");
  else printf("  Manual");
  if (RunTime->Batch) printf("  Batch");
  printf(".\n");
  if (ferror(stdout)) exit(1);
}

void Manual(struct RunTimeType *RunTime)
/* Get parameters from user */
{
  printf("METAFONT file name? ");
  if (ferror(stdout)) exit(1);
  if (gets(RunTime->FileName)==NULL) exit(1);
  if (strlen(RunTime->FileName)>FileNameDOS) {
    /* File name too long */
    printf("\7File name too long: %s...",RunTime->FileName);
    exit(1);
  }
  printf("\n");
  if (ferror(stdout)) exit(1);
  SimpleQuery("Fixed or proportional font width","Fixed","Proportional",
              &RunTime->FixedX);
  SimpleQuery("Fixed or proportional font height","Fixed","Proportional",
              &RunTime->FixedY);
  SimpleQuery("Standard or dictionary font","Standard","Dictionary",
              &RunTime->Standard);
  /* Batch mode intrinsically isn't manual */
  RunTime->Batch=False;
}

void FindBefore(FileNameType FileName)
/* No check for before kanjiaa */
{
  if (FileName[6]=='a') {
    FileName[6]='h';
    FileName[5]--;
  }
  else
    FileName[6]--;
}

void FindAfter(FileNameType FileName)
/* No check for above kanjihe */
{
  if (FileName[6]=='h') {
    FileName[6]='a';
    FileName[5]++;
  }
  else
    FileName[6]++;
}

void ScanMF(FileNameType FileName)
/* Scans backwards for the last JemTeX font generated */
/* Looks first for a .TFM and then for an .MF */
/* If no more fonts to generate, stops with error level 2 */
{
  FILE *TestFile;
  TotalNameType TotalName;
  int Found;

  strcpy(FileName,"kanjihf");
  do {
    FindBefore(FileName);
    strcpy(TotalName,FileName);
    strcat(TotalName,".tfm");
    TestFile=fopen(TotalName,"r");
    Found=(TestFile!=NULL);
    if (!Found) {
      strcpy(TotalName,FileName);
      strcat(TotalName,".mf");
      TestFile=fopen(TotalName,"r");
      Found=(TestFile!=NULL);
    }
  } while (!Found && strcmp(FileName,"kanjiaa"));
  if (Found) {
    if (fclose(TestFile)==EOF) exit(1);
    if (strcmp(FileName,"kanjihe")) FindAfter(FileName);
    else {
      printf("\7All JemTeX fonts generated!\n");
      exit(2);
    }
  }
}

void Automate(struct RunTimeType *RunTime, int argc, char *argv[])
/* Get parameters from command line */
/* Finds the next font to be generated if in batch mode */
{
  int ParamIndex,ParamLength,Index;
  ParamType Param;

  /* Defaults */
  strcpy(RunTime->FileName,"kanjiaa");
  RunTime->FixedX=False;
  RunTime->FixedY=False;
  RunTime->Standard=True;
  RunTime->Batch=False;
  /* Scan command line parameters */
  /* 0th is program's name, last is NULL */
  for (ParamIndex=1 ; ParamIndex<argc ; ParamIndex++) {
    ParamLength=strlen(argv[ParamIndex]);
    if (ParamLength>ParamMax) {
      /* Keyword too long */
      printf("\7Invalid command line parameter: %s...",argv[ParamIndex]);
      exit(1);
    }
    strcpy(Param,argv[ParamIndex]);
    if ((Param[0]==Flag1) || (Param[0]==Flag2)) {
      /* Not a font name */
      /* Delete 1 char at the 1st position */
      Delete(Param,1);
      /* Convert to upper case */
      for (Index=0 ; Index<ParamLength ; Index++)
        Param[Index]=toupper(Param[Index]);
      /* Scan known keywords */
      if ((!strcmp(Param,FixedX1)) || (!strcmp(Param,FixedX2)) || 
          (!strcmp(Param,FixedX3)) || (!strcmp(Param,FixedX4))) 
        RunTime->FixedX=True;
      else
      if ((!strcmp(Param,NoFixedX1)) || (!strcmp(Param,NoFixedX2)) ||
          (!strcmp(Param,NoFixedX3)) || (!strcmp(Param,NoFixedX4))) 
        RunTime->FixedX=False;
      else
      if ((!strcmp(Param,FixedY1)) || (!strcmp(Param,FixedY2)) || 
          (!strcmp(Param,FixedY3)) || (!strcmp(Param,FixedY4))) 
        RunTime->FixedY=True;
      else
      if ((!strcmp(Param,NoFixedY1)) || (!strcmp(Param,NoFixedY2)) || 
          (!strcmp(Param,NoFixedY3)) || (!strcmp(Param,NoFixedY4))) 
        RunTime->FixedY=False;
      else
      if (!strcmp(Param,Standard1)) 
        RunTime->Standard=True;
      else
      if (!strcmp(Param,NoStandard1)) 
        RunTime->Standard=False;
      else
      if (!strcmp(Param,Batch1)) 
        RunTime->Batch=True;
      else {
        /* Unknown keyword */
        printf("\7Invalid command line parameter: %s...\n",Param);
        exit(1);
      }
    }
    else {
      /* Must be a font name */
      if (ParamLength>FileNameDOS) {
        /* File name too long */
        printf("\7File name too long: %s...\n",Param);
        exit(1);
      }
      strcpy(RunTime->FileName,Param);
    }
  }
  if (RunTime->Batch) ScanMF(RunTime->FileName);
}

void GetParameters(struct RunTimeType *RunTime, int argc, char *argv[])
/* Get parameters from user or command line */
{
  /* 0th is program's name, last is NULL */
  if (argc==1) Manual(RunTime);
  else Automate(RunTime,argc,argv);
  GetMode(RunTime);
  EchoParameters(RunTime);
  printf("\n");
  if (ferror(stdout)) exit(1);
}

/*---------------------------------- Output ---------------------------------*/

void BeginOut(FILE *OutFile, struct RunTimeType *RunTime)
/* Writes initial METAFONT header */
/* Co-author is Mr. Masatoshi Watanabe */
{
  fprintf(OutFile,"%%JIS2MF Version 2.00 of 14 April 1991.\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"%% Font=%s\n",RunTime->FileName);
  if (RunTime->FixedX) fprintf(OutFile,"%% Fixed Width\n");
  else fprintf(OutFile,"%% Proportional Width\n");
  if (RunTime->FixedY) fprintf(OutFile,"%% Fixed Height\n");
  else fprintf(OutFile,"%% Proportional Height\n");
  if (RunTime->Standard) fprintf(OutFile,"%% Standard Positioning\n");
  else fprintf(OutFile,"%% Dictionary Positioning\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"tracingstats:=1;\n");
  fprintf(OutFile,"screen_cols:=640; %%VGA\n");
  fprintf(OutFile,"screen_rows:=480; %%VGA\n");
  fprintf(OutFile,"font_size 10pt#;\n");
  if (RunTime->Standard) {
    fprintf(OutFile,"u#:=12.7/36pt#;\n");
    fprintf(OutFile,"body_height#:=23.25u#;\n");
    fprintf(OutFile,"desc_depth#:=4.75u#;\n");
  }
  else {
    fprintf(OutFile,"u#:=13/36pt#;\n");
    fprintf(OutFile,"body_height#:=21u#;\n");
    fprintf(OutFile,"desc_depth#:=7u#;\n");
  }
  fprintf(OutFile,"\n");
  fprintf(OutFile,"letter_fit#:=0pt#;\n");
  fprintf(OutFile,"asc_height#:=0pt#;\n");
  fprintf(OutFile,"cap_height#:=0pt#;\n");
  fprintf(OutFile,"fig_height#:=0pt#;\n");
  fprintf(OutFile,"x_height#:=0pt#;\n");
  fprintf(OutFile,"math_axis#:=0pt#;\n");
  fprintf(OutFile,"bar_height#:=0pt#;\n");
  fprintf(OutFile,"comma_depth#:=0pt#;\n");
  fprintf(OutFile,"crisp#:=0pt#;\n");
  fprintf(OutFile,"tiny#:=0pt#;\n");
  fprintf(OutFile,"fine#:=0pt#;\n");
  fprintf(OutFile,"thin_join#:=0pt#;\n");
  fprintf(OutFile,"hair#:=1pt#;\n");
  fprintf(OutFile,"stem#:=1pt#;\n");
  fprintf(OutFile,"curve#:=1pt#;\n");
  fprintf(OutFile,"flare#:=1pt#;\n");
  fprintf(OutFile,"dot_size#:=0pt#;\n");
  fprintf(OutFile,"cap_hair#:=1pt#;\n");
  fprintf(OutFile,"cap_stem#:=1pt#;\n");
  fprintf(OutFile,"cap_curve#:=1pt#;\n");
  fprintf(OutFile,"rule_thickness#:=0pt#;\n");
  fprintf(OutFile,"vair#:=0pt#;\n");
  fprintf(OutFile,"notch_cut#:=0pt#;\n");
  fprintf(OutFile,"bar#:=1pt#;\n");
  fprintf(OutFile,"slab#:=1pt#;\n");
  fprintf(OutFile,"cap_bar#:=1pt#;\n");
  fprintf(OutFile,"cap_band#:=1pt#;\n");
  fprintf(OutFile,"cap_notch_cut#:=0pt#;\n");
  fprintf(OutFile,"serif_drop#:=0pt#;\n");
  fprintf(OutFile,"stem_corr#:=0pt#;\n");
  fprintf(OutFile,"vair_corr#:=0pt#;\n");
  fprintf(OutFile,"o#:=0pt#;\n");
  fprintf(OutFile,"apex_o#:=0pt#;\n");
  fprintf(OutFile,"hefty:=true;\n");
  fprintf(OutFile,"serifs:=true;\n");
  fprintf(OutFile,"monospace:=false;\n");
  fprintf(OutFile,"math_fitting:=false;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"mode_setup;\n");
  fprintf(OutFile,"font_setup;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"pair z;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def s(expr col,row)= %%square\n");
  fprintf(OutFile," z:=((col*u),(row*u));\n");
  fprintf(OutFile," fill unitsquare scaled u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def sul(expr col,row)= %%upper left square\n");
  fprintf(OutFile," z:=((col*u),(row*u)+.5u);\n");
  fprintf(OutFile," fill unitsquare scaled .5u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def sur(expr col,row)= %%upper right square\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill unitsquare scaled .5u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def sbr(expr col,row)= %%bottom right square\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u));\n");
  fprintf(OutFile," fill unitsquare scaled .5u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def sbl(expr col,row)= %%bottom left square\n");
  fprintf(OutFile," z:=((col*u),(row*u));\n");
  fprintf(OutFile," fill unitsquare scaled .5u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def c(expr col,row)= %%circle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill fullcircle scaled u shifted z;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def cul(expr col,row)= %%upper left circle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--quartercircle rotated 90 scaled u shifted z--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def cur(expr col,row)= %%upper right circle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--quartercircle scaled u shifted z--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def cbr(expr col,row)= %%bottom right circle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--quartercircle rotated 270 scaled u shifted z--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def cbl(expr col,row)= %%bottom left circle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--quartercircle rotated 180 scaled u shifted z--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def tul(expr col,row)= %%upper left triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tur(expr col,row)= %%upper right triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbr(expr col,row)= %%bottom right triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbl(expr col,row)= %%bottom left triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def rul(expr col,row)= %%upper left reverse triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rur(expr col,row)= %%upper right reverse triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbr(expr col,row)= %%bottom right reverse triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u));\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbl(expr col,row)= %%bottom left reverse triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u));\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def tuul(expr col,row)= %%upper left long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z-(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tull(expr col,row)= %%upper left long triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u));\n");
  fprintf(OutFile," fill z--z+(0,u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tuur(expr col,row)= %%upper right long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z+(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def turr(expr col,row)= %%upper right long triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u));\n");
  fprintf(OutFile," fill z--z+(0,u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbbr(expr col,row)= %%bottom right long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z+(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbrr(expr col,row)= %%bottom right long triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbbl(expr col,row)= %%bottom left long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u)+.5u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z-(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def tbll(expr col,row)= %%bottom left long triangle\n");
  fprintf(OutFile," z:=((col*u)+.5u,(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  fprintf(OutFile,"def ruul(expr col,row)= %%upper left reverse long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rull(expr col,row)= %%upper left reverse long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z+(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def ruur(expr col,row)= %%upper right reverse long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rurr(expr col,row)= %%upper right reverse long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u)+u);\n");
  fprintf(OutFile," fill z--z-(0,.5u)--z-(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbbr(expr col,row)= %%bottom right reverse long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u));\n");
  fprintf(OutFile," fill z--z+(0,u)--z-(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbrr(expr col,row)= %%bottom right reverse long triangle\n");
  fprintf(OutFile," z:=((col*u)+u,(row*u));\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z-(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbbl(expr col,row)= %%bottom left reverse long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u));\n");
  fprintf(OutFile," fill z--z+(0,u)--z+(.5u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"def rbll(expr col,row)= %%bottom left reverse long triangle\n");
  fprintf(OutFile," z:=((col*u),(row*u));\n");
  fprintf(OutFile," fill z--z+(0,.5u)--z+(u,0)--cycle;\n");
  fprintf(OutFile,"enddef;\n");
  fprintf(OutFile,"\n");
  if (ferror(OutFile)) exit(1);
}

void ActiveBitmap(FILE *OutFile, BitmapType Bitmap, int X, int Y, int XX, 
                  float YY)
/* Writes METAFONT code for an active cell */
/* Co-author is Mr. Masatoshi Watanabe */
{
  int SquareUR,SquareUL,SquareBR,SquareBL;
  int CircleUR,CircleUL,CircleBR,CircleBL;
  int LTryUUR,LTryURR,LTryUUL,LTryULL;
  int LTryBBR,LTryBRR,LTryBBL,LTryBLL;

  SquareUL=(Bitmap[X-1][Y] || Bitmap[X-1][Y+1] || Bitmap[X][Y+1]);
  SquareUR=(Bitmap[X+1][Y] || Bitmap[X+1][Y+1] || Bitmap[X][Y+1]);
  SquareBL=(Bitmap[X-1][Y] || Bitmap[X-1][Y-1] || Bitmap[X][Y-1]);
  SquareBR=(Bitmap[X+1][Y] || Bitmap[X+1][Y-1] || Bitmap[X][Y-1]);
  CircleUL=(!Bitmap[X-1][Y-1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y+1] &&
            !Bitmap[X][Y+1] && !Bitmap[X+1][Y+1]);
  CircleUR=(!Bitmap[X+1][Y-1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y+1] &&
            !Bitmap[X][Y+1] && !Bitmap[X-1][Y+1]);
  CircleBL=(!Bitmap[X-1][Y+1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y-1] &&
            !Bitmap[X][Y-1] && !Bitmap[X+1][Y-1]);
  CircleBR=(!Bitmap[X+1][Y+1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y-1] &&
            !Bitmap[X][Y-1] && !Bitmap[X-1][Y-1]);
  LTryUUL=(Bitmap[X-1][Y-1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y+1] &&
           !Bitmap[X][Y+1] && !Bitmap[X+1][Y+1] && Bitmap[X+1][Y]);
  LTryUUR=(Bitmap[X+1][Y-1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y+1] &&
           !Bitmap[X][Y+1] && !Bitmap[X-1][Y+1] && Bitmap[X-1][Y]);
  LTryBBL=(Bitmap[X-1][Y+1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y-1] &&
           !Bitmap[X][Y-1] && !Bitmap[X+1][Y-1] && Bitmap[X+1][Y]);
  LTryBBR=(Bitmap[X+1][Y+1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y-1] &&
           !Bitmap[X][Y-1] && !Bitmap[X-1][Y-1] && Bitmap[X-1][Y]);
  LTryULL=(!Bitmap[X-1][Y-1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y+1] &&
           !Bitmap[X][Y+1] && Bitmap[X+1][Y+1] && Bitmap[X][Y-1]);
  LTryURR=(!Bitmap[X+1][Y-1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y+1] &&
           !Bitmap[X][Y+1] && Bitmap[X-1][Y+1] && Bitmap[X][Y-1]);
  LTryBLL=(!Bitmap[X-1][Y+1] && !Bitmap[X-1][Y] && !Bitmap[X-1][Y-1] &&
           !Bitmap[X][Y-1] && Bitmap[X+1][Y-1] && Bitmap[X][Y+1]);
  LTryBRR=(!Bitmap[X+1][Y+1] && !Bitmap[X+1][Y] && !Bitmap[X+1][Y-1] &&
           !Bitmap[X][Y-1] && Bitmap[X-1][Y-1] && Bitmap[X][Y+1]);
  if (LTryUUL) fprintf(OutFile,"tuul(%d,%4.2f);",XX,YY);
  if (LTryULL) fprintf(OutFile,"tull(%d,%4.2f);",XX,YY);
  if (LTryUUR) fprintf(OutFile,"tuur(%d,%4.2f);",XX,YY);
  if (LTryURR) fprintf(OutFile,"turr(%d,%4.2f);",XX,YY);
  if (LTryBBL) fprintf(OutFile,"tbbl(%d,%4.2f);",XX,YY);
  if (LTryBLL) fprintf(OutFile,"tbll(%d,%4.2f);",XX,YY);
  if (LTryBBR) fprintf(OutFile,"tbbr(%d,%4.2f);",XX,YY);
  if (LTryBRR) fprintf(OutFile,"tbrr(%d,%4.2f);",XX,YY);
  if (SquareUL && SquareUR && SquareBL && SquareBR)
    fprintf(OutFile,"s(%d,%4.2f);",XX,YY);
  else
    if (CircleUL && CircleUR && CircleBL && CircleBR)
      fprintf(OutFile,"c(%d,%4.2f);",XX,YY);
    else {
      if (!LTryUUL && !LTryULL && !LTryUUR && !LTryBLL) 
        if (SquareUL) fprintf(OutFile,"sul(%d,%4.2f);",XX,YY);
        else
          if (CircleUL) fprintf(OutFile,"cul(%d,%4.2f);",XX,YY);
          else fprintf(OutFile,"tul(%d,%4.2f);",XX,YY);
      if (!LTryUUL && !LTryURR && !LTryUUR && !LTryBRR)
        if (SquareUR) fprintf(OutFile,"sur(%d,%4.2f);",XX,YY);
        else
          if (CircleUR) fprintf(OutFile,"cur(%d,%4.2f);",XX,YY);
          else fprintf(OutFile,"tur(%d,%4.2f);",XX,YY);
      if (!LTryBBL && !LTryULL && !LTryBBR && !LTryBLL)
        if (SquareBL) fprintf(OutFile,"sbl(%d,%4.2f);",XX,YY);
        else
          if (CircleBL) fprintf(OutFile,"cbl(%d,%4.2f);",XX,YY);
          else fprintf(OutFile,"tbl(%d,%4.2f);",XX,YY);
      if (!LTryBBL && !LTryURR && !LTryBBR && !LTryBRR) 
        if (SquareBR) fprintf(OutFile,"sbr(%d,%4.2f);",XX,YY);
        else
          if (CircleBR) fprintf(OutFile,"cbr(%d,%4.2f);",XX,YY);
          else fprintf(OutFile,"tbr(%d,%4.2f);",XX,YY);
    }
  if (ferror(OutFile)) exit(1);
}

void InactiveBitmap(FILE *OutFile, BitmapType Bitmap, int X, int Y, int XX, 
                    float YY, int *Active)
/* Writes METAFONT code for an inactive cell */
/* Co-author is Mr. Masatoshi Watanabe */
{
  if (Bitmap[X-1][Y] && Bitmap[X][Y+1])
    if (Bitmap[X-1][Y-1] && !Bitmap[X+1][Y+1])
      { *Active=True; fprintf(OutFile,"ruul(%d,%4.2f);",XX,YY); }
    else
      if (Bitmap[X+1][Y+1] && !Bitmap[X-1][Y-1])
        { *Active=True; fprintf(OutFile,"rull(%d,%4.2f);",XX,YY); }
      else
        { *Active=True; fprintf(OutFile,"rul(%d,%4.2f);",XX,YY); }
  if (Bitmap[X+1][Y] && Bitmap[X][Y+1])
    if (Bitmap[X+1][Y-1] && !Bitmap[X-1][Y+1])
      { *Active=True; fprintf(OutFile,"ruur(%d,%4.2f);",XX,YY); }
    else
      if (Bitmap[X-1][Y+1] && !Bitmap[X+1][Y-1])
        { *Active=True; fprintf(OutFile,"rurr(%d,%4.2f);",XX,YY); }
      else
        { *Active=True; fprintf(OutFile,"rur(%d,%4.2f);",XX,YY); }
  if (Bitmap[X-1][Y] && Bitmap[X][Y-1])
    if (Bitmap[X-1][Y+1] && !Bitmap[X+1][Y-1])
      { *Active=True; fprintf(OutFile,"rbbl(%d,%4.2f);",XX,YY); }
    else
      if (Bitmap[X+1][Y-1] && !Bitmap[X-1][Y+1])
        { *Active=True; fprintf(OutFile,"rbll(%d,%4.2f);",XX,YY); }
      else
        { *Active=True; fprintf(OutFile,"rbl(%d,%4.2f);",XX,YY); }
  if (Bitmap[X+1][Y] && Bitmap[X][Y-1])
    if (Bitmap[X+1][Y+1] && !Bitmap[X-1][Y-1])
      { *Active=True; fprintf(OutFile,"rbbr(%d,%4.2f);",XX,YY); }
    else
      if (Bitmap[X-1][Y-1] && !Bitmap[X+1][Y+1])
        { *Active=True; fprintf(OutFile,"rbrr(%d,%4.2f);",XX,YY); }
      else
        { *Active=True; fprintf(OutFile,"rbr(%d,%4.2f);",XX,YY); }
  if (ferror(OutFile)) exit(1);
}

void MiddleOut(FILE *OutFile, struct BitmapsType *Bitmaps, int Number, 
               int Standard)
/* Writes METAFONT code for a given Bitmap */
{
  int X,Y;
  int Active;

  fprintf(OutFile,"beginchar(%d,%du#,",Number,Bitmaps->XMax-Bitmaps->XMin+1);
  if (Standard) {
    if (Bitmaps->YMax>0.75) fprintf(OutFile,"%4.2fu#,",Bitmaps->YMax-0.75);
    else fprintf(OutFile,"0,");
    if (5.75>Bitmaps->YMin) fprintf(OutFile,"%4.2fu#);\n",5.75-Bitmaps->YMin);
    else fprintf(OutFile,"0);\n");
  }
  else {
    if (Bitmaps->YMax>3) fprintf(OutFile,"%du#,",Bitmaps->YMax-3);
    else fprintf(OutFile,"0,");
    if (8>Bitmaps->YMin) fprintf(OutFile,"%du#);\n",8-Bitmaps->YMin);
    else fprintf(OutFile,"0);\n");
  }
  fprintf(OutFile,"normal_adjust_fit(2u#,2u#);\n");
  for (X=Bitmaps->XMin ; X<=Bitmaps->XMax ; X++)
    for (Y=1 ; Y<=SizeMax ; Y++) {
      Active=Bitmaps->Bitmap[X][Y];
      if (Active)
        /* Current pixel is on */
        if (Standard) 
          ActiveBitmap(OutFile,Bitmaps->Bitmap,X,Y,X-Bitmaps->XMin,Y-3.75);
        else 
          ActiveBitmap(OutFile,Bitmaps->Bitmap,X,Y,X-Bitmaps->XMin,Y-6);
      else
        /* Current pixel is off */
        if (Standard) 
          InactiveBitmap(OutFile,Bitmaps->Bitmap,X,Y,X-Bitmaps->XMin,Y-3.75,
                         &Active);
        else 
          InactiveBitmap(OutFile,Bitmaps->Bitmap,X,Y,X-Bitmaps->XMin,Y-6,
                         &Active);
      /* Avoid METAFONT buffer overflow */
      if (Active) fprintf(OutFile,"\n");
  }
  fprintf(OutFile,"endchar;\n");
  fprintf(OutFile,"\n");
  if (ferror(OutFile)) exit(1);
} 

void EndOut(FILE *OutFile, struct RunTimeType *RunTime)
/* Writes final METAFONT header */
{
  fprintf(OutFile,"font_identifier \"%s\";\n",RunTime->FileName);
  if (RunTime->Standard) 
    fprintf(OutFile,"font_coding_scheme \"JemTeX Standard\";\n");
  else 
    fprintf(OutFile,"font_coding_scheme \"JemTeX Dictionary\";\n");
  fprintf(OutFile,"font_slant slant;\n");
  fprintf(OutFile,"font_normal_space 8u#;\n");
  fprintf(OutFile,"font_normal_stretch 4u#;\n");
  fprintf(OutFile,"font_normal_shrink 3u#;\n");
  fprintf(OutFile,"font_x_height 24u#; %%ex\n");
  fprintf(OutFile,"font_quad 24u#; %%em\n");
  fprintf(OutFile,"font_extra_space 0u#;\n");
  fprintf(OutFile,"\n");
  /* Must end with CR/LF because of a bug(?) in emTeX METAFONT */
  fprintf(OutFile,"bye\n");
  if (ferror(OutFile)) exit(1);
}

/*--------------------------------- Generate --------------------------------*/

void FindWantedBitmap(int Automatic, int *First, int *WantedBitmap, int *Number)
/* Finds number of the next desired Bitmap either automatically or manually */
/* The characters 0 and 1 in the first font kanjiaa are both set to Bitmap 1 */
{
  int Valid;
  AnswerType Answer;

  if (Automatic)
    /* Find automatically */
    if (*First)
      /* Early in font kanjiaa */
      if (*WantedBitmap==-1) *WantedBitmap=1;
      else {
        *WantedBitmap=1;
        *First=False;
      }
    else
      if ( (*Number==128) || (*WantedBitmap==BitmapMax) ) *WantedBitmap=0;
      else (*WantedBitmap)++;
  else
    /* Find manually */
    do {
      printf("Bitmap number? ");
      if (ferror(stdout)) exit(1);
      if (gets(Answer)==NULL) exit(1);
      if (1!=sscanf(Answer,"%d",WantedBitmap)) exit(1);
      Valid=( (0<=*WantedBitmap) && (*WantedBitmap<=BitmapMax) );
      if (!Valid) {
        printf("\7Bitmap %d out of range...\n",*WantedBitmap);
        if (ferror(stdout)) exit(1);
      }
    } while (!Valid);
  printf("Bitmap number %d.\n",*WantedBitmap);
  if (ferror(stdout)) exit(1);
}

void ScanBitmap(FILE *InFile, BitmapType Bitmap, int *Empty)
/* Reads the Bitmap in a logical grid */
/* (0,0) is the lower left corner of the Bitmap */
{
  int Y;
  BufferType Buffer;
  struct ColumnType *Column;

  /* Read the Bitmap */
  if (fread(Buffer,(unsigned)RecSize,1U,InFile)!=1U) exit(1);
  /* Find if the Bitmap is empty */
  *Empty=True;
  for (Y=1 ; Y<=SizeMax ; Y++) {
    Column=&Buffer[Y-1];
    if ( (Column->Data1!=(unsigned char)'\x00') || 
         (Column->Data2!=(unsigned char)'\x00') ||
         (Column->Data3!=(unsigned char)'\x00') ) {
      *Empty=False;
      break;
    }
  }
  /* Update logical grid */
  if (!*Empty)
    for (Y=1 ; Y<=SizeMax ; Y++) {            
      Column=&Buffer[SizeMax-Y];
      Bitmap[ 1][Y]=(int)(Column->Data1 & (unsigned char)'\x80');
      Bitmap[ 2][Y]=(int)(Column->Data1 & (unsigned char)'\x40');
      Bitmap[ 3][Y]=(int)(Column->Data1 & (unsigned char)'\x20');
      Bitmap[ 4][Y]=(int)(Column->Data1 & (unsigned char)'\x10');
      Bitmap[ 5][Y]=(int)(Column->Data1 & (unsigned char)'\x08');
      Bitmap[ 6][Y]=(int)(Column->Data1 & (unsigned char)'\x04');
      Bitmap[ 7][Y]=(int)(Column->Data1 & (unsigned char)'\x02');
      Bitmap[ 8][Y]=(int)(Column->Data1 & (unsigned char)'\x01');
      Bitmap[ 9][Y]=(int)(Column->Data2 & (unsigned char)'\x80');
      Bitmap[10][Y]=(int)(Column->Data2 & (unsigned char)'\x40');
      Bitmap[11][Y]=(int)(Column->Data2 & (unsigned char)'\x20');
      Bitmap[12][Y]=(int)(Column->Data2 & (unsigned char)'\x10');
      Bitmap[13][Y]=(int)(Column->Data2 & (unsigned char)'\x08');
      Bitmap[14][Y]=(int)(Column->Data2 & (unsigned char)'\x04');
      Bitmap[15][Y]=(int)(Column->Data2 & (unsigned char)'\x02');
      Bitmap[16][Y]=(int)(Column->Data2 & (unsigned char)'\x01');
      Bitmap[17][Y]=(int)(Column->Data3 & (unsigned char)'\x80');
      Bitmap[18][Y]=(int)(Column->Data3 & (unsigned char)'\x40');
      Bitmap[19][Y]=(int)(Column->Data3 & (unsigned char)'\x20');
      Bitmap[20][Y]=(int)(Column->Data3 & (unsigned char)'\x10');
      Bitmap[21][Y]=(int)(Column->Data3 & (unsigned char)'\x08');
      Bitmap[22][Y]=(int)(Column->Data3 & (unsigned char)'\x04');
      Bitmap[23][Y]=(int)(Column->Data3 & (unsigned char)'\x02');
      Bitmap[24][Y]=(int)(Column->Data3 & (unsigned char)'\x01');
  }
}

void ScanSides(struct BitmapsType *Bitmaps, int FixedX, int FixedY)
/* Determines the minimal size of the Bitmap for proportional spacing */
{
  int X,Y;

  if (FixedX) { 
    Bitmaps->XMin=1; 
    Bitmaps->XMax=SizeMax; 
  }
  else {
    Bitmaps->XMin=SizeMax1;
    for (X=SizeMax ; X>=1 ; X--)
      for (Y=1 ; Y<=SizeMax ; Y++)
        if (Bitmaps->Bitmap[X][Y]) Bitmaps->XMin=X;
    Bitmaps->XMax=0;
    for (X=1 ; X<=SizeMax ; X++)
      for (Y=1 ; Y<=SizeMax ; Y++)
        if (Bitmaps->Bitmap[X][Y]) Bitmaps->XMax=X;
  }
  if (FixedY) {
    Bitmaps->YMin=1; 
    Bitmaps->YMax=SizeMax; 
  }
  else {
    Bitmaps->YMin=SizeMax1;
    for (Y=SizeMax ; Y>=1 ; Y--)
      for (X=1 ; X<=SizeMax ; X++)
        if (Bitmaps->Bitmap[X][Y]) Bitmaps->YMin=Y;
    Bitmaps->YMax=0;
    for (Y=1 ; Y<=SizeMax ; Y++)
      for (X=1 ; X<=SizeMax ; X++)
        if (Bitmaps->Bitmap[X][Y]) Bitmaps->YMax=Y;
  }
}

void Generate(FILE *InFile, FILE *OutFile, int *Number, 
              struct RunTimeType *RunTime)
/* Generates the METAFONT code for the selected font */
{
  /* Bitmap pointers */
  int CurrentBitmap,WantedBitmap;
  /* Current Bitmap and its dimensions */
  struct BitmapsType Bitmaps;
  int X,Y;
  /* Indicates early in font kanjiaa */
  int First;
  /* Indicates current Bitmap is empty */
  int Empty;

  /* Clear the area outside the Bitmap once and for all */
  for (X=0 ; X<=SizeMax1 ; X++) {
    Bitmaps.Bitmap[X][0]=False; Bitmaps.Bitmap[X][SizeMax1]=False; }
  for (Y=1 ; Y<=SizeMax ; Y++) {
    Bitmaps.Bitmap[0][Y]=False; Bitmaps.Bitmap[SizeMax1][Y]=False; }
  /* Number of the Bitmap ready to be read */
  CurrentBitmap=1;
  /* First METAFONT character number */
  *Number=0;
  /* First Bitmap wanted */
  if (RunTime->Automatic) {
    WantedBitmap=1024 * ( toupper(RunTime->FileName[5])-'A' ) +
                 128 * ( toupper(RunTime->FileName[6])-'A' ) - 1;
    First=(WantedBitmap==-1);
  }
  do {
    FindWantedBitmap(RunTime->Automatic,&First,&WantedBitmap,Number);
    if (WantedBitmap) {
      /* Position pointer */
      if (WantedBitmap!=CurrentBitmap) {
        if ( fseek(InFile , ((WantedBitmap-1L)*RecSize) , 0) ) exit(1);
        CurrentBitmap=WantedBitmap;
      }
      printf("Reading Bitmap");
      if (ferror(stdout)) exit(1);
      ScanBitmap(InFile,Bitmaps.Bitmap,&Empty);
      CurrentBitmap++;
      printf(".\n");
      if (ferror(stdout)) exit(1);
      /* Process Bitmap */
      if (Empty) printf("Bitmap is empty, no METAFONT code %d.\n",*Number);
      else {
        printf("Writing METAFONT code %d",*Number);
        ScanSides(&Bitmaps,RunTime->FixedX,RunTime->FixedY);
        MiddleOut(OutFile,&Bitmaps,*Number,RunTime->Standard);
        printf(".\n");
      }
      printf("\n");
      if (ferror(stdout)) exit(1);
      /* Ready to generate next METAFONT character */
      (*Number)++;
    }
  } while (WantedBitmap);
}

/*----------------------------------- Main ----------------------------------*/

main(int argc, char *argv[])
{
  /* JIS24 and METAFONT file names */
  FILE *InFile;
  FILE *OutFile;
  TotalNameType TotalName;
  /* Current METAFONT character number */
  int Number;
  /* Run time parameters */
  struct RunTimeType RunTime;

  printf("\n");
  printf("Bitmaps to METAFONT Conversion Program.\n");  /*To make Borland happy*/
  printf("Version 2.00 Copyright F. Jalbert 1991.\n");
  printf("\n");
  if (ferror(stdout)) exit(1);

  printf("Opening Bitmap file JIS24");
  InFile=fopen("jis24","rb");
  if (InFile==NULL) exit(1);
  printf(".\n");
  printf("\n");
  if (ferror(stdout)) exit(1);

  GetParameters(&RunTime,argc,argv);
  strcpy(TotalName,RunTime.FileName);
  strcat(TotalName,".mf");
  printf("Creating METAFONT file %s",TotalName);
  OutFile=fopen(TotalName,"wt");
  if (OutFile==NULL) exit(1);
  printf(".\n");
  printf("\n");
  if (ferror(stdout)) exit(1);

  printf("Writing initial METAFONT header");
  #ifndef __TURBOC__
  printf("\n");
  #endif
  BeginOut(OutFile,&RunTime);
  printf(".\n");
  printf("\n");
  Generate(InFile,OutFile,&Number,&RunTime);
  printf("\n");
  if (ferror(stdout)) exit(1);

  printf("Writing final METAFONT header");
  #ifndef __TURBOC__
  printf("\n");
  #endif
  EndOut(OutFile,&RunTime);
  printf(".\n");
  printf("Closing METAFONT file %s",TotalName);
  if (fclose(OutFile)==EOF) exit(1);
  printf(".\n");
  printf("Closing Bitmap file JIS24");
  if (fclose(InFile)==EOF) exit(1);
  printf(".\n");
  printf("\n");
  if (ferror(stdout)) exit(1);

  printf("METAFONT code for %d Bitmap(s) generated.\n",Number);
  printf("\n");
  if (ferror(stdout)) exit(1);

  return(0);
}
