unit Main;

{ Structured Storage Browser Applet.
  (C)1995 By John Lam }

interface

uses WinTypes, WinProcs, Classes, Forms, Controls, Menus, Ole2ver,
  Dialogs, StdCtrls, Buttons, ExtCtrls, Grids, Outline, Sysutils, PCMOle2;

type
  TMyStatStg = class
    pszName: PChar;                 { Name of the Stream }
    pIStorage: IStorage;            { Pointer to the IStorage associated with this node }
    dwType: Longint;                { Type of this Stream }
    cbSize: ULargeint;              { The size of this stream- a 64 bit value }
    constructor Create(apszName: PChar; apIStorage: IStorage; adwType: Longint
      ; acbSize: ULargeint);
  end;

  TMainForm = class(TForm)
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    OpenItem: TMenuItem;
    ExitItem: TMenuItem;
    N1: TMenuItem;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    Help1: TMenuItem;
    About1: TMenuItem;
    StatusBar: TPanel;
    SpeedPanel: TPanel;
    OpenBtn: TSpeedButton;
    ExitBtn: TSpeedButton;
    Outline1: TOutline;
    DrawGrid1: TDrawGrid;
    Header1: THeader;
    procedure ShowHint(Sender: TObject);
    procedure ExitItemClick(Sender: TObject);
    procedure OpenItemClick(Sender: TObject);
    procedure SaveItemClick(Sender: TObject);
    procedure About1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Outline1DblClick(Sender: TObject);
    procedure DrawGrid1DrawCell(Sender: TObject; Col, Row: Longint;
      Rect: TRect; State: TGridDrawState);
    procedure Outline1Click(Sender: TObject);
    procedure Header1Sized(Sender: TObject; ASection, AWidth: Integer);
  private
    { Private declarations }
    m_fOleInitialized: Boolean;
    pIRootStorage: IStorage;  { pointer to Root Storage for Structured Storage File }
    m_dwStreamSize: Longint;    { size of the currently open stream }
    m_pStream: IStream;         { pointer to the currently open stream }
    m_lstIStorage: TList;       { maintain a list of pointers to IStorages in root Storage }
    m_iCharWidth: Integer;      { width of a Courier font character }
    procedure ViewStorage(inID: Integer; lpStorage: IStorage);
    procedure FreeIStorageList;
    function GetLowPart(X: Comp): Longint;
  public
    { Public declarations }
  end;

var
  MainForm: TMainForm;

const
  HEXCHARWIDTH = 20;

implementation

uses About;

{$R *.DFM}

constructor TMyStatStg.Create(apszName: PChar; apIStorage: IStorage
  ; adwType: Longint; acbSize: ULargeint);
begin
  inherited Create;
  pszName := apszName;
  pIStorage := apIStorage;
  dwType := adwType;
  cbSize := acbSize;
end;

procedure TMainForm.ShowHint(Sender: TObject);
begin
  StatusBar.Caption := Application.Hint;
end;

procedure TMainForm.ExitItemClick(Sender: TObject);
begin
  Close;
end;

{ Function to extract the Low Longint from a Comp Variable }
function TMainForm.GetLowPart(X: Comp): Longint;
begin
  { Copy the low order four bytes from the Comp type into the Result long integer }
  Move(X, Result, 4);
end;

procedure TMainForm.ViewStorage(inID: Integer; lpStorage: IStorage);
var
  lpEnum: IEnumStatStg;
  ss: TStatStg;
  ulCount, ldwType: Longint;
  sc: SCODE;
  pSubStg: IStorage;
  subID: Integer;
  sType: String[12];
  hr: HResult;
  snbExclude: PStr;
  pMyStatStg: TMyStatStg;
begin
  snbExclude := nil;
  pSubStg := nil;
  if FAILEDHR(lpStorage.EnumElements(0, nil, 0, lpEnum)) then
    ShowMessage('Could not get a Storage Enumerator at level: ' + IntToStr(inID))
  else begin
    sc := S_OK;
    while sc = S_OK do begin
      sc := GetSCode(lpEnum.Next(1, ss, ulCount));
      if (sc <> S_OK) and (sc <> S_FALSE) then begin
        ShowMessage('IEnumStatStg returned an error! SCODE = ' + IntToStr(Longint(sc)));
        Exit;
      end
      else
        ldwType := Longint(ss.dwType);
        case ldwType of
          STGTY_STREAM: sType := 'Stream';
          STGTY_STORAGE: sType := 'Storage';
          STGTY_LOCKBYTES: sType := 'LockBytes';
          STGTY_PROPERTY: sType := 'Property';
          else sType := '**Unknown**';
        end;

      { Add this record to the Outline Control. Note that ss.cbSize (the size of
        the stream object) is a 64 bit integer. }
      if sc = S_OK then begin

        { Add this TStatStg record to the outline control. We need to use a
          helper function: GetLowPart to bust up the ss.cbSize 64 bit integer. }
        subID := Outline1.AddChild(inID, StrPas(ss.pwcsName) + ', Type: ' +
          sType + ', Size: ' + IntToStr(GetLowPart(ss.cbSize)));
        if ldwType = STGTY_STORAGE then begin

          { We're in a Storage- Recursively traverse the storage by calling ourself }
          hr := lpStorage.OpenStorage(ss.pwcsName, nil, STGM_READ or
            STGM_SHARE_EXCLUSIVE, nil, Longint(nil), pSubStg);
          if SUCCEEDEDHR(hr) then begin
            pMyStatStg := TMyStatStg.Create(ss.pwcsName, pSubStg, ss.dwType, ss.cbSize);
            m_lstIStorage.Add(pMyStatStg);
            ViewStorage(subID, pSubStg);   { call ourself }
          end
          else
            ShowMessage('Failed substorage open! hr = ' + IntToStr(Longint(hr)));
        end
        else begin

          { This wasn't a storage, so we write the current lpStorage into the table }
          pMyStatStg := TMyStatStg.Create(ss.pwcsName, lpStorage, ss.dwType, ss.cbSize);
          m_lstIStorage.Add(pMyStatStg);
        end;
      end;
    end;
    lpEnum.Release;
    lpEnum := nil;
  end;
end;

procedure TMainForm.OpenItemClick(Sender: TObject);
var
  pszFilename, pszName: PChar;
  sc: SCODE;
  snbExclude: PStr;
  hr: HResult;
  dummyExclude: IStorage;
  pIStorage: ^IStorage;
  pMyStatStg: TMyStatStg;
  dummyULI: Comp;
begin
  if OpenDialog.Execute then begin

    { Try opening a structured storage document. Filename parsing is done in OpenDialog. }
    try
      pszFilename := StrAlloc(Length(OpenDialog.Filename) + 1);  { + 1 for NULL }
      StrPCopy(pszFilename, OpenDialog.Filename);
      sc := GetSCode(StgIsStorageFile(pszFilename));
      if sc = STG_E_FILENOTFOUND then
        ShowMessage(OpenDialog.Filename + ' was not found.')
      else if sc = S_FALSE then
        ShowMessage(OpenDialog.Filename + ' is not a valid OLE2 Structured Storage file.')
      else if sc = S_OK then begin

        { Now actually open the file, and obtain a pointer to IRootStorage }
        FreeIStorageList;
        snbExclude := nil;
        pIRootStorage := nil;

        if SUCCEEDEDHR(StgOpenStorage(pszFilename, nil, STGM_DIRECT or
          STGM_READWRITE or STGM_SHARE_EXCLUSIVE, snbExclude, Longint(nil),
          IStorage(pIRootStorage))) then begin

          { Create Root of TOutline. This *ALWAYS* has an ItemIndex of 0 }
          Outline1.AddChild(0, 'Root Storage: ' + StrPas(pszFilename));

          { Code to create a list of open storages so that I can open them when
            I need to view their contents later }
          pszName := StrAlloc(100);
          StrCopy(pszName, 'Root Storage: ');
          StrCat(pszName, pszFilename);
          dummyULI := 0;
          pMyStatStg := TMyStatStg.Create(pszName, pIRootStorage, 0, dummyULI);
          m_lstIStorage.Add(pMyStatStg);
          ViewStorage(1, pIRootStorage);    { call recursive outline filler }
        end;
      end;
    finally
      StrDispose(pszFilename);
    end;
  end;
end;

procedure TMainForm.SaveItemClick(Sender: TObject);
begin
  SaveDialog.Execute;
end;

procedure TMainForm.About1Click(Sender: TObject);
begin
  AboutBox.ShowModal;
end;

procedure TMainForm.FormCreate(Sender: TObject);
var
  dwVer, hr: HResult;
begin

  { Make sure that the OLE2 libraries are initialized, and that we're using the correct version }
  m_fOleInitialized := false;
  dwVer := CoBuildVersion;

  if rmm <> HIWORD(Longint(dwVer)) then
    ShowMessage('Invalid OLE Libraries')
  else begin
    hr := CoInitialize(nil);
    if FAILEDHR(hr) then
      ShowMessage('Failed OLE Library Initialization')
    else
      m_fOleInitialized := true;
  end;

  m_lstIStorage := TList.Create;      { create list to hold IStorages }
  m_pStream := nil;                   { current stream is NOT active }

  { Setup DrawGrid }
  with DrawGrid1.Canvas do begin
    Font.Name := 'Courier';
    Font.Size := 8;
    m_iCharWidth := TextWidth('0');
  end;

  DrawGrid1.ColWidths[0] := m_iCharWidth * 76;
  Application.OnHint := ShowHint;
end;

{ Release all of the IStorage interfaces maintained by m_lstIStorage }
procedure TMainForm.FreeIStorageList;
var
  i: Integer;
  pMyStatStg: TMyStatStg;
  pIMalloc: IMalloc;
begin
  if m_lstIStorage.Count <> 0 then
    for i := 0 to pred(m_lstIStorage.Count) do begin
      { Check to see if the pointer in the list is nil }
      pMyStatStg := TMyStatStg(m_lstIStorage.Items[i]);
      if pMyStatStg <> nil then begin
        if pMyStatStg.pIStorage <> nil then
          if pMyStatStg.dwType = Longint(STGTY_STORAGE) then begin
            pMyStatStg.pIStorage.Release;     { release each storage }
            pMyStatStg.pIStorage := nil;
          end;

        { The OLE2 SDK docs say that the owner of the TStatStg structure must free the memory
          that was allocated for the pwcsName PChar in the ss TStatStg structure. This memory
          was allocated using the task allocator from CoGetMalloc. Thus, we must use
          the IMalloc allocator to free that string. }
        CoGetMalloc(MEMCTX_TASK, pIMalloc);
        pIMalloc.Free(pMyStatStg.pszName);
        pIMalloc.Release;
        pMyStatStg.Free;
      end;  
    end;
  m_lstIStorage.Clear;  { clean up the TList }
  Outline1.Clear;       { clear everything in the outline too ... }
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  if m_pStream <> nil then begin
    m_pStream.Release;
    m_pStream := nil;
  end;

  FreeIStorageList;         { free the IStorages list }
  m_lstIStorage.Free;       { destroy the list }

  if m_fOleInitialized then
    CoUninitialize;
end;

procedure TMainForm.Outline1DblClick(Sender: TObject);
var
  pszName: PChar;
  pIStorage: IStorage;
  pIStream: IStream;
  pMyStatStg: TMyStatStg;
  aOffset, dummyLargeint: Comp;
begin
  if Outline1.SelectedItem > 0 then begin
    pMyStatStg := TMyStatStg(m_lstIStorage.Items[Outline1.SelectedItem-1]);
    StatusBar.Caption := 'Selected Item: ' + StrPas(pMyStatStg.pszName);

    { Now try and open the stream for this node, but first close existing stream }
    if m_pStream <> nil then begin
      m_pStream.Release;
      m_pStream := nil;
    end;

    if pMyStatStg.dwType = Longint(STGTY_STREAM) then begin
      pIStorage := pMyStatStg.pIStorage;
      pszName := pMyStatStg.pszName;

      { If there was already a stream open, release it first }
      if SUCCEEDEDHR(pIStorage.OpenStream(pszName, nil, STGM_READ
        or STGM_SHARE_EXCLUSIVE, 0, pIStream)) then begin

        { Setup the DrawGrid to Display this Stream }
        m_dwStreamSize := GetLowPart(pMyStatStg.cbSize);  { set the stream size variable }
        m_pStream := pIStream;                        { save the interface pointer }
        Drawgrid1.RowCount := m_dwStreamSize div 16;  { set the extent of the DrawGrid }
        Drawgrid1.Visible := True;

        { Seek to beginning of file }
        aOffset := 0;
        m_pStream.Seek(aOffset, Longint(STREAM_SEEK_SET), dummyLargeint);
      end;
    end
    else
      DrawGrid1.Visible := False;       { Hide Grid if not a Stream Object }
  end;
end;

procedure TMainForm.DrawGrid1DrawCell(Sender: TObject; Col, Row: Longint;
  Rect: TRect; State: TGridDrawState);
var
  TheText: string[16];
  i: Integer;
  byteArray: Array[0..15] of Byte;
  lBytesRead: Longint;
  aOffset, dummyLargeint: ULargeint;
begin
  begin

    { Calculate byte offset into Stream and seek to that byte position }
    aOffset := Row shl 4;
    if SUCCEEDEDHR(m_pStream.Seek(aOffset, Longint(STREAM_SEEK_SET), dummyLargeint)) then begin
      if SUCCEEDEDHR(m_pStream.Read(@byteArray, 16, lBytesRead)) then begin

        { Output xxxxxxxx: xx xx xx xx xx xx xx xx xx xx xx xx xx xx xx xx  abcdefghijklmnopq }
        TheText := IntToHex(Row shl 4, 6) + ':' ;
        SetBkColor(DrawGrid1.Canvas.Handle, GetSysColor(COLOR_WINDOW));
        DrawGrid1.Canvas.TextOut(Rect.Left + 2 * m_iCharWidth, Rect.Top, TheText);

        for i := 0 to lBytesRead - 1 do begin
          TheText := IntToHex(byteArray[i], 2);
          DrawGrid1.Canvas.TextOut(Rect.Left + 10 * m_iCharWidth + m_iCharWidth * i * 3
            , Rect.Top, TheText);
        end;

        { Format output string as printable characters or '.' if not printable }
        TheText := '';
        for i := 0 to lBytesRead - 1 do
          if IsCharAlphaNumeric(Char(byteArray[i])) then
            TheText := TheText + Char(byteArray[i])
          else
            TheText := TheText + '.';

        DrawGrid1.Canvas.TextOut(Rect.Left + 60 * m_iCharWidth, Rect.Top, TheText);
      end;
    end;
  end;
end;

{ On single click, clear drawgrid }
procedure TMainForm.Outline1Click(Sender: TObject);
begin
  DrawGrid1.Visible := False;
end;

procedure TMainForm.Header1Sized(Sender: TObject; ASection,
  AWidth: Integer);
begin
  Outline1.Width := Header1.SectionWidth[0];
end;

end.

