/*
 * MultiMail offline mail reader
 * message display and editing

 Copyright (c) 1996 Kolossvary Tamas <thomas@tvnet.hu>
 Copyright (c) 1997 John Zero <john@graphisoft.hu>
 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "interfac.h"
#include "isoconv.h"
#include "mysystem.h"

#include <sys/utsname.h>

extern mmail		mm;
extern HelpWindow	helpwindow;
extern AreaListWindow	areas;
extern LetterListWindow	letters;
extern letter_list	*letterList;
extern Interface	*interface;

Line::Line(int size)
{
	next = NULL;
	text = (size ? new char[size] : 0);
}

Line::~Line()
{
	delete text;
}

net_address LetterWindow::PickNetAddr()
{
	Line *line;
	net_address result;
	int i;

	line = linelist[0];
	for (i = 0; i < NumOfLines; i++)
		if (!strncmp(" * Origin:", linelist[i]->text, 10))
			break;

	if (i != NumOfLines) {
		line = linelist[i];
		i = line->length;
		while (((line->text[i - 1]) != '(') && i > 0)
			i--;
		//we have the opening bracket

		while ((i < (int) line->length) &&
			((line->text[i] < '0') || (line->text[i] > '9')))
				i++;
		//we have the begining of the address

		result.init(&line->text[i]);
	} else
		result = *(letterList->getNetAddr());

	return result;
}

void LetterWindow::ReDraw()
{
	touchwin(header);
	wnoutrefresh(header);
	touchwin(text);
	wnoutrefresh(text);
	touchwin(statbar);
	wnoutrefresh(statbar);
}

void LetterWindow::DestroyChain()
{
	if (linelist) {
		while (NumOfLines)
			delete linelist[--NumOfLines];
		delete linelist;
		linelist = NULL;
	}
	letter_in_chain = -1;
}

void LetterWindow::MakeChain(int columns)
{
	Line head(0), *curr;
	char *message, *j, *k, *begin;
	int c, end = 0;

	tagline[0] = '\0';

	DestroyChain();
	message = (char *) letterList->getBody();
	charconv_in(message);
	letter_in_chain = letterList->getCurrent();
	curr = &head;

	j = message;
	while (!end) {
		curr->next = new Line(columns + 1);
		curr = curr->next;

		NumOfLines++;

		begin = NULL;
		k = curr->text;
		c = 0;

		while ((*j != 10) && !end && (c < columns)) {
			if (*j == ' ')
				begin = NULL;
			else
				if (!begin)
					begin = j;
			if (*j) {
				*k++ = *j++;
				c++;
			} else
				end = 1;
		}
		if ((c == columns) && begin && ((j - begin) < columns)) {
			c -= j - begin;
			j = begin;
		}
		if (*j == 10)
			j++;

		curr->text[c] = '\0';
		curr->length = c;
	}

	linelist = new Line *[NumOfLines];
	curr = head.next;
	c = 0;
	while (curr) {
		linelist[c++] = curr;
		curr = curr->next;
	}
}

void LetterWindow::StatToggle(int mask)
{
	int stat = letterList->getStatus();
	stat ^= mask;
	letterList->setStatus(stat);
	DrawHeader();
}

void LetterWindow::Draw()
{
	letterList->setRead();	// nem ide kene? de.
	DrawHeader();
	DrawBody();
}

void LetterWindow::netAdd(char *tmp)
{
	net_address na = *(letterList->getNetAddr());
	if (na.zone) {
		strcat(tmp, " @ ");
		strcat(tmp, na.text());
	}
}

void LetterWindow::DrawHeader()
{
	char tmp[256];
	int stat = letterList->getStatus();

	wmove(header, 1, 0);
	wbkgdset(header, C_LHEADTEXT | ' ');
	wclrtobot(header);
	wbkgdset(header, ' ');
	wattrset(header, C_LHEADTEXT);

	mvwaddstr(header, 1, 2, "Msg#:");
	mvwaddstr(header, 2, 2, "From:");
	mvwaddstr(header, 3, 2, "  To:");
	mvwaddstr(header, 4, 2, "Subj:");

	mvwaddstr(header, 1, COLS - 33, "Date:");
	mvwaddstr(header, 2, COLS - 33, "Line:");
	mvwaddstr(header, 3, COLS - 33, "Stat:");

	wattrset(header, C_LHFLAGS);
	wattrset(header, (letterList->getPrivate()) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 27, "Pvt");
	wattrset(header, (stat & MS_READ) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 23, "Read");
	wattrset(header, (stat & MS_REPLIED) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 18, "Replied");
	wattrset(header, (stat & MS_MARKED) ? C_LHFLAGSHI : C_LHFLAGS);
	mvwaddstr(header, 3, COLS - 10, "Marked");

	wattrset(header, C_LHMSGNUM);
	mvwprintw(header, 1, 8, "%d (%d of %d)   ", letterList->getMsgNum(),
		letterList->getCurrent() + 1, mm.areaList->getNoOfLetters());

	mvwprintw(header, 2, COLS-27, "%5d/%-21d", position + 1, NumOfLines);

	wattrset(header, C_LHFROM);
	strcpy(tmp, letterList->getFrom());
	charconv_in(tmp);

	if (mm.areaList->isNetmail())
		netAdd(tmp);

	mvwaddstr(header, 2, 8, tmp);

	wattrset(header, C_LHTO);
	strcpy(tmp, letterList->getTo());
	charconv_in(tmp);

	if (mm.areaList->isReplyArea())
		netAdd(tmp);

	mvwaddstr(header, 3, 8, tmp);

	wattrset(header, C_LHSUBJ);
	sprintf(tmp, "%-71.71s", letterList->getSubject());
	charconv_in(tmp);
	mvwaddstr(header, 4, 8, tmp);

	for (int i = 0; i < COLS - 79; i++)
		waddch(header, ' ');

	wattrset(header, C_LHDATE);
	strcpy(tmp, letterList->getDate());
	charconv_in(tmp);
	mvwaddstr(header, 1, COLS - 27, tmp);

	//mvwaddch(header, 5, 0, ACS_LLCORNER);
	//whline(header, ACS_HLINE, COLS-2);
	//mvwaddch(header, 5, COLS-1, ACS_LRCORNER);

	wnoutrefresh(header);
}

void LetterWindow::lineCount()
{
	wattrset(header, C_LHMSGNUM);
	mvwprintw(header, 2, COLS-27, "%5d/%-21d", position + 1, NumOfLines);
	wnoutrefresh(header);
}

void LetterWindow::oneLine(int i)
{
	char test, *t, *r, *currtext;
	int j, c;

	currtext = linelist[position + i]->text;
	int length = (int) linelist[position + i]->length;

	wattrset(text, C_LTEXT);
	for (int m = 0; m < 5; m++)
		if (currtext[m] == '>')	//quoting
			wattrset(text, C_LQTEXT);
	test = currtext[0];
	if ((currtext[1] == test) && (currtext[2] == test) &&
	   ((currtext[3] == ' ') || length == 3))
		switch (test) {
		case '.':
			wattrset(text, C_LTAGLINE); //tagline
			t = &currtext[4];
			r = tagline;
			for (c = 0; (*t) && (*t != '\r') &&
			     (*t != '\n') && c < 79;
				t++, r++, c++)
				   *r = *t;
			*r = 0;
			break;
		case '-':
		case '~':
			wattrset(text, C_LTEAR);
			//mailer prog
		}
	else if (!strncmp(currtext, " * Origin:", 10))
		wattrset(text, C_LORIGIN);	//origin line

	mvwaddstr(text, i, 0, currtext);
	for (j = 0; j < (COLS - length); j++)
		waddch(text, ' ');
}

void LetterWindow::DrawBody()
{
	int i, j;

	if (letter_in_chain != letterList->getCurrent())
		MakeChain(COLS);

	lineCount();

	for (i = 0; i < y; i++)
		if ((position + i) < NumOfLines)
			oneLine(i);
		else
			for (j = 0; j < x; j++)
				mvwaddch(text, i, j, ' ');

	wnoutrefresh(text);
}

void LetterWindow::MakeActive()
{
	DestroyChain();

	header = newwin(6, COLS, 0, 0);
	text = newwin(LINES - 7, COLS, 6, 0);
	statbar = newwin(1, COLS, LINES - 1, 0);
	leaveok(header, TRUE);
	leaveok(text, TRUE);

	wattrset(header, C_LBOTTSTAT);

	wprintw(header, " " MM_TOPHEADER, MM_NAME, MM_MAJOR, MM_MINOR);
	for (int i = 0; i < COLS - 29; i++)
		waddch(header, ' ');

	getmaxyx(text, y, x);

	wattrset(statbar, C_LBOTTSTAT);
	char tmp[129];
	sprintf(tmp, " %-71.127s", mm.areaList->getDescription());
	charconv_in(tmp);
	waddstr(statbar, tmp);
	for (int i = 0; i < (COLS - 86); i++)
		waddch(statbar, ' ');
	mvwaddstr(statbar, 0, COLS - 15, "F1 or ? - Help ");
	wnoutrefresh(statbar);

	linelist = NULL;
	NumOfLines = position = 0;
	Draw();
}

void LetterWindow::Next()
{
	if (letterList->getCurrent() < (mm.areaList->getNoOfLetters() - 1)) {
		letters.Move(DOWN);
		letterList->gotoLetter(letterList->getCurrent() + 1);
		position = 0;
		Draw();
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->RIGHT_ARROW();
	}
}

void LetterWindow::Previous()
{
	if (letterList->getCurrent() > 0) {
		letters.Move(UP);
		letterList->gotoLetter(letterList->getCurrent() - 1);
		position = 0;
		Draw();
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->ikeypad(UP);
	}
}

void LetterWindow::Move(direction dir)
{
	switch (dir) {
	case UP:
		if (position > 0) {
			position--;
			scrollok(text, TRUE);
			wscrl(text, -1);
			scrollok(text, FALSE);
			oneLine(0);
			wnoutrefresh(text);
			lineCount();
		}
		break;
	case DOWN:
		if (position < NumOfLines - y) {
			position++;
			lineCount();
			scrollok(text, TRUE);
			wscrl(text, 1);
			scrollok(text, FALSE);
			oneLine(y - 1);
			wnoutrefresh(text);
		}
		break;
	case HOME:
		position = 0;
		DrawBody();
		break;
	case END:
		if (NumOfLines > y) {
			position = NumOfLines - y;
			DrawBody();
		}
		break;
	case PGUP:
		position -= ((y < position) ? y : position);
		DrawBody();
		break;
	case PGDN:
		if (position < NumOfLines - y) {
			position += y;
			if (position > NumOfLines - y)
				position = NumOfLines - y;
			DrawBody();
		}
		break;
	}
}

void LetterWindow::NextDown()
{
	position += y;
	if (position >= NumOfLines)
		Next();
	else
		DrawBody();
}

void LetterWindow::Delete()
{
	delwin(header);
	delwin(text);
	delwin(statbar);
}

void LetterWindow::Save(int all)
{
	FILE *fd;
	char filename[255];

	ShadowedWin question(5, 60, 10, 10, C_LLSAVETEXT);
	mvwaddstr(question.win, 1, 2, "Save to file:");
	mvwaddch(question.win, 2, 2, '<');
	mvwaddch(question.win, 2, 57, '>');
	mvwaddstr(question.win, 3, 30, "<ESC> (pause) to cancel");

	if (all)
		sprintf(filename, "%.8s.all", mm.areaList->getName());
	else
		sprintf(filename, "%.8s.%.03d", mm.areaList->getName(),
			letterList->getCurrent());

	for (unsigned c = 0; c < strlen(filename); c++)
		if (filename[c] == ' ')
			filename[c] = '_';

	if (question.getstring(2, 3, filename, 54, C_LLSAVETEXT, C_LLSAVEGET))
	{
		mychdir(mm.resourceObject->get(SaveDir));
		if ((fd = fopen(filename, "at"))) {
			if (all)
				for (int i = 0; i <
				   mm.areaList->getNoOfLetters(); i++) {
					letterList->gotoLetter(i);
					write_to_file(fd);
				}
			else
				write_to_file(fd);
			fclose(fd);
			if (!all)
				MakeChain(COLS);
		}
	}
}

void LetterWindow::set_Letter_Params(net_address *nm, const char *to)
{
	NM = *nm;
	if (to)
		strncpy(To, to, 29);
	else
		To[0] = '\0';
}

void LetterWindow::set_Letter_Params(int area, char param_key)
{
	key = param_key;
	if (key == 'N') {
		//find netmail area
		for (int i = 0; i < mm.areaList->noOfAreas(); i++) {
			mm.areaList->gotoArea(i++);
			if (mm.areaList->isNetmail())
				break;
		}
		if (mm.areaList->isNetmail())
			replyto_area = mm.areaList->getAreaNo();
		else
			//netmail area not found, we should warn the user
			replyto_area = area;
		areas.Select();
	} else
		replyto_area = area;
}

void LetterWindow::QuoteText(const char *reply_filename)
{
	FILE *reply;
	char TMP[26], TMP2[26], mg[4];
	int end, i;

	strcpy(TMP, letterList->getFrom());
	strcpy(TMP2, letterList->getTo());
	charconv_in(TMP);
	charconv_in(TMP2);
	reply = fopen(reply_filename, "wt");	//check!
	fprintf(reply, "-=> %s wrote to %s <=-\n\n", TMP, TMP2);

	strncpy(mg, letterList->getFrom(), 2);
	mg[2] = '\0';
	mg[3] = '\0';
	strcpy(TMP, letterList->getFrom());
	i = 1;
	for (int j = 1; j < 3; j++) {
		end = 0;
		while (TMP[i] && !end) {
			if ((TMP[i - 1] == ' ') && (TMP[i] != ' ')) {
				mg[j] = TMP[i];
				if (j == 1)
					end = 1;
			}
			i++;
		}
	}
	charconv_in(mg);

	MakeChain(75);

	for (i = 0; i < NumOfLines; i++)
		fprintf(reply, " %s> %s\n", mg, linelist[i]->text);

	MakeChain(COLS);

	fclose(reply);
}

int LetterWindow::HeaderLine(ShadowedWin *X, char *Y, int limit, int pos,
			int color)
{
	charconv_in(Y);
	int getit = X->getstring(pos, 8, Y, limit, color, color);
	charconv_out(Y);
	return getit;
}

int LetterWindow::EnterHeader(char *FROM, char *TO, char *SUBJ, int *privat)
{
	ShadowedWin *rep_header;
	char NETADD[25];
	int result, end = 0, current = 0, maxitems = 3;

	if (mm.areaList->isNetmail() || NM.zone) {
		strcpy(NETADD, NM.text());
		maxitems++;
	}

	rep_header = new ShadowedWin(maxitems + 2, COLS - 2, (LINES / 2) - 3,
		1, C_LETEXT);

	mvwaddstr(rep_header->win, 1, 2, "From:");
	mvwaddstr(rep_header->win, 2, 2, "  To:");
	if (maxitems == 4)
		mvwaddstr(rep_header->win, 3, 2, "Addr:");
	mvwaddstr(rep_header->win, maxitems, 2, "Subj:");
	wrefresh(rep_header->win);

	do {
		result = HeaderLine(rep_header, (current == (maxitems - 1)) ?
			SUBJ : (((current == 2) && (maxitems == 4)) ? 
			NETADD : (current ? TO : FROM)),
			(current == (maxitems - 1)) ? 69 : 25,
			current + 1, (current & 1) ? C_LEGET1 : C_LEGET2);

		switch (result) {
		case 0:
			end = 1;
			break;
		case 1:
			current++;
			if (current == maxitems)
				end = 1;
			break;
		case 2:
			if (current > 0)
				current--;
			break;
		case 3:
			if (current < maxitems - 1)
				current++;
		}
	} while (!end);

	if (result) {
		if (mm.areaList->isNetmail() || NM.zone)
			NM.init(NETADD);
		if (!interface->WarningWindow(
			"Make letter private?", *privat ? "Yes" : "No",
				*privat ? "No" : "Yes"))
					*privat = !*privat;
	}

	delete rep_header;

	switch (interface->active()) {
	case letter:
		DrawHeader();
		ReDraw();
		break;
	case letterlist:
	case arealist:
		interface->screentouch();
		interface->activeList()->ReDraw();
		helpwindow.redraw();
	default:;
	}

	return result;
}

void LetterWindow::editletter(const char *reply_filename)
{
	char command[256];

	sprintf(command, "%s %s", mm.resourceObject->get(editor),
		canonize(reply_filename));
	mysystem(command);
}

long LetterWindow::reconvert(const char *reply_filename)
{
	FILE *reply;
	long replen;
	char *body;

	reply = fopen(reply_filename, "rt");
	fseek(reply, 0, SEEK_END);
	replen = ftell(reply);
	rewind(reply);

	body = new char[replen + 1];
	replen = (long) fread(body, 1, replen, reply);
	fclose(reply);

	body[replen] = '\0';
	charconv_out(body);
	while (body[replen - 1] == '\n')
		replen--;

	reply = fopen(reply_filename, "wt");
	fwrite(body, 1, replen, reply);
	fclose(reply);

	delete body;
	return replen;
}

int LetterWindow::EnterLetter()
{
	FILE *reply;
	TaglineWindow *taglines;
	char reply_filename[256], FROM[26], TO[26], SUBJ[78];
	long replen;
	int privat, replyto_num;

	mm.areaList->gotoArea(replyto_area);

	// HEADER

	int usealias = mm.areaList->getUseAlias();
	if (usealias) {
		sprintf(FROM, "%.25s", mm.resourceObject->get(AliasName));
		if (!FROM[0])
			usealias = 0;
	}
	if (!usealias)
		sprintf(FROM, "%.25s", mm.resourceObject->get(LoginName));

	if (key == 'E') {
		strcpy(TO, (To[0] ? To : "All"));
		SUBJ[0] = '\0';	//we don't have subject yet
	} else {
		sprintf(TO, "%.25s", (key == 'O') ? letterList->getTo() :
			letterList->getFrom());
		sprintf(SUBJ, "Re: %.65s", letterList->getSubject());
	}

	if ((key == 'N') || mm.areaList->isNetmail())
		privat = 1;
	else
		privat = (key == 'E') ? 0 : letterList->getPrivate();

	if (!EnterHeader(FROM, TO, SUBJ, &privat))
		return 0;

	replyto_num = (key == 'E') ? 0 : letterList->getMsgNum();

	// BODY

	mytmpnam(reply_filename);

	// Quote the old text, mark original as replied
	if (key != 'E') {
		QuoteText(reply_filename);
		letterList->setStatus(letterList->getStatus() | MS_REPLIED);
	}

	// Edit the reply
	editletter(reply_filename);

	reply = fopen(reply_filename, "at");

	// Signature
	const char *sg = mm.resourceObject->get(sigFile);
	if (sg) {
		FILE *s = fopen(sg, "rt");
		if (s) {
			char bufx[81];

			fputc('\n', reply);
			while (fgets(bufx, 79, s))
				fputs(bufx, reply);
			fputc('\n', reply);
			fclose(s);
		}
	}

	// Tagline
	tagline[0] = '\0';
	taglines = new TaglineWindow;
	if (tagline[0])
		fprintf(reply, "\n... %s\n", tagline);

	// Tearline (not for Blue Wave -- it does its own)
	if (mm.driverList->useTearline()) {
		struct utsname buf;
		uname(&buf);
		if (!tagline[0])
			fputc('\n', reply);
		fprintf(reply, "--- %s/%s v%d.%d\n", MM_NAME, buf.sysname,
			MM_MAJOR, MM_MINOR);
	}

	fclose(reply);

	// Reconvert the text
	replen = reconvert(reply_filename);

	if (interface->active() == letter) {
		DrawHeader();
		ReDraw();
	}

	mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, replyto_num,
		privat, &NM, reply_filename, (int) replen);

	areas.Select();

	NM.zone = 0;
	To[0] = '\0';

	return 1;
}

int LetterWindow::EditLetter()
{
	FILE *reply;
	char reply_filename[256], FROM[26], TO[26], SUBJ[78], *body;
	long siz;
	int privat, replyto_num, replyto_area;

	strcpy(FROM, letterList->getFrom());
	strcpy(TO, letterList->getTo());
	strcpy(SUBJ, letterList->getSubject());
	privat = letterList->getPrivate();

	NM = *(letterList->getNetAddr());

	if (!EnterHeader(FROM, TO, SUBJ, &privat))
		return 0;

	DestroyChain();		// current letter's chain reset

	replyto_num = letterList->getReplyTo();
	replyto_area = letterList->getAreaID();

	mytmpnam(reply_filename);

	body = (char *) letterList->getBody();
	charconv_in(body);
	reply = fopen(reply_filename, "wt");
	fwrite(body, strlen(body), 1, reply);
	fclose(reply);
	body = NULL;		// it will be auto-dealloc'd by next getBody

	editletter(reply_filename);
	siz = reconvert(reply_filename);

	mm.areaList->killLetter(letterList->getMsgNum());
	mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, replyto_num,
		privat, &NM, reply_filename, (int) siz);

	letterList->rrefresh();
	areas.Select();
	if (interface->active() == letter)
		interface->back();
	else {
		letters.Delete();
		letters.MakeActive();
		letters.Draw();
	}

	NM.zone = 0;

	return 1;
}

void LetterWindow::set_Tagline(const char *tl)
{
	strncpy(tagline, tl, 76);
}

void LetterWindow::GetTagline()
{
	TaglineWindow *tags;

	tags = new TaglineWindow(1);
	tags->EnterTagline(tagline);
	delete tags;
}

void LetterWindow::write_to_file(FILE *fd)
{
	char Header[400];
	int j;

	//write header to file
	for (j = 0; j < 80; j++)
		fputc('=', fd);

	sprintf(Header,
		"\n System: %.71s\n   Area: %.71s\n   Date: %.25s\n"
		"   From: %.45s\n     To: %.45s\n   Subj: %.71s\n",
		mm.resourceObject->get(BBSName),
		mm.areaList->getDescription(), letterList->getDate(),
		letterList->getFrom(), letterList->getTo(),
		letterList->getSubject());

	charconv_in(Header);
	fputs(Header, fd);

	for (j = 0; j < 80; j++)
		fputc('-', fd);
	fputc('\n', fd);

	//write chain to file
	MakeChain(80);
	for (j = 0; j < NumOfLines; j++)
		fprintf(fd, "%s\n", linelist[j]->text);
	fputc('\n', fd);
}

void LetterWindow::ansi_dump()
{
	AnsiWindow *ansi;
	unsigned char *source;

	source = (unsigned char *) letterList->getBody();
	ansi = new AnsiWindow(source, letterList->getSubject());
	ansi->KeyHandle();
	delete ansi;
	ReDraw();
}
