/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPEVENTS.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:
**     CREATION DATE:       1990/5/14
**
** DESCRIPTION: Contains the datastructures and functions
**              which implement the event and timer processing
**              for the IP package. All IP events, except those
**              related to the sending part, are initiated
**              by this event processing.
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPEVENTS.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPEVENTS.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:23:28   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:14:40   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPEVENTS.C_V   1.1   21 Nov 1990 14:23:28   etstjan  $";
#endif

#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <dos.h>                  /* interrupt handling */
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "ipif.h"                 /* interface layer */
#include      "ipevents.h"


typedef struct _event_t {              /* structure contains event info */
    struct _event_t  *NextEvent;       /* next event in chain */
    void (*FuncPoint)(EVENTUNION *);   /* pointer to scheduled function */
    EVENTUNION       Param;            /* parameter for scheduled function */
    USHORT           *TimeToGo;        /* countdown for timer purposes */
    USHORT           PriorSave;        /* store priority on count down */
    } EVENT_T;

typedef struct _event_list {           /* structure contains event info */
    EVENT_T          *HeadOfList;      /* head of event list */
    EVENT_T          *TailOfList;      /* tail of event list */
    } EVENT_LIST;


EVENT_LIST   PriorityLst[LOW+1] = {    /* a high and low priority list */
               {NULL, NULL},
               {NULL, NULL}
               };

EVENT_T      *TimerQueue = NULL;       /* start of timer queue */

/**************************************************************
** NAME:        Schedule
** SYNOPSIS:    int Schedule(PRIORITY Priority,
**                           void (*ActFunc)(EVENTUNION *),
**                           EVENTUNION *ThisParam);
** DESCRIPTION: Places a function in the event list.
**              Priority can be HIGH or LOW.
**              ActFunc is a pointer to the function,
**              ThisParam is a pointer to the parameter
**              used when calling ActFunc.
** RETURNS:     NO_ERR -->   no error
**              else    error code
**************************************************************/
int  Schedule(PRIORITY Priority,
              void (*ActFunc)(EVENTUNION *),
              EVENTUNION *ThisParam)
{
  EVENT_T     *EventPtr;
  EVENT_LIST  *CurList;
  int         RetCode = NO_ERR;

  if (Priority > LOW) {
    RetCode = ILL_REQUEST;
    }

  else {
    CurList = PriorityLst + Priority;    /* get actual event list */

    if ((EventPtr = IPBufGet(sizeof(EVENT_T))) == NULL) {
      RetCode = NOSPACE;
      }
    else {
      EventPtr->FuncPoint = ActFunc;
      if (ThisParam != NULL) {
        EventPtr->Param = *ThisParam;
        }
      EventPtr->NextEvent = NULL;
      EventPtr->TimeToGo = NULL;

      _disable();                      /* enter critical section */
      if (CurList->HeadOfList == NULL) {
        CurList->HeadOfList = EventPtr;
        }
      else {
        CurList->TailOfList->NextEvent = EventPtr;
        }
      CurList->TailOfList = EventPtr;
      _enable();
      }
    }
  return RetCode;
}


/**************************************************************
** NAME:        ProcessEvents
** SYNOPSIS:    void ProcessEvents(void);
**
** DESCRIPTION: Processes events placed in the event lists.
**              The events are processed in priority order.
**              In a priority class, the precedence is
**              first come, first serve.
**              After an event has been served, the
**              eventlists are scanned again, starting
**              with the highest priority.
** RETURNS:
**
**************************************************************/
void ProcessEvents(void)
{
  EVENT_LIST *CurrentList;
  EVENT_T    *CurrentEvent;
  int        i = 0;

  while (i <= LOW) {
    _disable();
    CurrentList = PriorityLst + i;
    /* enter the critical section, keep it as short as possible */
    if ((CurrentEvent = CurrentList->HeadOfList) != NULL) {
      CurrentList->HeadOfList = CurrentEvent->NextEvent;
      }
    _enable();
    if (CurrentEvent != NULL) {
      (*CurrentEvent->FuncPoint)(&(CurrentEvent->Param));
      IPBufFree(CurrentEvent);
      i = 0;                           /* after an event, search again */
      }
    else {
      i++;                             /* else, try next list */
      }
    }
}


/**************************************************************
** NAME:        Schedule2
** SYNOPSIS:    static void Schedule2(EVENT_T *CurEvent);
**
** DESCRIPTION: Places a function in the event list.
**              All parameters needed should be placed in
**              the passed parameter.
**              When not used in interrupt processing,
**              interrupts should be disabled before calling
**              this function.
** RETURNS:
** SEE ALSO:    Schedule, ProcessEvents
**************************************************************/
static void Schedule2(EVENT_T *CurEvent)
{
  EVENT_LIST  *CurList;

  CurList = PriorityLst + CurEvent->PriorSave;
  CurEvent->NextEvent = NULL;

  if (CurList->HeadOfList == NULL) {
    CurList->HeadOfList = CurEvent;
    }
  else {
    CurList->TailOfList->NextEvent = CurEvent;
    }
  CurList->TailOfList = CurEvent;

  return;
}


/**************************************************************
** NAME:        SetTimer
** SYNOPSIS:    int  SetTimer(PRIORITY Priority,
**                            void (*ActFunc)(EVENTUNION *),
**                            EVENTUNION *ThisParam,
**                            WORD *TimePlace,
**                            WORD Seconds);
**
** DESCRIPTION: Sets a timer. Priority determines the priority
**              when the function associated with the timer
**              is scheduled. ThisParam is the parameter used
**              when ActFunc is called. The caller should
**              deliver a pointer to an WORD, which
**              will be used to store the countdown timer.
**              Reading this value makes no sense. Seconds
**              determines the period before the timer
**              expires (in seconds).
**
** RETURNS:     NO_ERR -->   no error
**              else    error code
**
** SEE ALSO:    Schedule, ResetTimer
**************************************************************/
int  SetTimer(PRIORITY Priority, void (*ActFunc)(EVENTUNION *),
              EVENTUNION *ThisParam, WORD *TimePlace,
              WORD Seconds)
{
  EVENT_T       *EventPtr;
  EVENT_T       *NextPtr;
  EVENT_T       **PriorPtr = &TimerQueue;
  USHORT        TotalTime = 0;

  if (Priority > LOW) {
    return ILL_REQUEST;
    }
  if ((EventPtr = IPBufGet(sizeof(EVENT_T))) == NULL) {
    return NOSPACE;
    }
  EventPtr->FuncPoint = ActFunc;
  if (ThisParam != NULL) {
    EventPtr->Param = *ThisParam;
    }
  EventPtr->TimeToGo = TimePlace;
  EventPtr->PriorSave = Priority;

  if (Seconds == 0) {
    _disable();
    Schedule2(EventPtr);
    _enable();
    }
  else {
    /* enter the critical section */
    _disable();
    NextPtr = TimerQueue;
    while ((NextPtr != NULL) &&
           ((TotalTime + *NextPtr->TimeToGo) <= Seconds)) {
      TotalTime += *NextPtr->TimeToGo;
      PriorPtr = &NextPtr->NextEvent;
      NextPtr = NextPtr->NextEvent;
      }

    *PriorPtr = EventPtr;
    EventPtr->NextEvent = NextPtr;
    *EventPtr->TimeToGo = Seconds - TotalTime;

    if (NextPtr != NULL) {
      *NextPtr->TimeToGo -= *EventPtr->TimeToGo;
      }

    /* clear the critical section */
    _enable();
    }
  return NO_ERR;

}

/**************************************************************
** NAME:        SearchTimer
** SYNOPSIS:    EVENT_T  *SearchTimer(EVENT_T **PriorPtr,
**                                    WORD *SearchAdd)
**
** DESCRIPTION: Searches in the event list starting at
**              *PriorPtr for the timer address SearchAdd.
**
** RETURNS:     The address of the EVENT_T containing
**              SearchAdd, NULL if SearchAdd is not found.
**              PriorPtr contains a pointer to the EVENT_T
**              prior to the return value.
**************************************************************/
EVENT_T  *SearchTimer(EVENT_T ***PriorPtr,
                      WORD *SearchAdd)
{
  EVENT_T *NextPtr = **PriorPtr;

  while ((NextPtr != NULL) &&          /* compare by address of timer */
         ((NextPtr->TimeToGo) != SearchAdd)) {
    *PriorPtr = &NextPtr->NextEvent;
    NextPtr = NextPtr->NextEvent;
    }
  return NextPtr;
}

/**************************************************************
** NAME:        ResetTimer
** SYNOPSIS:    int ResetTimer(WORD *TimePlace);
**
** DESCRIPTION: Resets the timer associated with TimePlace.
**              First the timerlist is scanned. If the timer
**              is not found, the eventlists are scanned
**              as well.
**
** RETURNS:     NO_ERR    --> no error
**              NOT_FOUND --> timer could not be located
**              else      --> IP buffer management return code
**
** SEE ALSO:    SetTimer, Schedule, Schedule2, TimerDriver
**************************************************************/
int  ResetTimer(WORD *TimePlace)
{
  EVENT_LIST    *CurrentList;
  EVENT_T       *NextPtr;
  EVENT_T       **PriorPtr;
  int           i;

  /* first, search if timer has not expired yet */
  PriorPtr = &TimerQueue;
  _disable();
  NextPtr = SearchTimer(&PriorPtr, TimePlace);
  if (NextPtr != NULL) {
    *PriorPtr = NextPtr->NextEvent;
    }
  _enable();

  /* if the timer has not been found, search in the event lists */
  for (i = 0; (i <= LOW) && (NextPtr == NULL); i++) {
    CurrentList = PriorityLst + i;
    PriorPtr = &CurrentList->HeadOfList;
    _disable();
    NextPtr = SearchTimer(&PriorPtr, TimePlace);
    if (NextPtr != NULL) {
      *PriorPtr = NextPtr->NextEvent;
      }
    _enable();
    }
  return (NextPtr == NULL) ? NOT_FOUND : IPBufFree(NextPtr);
}

/**************************************************************
** NAME:        TimerDriver
** SYNOPSIS:    void TimerDriver(void);
**
** DESCRIPTION: Entry point for timer countdown.
**              This function should be activated every
**              second for correct functioning.
**
** RETURNS:
** SEE ALSO:    SetTimer, ResetTimer, Schedule, Schedule2
**************************************************************/
void TimerDriver(void)
{
  EVENT_T  *PassPtr;

  if (TimerQueue != NULL) {
    (*TimerQueue->TimeToGo)--;
    while ((TimerQueue != NULL) && (*TimerQueue->TimeToGo == 0)) {
      PassPtr = TimerQueue;
      TimerQueue = TimerQueue->NextEvent;
      Schedule2(PassPtr);
      }
    }
  return;
}


