/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       DPFPKT1.C
**     SYSTEM   NAME:       GOBBLER
**     ORIGINAL AUTHOR(S):  Tirza van Rijn
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1991/11/11
**
** DESCRIPTION: Packet filter for the remote GOBBLER
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   DPFPKT1.C  $
** LOGINFO:     $Log:   I:/TIRZA/CPROG/BEHOLDER/APS/PKTCAPT/VCS/DPFPKT1.C_V  $
**              
**                 Rev 1.0   11 Nov 1991 15:26:54   tirza
**              Initial revision.
*************************************************************************/
#if !defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/TIRZA/CPROG/BEHOLDER/APS/PKTCAPT/VCS/DPFPKT.C_V   1.0   16 May 1991 15:26:54   tirza  $";
#endif


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <snmp.h>
#include <mib2.h>
#include <error.h>
#include <dp.h>

#define P_PktF_start            (StartOf_FilterList[0])
#define N_PktF_start            (StartOf_FilterList[1])
#define P_StartF_start          (StartOf_FilterList[2])
#define N_StartF_start          (StartOf_FilterList[3])
#define P_StopF_start           (StartOf_FilterList[4])
#define N_StopF_start           (StartOf_FilterList[5])
#define NrOf_P_PktFs            (NrOf_Filters[0])
#define NrOf_N_PktFs            (NrOf_Filters[1])
#define NrOf_P_StartFs          (NrOf_Filters[2])
#define NrOf_N_StartFs          (NrOf_Filters[3])
#define NrOf_P_StopFs           (NrOf_Filters[4])
#define NrOf_N_StopFs           (NrOf_Filters[5])
#define OjiNumber(x) (sizeof((x))/sizeof(unsigned))


DPHDRINFO EthHdrInfo = {            
    0,                              
    6,                              
    12,                             
    14,                             
    6,                              
    2                               
};
DPHDRINFO *hdrinfo = &EthHdrInfo;

typedef struct _FILTER {
    unsigned int FilterNo;
    int UseDestAddr;
    BYTE DestAddr [6];
    int UseSrcAddr;
    BYTE SrcAddr [6];
    int UsePktType;
    unsigned int PktTypeInt;
    BYTE PktType [2];
    int UseContents;
    int ContentsOff;
    int ContentsLen;
    BYTE *Contents;
    int UseDevNr;
    int DevNr;
    int UsePktSize;
    unsigned int PktSizeLL;
    unsigned int PktSizeUL;
    struct _FILTER *FNext;
} FILTER;


/* Function prototypes */
static int  ProcessEventPacket(DPEVENT Event);
static int  ReceivePkt(void);
static int  Init(void);
static int  Start(void);
static void Stop(void);
static void Reset(void);
static unsigned FilterGetVars(pkt_obj *Obj, void *Arg);
static unsigned FilterGetNxtVars(pkt_obj *Obj, void *Arg);
static unsigned FilterSetVars(pkt_obj *Obj, void *Arg);
static unsigned FilterGetFilterStat(pkt_obj *Obj, void *Arg);
static int FilterPacket(FILTER *fp, DPBUF *frame);
static FILTER *AddFilter(unsigned FilterType);
static void ResetFilters(void);
static void DisposeFilterList(FILTER *fp);
static void Swap(int *a, int *b);
static FILTER *StartOf_FilterList[6] = {
    NULL,           /* P_PktF_start */
    NULL,           /* N_PktF_start */
    NULL,           /* P_StartF_start */
    NULL,           /* N_StartF_start */
    NULL,           /* P_StopF_start */
    NULL,           /* N_StopF_start */
};
unsigned int NrOf_Filters[6] = {
    0,              /* NrOf_P_PktFs */
    0,              /* NrOf_N_PktFs */
    0,              /* NrOf_P_StartFs */
    0,              /* NrOf_N_StartFs */
    0,              /* NrOf_P_StopFs */
    0,              /* NrOf_N_StopFs */
};
DPFILT DpfPkt   = { "Pkt",   ProcessEventPacket,   0 };
static int r;
static unsigned GetVars[] = {1,3,6,1,4,1,99,99,99};
static unsigned GetFilterStat[] = {1,3,6,1,4,1,99,99,96,1};
static int Filtering;
static int FilterStat;


static int ProcessEventPacket(DPEVENT Event)
{
    int ret = 0;

    switch(Event)
    {
        case DPE_RECEIVEPKT:  ret = ReceivePkt();       break;
        case DPE_INIT:          r = Init();             break;
        case DPE_START:             Start(); ret = r;  break;
        case DPE_STOP:              Stop();             break;
        case DPE_RESET:             Reset();            break;
    }
    return ret;
}

static int ReceivePkt(void)
{
    DPBUF *frame;
    int ret;
    FILTER *fp;

    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Starting..."); 
    frame = DpReceivePkt();

    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Filtering = %d", Filtering); 
    if (Filtering == 0)
    {
        if (P_StartF_start == NULL && N_StartF_start == NULL)
            Filtering = 1;
        else
        {
            fp = P_StartF_start;
            while ((Filtering == 0) && (fp != NULL))
            {
                Filtering = Filtering || FilterPacket(fp, frame);
                fp = fp->FNext;
            }
            if (Filtering == 0)
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass p-START-filter");
            else
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed p-START-filter");
            fp = N_StartF_start;
            while ((Filtering == 1) && (fp != NULL))
            {
                Filtering = Filtering && !(FilterPacket(fp, frame));
                fp = fp->FNext;
            }
            if (Filtering == 0)
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass n-START-filter");
            else
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed n-START-filter");
        }
        if (Filtering == 1)
            FilterStat = 1;
    }
    if (Filtering == 0)
        return (0);
    else
    {
        if (P_PktF_start == NULL && N_PktF_start == NULL)
            ret = 1;
        else
        {
            ret = 0;
            fp = P_PktF_start;
            while ((ret == 0) && (fp != NULL))
            {
                ret = ret || FilterPacket(fp, frame);
                fp = fp->FNext;
            }
            if (ret == 0)
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass p-PKT-filter");
            else
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed p-PKT-filter");
            fp = N_PktF_start;
            if (P_PktF_start == NULL)
                ret = 1;
            while ((ret == 1) && (fp != NULL))
            {
                ret = ret && !(FilterPacket(fp, frame));
                fp = fp->FNext;
            }
            if (ret == 0)
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass n-PKT-filter");
            else
                ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed n-PKT-filter");
            if (P_StopF_start == NULL && N_StopF_start == NULL)
                ;
            else
            {
                fp = P_StopF_start;
                while ((Filtering == 1) && (fp != NULL))
                {
                    Filtering = Filtering && !(FilterPacket(fp, frame));
                    fp = fp->FNext;
                }
                if (Filtering == 1)
                    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass p-STOP-filter");
                else
                    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed p-STOP-filter");
                fp = N_StopF_start;
                while ((Filtering == 0) && (fp != NULL))
                {
                    Filtering = Filtering || FilterPacket(fp, frame);
                    fp = fp->FNext;
                }
                if (Filtering == 1)
                    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet didn't pass n-STOP-filter");
                else
                {
                    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Packet passed n-STOP-filter");
                    Filtering  = 2;
                    FilterStat = 2;
                }
            }
        }
    }
    ERR_DEB(ERR_PCA, 1, "DPFPKT=ReceivePkt: Ended."); 
    return ret;
}

static int Init(void)
{
    mib_prf *Prf;

    ERR_DEB(ERR_PCA, 1, "DPFPKT=Init: starting"); 
    if ((Prf = MibAddPrf ("test", strlen("test"))) == NULL)
        return -1;
    if ((MibAddObj (Prf, GetVars, OjiNumber(GetVars), FilterGetVars, FilterGetNxtVars, FilterSetVars, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetFilterStat, OjiNumber(GetFilterStat), FilterGetFilterStat, NULL, NULL, NULL)) < 0)
        return -1;
    ERR_DEB(ERR_PCA, 1, "DPFPKT=Init: successfully added SNMP-variables"); 
    return 0;
}

static int Start(void)
{
    ERR_DEB(ERR_PCA, 1, "DPFPKT=Start: starting");
    Filtering = 0;
    FilterStat = 0;
    return 0;
}

static void Stop(void)
{
    ERR_DEB(ERR_PCA, 1, "DPFPKT=Stop: starting");
    ResetFilters();
    Filtering = 0;
    FilterStat = 3;
}

static void Reset(void)
{
    ERR_DEB(ERR_PCA, 1, "DPFPKT=Reset: starting");
    ResetFilters();
    Filtering = 0;
    FilterStat = 0;
}

/*----------------------------------------------------------------*/
/* <GetVars>.<Var>.<FilterType>.<FilterNr>                        */
/* Length:     1       1             1                            */
/* Offset:     0       1             2                            */
/* GetVars    = 1.3.6.1.4.1.99.99.99                              */
/* Var        = 1..8                                              */
/*      1 = destination address     5 = contents offset           */
/*      2 = source address          6 = device number             */
/*      3 = protocol type           7 = packet length lower limit */
/*      4 = contents                8 = packet length upper limit */
/* FilterType = 1..6                                              */
/*      1 = positive PKT-filter     4 = negative START-filter     */
/*      2 = negative PKT-filter     5 = positive STOP-filter      */
/*      3 = positive START-filter   6 = negative STOP-filter      */
/* FilterNr   = 1..NrOf-?-?-Filters                               */
/*----------------------------------------------------------------*/
              
static unsigned FilterGetVars(pkt_obj *Obj, void *Arg)
{
    FILTER *fp;
    unsigned Var, FilterType, FilterNr, NrOfFilters;
    int i;

    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Starting...");
    Var = Obj->ObjOji[OjiNumber(GetVars)+0];
    if ((Var < 1) || (Var > 8))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Var (=%d) out of range", Var);
        return(PKT_NOSUCHNAME);
    }
    FilterType = Obj->ObjOji[OjiNumber(GetVars)+1];
    if ((FilterType < 1) || (FilterType > 6))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: FilterType (=%d) out of range", FilterType);
        return(PKT_NOSUCHNAME);
    }
    FilterNr = Obj->ObjOji[OjiNumber(GetVars)+2];
    fp = StartOf_FilterList[FilterType-1];
    NrOfFilters = NrOf_Filters[FilterType-1];
    if ((FilterNr < 1) || (FilterNr > NrOfFilters))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Filter (type=%d,nr=%d) out of range", FilterType, FilterNr);
        return(PKT_NOSUCHNAME);
    }
    while ((fp != NULL) && (fp->FilterNo != FilterNr))
        fp = fp->FNext;
    if (fp == NULL)
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Couldn't find filter (type=%d,nr=%d)", FilterType, FilterNr);
        return(PKT_NOSUCHNAME);
    }
    switch (Var)
    {
        case 1: /* Destination address */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting destination address, starting...");
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT FilterGetVars: Destination address was %02x %02x %02x %02x %02x %02x", fp->DestAddr[0], fp->DestAddr[1], fp->DestAddr[2], fp->DestAddr[3], fp->DestAddr[4], fp->DestAddr[5]);
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->DestAddr, hdrinfo->AddrLen);
            Obj->ObjSynLen = hdrinfo->AddrLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting destination address, ended."); 
            break;
        case 2: /* Source address */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting source address, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Source address was %02x %02x %02x %02x %02x %02x", fp->SrcAddr[0], fp->SrcAddr[1], fp->SrcAddr[2], fp->SrcAddr[3], fp->SrcAddr[4], fp->SrcAddr[5]);
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->SrcAddr, hdrinfo->AddrLen);
            Obj->ObjSynLen = hdrinfo->AddrLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting source address, ended."); 
            break;
        case 3: /* Protocol type */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting protocol type, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Protocol type was %02x %02x", fp->PktType[0], fp->PktType[1]); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->PktTypeInt;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting protocol type, ended."); 
            break;
        case 4: /* Contents */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting contents, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Contents were"); 
            for (i=0;i<fp->ContentsLen;i++)
            ERR_DEB(ERR_PCA, 2, "%02x", fp->Contents[i]); 
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->Contents, fp->ContentsLen);
            Obj->ObjSynLen = fp->ContentsLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting contents, ended."); 
            break;
        case 5: /* Contents offset */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting contents-offset, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Contents-offset was %d", fp->ContentsOff); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long) fp->ContentsOff;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting contents-offset, ended."); 
            break;
        case 6: /* Device number */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting device number, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Device number was %d", fp->DevNr); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->DevNr;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting device number, ended."); 
            break;
        case 7: /* Packet length lower limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting packet size lower limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Packet size was %d", fp->PktSizeLL); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->PktSizeLL;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting packet size lower limit, ended."); 
            break;
        case 8: /* Packet length upper limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting packet size upper limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Packet size was %d", fp->PktSizeUL); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->PktSizeUL;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Getting packet size upper limit, ended."); 
            break;
    }
    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Ended."); 
    return(PKT_NOERROR);
}

static unsigned FilterGetNxtVars(pkt_obj *Obj, void *Arg)
{
    FILTER *fp;
    unsigned Var, FilterType, FilterNr, NrOfFilters;
    int i;

    if (Obj->ObjOjiLen <= OjiNumber(GetVars))
    {
        Var = Obj->ObjOji[OjiNumber(GetVars)+0] = 1;
        (Obj->ObjOjiLen)++;
    }
    else
    {
        Var = Obj->ObjOji[OjiNumber(GetVars)+0];
        if ((Var < 1) || (Var > 8))
            return(PKT_NOSUCHNAME);
    }
    if (Obj->ObjOjiLen <= (OjiNumber(GetVars)+1))
    {
        FilterType = Obj->ObjOji[OjiNumber(GetVars)+1] = 1;
        (Obj->ObjOjiLen)++;
    }
    else
    {
        FilterType = Obj->ObjOji[OjiNumber(GetVars)+1];
        if ((FilterType < 1) || (FilterType > 6))
            return(PKT_NOSUCHNAME);
    }
    fp = StartOf_FilterList[FilterType-1];
    NrOfFilters = NrOf_Filters[FilterType-1];
    if (Obj->ObjOjiLen <= (OjiNumber(GetVars)+2))
    {
        FilterNr = Obj->ObjOji[OjiNumber(GetVars)+2] = 1;
        (Obj->ObjOjiLen)++;
    }
    else
    {
        FilterNr = Obj->ObjOji[OjiNumber(GetVars)+2];
        if ((FilterNr < 1) || (FilterNr > NrOfFilters))
            return(PKT_NOSUCHNAME);
    }
    if (++FilterNr > NrOfFilters)
    {
        FilterNr = 1;
        if (++FilterType > 6)
        {
            FilterType = 1;
            if (++Var > 8)
                return(PKT_NOSUCHNAME);
        }
    }
    Obj->ObjOji[OjiNumber(GetVars)+0] = Var;
    Obj->ObjOji[OjiNumber(GetVars)+1] = FilterType;
    Obj->ObjOji[OjiNumber(GetVars)+2] = FilterNr;
    while ((fp != NULL) && (fp->FilterNo != FilterNr))
        fp = fp->FNext;
    if (fp == NULL)
        return(PKT_NOSUCHNAME);
    switch (Var)
    {
        case 1: /* Destination addres */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting destination address, starting...");
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT FilterGetNxtVars: Destination address was %02x %02x %02x %02x %02x %02x", fp->DestAddr[0], fp->DestAddr[1], fp->DestAddr[2], fp->DestAddr[3], fp->DestAddr[4], fp->DestAddr[5]);
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->DestAddr, hdrinfo->AddrLen);
            Obj->ObjSynLen = hdrinfo->AddrLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting destination address, ended."); 
            break;
        case 2: /*Source address */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting source address, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Source address was %02x %02x %02x %02x %02x %02x", fp->SrcAddr[0], fp->SrcAddr[1], fp->SrcAddr[2], fp->SrcAddr[3], fp->SrcAddr[4], fp->SrcAddr[5]);
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->SrcAddr, hdrinfo->AddrLen);
            Obj->ObjSynLen = hdrinfo->AddrLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting source address, ended."); 
            break;
        case 3: /* Protocol type */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting protocol type, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Protocol type was %02x %02x", fp->PktType[0], fp->PktType[1]); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long) fp->PktTypeInt;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting protocol type, ended."); 
            break;
        case 4: /* Contents */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting contents, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Contents were"); 
            for (i=0;i<fp->ContentsLen;i++)
            ERR_DEB(ERR_PCA, 2, "%02x", fp->Contents[i]); 
            Obj->ObjTag = PKT_OCTETSTRING;
            memcpy (Obj->ObjSyn.SynBufChr, fp->Contents, fp->ContentsLen);
            Obj->ObjSynLen = fp->ContentsLen;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting contents, ended."); 
            break;
        case 5: /* Contents offset */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting contents-offset, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Contents-offset was %d", fp->ContentsOff); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long) fp->ContentsOff;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting contents-offset, ended."); 
            break;
        case 6: /* Device number */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting device number, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Device number was %d", fp->DevNr); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->DevNr;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting device number, ended."); 
            break;
        case 7: /* Packet length lower limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting packet size lower limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Packet size was %d", fp->PktSizeLL); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->PktSizeLL;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting packet size lower limit, ended."); 
            break;
        case 8: /* Packet length upper limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting packet size upper limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Packet size was %d", fp->PktSizeUL); 
            Obj->ObjTag = PKT_INTEGER;
            Obj->ObjSyn.SynLngInt = (long)fp->PktSizeUL;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetNxtVars: Getting packet size upper limit, ended."); 
            break;
    }
    return(PKT_NOERROR);
}

static unsigned FilterSetVars(pkt_obj *Obj, void *Arg)
{
    FILTER *fp;
    unsigned Var, FilterType, FilterNr, *NrOfFilters;
    int i;

    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Starting...");
    Var = Obj->ObjOji[OjiNumber(GetVars)+0];
    if ((Var < 1) || (Var > 8))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Var (=%d) out of range", Var);
        return(PKT_NOSUCHNAME);
    }
    FilterType = Obj->ObjOji[OjiNumber(GetVars)+1];
    if ((FilterType < 1) || (FilterType > 6))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: FilterType (=%d) out of range", FilterType);
        return(PKT_NOSUCHNAME);
    }
    FilterNr = Obj->ObjOji[OjiNumber(GetVars)+2];
    fp = StartOf_FilterList[FilterType-1];
    NrOfFilters = &NrOf_Filters[FilterType-1];
    if ((FilterNr < 1) || (FilterNr > *NrOfFilters+1))
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Filter (type=%d,nr=%d) out of range", FilterType, FilterNr);
        return(PKT_NOSUCHNAME);
    }
    if (FilterNr == *NrOfFilters+1)
        fp = AddFilter(FilterType-1);
    while ((fp != NULL) && (fp->FilterNo != FilterNr))
        fp = fp->FNext;
    if (fp == NULL)
    {
        ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Couldn't find filter (type=%d,nr=%d)", FilterType, FilterNr);
        return(PKT_NOSUCHNAME);
    }
    switch (Var)
    {
        case 1: /* Destination address */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting destination address, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_OCTETSTRING) || ((Obj->ObjSynLen != 0) && (Obj->ObjSynLen != hdrinfo->AddrLen)))
                return (PKT_BADVALUE);
            fp->UseDestAddr = (Obj->ObjSynLen != 0);
            memcpy(fp->DestAddr, Obj->ObjSyn.SynBufChr, hdrinfo->AddrLen);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UseDestAddr is %d", fp->UseDestAddr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Destination address is %02x %02x %02x %02x %02x %02x", fp->DestAddr[0], fp->DestAddr[1], fp->DestAddr[2], fp->DestAddr[3], fp->DestAddr[4], fp->DestAddr[5]);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting destination address, ended."); 
            break;
        case 2: /* Source address */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting source address, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_OCTETSTRING) || ((Obj->ObjSynLen != 0) && (Obj->ObjSynLen != hdrinfo->AddrLen)))
                return (PKT_BADVALUE);
            fp->UseSrcAddr = (Obj->ObjSynLen != 0);
            memcpy(fp->SrcAddr, Obj->ObjSyn.SynBufChr, hdrinfo->AddrLen);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UseSrcAddr is %d", fp->UseSrcAddr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetVars: Source address is %02x %02x %02x %02x %02x %02x", fp->SrcAddr[0], fp->SrcAddr[1], fp->SrcAddr[2], fp->SrcAddr[3], fp->SrcAddr[4], fp->SrcAddr[5]);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting source address, ended."); 
            break;
        case 3: /* Protocol type */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting protocol type, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_INTEGER) || (Obj->ObjSyn.SynLngInt > INT_MAX))
                return (PKT_BADVALUE);
            fp->UsePktType = 1;
            fp->PktTypeInt = (unsigned int) Obj->ObjSyn.SynLngInt;
            fp->PktType [0] = (BYTE) (fp->PktTypeInt >> 8);
            fp->PktType [1] = (BYTE) (fp->PktTypeInt & 0xff);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UsePktType is %d", fp->UsePktType);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Protocol type is %02x %02x", fp->PktType[0], fp->PktType[1]); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting protocol type, ended."); 
            break;
        case 4: /* Contents */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting contents, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_OCTETSTRING) || (Obj->ObjSynLen > 1514))
                return (PKT_BADVALUE);
            fp->ContentsLen = Obj->ObjSynLen;
            if ((fp->Contents = IPBufGet(fp->ContentsLen * sizeof(BYTE))) != NULL)
            {
                fp->UseContents = (Obj->ObjSynLen != 0);
                memcpy(fp->Contents, Obj->ObjSyn.SynBufChr, fp->ContentsLen);
                ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Successfully added new contents");
            }
            else
                ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Couldn't allocate new contents");
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UseContents is %d", fp->UseContents);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Contents are"); 
            for (i=0;i<fp->ContentsLen;i++)
                ERR_DEB(ERR_PCA, 2, "%02x", fp->Contents[i]); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting contents, ended."); 
            break;
        case 5: /* Contents offset */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting contents-offset, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_INTEGER) || (Obj->ObjSyn.SynLngInt > 1513))
                return (PKT_BADVALUE);
            fp->ContentsOff = (int) Obj->ObjSyn.SynLngInt;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Contents-offset is %d", fp->ContentsOff); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting contents-offset, ended."); 
            break;
        case 6: /* Device number */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting device number, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if (Obj->ObjTag != PKT_INTEGER)
                return (PKT_BADVALUE);
            fp->UseDevNr = 1;
            fp->DevNr = (int) Obj->ObjSyn.SynLngInt;
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UseDevNr is %d", fp->UseDevNr);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Device number is %d", fp->DevNr); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting device number, ended."); 
            break;
        case 7: /* Packet length lower limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting packet size lower limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_INTEGER) || (Obj->ObjSyn.SynLngInt > 1514))
                return (PKT_BADVALUE);
            fp->UsePktSize = 1;
            fp->PktSizeLL = (unsigned int) Obj->ObjSyn.SynLngInt;
            if (fp->PktSizeLL > fp->PktSizeUL)
                Swap(&fp->PktSizeLL, &fp->PktSizeUL);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UsePktSize is %d", fp->UsePktSize);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Packet size is %d", fp->PktSizeLL); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting packet size lower limit, ended."); 
            break;
        case 8: /* Packet length upper limit */
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting packet size upper limit, starting..."); 
            ERR_DEB(ERR_PCA, 2, "Filter (type=%d,nr=%d)", FilterType, FilterNr);
            if ((Obj->ObjTag != PKT_INTEGER) || (Obj->ObjSyn.SynLngInt > 1514))
                return (PKT_BADVALUE);
            fp->UsePktSize = 1;
            fp->PktSizeUL = (unsigned int) Obj->ObjSyn.SynLngInt;
            if (fp->PktSizeLL > fp->PktSizeUL)
                Swap(&fp->PktSizeLL, &fp->PktSizeUL);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: UsePktSize is %d", fp->UsePktSize);
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Packet size is %d", fp->PktSizeUL); 
            ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Setting packet size upper limit, ended."); 
            break;
    }
    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterSetVars: Ended."); 
    return(PKT_NOERROR);
}

static unsigned FilterGetFilterStat(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetFilterStat: Starting...");
    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetFilterStat: Filter status was %d", FilterStat);
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = FilterStat;
    ERR_DEB(ERR_PCA, 2, "DPFPKT=FilterGetFilterStat: Ended.");
    return(PKT_NOERROR);
}

static int FilterPacket(FILTER *fp, DPBUF *frame)
{
  return(((!fp->UseDestAddr)  || (memcmp(frame->pBuf+hdrinfo->DestOff, fp->DestAddr, hdrinfo->AddrLen) == 0)) &&
         ((!fp->UseSrcAddr)   || (memcmp(frame->pBuf+hdrinfo->SrcOff,  fp->SrcAddr,  hdrinfo->AddrLen) == 0)) &&
         ((!fp->UsePktType)   || (memcmp(frame->pBuf+hdrinfo->TypOff,  fp->PktType,  hdrinfo->TypLen)  == 0)) &&
         ((!fp->UseContents)  || (memcmp(frame->pBuf+fp->ContentsOff, fp->Contents, fp->ContentsLen) == 0)) &&
         ((!fp->UseDevNr)     || (frame->Dev == fp->DevNr)) &&
         ((!fp->UsePktSize)   || (frame->Size >= fp->PktSizeLL && frame->Size <= fp->PktSizeUL)));
}

static FILTER *AddFilter(unsigned FilterType)
{
    FILTER **temp, *fp;
    unsigned int *NrOfFilters;

    temp = &StartOf_FilterList[FilterType];
    NrOfFilters = &NrOf_Filters[FilterType];
    if ((fp = IPBufGet(sizeof(FILTER))) != NULL)
    {
        fp->FilterNo = (*NrOfFilters)+1;
        fp->UseDestAddr = 0;
        fp->DestAddr[0] = 0x00;
        fp->UseSrcAddr = 0;
        fp->SrcAddr[0] = 0x00;
        fp->UsePktType = 0;
        fp->PktTypeInt = 0;
        fp->UseContents = 0;
        fp->ContentsOff = 0;
        fp->ContentsLen = 0;
        fp->Contents = NULL;
        fp->UseDevNr = 0;
        fp->DevNr = 0;
        fp->UsePktSize = 0;
        fp->PktSizeLL = 0;
        fp->PktSizeUL = 1514;
        fp->FNext = NULL;

        while ((*temp) != NULL)
            temp = &((*temp)->FNext);
        (*temp) = fp;
        (*NrOfFilters)++;
        ERR_DEB(ERR_PCA, 2, "DPFPKT=AddFilter: Successfully added new filter(type=%d,nr=%d)", FilterType, *NrOfFilters);
    }
    else
        ERR_DEB(ERR_PCA, 2, "Couldn't allocate new filter");
    return (fp);
}

static void ResetFilters(void)
{
    int i;

    for (i=0; i<6; i++)
    {
        DisposeFilterList(StartOf_FilterList[i]);
        StartOf_FilterList[i] = NULL;
        NrOf_Filters[i] = 0;
    }
}

static void DisposeFilterList(FILTER *fp)
{
    if (fp != NULL)
        DisposeFilterList(fp->FNext);
    IPBufFree(fp->Contents);
    IPBufFree(fp);
}

static void Swap(int *a, int *b)
{
    int t;

     t = *a;
    *a = *b;
    *b =  t;
}
                       
