/******************************************************************************
* Select the colors to be used according to specified device.		      *
*									      *
*					Written by Gershon Elber,  Oct. 1990  *
*******************************************************************************
* History:								      *
*  3 Oct 90 - Version 1.0 by Gershon Elber.				      *
******************************************************************************/

#include "intr_loc.h"
#include "intr_gr.h"

#define MAX_COLORS (INTR_NUM_COLORS * 4)

static IntrBType
#ifdef DJGCC
    Allow256 = TRUE;
#else
    Allow256 = FALSE;
#endif /* DJGCC */
static int
    InterColorTable[MAX_COLORS],
    NumOfIntensities = 1;

/******************************************************************************
* Given the device, reset the color table used by this library. Each color in *
* InterColorType has up to 4 intensity levels that may be used. If the system *
* does not enough colors BLACK color is substituted instead. Only the highest *
* intensity of the four is guarantee to be non BLACK.			      *
* This routine should be called once as part of initialization.		      *
******************************************************************************/
void _IntrAllocColors(void)
{
    int i, j, k, l, Red, Green, Blue, Index, UseColors;

    if (GRScreenMaxColors == 256 && !Allow256)
	UseColors = 16;
    else
	UseColors = GRScreenMaxColors;

    switch (UseColors) {
	case 2:
	    /* Its a monocrome system. Allocate highest intensity as white   */
	    /* and all others as black. This is not going to look good...    */
	case 4:
	    /* Its probably a monocrome system with gray levels.	     */
	    for (i = 0; i < INTR_NUM_COLORS; i++) {
		j = l = (i << 2);
		k = j + 4;
		for ( ; j < k ; j++ )
		    InterColorTable[j] = j == l;
	    }
	    NumOfIntensities = 1;
	    break;
	case 16:
	    /* Its an assumed to be EGA/VGA system and the default EGA/VGA   */
	    /* Pallette is assumed as well as follows:			     */
	    /* {   0,   0,   0 },   Color 0. Black 			     */
	    /* {   0,   0, 170 },   Color 1. Blue 			     */
	    /* {   0, 170,   0 },   Color 2. Green 			     */
	    /* {   0, 170, 170 },   Color 3. Cyan 			     */
	    /* { 170,   0,   0 },   Color 4. Red 			     */
	    /* { 170,   0, 170 },   Color 5. Magenta 			     */
	    /* { 170, 170,   0 },   Color 6. Brown 			     */
	    /* { 170, 170, 170 },   Color 7. LightGray 			     */
	    /* {  85,  85,  85 },   Color 8. DarkGray 			     */
	    /* {  85,  85, 255 },   Color 9. LightBlue 			     */
	    /* {  85, 255,  85 },   Color 10. LightGreen 		     */
	    /* {  85, 255, 255 },   Color 11. LightCyan 		     */
	    /* { 255,  85,  85 },   Color 12. LightRed 			     */
	    /* { 255,  85, 255 },   Color 13. LightMagenta 		     */
	    /* { 255, 255,  85 },   Color 14. Yellow 			     */
	    /* { 255, 255, 255 },   Color 15. White 			     */
	    /* At this time I prefer not to change the pallette to ensure    */
    	    /* these colors are in fact used as user may attempt and use     */
    	    /* color directly.						     */
	    /* Only two intensities are assumed for each color although the  */
    	    /* third is set to be thsame as the second...		     */
    	    for (i = 0; i < INTR_NUM_COLORS; i++) {
		j = (i << 2);
    	    	InterColorTable[j + 3] = 0;
		switch (i) {
		    case INTR_COLOR_WHITE:
			InterColorTable[j] = 15;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 7;
			break;
		    case INTR_COLOR_BLACK:
			InterColorTable[j] = 0;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 0;
			break;
		    case INTR_COLOR_RED:
			InterColorTable[j] = 12;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 4;
			break;
		    case INTR_COLOR_GREEN:
			InterColorTable[j] = 10;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 2;
			break;
		    case INTR_COLOR_BLUE:
			InterColorTable[j] = 9;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 1;
			break;
		    case INTR_COLOR_YELLOW:
			InterColorTable[j] = 14;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 6;
			break;
		    case INTR_COLOR_CYAN:
			InterColorTable[j] = 11;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 3;
			break;
		    case INTR_COLOR_MAGENTA:
			InterColorTable[j] = 13;
			InterColorTable[j + 2] = InterColorTable[j + 1] = 5;
			break;
		}
    	    }

	    /* Init. to WHITE. */
    	    for (Index = 16; Index < GRScreenMaxColors; )
		GRSetRGBPalette(Index++, 255, 255, 255);

	    NumOfIntensities = 3;
	    break;
	case 256:
	    /* Wow. This is going to be fun. We can actually allocate our    */
	    /* own colors, so 4 levels are allocated for each color.	     */
	    Index = 16;		   /* Start allocating colors from Index 16. */
    	    for (i = 0; i < INTR_NUM_COLORS; i++) {
		j = (i << 2);
		Red = Green = Blue = 0;
		switch (i) {
		    case INTR_COLOR_WHITE:
			Red = Green = Blue = 63;
			break;
		    case INTR_COLOR_BLACK:
			break;
		    case INTR_COLOR_RED:
			Red = 63;
			break;
		    case INTR_COLOR_GREEN:
			Green = 63;
			break;
		    case INTR_COLOR_BLUE:
			Blue = 63;
			break;
		    case INTR_COLOR_YELLOW:
			Red = Green = 63;
			break;
		    case INTR_COLOR_CYAN:
			Green = Blue = 63;
			break;
		    case INTR_COLOR_MAGENTA:
			Red = Blue = 63;
			break;
		}

		for (k = 4; k > 0; k--) {
#ifdef DJGCC
		    GRSetRGBPalette(Index, Red * k, Green * k, Blue * k);
#else
		    GRSetRGBPalette(Index,
                    		    (Red * k) >> 2,
                                    (Green * k) >> 2,
                                    (Blue * k) >> 2);
#endif /* DJGCC */
		    InterColorTable[j + 4 - k] = Index++;
		}
    	    }

	    while (Index < 256)
		GRSetRGBPalette(Index++, 255, 255, 255);  /* Init. to WHITE. */

	    NumOfIntensities = 4;
	    break;
        default:
            IntrFatalError("Undefined number of colors in device.");
            break;
    }
}

/******************************************************************************
* Allow 256 colors which enable 4 shades of each color.			      *
* This mode is incompatible with the EGA/VGA 16 colors and colors will look   *
* somewhat different in this mode.					      *
******************************************************************************/
void IntrAllow256Colors(IntrBType Allow256Colors)
{
#ifndef DJGCC
    Allow256 = Allow256Colors;
#endif /* DJGCC */
}

/******************************************************************************
* Select a color with intensity.					      *
******************************************************************************/
int IntrAllocColor(IntrColorType Color, IntrIntensityType Intensity)
{
    int SelectedIntensity, GRColor;

    switch (Intensity) {
	case INTR_INTENSITY_VHIGH:
	    SelectedIntensity = 0;
            break;
	case INTR_INTENSITY_HIGH:
	    SelectedIntensity = 1;
            break;
	case INTR_INTENSITY_LOW:
	    SelectedIntensity = 2;
            break;
	case INTR_INTENSITY_VLOW:
	    SelectedIntensity = 3;
            break;
    }

    GRColor = InterColorTable[(((int) Color) << 2) + SelectedIntensity];
    GRSetColor(GRColor);
    GRSetFillStyle(INTR_FILL_SOLID, GRColor);

    return GRColor;
}
