/* noise.c - calculates noise figure for deviation of Gamma optimum.
 *
 * 1-jun-1993
 * Peter Beyer -PA3AEF-
 * beyer@athena.uto.dec.com
 *
 * Evironment variables:
 *      NOISE_RN - Normalized noise resistance of two port
 *      NOISE_GO - Gamma optimum of two port
 *      NOISE_F  - Noise figure (dB) at Gamma optimum
 *
 *
 * Some brain massage:
 *
 * In general the noise temperature for a lineair two-port is defined as:
 *                                                   2
 *                                          |Gs - Go|
 *            T = Tmin + To * 4 * rn * -------------------------
 *                                               2            2
 *                                      (1 - |Gs| ) * |1 + Go| 
 *
 * Where: Tmin is the two-port minimum noise temperature,
 *        To is room temperature (293K),
 *        rn is normalized noise resistance,
 *        Gs is Gamma source and,
 *        Go is Gamma optimum
 *
 * This will calculate the expected noise figure when the two port is NOT
 * properly matched to Gamma optimum. Gamma optimum is the impedance the 
 * two-port wants to see for minimum noise temperature (Tmin). 
 * 
 * The equation actually has the form of a circle. i.e. for a lot of
 * combinations of Gs the same noise temperature can be reached. These 
 * circles are refered to as Constant Noise Circles and can be used together 
 * with Constant Gain Circles to find the best compromis between unilateral
 * transducer gain and minimum noise figure.
 *
 * The equations for the Noise circles can be found given the parameters
 * Gamma optimum, Tmin and rn. To determine a family of noise figure circles
 * a noise figure parameter Ni is defined:
 *
 *       		Ti - Tmin          |        | 2
 *		Ni = ------------------- * | 1 + Go |
 *                        4 * rn           |        |
 *
 * where Ti is the value of the desired noise temperature circle.
 *
 * Now the center and the radius of the circle in the smithchart can be
 * calculated:
 *                  Go
 *	Center = --------
 *                1 + Ni
 *
 *                  1              2                 2
 * 	Radius = -------- * sqrt(Ni  + Ni * (1 - |Go| ))
 *                1 + Ni
 *
 * Where Radius and Center lie along the Gamma optimum vector.
 * 
 * In these calculations, the losses in the input and output matching 
 * networks are not incorporated. They have to be added to the resulting noise
 * temperature: 
 *                                     Toutput
 *      Ttot = Tinput + Ttwo-port + -------------
 *                                  Gain two-port
 *
 * 
 */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#define _PROTO_
#include "complex.c"

#define To 293                              /* Room temperature */
main()
{
    struct polair Gs, Go, pt1;
    struct complex ct1, ct2, ct3, ct4;
    double rn, F, Fmin, T, Tmin, tmp1, tmp2, tmp3, i;
    char *env;

    printf("NOISE - Calculate NF for deviation of Gamma optimum (c) PA3AEF\n");

    /* Get Gamma optimum */
    if ((env = getenv("NOISE_GO")) != NULL) {
	sscanf(env, "%le,%le", &Go.m, &Go.a);
	printf("Enter Gamma optimum (Mag, Angle) : %0.2f,%0.2f\n", Go.m, Go.a);
    } else {
        printf("Enter Gamma optimum (Mag, Angle) : ");
	scanf("%le,%le",&Go.m, &Go.a);
    }

    /* We want to know the complex conjugate of Gamma optimum as well */
    pol2c(&Go, &ct1);
    ct2.x = 1; ct2.y = 0;
    cadd(&ct2, &ct1, &ct3);
    csub(&ct2, &ct1, &ct4);
    cdiv(&ct3, &ct4, &ct1);
    printf("Complex conjugate for Zo=50: %0.2f j%0.2f ohms\n",
        50 * ct1.x, -50 * ct1.y);
 
    /* Get normalized noise resistance */
    if ((env = getenv("NOISE_RN")) != NULL) {
	sscanf(env, "%le", &rn);
	printf("Enter Rn/50 : %0.2f\n", rn);
    } else {
	printf("Enter Rn/50 : ");
	scanf("%le", &rn);
    }

    /* Get minimum noise figure and convert to temperature */
    if ((env = getenv("NOISE_F")) != NULL) {
	sscanf(env, "%le", &Fmin);
	printf("Enter Fmin (dB) at Gamma optimum : %0.2f\n", Fmin);
    } else {
        printf("Enter Fmin (dB) at Gamma optimum :");
        scanf("%le", &Fmin);
    }
    Tmin = To * (pow(10, Fmin/10) -1);

    /* Get Gamma source */
    printf("Enter Gamma source (Mag, Angle) : ");
    scanf("%le,%le",&Gs.m, &Gs.a);

    /* Now calculate the noise figure for this mismatch */
    polsub(&Gs, &Go, &pt1);
    tmp1 = pt1.m * pt1.m;
    tmp2 = 1 - Gs.m * Gs.m;
    pol2c(&Go, &ct1);
    ct2.x = 1; ct2.y = 0;
    cadd(&ct1, &ct2, &ct3);
    tmp3 = cabs(ct3) * cabs(ct3);
    T = Tmin + 4 * rn * To * (tmp1 / (tmp2 * tmp3));
    F = 10 * log10(1+(T/To));
    printf("Noise Figure: %0.2fdB (%3.0fK)\n", F, T);

    /* Calculate the noise figure circles */
    printf("Constant Noise Circles in Smithchart:\n");
    pol2c(&Go, &ct1);
    ct1.x++;
    tmp1 = cabs(ct1) * cabs(ct1);
    for (i=0.2; i<Fmin+3; i=i+0.2) {
	double Ni, Cfi, Rfi, Ti;

	if (i < Fmin)
	    continue;
	Ti = To * (pow(10, i/10) - 1);
	Ni = ((Ti - Tmin) / (4 * rn * To)) * tmp1;
	Cfi = Go.m / (1 + Ni);
	Rfi = (1 / (1 + Ni)) * sqrt(Ni*Ni + Ni * (1 - Go.m * Go.m));
	printf("\tNF: %0.2fdB  (%3.0fK)  C: %0.3f, %0.1fdeg  R: %0.3f\n",
	    i, Ti, Cfi, Go.a, Rfi,Ni);
    }
}

