/*******************  start of original comments  ********************/
/*
 * Written by Douglas Thomson (1989/1990)
 *
 * This source code is released into the public domain.
 */

/*
 * Name:    dte - Doug's Text Editor program - miscellaneous utilities
 * Purpose: This file contains miscellaneous functions that were required
 *           in more than one of the other files, or were thought to be
 *           likely to be used elsewhere in the future.
 * File:    utils.c
 * Author:  Douglas Thomson
 * System:  this file is intended to be system-independent
 * Date:    October 1, 1989
 */
/*********************  end of original comments  ********************/


/*
 * The utility routines have been EXTENSIVELY rewritten.  Update screens as
 * needed.  Most times, only one line has changed.  Just show changed line
 * in all windows if it is on screen.
 *
 * Support routines for text lines longer than screen width have been added.
 * Currently support lines as long as 1040 characters.
 *
 * In DTE, Doug chose to test whether characters are part of a word.  In TDE,
 * we will test whether characters are not part of a word.  The character
 * set not part of a word will not change as much as the characters that
 * are part of a word.  In most languages, human and computer, the delimiters
 * are much more common across languages than the tokens that make up a word.
 * Thanks to Pierre Jelenc, pcj1@columbia.edu, for recommending looking for
 * delimiters.
 *
 * New editor name:  TDE, the Thomson-Davis Editor.
 * Author:           Frank Davis
 * Date:             June 5, 1991, version 1.0
 * Date:             July 29, 1991, version 1.1
 * Date:             October 5, 1991, version 1.2
 * Date:             January 20, 1992, version 1.3
 * Date:             February 17, 1992, version 1.4
 * Date:             April 1, 1992, version 1.5
 * Date:             June 5, 1992, version 2.0
 * Date:             October 31, 1992, version 2.1
 * Date:             April 1, 1993, version 2.2
 * Date:             June 5, 1993, version 3.0
 * Date:             August 29, 1993, version 3.1
 * Date:             November 13, 1993, version 3.2
 * Date:             June 5, 1994, version 4.0
 * Date:             December 5, 1998, version 5.0 (jmh)
 *
 * This modification of Douglas Thomson's code is released into the
 * public domain, Frank Davis.  You may distribute it freely.
 */

#include "tdestr.h"
#include "common.h"
#include "define.h"
#include "tdefunc.h"


/*
 * Name:    myiswhitespc
 * Purpose: To determine whether or not a character is *NOT* part of a "word".
 * Date:    July 4, 1992
 * Passed:  c: the character to be tested
 * Returns: TRUE if c is in the character set *NOT* part of a word
 * Notes:   The characters in the set not part of a word will not change as
 *           as much as the characters that are part of a word.  In most
 *           languages, human and computer, the delimiters are much more
 *           common than the tokens that make up a word.  For example,
 *           the set of punction characters don't change as much across
 *           languages, human and computer, as the characters that make
 *           up the alphabet, usually.  In other words, the delimiters
 *           are fairly constant across languages.
 */
int  myiswhitespc( int c )
{
   return( c == ' ' || (bj_ispunct( c ) && c != '_') || bj_iscntrl( c ) );
}


/*
 * Name:    check_virtual_col
 * Purpose: ensure integrity of rcol, ccol, and bcol
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *          rcol: real column of cursor
 *          ccol: current or logical column of cursor
 * jmh 980724: update cursor cross
 */
void check_virtual_col( TDE_WIN *window, int rcol, int ccol )
{
register int bcol;
int  start_col;
int  end_col;
file_infos *file;

   file      = window->file_info;
   bcol      = window->bcol;
   start_col = window->start_col;
   end_col   = window->end_col;

   /*
    * is logical column past end of screen?
    */
   if (ccol > end_col) {
      ccol = rcol - bcol + start_col;
      if (ccol > end_col) {
         ccol = end_col;
         bcol = rcol - (ccol - start_col);
         file->dirty = LOCAL;
      }
   }

   /*
    * is logical column behind start of screen?
    */
   if (ccol < start_col) {
      if (bcol >= (start_col - ccol))
         bcol -= (start_col - ccol);
      ccol = start_col;
      file->dirty = LOCAL;
   }

   /*
    * is real column < base column?
    */
   if (rcol < bcol) {
      ccol = rcol + start_col;
      bcol = 0;
      if (ccol > end_col) {
         bcol = rcol;
         ccol = start_col;
      }
      file->dirty = LOCAL;
   }

   /*
    * current column + base column MUST equal real column
    */
   if ((ccol - start_col) + bcol != rcol) {
      if (bcol < 0 || bcol > rcol) {
         bcol = rcol;
         file->dirty = LOCAL;
      }
      ccol = rcol - bcol + start_col;
      if (ccol > end_col) {
         bcol = rcol;
         ccol = start_col;
         file->dirty = LOCAL;
      }
   }

   /*
    * rcol CANNOT be negative
    */
   if (rcol < 0) {
      rcol = bcol = 0;
      ccol = start_col;
      file->dirty = LOCAL;
   }

   if (rcol >= MAX_LINE_LENGTH) {
      rcol = MAX_LINE_LENGTH - 1;
      bcol = rcol - (ccol - start_col);
   }

   assert( rcol >= 0 );
   assert( rcol < MAX_LINE_LENGTH );
   assert( bcol >= 0 );
   assert( bcol < MAX_LINE_LENGTH );
   assert( ccol >= start_col );
   assert( ccol <= end_col );

   window->bcol = bcol;
   window->ccol = ccol;
   window->rcol = rcol;

   /*
    * If the cursor cross is on, set LOCAL update. This is not as
    * efficient as it should be, but is a whole lot easier than
    * modifying every function that could move the cursor left or right.
    */
   if (mode.cursor_cross)
      file->dirty = LOCAL;
}


/*
 * Name:    copy_line
 * Purpose: To copy the cursor line, if necessary, into the current line
 *           buffer, so that changes can be made efficiently.
 * Date:    June 5, 1991
 * Passed:  text_line: line to be copied to line buffer
 *          line: line to display error message
 * Notes:   See un_copy_line, the reverse operation.
 *          DO NOT use the C library string functions on text in
 *           g_status.line_buff, because Null characters are allowed as
 *           normal text in the file.
 */
void copy_line( line_list_ptr ll )
{
register unsigned int len;
text_ptr text_line;

   if (g_status.copied == FALSE  &&  ll->len != EOF) {

      assert( ll != NULL );

      len = ll->len;
      text_line = ll->line;
      g_status.buff_node = ll;

      assert( len < MAX_LINE_LENGTH );

      if (text_line != NULL)
         my_memcpy( g_status.line_buff, text_line, len );

      g_status.line_buff_len = len;
      g_status.copied = TRUE;
   }
}


/*
 * Name:    un_copy_line
 * Purpose: To copy the cursor line, if necessary, from the current line
 *           buffer, shifting the main text to make the right amount of
 *           room.
 * Date:    June 5, 1991
 * Passed:  test_line:  location in file to copy line buffer
 *          window:  pointer to current window
 *          del_trailing:  delete the trailing blanks at eol? TRUE or FALSE
 * Notes:   For some functions, trailing spaces should not be removed when
 *           returning the line buffer to the main text.  The JoinLine function
 *           is a good example.  We need to leave trailing space so when we
 *           join lines - the current line will extend at least up to
 *           the column of the cursor.  We need to leave trailing space
 *           during BOX block operations.
 *          See copy_line, the reverse operation.
 */
int  un_copy_line( line_list_ptr ll, TDE_WIN *window, int del_trailing )
{
text_ptr p;
size_t len;     /* length of line buffer text */
size_t ll_len;  /* length of ll->line */
int  net_change;
int  rc;
char c;
file_infos *file;
TDE_WIN *wp;

   rc = OK;
   if (mode.do_backups == TRUE)
      rc = backup_file( window );

   if (g_status.copied == TRUE  &&  ll->len != EOF) {

      file = window->file_info;

      /*
       * if we are deleting the entire line, don't worry about the
       *  deleting the trailing space, since we're deleting entire line.
       */
      if (g_status.command == DeleteLine)
         del_trailing = FALSE;

      if (del_trailing  &&  mode.trailing  &&  file->crlf != BINARY) {
         len = g_status.line_buff_len;
         for (p=(text_ptr)(g_status.line_buff+len); len > 0; len--, p--) {
            c = *(p - 1);
            if (c != ' '  &&  c != '\t')
               break;
            if (!mode.inflate_tabs && c == '\t')
               break;
         }
         g_status.line_buff_len = len;
         file->dirty = GLOBAL;
         if (window->visible == TRUE)
            show_changed_line( window );
      }
      len = g_status.line_buff_len;
      ll_len =  (ll->line == NULL) ? 0 : ll->len;


      assert( len < MAX_LINE_LENGTH );
      assert( ll_len < MAX_LINE_LENGTH );

      net_change = len - ll_len;

      if (ll_len != len  ||  ll->line == NULL) {
         /*
          * let malloc space for the new line before we free the old line.
          */
         p = my_malloc( len, &rc );
         if (rc == ERROR)
            error( WARNING, window->bottom_line, main4 );

         /*
          * free the space taken up by current line in FAR heap.
          */
         if (rc != ERROR  &&  ll->line != NULL)
            my_free( ll->line );
      } else
         p = ll->line;

      if (rc != ERROR) {
         if (len > 0)
            my_memcpy( p, g_status.line_buff, len );
         ll->line = p;
         ll->len = len;

         if (net_change != 0) {
            for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
               if (wp->file_info == file && wp != window)
                  if (wp->rline > window->rline)
                     wp->bin_offset += net_change;
            }
         }

         file->modified = TRUE;
         show_avail_mem( );
      }
   }
   g_status.copied = FALSE;
   return( rc );
}


/*
 * jmh 980729: This function is currently unused.
 */
#if 0
/*
 * Name:    un_copy_tab_buffer
 * Purpose: To copy the tab buffer line the main text buffer
 * Date:    October 31, 1992
 * Passed:  line_number:  line number to copy line tab out buffer
 *          window:       pointer to current window
 */
int  un_copy_tab_buffer( line_list_ptr ll, TDE_WIN *window )
{
text_ptr p;
int  len;               /* length of current line buffer text */
int  net_change;
int  rc;
file_infos *file;
TDE_WIN *wp;

   rc = OK;
   file = window->file_info;
   /*
    * file has changed.  lets create the back_up if needed
    */
   if (mode.do_backups == TRUE) {
      window->file_info->modified = TRUE;
      rc = backup_file( window );
   }

   len = g_status.tabout_buff_len;

   assert( len >= 0 );
   assert( len < MAX_LINE_LENGTH );
   assert( ll->len >= 0 );
   assert( ll->len < MAX_LINE_LENGTH );

   /*
    * if the FAR heap has run out of space, then only part of the
    *  current line can be moved back into the FAR heap. Warn the user
    *  that some of the current line has been lost.
    */
   p = my_malloc( len, &rc );
   if (rc == ERROR)
      error( WARNING, window->bottom_line, main4 );

   if (rc == OK) {
      net_change = len - ll->len;

      if (ll->line != NULL)
         my_free( ll->line );
      if (len > 0)
         my_memcpy( p, g_status.line_buff, len );
      ll->line = p;
      ll->len  = len;

      if (net_change != 0) {
         for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
            if (wp->file_info == file && wp != window)
               if (wp->rline > window->rline)
                  wp->bin_offset += net_change;
         }
      }

      file->modified = TRUE;
   }
   return( rc );
}
#endif


/*
 * Name:    load_undo_buffer
 * Purpose: To copy the cursor line to the undo buffer.
 * Date:    September 26, 1991
 * Passed:  file:          pointer to file
 *          line_to_undo:  pointer to line in file to save
 * Notes:   save the last mode.undo_max lines in a stack.  when we overflow
 *           the stack, dump the oldest line.
 */
void load_undo_buffer( file_infos *file, text_ptr line_to_undo, int len )
{
int  rc;
text_ptr l;
line_list_ptr temp_ll;

   rc = OK;
   if (file->undo_count >= mode.undo_max) {
      --file->undo_count;
      temp_ll = file->undo_bot->prev;
      temp_ll->prev->next = file->undo_bot;
      file->undo_bot->prev = temp_ll->prev;
      if (temp_ll->line != NULL)
         my_free( temp_ll->line );
   } else
      temp_ll = (line_list_ptr)my_malloc( sizeof(line_list_struc), &rc );

   assert( len >= 0 );
   assert( len < MAX_LINE_LENGTH );

   l = my_malloc( len, &rc );

   if (rc == ERROR) {
      if (l != NULL)
         my_free( l );
      if (temp_ll != NULL)
         my_free( temp_ll );
   } else {
      if (len > 0)
         my_memcpy( l, line_to_undo, len );
      temp_ll->line = l;
      temp_ll->len  = len;
      temp_ll->type = DIRTY;

      temp_ll->prev = NULL;
      temp_ll->next = file->undo_top;
      file->undo_top->prev = temp_ll;
      file->undo_top = temp_ll;

      ++file->undo_count;
   }
}


/*
 * Name:    set_prompt
 * Purpose: To display a prompt, highlighted, at the bottom of the screen.
 * Date:    October 1, 1989
 * Passed:  prompt: prompt to be displayed
 *          line:   line to display prompt
 * jmh 980726: changed eol_clear to text color rather than message.
 */
void set_prompt( char *prompt, int line )
{
register int prompt_col;

   /*
    * work out where the answer should go
    */
   prompt_col = strlen( prompt );

   /* jmh 981002: truncate the message if it's too long, instead of failing an
    *             assertion.
   assert( prompt_col <= g_display.ncols );
    */
   if (prompt_col > g_display.ncols) prompt[prompt_col = g_display.ncols-1] = 0;

   /*
    * output the prompt
    */
   s_output( prompt, line, 0, g_display.message_color );
   eol_clear( prompt_col, line, g_display.text_color );

   /*
    * put cursor at end of prompt
    */
   xygoto( prompt_col, line );
}


/*
 * Name:    show_eof
 * Purpose: display eof message
 * Date:    September 16, 1991
 * Notes:   line:  ususally, line to display is "<=== eof ===>"
 *
 * jmh 980702: also display the top of file message
 * jmh 980703: cover the entire line
 */
void show_eof( TDE_WIN *window )
{
register int color;
char temp[MAX_COLS+2];
char *msg;
int  len;
int  j;
int  spaces;

   /*
    * 15 for the minimum window width
    * 2 each side for character and space
    */
   assert( strlen( mode.tof ) <= 15-4 );
   assert( strlen( mode.eof ) <= 15-4 );

   color = window->end_col + 1 - window->start_col;
   memset( temp, EOF_CHAR, color );
   temp[color] = '\0';
   spaces = (color > 30) ? 3 : (color > 20) ? 2 : 1;
   msg = (window->ll->prev == NULL) ? mode.tof : mode.eof;
   len = strlen( msg );
   color = (color - len) / 2;
   memcpy( temp + color, msg, len );
   for (j = 0; j < spaces; ++j)
      temp[color-1-j] = temp[color+len+j] = ' ';

   color = g_display.eof_color;
   window_eol_clear( window, color );
   s_output( temp, window->cline, window->start_col, color );
}


/*
 * Name:    display_current_window
 * Purpose: display text in current window
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   use a temporary window structure, "w", to do the dirty work.
 */
void display_current_window( TDE_WIN *window )
{
int  count;     /* number of lines updated so far */
int  number;    /* number of lines visible in window */
register int i; /* register variable */
TDE_WIN w;      /* scratch window structure */
int  curl;      /* current line on screen, window->cline */
int  eof;

   /*
    * jmh 980801: reset the current line count, since it wasn't required.
    */
   window->cur_count = 0;

   /*
    * initialize the scratch variables
    */
   dup_window_info( &w, window );
   number   = w.bottom_line - ((w.top_line + w.ruler) - 1);
   count    = w.cline - (w.top_line + w.ruler);
   w.cline -= count;
   w.rline -= count;
   for (eof=count; eof > 0; eof--)
      w.ll = w.ll->prev;

   /*
    * start at the top of the window and display a window full of text
    */
   eof =  w.ll->prev != NULL;
   curl = window->cline;
   for (i=number; i>0; i--) {
      if (w.ll->len != EOF) {
         /*
          * if this is window->cline, do not show the line because we
          *  show the curl at the end of this function.  don't show it twice
          */
         if (w.cline != curl)
            update_line( &w );
         w.ll = w.ll->next;
      } else if (eof < 2) {
         show_eof( &w );
         if (++eof == 1)
            w.ll = w.ll->next;
      } else
         window_eol_clear( &w, g_display.text_color );
      ++w.cline;
      ++w.rline;
   }
   show_asterisk( window );
   show_curl_line( window );
}


/*
 * Name:    redraw_screen
 * Purpose: display all visible windows, modes, and headers
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
int  redraw_screen( TDE_WIN *window )
{
register TDE_WIN *above;        /* window above current */
register TDE_WIN *below;        /* window below current */

#if defined( __UNIX__ )
   touchwin( curscr );
#endif

   /*cls( );*/                  /* removed by jmh 980528 */

   /*
    * display the current window
    */
   redraw_current_window( window );

   /*
    * now update all the other windows
    */
   above = below = window;
   while (above->prev || below->next) {
      if (above->prev) {
         above = above->prev;
         redraw_current_window( above );
      }
      if (below->next) {
         below = below->next;
         redraw_current_window( below );
      }
   }
   window->file_info->dirty = FALSE;
   show_modes( );

#if defined( __UNIX__ )
   refresh( );
#endif

   return( OK );
}


/*
 * Name:    redraw_current_window
 * Purpose: redraw all info in window
 * Date:    July 13, 1991
 * Passed:  window:  pointer to current window
 */
void redraw_current_window( TDE_WIN *window )
{

   /*
    * display the current window
    */
   if (window->visible) {
      display_current_window( window );
      show_window_header( window );
      show_ruler( window );
      show_ruler_pointer( window );
      if (window->vertical)
         show_vertical_separator( window );
   }
}


/*
 * Name:    show_changed_line
 * Purpose: Only one line was changed in file, just show it
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 *
 * jmh 980728: Even though only one line has been changed, that change may
 *              affect other lines' colors in the syntax highlighting.
 * jmh 981125: Corrected bugs with the above.
 */
void show_changed_line( TDE_WIN *window )
{
TDE_WIN *above;                 /* window above current */
TDE_WIN *below;                 /* window below current */
TDE_WIN w;                      /* scratch window structure */
long changed_line;              /* line number in file that was changed */
long top_line, bottom_line;     /* top and bottom line in file on screen */
int  line_on_screen;            /* is changed line on screen? */
file_infos *file;               /* file pointer */
long count;                     /* number of additional lines to display */
long rline;                     /* original rline in other window */
int  i;

   file  = window->file_info;
   count = syntax_check_lines( window->ll, file->syntax );

   if (window->visible) {
      window->cur_count = window->bottom_line - window->cline;
      if (count < window->cur_count)
         window->cur_count = (int)count;
      if (file->dirty == LOCAL || file->dirty == GLOBAL)
         show_curl_line( window );
   }

   changed_line = window->rline;

   /*
    * now update the line in all other windows
    */
   if (file->dirty != LOCAL) {
      above = below = window;
      while (above->prev || below->next) {
         if (above->prev) {
            above = above->prev;
            dup_window_info( &w, above );
         } else if (below->next) {
            below = below->next;
            dup_window_info( &w, below );
         }

         /*
          * is this window the changed file and is it visible?
          */
         if (w.file_info == file && w.visible) {

            /*
             * calculate file lines at top and bottom of screen.
             * the changed line may not be the curl in other windows.
             */
            line_on_screen = FALSE;
            rline = 0;
            top_line = w.rline - (w.cline - (w.top_line + w.ruler));
            bottom_line = w.rline + (w.bottom_line - w.cline);
            if (changed_line == w.rline)
               line_on_screen = CURLINE;
            else if (changed_line < w.rline &&
                     changed_line + count >= top_line) {
               line_on_screen = NOTCURLINE;
               rline = w.rline;
               while (w.rline > changed_line && w.rline > top_line) {
                  w.ll = w.ll->prev;
                  --w.rline;
                  --w.cline;
               }
               count -= w.rline - changed_line;
            } else if (changed_line > w.rline && changed_line <= bottom_line) {
               line_on_screen = NOTCURLINE;
               while (w.rline < changed_line) {
                  w.ll = w.ll->next;
                  ++w.rline;
                  ++w.cline;
               }
            }

            /*
             * display the changed line if on screen
             */
            if (line_on_screen) {
               if (line_on_screen == NOTCURLINE)
                  update_line( &w );
               else
                  show_curl_line( &w );

               i = w.bottom_line - w.cline;
               if (count < i)
                  i = (int)count;
               while (i > 0) {
                  w.ll = w.ll->next;
                  ++w.rline;
                  ++w.cline;
                  if (w.rline == rline)
                     show_curl_line( &w );
                  else
                     update_line( &w );
                  --i;
               }
            }
         }
      }
   }
   file->dirty = FALSE;
}


/*
 * Name:    show_curl_line
 * Purpose: show current line in curl color
 * Date:    January 16, 1992
 * Passed:  window:  pointer to current window
 * jmh 980724: use a global variable to indicate the current line
 * jmh 980801: display the lines changed by syntax highlighting (caused by
 *              the NOT_LOCAL update in show_changed_line).
 */
void show_curl_line( TDE_WIN *window )
{
TDE_WIN w;
int cur_count;

   if (window->visible  &&  g_status.screen_display) {
      g_status.cur_line = TRUE;
      update_line( window );
      g_status.cur_line = FALSE;
      if (window->cur_count > 0) {
         dup_window_info( &w, window );
         for (cur_count = w.cur_count; cur_count > 0; --cur_count) {
            w.ll = w.ll->next;
            ++w.rline;
            ++w.cline;
            update_line( &w );
         }
         window->cur_count = 0;
      }
   }
}


/*
 * Name:    dup_window_info
 * Purpose: Copy window info from one window pointer to another
 * Date:    June 5, 1991
 * Passed:  dw: destination window
 *          sw: source window
 */
void dup_window_info( TDE_WIN *dw, TDE_WIN *sw )
{
   memcpy( dw, sw, sizeof( TDE_WIN ) );
}


/*
 * Name:    adjust_windows_cursor
 * Purpose: A change has been made, make sure pointers are not ahead of
 *           or behind file.
 * Date:    June 5, 1991
 * Passed:  window:       pointer to current window
 *          line_change:  number of lines add to or subtracted from file
 * Notes:   If a file has been truncated in one window and there is another
 *           window open to the same file and its current line is near the
 *           end, the current line is reset to the last line of the file.
 */
void adjust_windows_cursor( TDE_WIN *window, long line_change )
{
register TDE_WIN *next;
long i;
file_infos *file;
MARKER *marker;
long length;

   file = window->file_info;
   length = file->length;
   next = g_status.window_list;
   while (next != NULL) {
      if (next != window) {
         if (next->file_info == file) {
            if (next->rline > length + 1) {
               next->rline = length + 1;
               next->ll    = file->line_list_end;
               file->dirty = NOT_LOCAL;
            } else if (next->rline < 0) {
               next->rline = 0;
               next->cline = next->top_line + next->ruler;
               next->ll    = file->line_list;
               next->bin_offset = 0;
               file->dirty = NOT_LOCAL;
            }
            if (next->rline > window->rline  &&  line_change) {
               file->dirty = NOT_LOCAL;
               if (line_change < 0) {
                  for (i=line_change; i < 0 && next->ll->next != NULL; i++) {
                     next->bin_offset += next->ll->len;
                     next->ll = next->ll->next;
                  }
               } else if (line_change > 0) {
                  for (i=line_change; i > 0 && next->ll->prev != NULL; i--) {
                     next->ll = next->ll->prev;
                     next->bin_offset -= next->ll->len;
                  }
               }
            }
            if (next->rline < (next->cline - (next->top_line+next->ruler))) {
               next->cline = (int)next->rline+(next->top_line+next->ruler);
               file->dirty = NOT_LOCAL;
            }
         }
      }
      next = next->next;
   }

   /*
    * now adjust any markers.
    */
   for (i = 0; i < NO_MARKERS; i++) {
      marker = &file->marker[ (int) i ];
      if (marker->rline > window->rline) {
         marker->rline += line_change;
         if (marker->rline < 0L)
            marker->rline = 0L;
         else if (marker->rline > length)
            marker->rline = length;
      }
   }

   /*
    * adjust the break point.
    */
   if (file->break_point > window->rline) {
      file->break_point += line_change;
      if (file->break_point < 1L || file->break_point >= length)
         file->break_point = 0;
   }
}


/*
 * Name:    first_non_blank
 * Purpose: To find the column of the first non-blank character
 * Date:    June 5, 1991
 * Passed:  s:    the string to search
 *          len:  length of string
 * Returns: the first non-blank column
 */
int  first_non_blank( text_ptr s, int len )
{
register int count = 0;

   if (s != NULL) {
      if (mode.inflate_tabs) {
         for (; len > 0 && (*s == ' ' || *s == '\t'); s++, len--) {
            if (*s != '\t')
               ++count;
            else
               count += mode.ptab_size - (count % mode.ptab_size);
         }
      } else {
         while (len-- > 0  &&  *s++ == ' ')
           ++count;
      }
   }
   return( count );
}


/*
 * Name:    find_end
 * Purpose: To find the last character in a line
 * Date:    October 31, 1992
 * Passed:  s:    the string to search
 *          len:  length of string
 * Returns: the first non-blank column
 */
int  find_end( text_ptr s, int len )
{
register int count = 0;

   if (s != NULL) {
      if (mode.inflate_tabs) {
         for (;len > 0; s++, len--) {
            if (*s == '\t')
               count += mode.ptab_size - (count % mode.ptab_size);
            else
               ++count;
         }
      } else
         count = len;
   }
   return( count );
}


/*
 * Name:    is_line_blank
 * Purpose: is line empty or does it only contain spaces?
 * Date:    November 28, 1991
 * Passed:  s:    the string to search
 *          len:  length of string
 * Returns: TRUE if line is blank or FALSE if something is in line
 */
int is_line_blank( text_ptr s, int len )
{
   if (s != NULL) {
      if (mode.inflate_tabs) {
        while (len > 0  &&  (*s == ' ' || *s == '\t')) {
           ++s;
           --len;
        }
      } else {
         while (len > 0  &&  *s == ' ') {
            ++s;
            --len;
         }
      }
   } else
      len = 0;
   return( len == 0 );
}


/*
 * Name:    page_up
 * Purpose: To move the cursor one page up the window
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   The cursor line is moved back the required number of lines
 *           towards the start of the file.
 *          If the start of the file is reached, then movement stops.
 */
int  page_up( TDE_WIN *window )
{
int  i;                 /* count of lines scanned */
int  rc = OK;           /* default return code */
register TDE_WIN *win;  /* put window pointer in a register */
long number;
long len;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->rline != (win->cline - (win->top_line + win->ruler))) {
      i = win->cline - (win->top_line + win->ruler);
      number = win->rline;
      if (( win->rline - i) < win->page)
         win->rline = (win->cline-(win->top_line + win->ruler)) + win->page;
      win->rline -= win->page;
      for (len =0, i=(int)(number - win->rline); i>0; i--)
         if (win->ll->prev != NULL) {
            win->ll = win->ll->prev;
            len -= win->ll->len;
         }
      win->file_info->dirty = LOCAL;
      win->bin_offset += len;
      if (win->rline == 0)
         win->bin_offset += EOF;
   } else
      rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    page_down
 * Purpose: To move the cursor one page down the window
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   The cursor line is moved forwards the required number of lines
 *           towards the end of the file.
 *          If the end of the file is reached, then movement stops.
 */
int  page_down( TDE_WIN *window )
{
int  i;                 /* count of lines scanned so far */
int  k;
int  rc = OK;
long len;
register TDE_WIN *win;  /* put window pointer in a register */
line_list_ptr p;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   p = win->ll;
   k = win->cline - (win->top_line + win->ruler);
   for (len=i=0; i < win->page && p->next != NULL; i++, k++, p=p->next)
      if (p->len != EOF)
         len += p->len;
   if (k >= win->page) {
      win->rline += i;
      win->cline += i - win->page;
      win->bin_offset += len;
      win->ll = p;
      win->file_info->dirty = LOCAL;
   } else
      rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    scroll_down
 * Purpose: scroll window down one line
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If there is a line to scroll_down, make the window LOCAL dirty.
 *          We have to redraw the screen anyway, so don't update here.
 */
int  scroll_down( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;   /* put window pointer in a register */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->cline == win->top_line + win->ruler) {
      if (win->ll->next != NULL) {
         if (win->rline++ != 0)
            win->bin_offset += win->ll->len;
         win->ll = win->ll->next;
         win->file_info->dirty = LOCAL;
      } else
         rc = ERROR;
   } else {
      --win->cline;
      win->file_info->dirty = LOCAL;
   }
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    scroll_up
 * Purpose: To scroll the window up one line
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If this is the first page, then update screen here.  Else, make
 *           the window LOCAL dirty because we have to redraw screen.
 */
int  scroll_up( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;   /* put window pointer in a register */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->rline > 0) {
      if (win->rline == (win->cline - (win->top_line + win->ruler))) {
         if (!mode.sync)
            update_line( win );
         win->ll = win->ll->prev;
         --win->cline;
         if (--win->rline != 0)
            win->bin_offset -= win->ll->len;
         if (!mode.sync)
            show_curl_line( win );
      } else {
         if (win->cline == win->bottom_line) {
            win->ll = win->ll->prev;
            if (--win->rline != 0)
               win->bin_offset -= win->ll->len;
            win->file_info->dirty = LOCAL;
         } else {
            ++win->cline;
            win->file_info->dirty = LOCAL;
         }
      }
   } else
     rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    pan_up
 * Purpose: To leave cursor on same logical line and scroll text up
 * Date:    September 1, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If cursor is on first page then do not scroll.
 */
int  pan_up( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;   /* put window pointer in a register */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );

   /*
    * see if cursor is on the first page. if it's not then pan_up.
    */
   if (win->rline != (win->cline - (win->top_line + win->ruler))) {
      if (win->rline > 0) {
         win->ll = win->ll->prev;
         if (--win->rline != 0)
            win->bin_offset -= win->ll->len;
         win->file_info->dirty = LOCAL;
      }
   } else
      rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    pan_down
 * Purpose: To leave cursor on same logical line and scroll text down
 * Date:    September 1, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If cursor is on last line in file then do not scroll.
 */
int  pan_down( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;   /* put window pointer in a register */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->ll->next != NULL) {
      if (win->rline++ != 0)
         win->bin_offset += win->ll->len;
      win->ll = win->ll->next;
      win->file_info->dirty = LOCAL;
   } else
      rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    show_window_header
 * Purpose: show file stuff in window header
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Clear line and display header in a lite bar
 */
void show_window_header( TDE_WIN *window )
{
register TDE_WIN *win;          /* put window pointer in a register */
int  len;
char temp[MAX_COLS+2];

   win = window;
   len = win->vertical ? win->end_col + 1 - win->start_col : win->end_col;

   assert( len >= 0 );
   assert( len <= g_display.ncols );
   memset( temp, ' ', len );
   temp[len] = '\0';
   s_output( temp, win->top_line-1, win->start_col, g_display.head_color );
   show_window_number_letter( win );
   show_window_fname( win );
   show_crlf_mode( win );
   show_size( win );
   show_line_col( win );
}


/*
 * Name:    show_window_number_letter
 * Purpose: show file number and letter of window in lite bar
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_window_number_letter( TDE_WIN *window )
{
int  col;
char temp[10];
register TDE_WIN *win;   /* put window pointer in a register */

   win = window;
   col = win->start_col;
   s_output( "   ", win->top_line-1, col, g_display.head_color );
   my_ltoa( win->file_info->file_no, temp, 10 );
   s_output( temp, win->top_line-1, strlen( temp ) > 1 ? col : col+1,
             g_display.head_color );
   c_output( win->letter, col+2, win->top_line-1, g_display.head_color );
}


/*
 * Name:     show_window_fname
 * Purpose:  show file name in window header.
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:   window:  pointer to current window
 * Notes:    Clear name field and display name in a lite bar
 *
 * jmh 980503: assume if the name is "" then it's a pipe and display "(pipe)"
 *     980504: if the current directory matches the start of the name, display
 *             only the remainder (ie. if cd is "d:/utils/tde/" and file is
 *             "d:/utils/tde/utils.c" only display "utils.c").
 *     980511: if the current has only one extra directory, display the parent
 *             shortcut (ie. if cd is "d:/utils/tde/config/" and file is
 *             "d:/utils/tde/utils.c" display "../utils.c").
 *     980614: corrected parent shortcut bug - test for slash in the filename.
 */
void show_window_fname( TDE_WIN *window )
{
register fattr_t fattr;
char *p;
register TDE_WIN *win;          /* put window pointer in a register */
int  col;
int  len;
char temp[PATH_MAX+2];
char cwd[PATH_MAX];

   win = window;
   col = win->start_col;
   len = win->vertical ? 11 : FNAME_LENGTH;

   assert( len >= 0 );
   assert( len <= g_display.ncols );

   memset( temp, ' ', len );
   temp[len] = '\0';
   s_output( temp, win->top_line-1, col+5, g_display.head_color );

   /*
    * jmh - I don't think this assertion is necessary, especially since the
    * name can legitimately be greater and the length is tested below.
    *
   assert( strlen( win->file_info->file_name ) < (size_t)g_display.ncols );
    */

   p = win->file_info->file_name;
   if (*p == '\0')
      strcpy( temp, "(pipe)" );
   else {
      *temp = len = 0;
      if (get_current_directory( cwd, 0 ) == OK)
      {
         len = strlen( cwd );
         if (strncmp( cwd, p, len ) != 0) {
            /*
             * Go back a directory. If that matches, use the parent shortcut.
             */
            for (len -= 2; len > 0 && cwd[len] != '/'; --len) ;
            if (len && strncmp( cwd, p, len ) == 0 && p[len] == '/')
               strcpy( temp, ".." );
            else
               len = 0;
         }
      }
      strcat( temp, p + len );
   }
   p = temp;
   len = strlen( temp );
   if (win->vertical) {
      for (p = temp + len; *(p-1) != '/' && p>temp; )
         --p;
      if (strlen( p ) > 11)
         *(p + 11) = '\0';
   } else {
      if (len > FNAME_LENGTH)
         p = temp + len - (FNAME_LENGTH-1);
   }
   s_output( p, win->top_line-1, col+5, g_display.head_color );
   if (!win->vertical) {
      fattr = win->file_info->file_attrib;
      p = temp;
#if defined( __UNIX__ )
      *p++ = (char)(fattr & S_IRUSR  ? L_UNIX_READ    : '-');
      *p++ = (char)(fattr & S_IWUSR  ? L_UNIX_WRITE   : '-');
      *p++ = (char)(fattr & S_IXUSR  ? L_UNIX_EXECUTE : '-');
#else
      *p++ = (char)(fattr & ARCHIVE   ? L_DOS_ARCHIVE   : '-');
      *p++ = (char)(fattr & SYSTEM    ? L_DOS_SYSTEM    : '-');
      *p++ = (char)(fattr & HIDDEN    ? L_DOS_HIDDEN    : '-');
      *p++ = (char)(fattr & READ_ONLY ? L_DOS_READ_ONLY : '-');
#endif
      *p   = '\0';
      s_output( temp, win->top_line-1, col+51, g_display.head_color );
   }
}



/*
 * Name:     show_crlf_mode
 * Purpose:  display state of crlf flag
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 */
void show_crlf_mode( TDE_WIN *window )
{
char temp[MAX_COLS+2];

   if (!window->vertical) {
      switch (window->file_info->crlf) {
         case LF :
            strcpy( temp, mode_lf );
            break;
         case CRLF :
            strcpy( temp, mode_crlf );
            break;
         case BINARY :
            strcpy( temp, mode_bin );
            break;
         default :
            assert( FALSE );
      }
      s_output( temp, window->top_line-1, window->start_col+56,
                g_display.head_color );
   }
}


/*
 * Name:    show_size
 * Purpose: show number of lines in file
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
void show_size( TDE_WIN *window )
{
char csize[20];

   if (!window->vertical  &&  window->file_info->crlf != BINARY) {
      s_output( "       ", window->top_line-1, 61, g_display.head_color );
      my_ltoa( window->file_info->length, csize, 10 );
      s_output( csize, window->top_line-1, 61, g_display.head_color );
   }
}


/*
 * Name:    quit
 * Purpose: To close the current window without saving the current file.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the file has been modified but not saved, then the user is
 *           given a second chance before the changes are discarded.
 *          Note that this is only necessary if this is the last window
 *           that refers to the file.  If another window still refers to
 *           the file, then the check can be left until later.
 *          jmh - if output has been redirected then quitting will send
 *                 nothing - file (function file_file) should be used instead.
 *                 This is so windows can be closed without redirection taking
 *                 place multiple times.
 */
int  quit( TDE_WIN *window )
{
register file_infos *file;
TDE_WIN *wp;
int  count = 0;
int  rc = OK;

   assert( window != NULL );

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   file = window->file_info;
   for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
      if (wp->file_info == file && wp->visible)
         ++count;
   }
   if (file->modified && count == 1) {
      /*
       * abandon changes?
       */
      if (get_yn( utils12, window->bottom_line, R_PROMPT | R_ABORT ) != A_YES)
         rc = ERROR;
   }

   /*
    * remove window, allocate screen lines to other windows etc
    */
   if (rc == OK)
      finish( window );
   return( OK );
}


/*
 * Name:     quit_all
 * Purpose:  To abandon all windows and leave the editor
 * Author:   Jason Hood
 * Date:     29 December, 1996
 * Modified: 27 August, 1997
 * Passed:   arg_filler to satisfy the prototype
 * Notes:    if any window has been modified, prompt before leaving
 *
 * Change:   added disabled message during output redirection
 */
int  quit_all( TDE_WIN *arg_filler )
{
int  prompt_line;
register TDE_WIN *wp;
int  mod = FALSE;
int  rc  = A_YES;

   prompt_line = g_display.nlines;
   if (g_status.input_redir) {
      error( WARNING, prompt_line, main22 );
      return( ERROR );
   }

   for (wp=g_status.window_list; wp != NULL; wp=wp->next) {
     if (wp->file_info->modified) {
       mod = TRUE;
       break;
     }
   }
   if (mod) {
      /*
       * abandon all files?
       */
      rc = get_yn( utils12a, prompt_line, R_PROMPT | R_ABORT );
   }
   if (rc == A_YES)
      g_status.stop = TRUE;
   return( OK );
}


/*
 * Name:    move_up
 * Purpose: To move the cursor up one line
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the cursor is at the top of the window, then the file must
 *           be scrolled down.
 */
int  move_up( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;   /* put window pointer in a register */
int  at_top = FALSE;    /* is cline at top of screen? */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );

   /*
    * if no previous line, give up
    */
   if (win->rline > 0) {
      if (win->cline == win->top_line + win->ruler) {
         win->file_info->dirty = LOCAL;
         at_top = TRUE;
      }
      if (!at_top)
         update_line( win );
      win->ll = win->ll->prev;
      if (--win->rline != 0)    /* ALWAYS decrement line counter */
         win->bin_offset -= win->ll->len;
      if (!at_top) {
         --win->cline;          /* we aren't at top of screen - so move up */
         show_curl_line( win );
      }
   } else
      rc = ERROR;
   cursor_sync( win );
   return( rc );
}


/*
 * Name:    move_down
 * Purpose: To move the cursor down one line
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the cursor is at the bottom of the window, then the file must
 *           be scrolled up.   If the cursor is at the bottom of the file,
 *           then scroll line up until it is at top of screen.
 */
int  move_down( TDE_WIN *window )
{
int  rc;

   rc = prepare_move_down( window );
   cursor_sync( window );
   return( rc );
}


/*
 * Name:    prepare_move_down
 * Purpose: Do the stuff needed to move the cursor down one line.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Put all the stuff needed to move the cursor down one line in
 *           one function, so several functions can use the guts of the
 *           algorithm.
 */
int  prepare_move_down( TDE_WIN *window )
{
int  rc = OK;
register TDE_WIN *win;  /* put window pointer in a register */
int  at_bottom = FALSE; /* is cline at bottom of screen */

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->cline == win->bottom_line) {
      win->file_info->dirty = LOCAL;
      at_bottom = TRUE;
   }
   if (!at_bottom)
      update_line( win );
   if (win->ll->next != NULL) {
      if (win->rline++ != 0)    /* ALWAYS increment line counter */
         win->bin_offset += win->ll->len;
      win->ll = win->ll->next;
      if (!at_bottom) {
         ++win->cline;          /* if not at bottom of screen move down */
         show_curl_line( win );
      }
   } else if (win->cline > win->top_line + win->ruler) {
      --win->cline;
      win->file_info->dirty = LOCAL;
      rc = ERROR;
   } else
      rc = ERROR;
   return( rc );
}


/*
 * Name:    move_left
 * Purpose: To move the cursor left one character
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the cursor is already at the left of the screen, then
 *           scroll horizontally if we're not at beginning of line.
 * jmh 980724: update cursor cross
 */
int  move_left( TDE_WIN *window )
{
int  new_ruler = FALSE;

   if (mode.cursor_cross)
      window->file_info->dirty = LOCAL;

   if (window->ccol > window->start_col) {
      show_ruler_char( window );
      --window->ccol;
      --window->rcol;
   } else if (window->ccol == window->start_col && window->rcol > 0) {
      --window->rcol;
      --window->bcol;
      window->file_info->dirty = LOCAL;
      new_ruler = TRUE;
   }
   cursor_sync( window );
   if (new_ruler) {
      make_ruler( window );
      show_ruler( window );
   }
   return( OK );
}


/*
 * Name:    move_right
 * Purpose: To move the cursor right one character
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   If the cursor is already at the right of the screen (logical
 *          column 80) then scroll horizontally right.
 * jmh 980724: update cursor cross
 */
int  move_right( TDE_WIN *window )
{
int  new_ruler = FALSE;

   if (mode.cursor_cross)
      window->file_info->dirty = LOCAL;

   if (window->rcol < g_display.line_length - 1) {
      if (window->ccol < window->end_col) {
         show_ruler_char( window );
         ++window->ccol;
         ++window->rcol;
      } else if (window->ccol == window->end_col) {
         ++window->rcol;
         ++window->bcol;
         window->file_info->dirty = LOCAL;
         new_ruler = TRUE;
      }
   }
   cursor_sync( window );
   if (new_ruler) {
      make_ruler( window );
      show_ruler( window );
   }
   return( OK );
}


/*
 * Name:    pan_left
 * Purpose: To pan the screen left one character
 * Date:    January 5, 1992
 * Passed:  window:  pointer to current window
 */
int  pan_left( TDE_WIN *window )
{
   if (window->bcol > 0 ) {
      --window->bcol;
      --window->rcol;
      window->file_info->dirty = LOCAL;
      make_ruler( window );
      show_ruler( window );
   }
   cursor_sync( window );
   return( OK );
}


/*
 * Name:    pan_right
 * Purpose: To pan the screen right one character
 * Date:    January 5, 1992
 * Passed:  window:  pointer to current window
 */
int  pan_right( TDE_WIN *window )
{
   if (window->rcol < g_display.line_length - 1) {
      ++window->rcol;
      ++window->bcol;
      window->file_info->dirty = LOCAL;
      make_ruler( window );
      show_ruler( window );
   }
   cursor_sync( window );
   return( OK );
}


/*
 * These two functions are used to invert the sense of comparison for the
 * WordEndLeft functions. jmh 980521.
 */
static int not_myiswhitespc( int c ) {
   return( !myiswhitespc( c ) );
}

static int not_bj_isspc( int c ) {
   return( !bj_isspc( c ) );
}


/*
 * Name:    word_left
 * Purpose: To move the cursor left one word
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Words are considered strings of letters, numbers and underscores,
 *          which must be separated by other characters.
 */
int  word_left( TDE_WIN *window )
{
text_ptr p;             /* text pointer */
int  len;               /* length of current line */
int  rc;
int  tabs;
int  rcol;
register int r1;
long rline;
line_list_ptr ll;
TDE_WIN w;
int  (*space)( int );   /* Function to determine what a word is */
int  (*spc)( int );     /* Function to determine what is a space */
int  next;              /* Go forward a character? */

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   rc = OK;
   dup_window_info( &w, window );
   rline = window->rline;
   rcol  = window->rcol;
   ll = window->ll;
   if (ll->len != EOF) {
      p   = ll->line;
      len = ll->len;

      tabs = mode.inflate_tabs;
      r1 =  tabs ? entab_adjust_rcol( p, len, rcol ) : rcol;

      switch (g_status.command) {
         default:
         case WordLeft:        space = myiswhitespc;     break;
         case WordEndLeft:     space = not_myiswhitespc; break;
         case FullWordLeft:    space = bj_isspc;         break;
         case FullWordEndLeft: space = not_bj_isspc;
      }
      next = (g_status.command == WordLeft || g_status.command == FullWordLeft);
      spc  = (g_status.command == WordLeft || g_status.command == WordEndLeft) ?
             myiswhitespc : bj_isspc;

      if (p != NULL  &&  r1 > 0  &&  r1 >= len  &&  !spc( *(p + len - 1) )) {
         r1 = len - 1;
         p += r1;
         for (; r1 >= 0 && !space( *p ); r1--, p--);
         r1 += next;
         rcol = tabs ? detab_adjust_rcol( ll->line, r1 ) : r1;
         check_virtual_col( window, rcol, rcol );
         make_ruler( window );
         show_ruler( window );
      } else {
         r1 =  r1 >= len ? len-1 : r1;
         if (r1 >= 0)
            p += r1;
         if (p != NULL  &&  r1 > 0  &&  next  &&  !space( *p )  &&
                                                  !space( *(p-1) )) {
            for (; r1 >= 0 && !space( *p ); r1--, p--);
            ++r1;
            rcol = tabs ? detab_adjust_rcol( ll->line, r1 ) : r1;
            check_virtual_col( window, rcol, rcol );
            make_ruler( window );
            show_ruler( window );
         } else {

            /*
             * if we are on the first letter of a word, get off.
             */
            if (p != NULL)
               for (; r1 >= 0 && !spc( *p ); r1--, p--);

            /*
             * go to the next line if word begins at 1st col in line.
             */
            if (r1 < 0) {
               if (ll->prev != NULL) {
                  --rline;
                  ll = ll->prev;
                  p  = ll->line;
                  r1 = ll->len - 1;
                  if (r1 >= 0)
                     p += r1;
               } else
                  rc = ERROR;
            }

            /*
             * skip all blanks until we get to a previous word
             */
            while (rc == OK  &&  (p == NULL  ||  (p != NULL  &&
                                                  spc( *p )))) {
               for (; r1 >= 0 && spc( *p ); r1--, p--);
               if (r1 < 0) {
                  if (ll->prev != NULL) {
                     --rline;
                     ll = ll->prev;
                     p  = ll->line;
                     r1 = ll->len - 1;
                     if (r1 >= 0)
                        p += r1;
                  } else
                     rc = ERROR;
               } else
                  break;
            }

            /*
             * now, find the beginning of the word.
             */
            if (rc == OK  &&  p != NULL) {
               for (; r1 >= 0 && !space( *p ); r1--, p--);
               r1 += next;
               bin_offset_adjust( window, rline );
               find_adjust( window, ll, rline, r1 );
               if (rline != w.rline && !window->file_info->dirty) {
                  update_line( &w );
                  show_curl_line( window );
               }
               make_ruler( window );
               show_ruler( window );
            } else
               rc = ERROR;
         }
      }
   } else
      rc = ERROR;

   cursor_sync( window );
   return( rc );
}


/*
 * Name:    word_right
 * Purpose: To move the cursor right one word
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Words are considered strings of letters, numbers and underscores,
 *           which must be separated by other characters.
 */
int  word_right( TDE_WIN *window )
{
int  len;               /* length of current line */
text_ptr p;             /* text pointer */
int  rc;
int  tabs;
int  rcol;
register int r1;
TDE_WIN w;
line_list_ptr ll;
long rline;
int  (*space)( int );   /* Function to determine what a word is */
int  end;               /* Searching for end of word? */

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   rc = OK;
   dup_window_info( &w, window );
   rline = window->rline;
   r1    = rcol = window->rcol;
   ll    = window->ll;
   if (ll->len != EOF) {
      p   = ll->line;
      len = ll->len;

      tabs = mode.inflate_tabs;
      r1   =  tabs ? entab_adjust_rcol( p, len, rcol ) : rcol;

      end   = (g_status.command == WordEndRight ||
               g_status.command == FullWordEndRight);
      space = (g_status.command == WordRight ||
               g_status.command == WordEndRight) ? myiswhitespc : bj_isspc;

      /*
       * if rcol is past EOL, move it to EOL
       */
      r1 =  r1 >= len ? len-1 : r1;
      if (r1 >= 0)
         p += r1;

      /*
       * if cursor is on a word, find end of word.
       */
      if (p != NULL && r1 < len-1 && end && !space( *p ) && !space( *(p+1) )) {
         for (; r1 < len && !space( *p ); r1++, p++);
         --r1;
      } else {
         if (p != NULL)
            for (; r1 < len && !space( *p ); r1++, p++);
         else
            r1 = len;

         /*
          * go to the next line if word ends at eol.
          */
         if (r1 == len) {
            ++rline;
            ll = ll->next;
            if (ll->len != EOF) {
               p   = ll->line;
               len = ll->len;
               r1  = 0;
            } else
               rc = ERROR;
         }

         /*
          * now, go forward thru the file looking for the first letter of word.
          */
         while (rc == OK && (p == NULL  ||  (p != NULL && space( *p )))) {
            for (; r1 < len && space( *p ); r1++, p++);
            if (r1 == len) {
               ++rline;
               ll = ll->next;
               if (ll->len != EOF) {
                  p   = ll->line;
                  len = ll->len;
                  r1  = 0;
               } else
                  rc = ERROR;
            } else
               break;
         }
         /*
          * now find the end of the word.
          */
         if (end) {
            for (; r1 < len && !space( *p ); r1++, p++);
            --r1;
         }
      }
   } else
      rc = ERROR;

   if (rc == OK) {
      bin_offset_adjust( window, rline );
      find_adjust( window, ll, rline, r1 );
      make_ruler( window );
      show_ruler( window );
   }

   if (rline != w.rline && !window->file_info->dirty) {
      update_line( &w );
      show_curl_line( window );
   }
   cursor_sync( window );
   return( rc );
}


/*
 * Name:     next_dirty_line
 * Purpose:  To move the cursor to the next dirty line, if it exists
 * Date:     April 1, 1993
 * Modified: August 12, 1997, Jason Hood - reposition for real tabs
 * Passed:   window:  pointer to current window
 *
 * jmh 980824: skip groups of dirty lines.
 */
int  next_dirty_line( TDE_WIN *window )
{
int  rc;
line_list_ptr ll;
long rline;
long bin_offset;       /* binary offset */
TDE_WIN w;

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   dup_window_info( &w, window );
   rline = window->rline;
   ll = window->ll;
   bin_offset = window->bin_offset;
   rc = OK;
   if (ll->prev == NULL)
      bin_offset -= EOF;
   while (ll->type & DIRTY) {
      ++rline;
      bin_offset += ll->len;
      ll = ll->next;
   }
   while (!(ll->type & DIRTY)) {
      ++rline;
      bin_offset += ll->len;
      ll = ll->next;
      if (ll == NULL) {
         rc = ERROR;
         break;
      }
   }

   if (rc == OK) {
      int rcol = window->rcol;
      set_marker( window );             /* remember previous position */
      window->bin_offset = bin_offset;
      find_adjust( window, ll, rline, window->rcol );
      if (mode.inflate_tabs)
         check_virtual_col( window, rcol, rcol );
      make_ruler( window );
      show_ruler( window );
   } else
      error( WARNING, window->bottom_line, utils16 );

   if (rline != w.rline && !window->file_info->dirty) {
      update_line( &w );
      show_curl_line( window );
   }
   cursor_sync( window );
   return( rc );
}


/*
 * Name:     prev_dirty_line
 * Purpose:  To move the cursor to the prev dirty line, if it exists
 * Date:     April 1, 1993
 * Modified: August 10, 1997, Jason Hood - reposition for real tabs
 * Passed:   window:  pointer to current window
 *
 * jmh 980824: skip groups of dirty lines.
 */
int  prev_dirty_line( TDE_WIN *window )
{
int  rc;
line_list_ptr ll;
long rline;
long bin_offset;        /* binary offset */
TDE_WIN w;

   entab_linebuff( );
   if (un_copy_line( window->ll, window, TRUE ) == ERROR)
      return( ERROR );
   dup_window_info( &w, window );
   rline = window->rline;
   ll = window->ll;
   bin_offset = window->bin_offset;
   rc = OK;
   while (ll->type & DIRTY) {
      --rline;
      ll = ll->prev;
      bin_offset -= ll->len;
   }
   while (!(ll->type & DIRTY)) {
      if (--rline <= 0) {
         rc = ERROR;
         break;
      }
      ll = ll->prev;
      bin_offset -= ll->len;
   }

   if (rc == OK) {
      int rcol = window->rcol;
      set_marker( window );             /* remember previous position */
      window->bin_offset = bin_offset;
      find_adjust( window, ll, rline, window->rcol );
      if (mode.inflate_tabs)
         check_virtual_col( window, rcol, rcol );
      make_ruler( window );
      show_ruler( window );
   } else
      error( WARNING, window->bottom_line, utils16 );

   if (rline != w.rline && !window->file_info->dirty) {
      update_line( &w );
      show_curl_line( window );
   }
   cursor_sync( window );
   return( rc );
}


/*
 * Name:    center_window
 * Purpose: To place the current line or cursor in the center of a window.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
int  center_window( TDE_WIN *window )
{
int  center;
int  center_line;
int  diff;
register file_infos *file;
register TDE_WIN *win;           /* put window pointer in a register */

   win = window;
   file = win->file_info;
   center = (win->bottom_line + 1 - win->top_line) / 2 - win->ruler;
   center_line = win->top_line + win->ruler + center;
   diff = center_line - win->cline;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (g_status.command == CenterWindow) {
      if (diff > 0) {
         if (win->rline + diff <= file->length) {
            update_line( win );
            win->cline += diff;
            win->rline += diff;
            for (; diff > 0; diff--) {
               win->bin_offset += win->ll->len;
               win->ll = win->ll->next;
            }
            show_curl_line( win );
         }
      } else if (diff < 0) {
         update_line( win );
         win->cline += diff;
         win->rline += diff;
         for (; diff < 0; diff++) {
            win->ll = win->ll->prev;
            win->bin_offset -= win->ll->len;
         }
         show_curl_line( win );
      }
   } else {
      if (diff > 0) {
         win->cline += diff;
         if ((long)(win->cline - (win->top_line + win->ruler)) > win->rline)
            win->cline = (win->top_line + win->ruler) + (int)win->rline;
         file->dirty = LOCAL;
      } else if (diff < 0) {
         win->cline += diff;
         file->dirty = LOCAL;
      }
   }
   if (g_status.command == CenterWindow  ||  g_status.command == CenterLine)
      cursor_sync( win );
   return( OK );
}


/*
 * Name:    horizontal_screen_right
 * Purpose: To move the cursor one screen to the right
 * Date:    September 13, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Add 80 columns to the real cursor.  If the cursor is past the
 *          maximum line length then move it back.
 * jmh 980910: if not-eol display, add 79 columns.
 */
int  horizontal_screen_right( TDE_WIN *window )
{
int  col;
int  screen_width;

   screen_width = window->end_col - window->start_col + (mode.show_eol != 2);
   col = window->rcol + screen_width;
   if (col < MAX_LINE_LENGTH) {
      window->rcol = col;
      window->bcol += screen_width;
      window->file_info->dirty = LOCAL;
      check_virtual_col( window, window->rcol, window->ccol );
      cursor_sync( window );
      make_ruler( window );
      show_ruler( window );
   }
   return( OK );
}


/*
 * Name:    horizontal_screen_left
 * Purpose: To move the cursor one screen to the left
 * Date:    September 13, 1991
 * Passed:  window:  pointer to current window
 * Notes:   Subtract screen width from the real cursor.  If the cursor is less
 *           than zero then see if bcol is zero.  If bcol is not zero then make
 *           bcol zero.
 * jmh 980910: if not-eol display, include one-column "overlap".
 */
int  horizontal_screen_left( TDE_WIN *window )
{
int  screen_width;

   screen_width = window->end_col - window->start_col + (mode.show_eol != 2);
   if (window->rcol - screen_width < 0) {
      if (window->bcol != 0) {
         window->bcol = 0;
         window->file_info->dirty = LOCAL;
      }
   } else {
      window->rcol -= screen_width;
      window->bcol -= screen_width;
      if (window->bcol < 0)
         window->bcol = 0;
      window->file_info->dirty = LOCAL;
   }
   check_virtual_col( window, window->rcol, window->ccol );
   cursor_sync( window );
   make_ruler( window );
   show_ruler( window );
   return( OK );
}


/*
 * Name:    goto_top_file
 * Purpose: To move the cursor to the top of the file.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
int  goto_top_file( TDE_WIN *window )
{
register TDE_WIN *win;   /* put window pointer in a register */
long num;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   if (win->rline != win->cline - (win->top_line+win->ruler)) {
      set_marker( win );                /* remember previous position */
      win->rline = win->cline - (win->top_line+win->ruler);
      win->bin_offset = (win->rline > 0) ? -EOF : 0;
      win->ll = win->file_info->line_list;
      for (num=0; num < win->rline; num++) {
         win->bin_offset += win->ll->len;
         win->ll = win->ll->next;
      }
      display_current_window( win );
   }
   cursor_sync( win );
   return( OK );
}


/*
 * Name:    goto_end_file
 * Purpose: To move the cursor to the end of the file.
 * Date:    June 5, 1991
 * Passed:  window:  pointer to current window
 */
int  goto_end_file( TDE_WIN *window )
{
register TDE_WIN *win;  /* put window pointer in a register */
line_list_ptr ll;
long length;

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   length = win->file_info->length;
   if (length >= win->rline + win->bottom_line - win->cline) {
      set_marker( win );                /* remember previous position */
      win->rline = length - (win->bottom_line - win->cline) + 1;
      win->ll = win->file_info->line_list_end;
      for (;length >= win->rline; length--)
         win->ll = win->ll->prev;

      win->bin_offset = 0;
      ll = win->file_info->line_list->next;
      for (length = 1; length < win->rline; length++) {
         win->bin_offset += ll->len;
         ll = ll->next;
      }

      display_current_window( win );
   }
   cursor_sync( win );
   return( OK );
}


/*
 * Name:     goto_line
 * Purpose:  To move the cursor to a particular line in the file
 * Date:     June 5, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 *           August 11, 1997, Jason Hood - can use -ve numbers to go from eof
 *                                         -1 is last line, -2 second last...
 *           August 12, 1997, Jason Hood - reposition for real tabs
 * Passed:   window:  pointer to current window
 */
int  goto_line( TDE_WIN *window )
{
long number;             /* line number selected */
long n;
register TDE_WIN *win;   /* put window pointer in a register */
line_list_ptr ll;
int  rcol;
int  rc;
char answer[MAX_COLS+2];
char temp[MAX_COLS+2];

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );
   /*
    * find out where we are going
    */

   answer[0] = '\0';
   /*
    * line number:
    */
   if (get_name( find11, win->bottom_line, answer ) != OK  ||  *answer == '\0')
      return( ERROR );
   number = atol( answer );
   if (number < 0) number += (long)win->file_info->length + 1;
   if (number > 0  && number <= (long)win->file_info->length) {
      update_line( win );
      ll = win->ll;
      n = win->rline;
      if (number < win->rline) {
         if (n - number < number - 1) {
            for (; n > number; n--) {
               ll = ll->prev;
               win->bin_offset -= ll->len;
            }
         } else {
            ll = win->file_info->line_list->next;
            n = 1;
            for (; n < number; n++) {
               win->bin_offset += ll->len;
               ll = ll->next;
            }
         }
      } else if (number > win->rline) {
         for (; n < number; n++) {
            win->bin_offset += ll->len;
            ll = ll->next;
         }
      }
      set_marker( win );                /* remember previous position */
      rcol = win->rcol;
      find_adjust( win, ll, number, win->rcol );
      if (mode.inflate_tabs)
         check_virtual_col( win, rcol, rcol );
      if (!win->file_info->dirty)
         show_curl_line( win );
      rc = OK;
   } else {
      /*
       * out of range.  must be in the range 1 -
       */
      strcpy( temp, find12 );
      my_ltoa( win->file_info->length, temp + strlen( temp ), 10 );
      error( WARNING, win->bottom_line, temp );
      rc = ERROR;
   }
   return( rc );
}


/*
 * Name:     set_marker
 * Purpose:  To set file marker
 * Date:     December 28, 1991
 * Passed:   window:  pointer to current window
 * Modified: August 9, 1997, Jason Hood - added block markers
 *           September 13, 1997, Jason Hood - added previous position
 *           July 18, 1998, Jason Hood - added macro marker
 *           July 28, 1998, Jason Hood - removed block markers (see goto_marker)
 */
int  set_marker( TDE_WIN *window )
{
register MARKER  *marker;       /* put the marker in a register */

   if (g_status.command >= SetMark1 && g_status.command <= SetMark3)
      marker = &window->file_info->marker[g_status.command - SetMark1 + 1];
   else if (g_status.command == MacroMark)
      marker = &g_status.macro_mark;
   else
      marker = &window->file_info->marker[0];

   marker->rline  = window->rline;
   marker->rcol   = window->rcol;
   marker->ccol   = window->ccol;
   marker->bcol   = window->bcol;
   marker->marked = TRUE;
   return( OK );
}


/*
 * Name:     goto_marker
 * Purpose:  To goto a file marker
 * Date:     December 28, 1991
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Modified: August 9, 1997, Jason Hood - added block markers
 *           August 11, 1997, Jason Hood - move cursor to line if on screen
 * Passed:   window:  pointer to current window
 *
 * jmh 980728: Create block markers on the fly.
 * jmh 980801: If already at the block marker, move to the other one (ie. if
 *              at top-left corner, move to top-right; if at bottom-right,
 *              move to bottom-left).
 */
int  goto_marker( TDE_WIN *window )
{
int  m;
file_infos *file;
long new_rline;
long n;
MARKER *marker, prev;
register TDE_WIN *win;   /* put window pointer in a register */
line_list_ptr ll;
int  rc;

   win  = window;
   file = win->file_info;
   m = -1;
   if (g_status.command == PreviousPosition) {
      prev   = file->marker[0];
      marker = &prev;
   } else if (g_status.command >= GotoMark1 && g_status.command <= GotoMark3) {
      m = g_status.command - GotoMark1 + 1;
      marker = &file->marker[m];
   } else if (g_status.command == MacroMark)
      marker = &g_status.macro_mark;
   else if (g_status.command == BlockBegin || g_status.command == BlockEnd) {
      marker = &prev;
      if (g_status.marked && g_status.marked_file == file) {
         prev.marked = TRUE;
         prev.ccol   = win->ccol;
         prev.bcol   = win->bcol;
         if (g_status.command == BlockBegin) {
            prev.rline = file->block_br;
            prev.rcol  = (win->rline == file->block_br &&
                          win->rcol  == file->block_bc) ? file->block_ec :
                                                          file->block_bc;
         } else {
            prev.rline = file->block_er;
            prev.rcol  = (win->rline == file->block_er &&
                          win->rcol  == file->block_ec) ? file->block_bc:
                                                          file->block_ec;
         }
      } else
         prev.marked = FALSE;
   } else
      return( ERROR );

   if (marker->marked) {
      entab_linebuff( );
      if (un_copy_line( win->ll, win, TRUE ) == ERROR)
         return( ERROR );
      /*
       * Remember the previous position, but not if it's a macro mark
       */
      if (g_status.command != MacroMark)
         set_marker( win );
      file->dirty = LOCAL;
      if (marker->rline > file->length)
         marker->rline = file->length;
      if (marker->rline < 1l)
         marker->rline = 1l;
      new_rline = marker->rline;
      ll = win->ll;
      if (new_rline < win->rline) {
         n = win->rline - new_rline;
         if ((long)win->cline - n > (long)(win->top_line+win->ruler-1))
            win->cline -= (int)n;
         if (n < new_rline - 1) {
            for (n=win->rline; n > new_rline; n--) {
               ll = ll->prev;
               win->bin_offset -= ll->len;
            }
         } else {
            ll = win->file_info->line_list->next;
            win->bin_offset = 0;
            n = 1;
            for (; n < new_rline; n++) {
               win->bin_offset += ll->len;
               ll = ll->next;
            }
         }
      } else if (new_rline > win->rline) {
         n = new_rline - win->rline;
         if ((long)win->cline + n <= (long)win->bottom_line)
            win->cline += (int)n;
         n = win->rline;
         for (; n < new_rline; n++) {
            win->bin_offset += ll->len;
            ll = ll->next;
         }
      }
      win->rline  = new_rline;
      win->ll     = ll;
      win->rcol   = marker->rcol;
      win->ccol   = marker->ccol;
      win->bcol   = marker->bcol;
      if (win->rline < (win->cline - ((win->top_line + win->ruler))))
         win->cline = (int)win->rline + (win->top_line + win->ruler);
      check_virtual_col( win, win->rcol, win->ccol );
      make_ruler( window );
      show_ruler( window );
      rc = OK;
   } else {
      /*
       * Don't display a warning for the block markers and previous position
       */
      if (m > 0) {
         if (m == 10)
            m = 0;
         *(utils13 + UTILS13_NO_SLOT) = (char)('0' + m);
         /*
          * marker not set
          */
         error( WARNING, win->bottom_line, utils13 );
      }
      rc = ERROR;
   }
   return( rc );
}


/*
 * Name:     date_time_stamp
 * Purpose:  put system date and time into file at cursor position
 * Date:     June 5, 1992
 * Modified: November 13, 1993, Frank Davis per Byrial Jensen
 * Passed:   window:  pointer to current window
 *
 * jmh 980702: test for EOF
 */
int  date_time_stamp( TDE_WIN *window )
{
char answer[MAX_COLS+2];

   if (window->ll->len == EOF)
      return( ERROR );

   format_time( mode.stamp, answer );
   return( add_chars( answer, window ) );
}


/*
 * Name:     stamp_format
 * Purpose:  to set a format for the date and time stamp
 * Date:     May 21, 1998
 * Author:   Jason Hood
 * Passed:   window:  pointer to current window
 */
int  stamp_format( TDE_WIN *window )
{
register TDE_WIN *win;   /* put window pointer in a register */
char answer[MAX_COLS+2];

   win = window;
   entab_linebuff( );
   if (un_copy_line( win->ll, win, TRUE ) == ERROR)
      return( ERROR );

   strcpy( answer, mode.stamp );
   /*
    * stamp format (F1 = help):
    */
   if (get_name( utils17, g_display.nlines, answer ) != OK || *answer == '\0')
      return( ERROR );

   strcpy( mode.stamp, answer );
   return( OK );
}


/*
 * Name:    add_chars
 * Purpose: insert string into file
 * Date:    June 5, 1992
 * Passed:  string:  string to add to file
 *          window:  pointer to current window
 *
 * jmh 980524: treat the tab character as though the key was pressed.
 * jmh 980526: treat the newline character as the Rturn function.
 * jmh 980728: test rc in the while.
 */
int  add_chars( char *string, TDE_WIN *window )
{
int  rc = OK;

   while (*string && rc == OK) {
      if (*string == '\t')
         rc = tab_key( window );
      else if (*string == '\n')
         rc = insert_newline( window );
      else {
         g_status.key_pressed = *string;
         rc = insert_overwrite( window );
      }
      ++string;
   }
   return( rc );
}


/*
 * Name:    dos_shell
 * Purpose: shell to DOS (or whatever)
 * Author:  Jason Hood
 * Date:    November 27, 1996
 * Modified: August 28, 1997 - tested for redirection
 * Passed:  window:  pointer to current window
 * Notes:   the video mode should not be changed on return to TDE
 */
int  dos_shell( TDE_WIN *window )
{
int  stdoutput;

#if defined( __UNIX__ )
   cls( );
   xygoto( 0, 0 );
#else
   page( 0 );
#endif
   if (!g_status.output_redir)
      system( "" );
   /*
    * If output has been redirected then put it back to the screen for the
    * shell and restore it afterwards.
    */
   else {
      stdoutput = dup( fileno( stdout ) );
      freopen( STDFILE, "w", stdout );
      system( "" );
      dup2( stdoutput, fileno( stdout ) );
   }
#if !defined( __UNIX__ )
   page( 1 );
#endif
   redraw_screen( window );
   return( OK );
}


/* Name:    dos_screen
 * Purpose: to display the DOS screen
 * Author:  Jason Hood
 * Date:    December 28, 1996
 * Passed:  arg_filler to satisfy the prototype
 * Notes:   DOS only
 *
 */
int  dos_screen( TDE_WIN *arg_filler )
{
#if !defined( __UNIX__ )
   page( 0 );
   getkey( );
   page( 1 );
#endif
   return( OK );
}
