
(*$B-*)  (* No Ctrl-Break handling *)
(*$S 800*)  (* Stack size 2K *)

MODULE xasctsr1; (* TSR ascii table, JPI TopSpeed M-2 V1.16, Ron Buelow 5/90 *)

 FROM IO     IMPORT  WrCard, WrChar, WrCharRep, WrHex, WrLn, WrStr, WrStrAdj;
 FROM Window IMPORT  Black, LightGray, DoubleFrame, GotoXY, Hide, Open,
                     PutOnTop, WinDef, WinType;

 IMPORT Lib, Str, SYSTEM, TSR;

 TYPE
   DefType = ARRAY [0..32] OF ARRAY [0..2] OF CHAR;
   AttType = ARRAY [0..15] OF ARRAY [0..1] OF CHAR;

 CONST
   xDef = DefType('NUL','SOH','STX','ETX','EOT','ENQ','ACK','BEL',' BS',
                  ' HT',' LF',' VT',' FF',' CR',' SO',' SI','DLE','DC1',
                  'DC2','DC3','DC4','NAK','SYN','ETB','CAN',' EM','SUB',
                  'ESC',' FS',' GS',' RS',' US','SPC');

   xAtt = AttType('bk','bu','gn','cn','rd','vt','bn','wh','GY','BU','GN',
                  'CN','RD','VT','YW','WH');
   ScrWidth = 80;
   ScrDepth = 25;

   Depth = 23;  (* window *)
   Width = 36;
   Left = ScrWidth - Width;
   Top = (ScrDepth - Depth) -2;
   Right = Left + Width-1;
   Bottom = Top + Depth;

   pointup = 36C;  (* char *)
   pointdn = 37C;  (* char *)
   rev = 160C;  (* attr *)
   nor = 7C;  (* attr *)
   cursor = 17C;  (* attr *)

   esc = 33C;     f1 = 273C;     f2 = 274C;
   home = 307C;   up = 310C;     pgup = 311C;
   left = 313C;   right = 315C;  end = 317C;
   dn = 320C;     pgdn = 321C;   ctrlL = 363C;
   ctrlR = 364C;

   altX = 255C;

 VAR
   screen [0B800H:0]: ARRAY [1..ScrDepth], [1..ScrWidth] OF
                        RECORD
                          char : CHAR;
                          attr : CHAR
                        END;

   x, y, x1, y1, x2, y2 : CARDINAL;
   scnNum, chrNum : CARDINAL;
   c, d, e, i : SHORTCARD;
   kbmode, tsr : BOOLEAN;
   sc, ch : CHAR;

   AscWin : WinType;

 PROCEDURE GetScanKey(VAR sc:CHAR): CHAR;
 VAR r:SYSTEM.Registers;
 BEGIN
   WITH r DO
     AH := 0;
     Lib.Intr(r,16H);
     IF AH # 0 THEN
       sc := CHR(AH)
     ELSE
       sc := CHR(AH-128)
     END(*IF*);
   IF sc > CHR(127) THEN
     AL := AH
   END(*IF*);
     IF AL # 0 THEN
       RETURN CHR(AL)
     ELSE
       RETURN CHR(AH+128)
     END(*IF*)
   END(*WITH*)
 END GetScanKey;

 PROCEDURE WrOct(n:CARDINAL; Length:INTEGER);
 VAR
   s : ARRAY[0..80] OF CHAR;
   OK : BOOLEAN;
 BEGIN
   Str.CardToStr(VAL(LONGCARD,n),s,8,OK);
   IF OK THEN
     WrStrAdj(s,Length)
   END(*IF*)
 END WrOct;

 PROCEDURE MoveCursor(newX, newY : INTEGER);
 BEGIN
   IF newX < Left+3 THEN  (* wrap cursor *)
     newX := Right-1;
     newY := newY-2
   END;(*IF*)
   IF newX > Right-1 THEN
     newX := Left+3;
     newY := newY+2
   END;(*IF*)
   IF newY < Top+3 THEN
     newY := Bottom-6
   END;(*IF*)
   IF newY > Bottom-6 THEN
     newY := Top+3
   END;(*IF*)
   screen[y-1,x].char := ' ';  (* erase & move cursor *)
   screen[y,x].attr := rev;
   screen[y+1,x].char := ' ';
   x := newX;
   y := newY;
   screen[y-1,x].char := pointdn;
   screen[y,x].attr := cursor;
   screen[y+1,x].char := pointup
 END MoveCursor;

 PROCEDURE WrMode;
 BEGIN
   GotoXY(18,22);
   IF kbmode = TRUE THEN
     WrStr(" Keybd ")
   ELSE
     WrStr(" Table ")
   END;(*IF*)
   x1 := Left+19;
   y1 := Top+23;
   FOR i := 1 TO 7 DO
     screen[y1,x1].attr := nor;
     INC(x1)
   END(*FOR*)
 END WrMode;

 PROCEDURE WrTable;
 BEGIN
   PutOnTop(AscWin);
   c := 0;
   d := Width-5;
   x := Left+3;
   y := Top+3;
   FOR i := 1 TO 8 DO  (* write table *)
     FOR e := c TO c + d DO
       screen[y,x].attr := rev;
       screen[y,x].char := CHR(e);
       INC(x)
     END;(*FOR*)
     INC(y);
     INC(y);
     x := Left+3;
     c := e+1
   END;(*FOR*)
   x := Left+1;  (* adjust borders *)
   y := Bottom-4;
   FOR i := 1 TO 2 DO
     screen[y,x].char := CHR(199);
     INC(x);
       FOR e := 1 TO 34 DO
         screen[y,x].char := CHR(196);
         INC(x)
       END;(*FOR*)
     screen[y,x].char := CHR(182);
     x := Left+1;
     y := Bottom-1
   END;(*FOR*)           (* write text *)
   GotoXY(1,20);
   WrStr(" CHR DEC HEX OCT DEF SCD SCH ATTR");
   GotoXY(1,22);
   WrStr(" XASCTSR F2-Mode  Table  Esc-Exit ");
   x := Left+3;
   y := Top+3;
   MoveCursor(x,y);  (* mark cursor *)
   WrMode
 END WrTable;

 PROCEDURE WrOrd;  (* tag ^2 nul *)
 BEGIN
   IF scnNum = 3 THEN
     IF chrNum = 131 THEN
       chrNum := 0;
       x1 := Left+3;
       y1 := Top+3;
       MoveCursor(x1,y1)
     END(*IF*)
   END(*IF*);

   screen[Bottom-3,Left+4].char := CHR(chrNum);  (* write character *)

   GotoXY(6,19);  (* write ordinals *)
   WrCard(chrNum,3);
   WrHex(chrNum,3);
   WrStr('h ');
   WrOct(chrNum,3);
   WrChar(' ');

   IF chrNum < 33 THEN  (* write definition *)
     WrStr(xDef[chrNum])
   ELSIF chrNum = 127 THEN
     WrStr("DEL")
   ELSE
     WrStr("   ")
   END;(*IF*)

   IF kbmode = FALSE THEN  (* write control character *)
     GotoXY(22,20);
     WrStr("       ");
     GotoXY(22,19);
     CASE chrNum OF
     0..31   : WrChar('^');
               WrChar(CHR(chrNum+64));
               WrStr("     ")
     ELSE
       WrStr("       ")
     END;(*CASE*)
   ELSE
     GotoXY(22,20);
     WrStr("SCD SCH")
   END;(*IF*)

   GotoXY(30,19);  (* write attribute *)
   WrStr(xAtt[chrNum MOD 16]);
   WrStr(xAtt[chrNum DIV 16]);
   x1 := Right-4;
   y1 := Bottom-2;
   FOR i := 1 TO 4 DO
     screen[y1,x1].attr := CHR(chrNum);
     INC(x1)
   END(*FOR*)
 END WrOrd;

 PROCEDURE WrCode;
 BEGIN
   scnNum := ORD(sc);
   chrNum := ORD(ch);
   MoveCursor((Left+3)+chrNum MOD 32,
             ((chrNum DIV 32)*2)+Top+3);
   WrOrd;
   GotoXY(22,19);
   WrCard(scnNum,3);  (* write scancode *)
   WrHex(scnNum,3);
   WrChar('h');
   ch := GetScanKey(sc)
 END WrCode;

 PROCEDURE RunAsc;
 BEGIN
   WrTable;
   WrMode;
   kbmode := FALSE;
   IF tsr = TRUE THEN
     MoveCursor(x2,y2)
   END;(*IF*)
   LOOP
     chrNum := ORD(screen[y,x].char);
     WrOrd;
     WHILE kbmode = TRUE DO
       CASE ch OF
         f2 : WrCode;
              IF ch = f2 THEN
                MoveCursor(Left+3,Top+3);
                kbmode := FALSE
              END;(*IF*)
              WrMode
      | esc : WrCode;
              IF ch = esc THEN
                MoveCursor(Left+3,Top+3);
                kbmode := FALSE
              END;(*IF*)
              WrMode
       ELSE
         WrCode
       END;(*CASE*)
     END;(*WHILE*)
     chrNum := ORD(screen[y,x].char);
     WrOrd;

     ch := GetScanKey(sc);

     CASE ch OF  (* handle input *)
     | f2    : IF kbmode = TRUE THEN
                 kbmode := FALSE;
               ELSE
                 kbmode := TRUE
               END;
               WrMode
     | esc   : EXIT (* end program *)
     | left  : MoveCursor(x-1,y)  (* cursor controls *)
     | up    : MoveCursor(x,y-2)
     | right : MoveCursor(x+1,y)
     | dn    : MoveCursor(x,y+2)
     | home  : MoveCursor(Left+3,Top+3)
     | end   : MoveCursor(Right-1,Bottom-6)
     | pgup  : MoveCursor(x,Top+3)
     | pgdn  : MoveCursor(x,Bottom-6)
     | ctrlL : MoveCursor(Left+3,y)
     | ctrlR : MoveCursor(Right-1,y)
     | altX  : TSR.DeInstall; EXIT
     ELSE
       IF kbmode = FALSE THEN
         WrChar(7C)  (* honk if invalid key is hit *)
       END(*IF*)
     END(*CASE*)
   END;(*LOOP*)
   Hide(AscWin);
   x2 := x;
   y2 := y;
   tsr := TRUE
 END RunAsc;

 BEGIN
   tsr := FALSE;
   WrLn;
   WrStr("XASCTSR1 ...Installing");
   WrLn;
   WrStr("Activate with: <Alt-K>");
   WrLn;
   WrLn;
   AscWin := Open(WinDef(Left,Top,Right,Bottom,Black,LightGray,
                  FALSE,FALSE,TRUE,TRUE,DoubleFrame,Black,LightGray));
   TSR.Install(RunAsc,TSR.KBFlagSet{TSR.Alt},37,175H) (* Alt-K *)
 END xasctsr1.