#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "cmdline.h"
#include "umem.h"
#include "module.h"
#include "data.h"
#include "section.h"
#include "errors.h"

extern char modname[];

SECTION **sectiontable;
LIST *sectionlist = 0;

static SECTION **section_ptrs=0;
static int numsections = 0;

static char SectionAttribs[] = "ACEMNPRSUWXZ";

SECTION *StepToSection(char *buf, uint flags)
{
	int i;
	for (i=0; i < numsections; i++) {
		if (section_ptrs[i]) {
			if (!strcmp(section_ptrs[i]->name, buf)) {
				if ((section_ptrs[i]->flags & ~SECTION_ABSOLUTE)== flags && !(flags & SECTION_SEPERATE)) {
					SECTION *p = section_ptrs[i];
					section_ptrs[i] = section_ptrs[i]->link;
					return(p);
				}
			}
		}
	}
	return(0);
}
void AddNewSection(SECTION *sect)
{
	LIST **l = &sectionlist, *new;
	while (*l) {
		SECTION *p = (*l)->data;
		if (!strcmp(p->name, sect->name)) {
			if (p->flags == sect->flags && !(sect->flags &SECTION_SEPERATE)) {
				if (sect->flags & SECTION_UNIQUE) {
					Error("Duplicate Unique Section %s, modules %s, %s",
							sect->name, sect->modname, p->modname);
				}
				sect->parent = p;
				while (p->link) 
					p = p->link;
				p->link = sect;
				return;
			}
			if ((p->flags ^ sect->flags) & 4)
					Error("Section %s Attribute Mismatch, modules %s, %s",
							sect->name, sect->modname, p->modname);
		}
		l = *l;
	}
	sect->parent = sect;
	new = AllocateMemory(sizeof(LIST));
	new->link = 0;
	new->data = sect;
	*l = new;
}
SECTION *GetSection(int num)
{
	return(sectiontable[num]);
}
void SectionDefinitions(int mode)
{
	int id;
	BYTE ch;
	char buf[MAX_NAMESIZE+1];
	uint flags=0;
	buf[0] = 0;
	id = ReadNumber(0);
  CheckForComma(TRUE);
	while ((ch = ReadChar()) != '.') {
		char *p = strchr(SectionAttribs, ch);
		if (p) {
			flags |= 1 << ((uint)(p - SectionAttribs));
			CheckForComma(TRUE);
		}
		else
			break;
	}
	if (!(flags & SECTION_POSTPONE))
		flags |= SECTION_NOW;
	if (ch != '.'){
		putbackch(ch);
  	ReadString(buf,MAX_NAMESIZE+1);
		CheckForPeriod();
	}
	if (id >= MAX_SECTIONS)
		fatal("Section ID too big in module %s", modname);
	if (mode == SCAN) {
		SECTION *q = AllocateMemory(sizeof(SECTION));
		q->name = AllocateMemory(strlen(buf)+1);
		q->modname = AllocateMemory(strlen(modname)+1);
		strcpy(q->name, buf);
		strcpy(q->modname, modname);
		q->virtual = 0;
		q->flags = flags;
		q->base = 0;
		q->size = 0;
		q->align = 0;
		q->link = 0;
		q->parent = 0;
		q->absbase = 0;
		q->abssize = 0;
		q->fixups = 0;
		q->fixupptr = &q->fixups;
		q->loaded = FALSE;
		AddNewSection(q);
		sectiontable[id] = q;
	}
	else {
		sectiontable[id] = StepToSection(buf,flags);
	}
}
void SectionAlignment(void)
{
	SECTION *q;
	int id;
	int align;
	id = ReadNumber(0);
	CheckForComma(TRUE);
	align = ReadNumber(0);
	CheckForPeriod();
	if ((q = sectiontable[id]) != 0)
		q->align = align;
}
void SectionSetSize(void)
{
	SECTION *q;
	int id;
	long size;
	id = ReadNumber(0);
	CheckForComma(TRUE);
	size = ReadNumber(0);
	CheckForPeriod();
	if ((q = sectiontable[id]) != 0)
		q->size = size;
}
void SectionModuleRundown(uint mode)
{
	int i;
	SECTION **p= sectiontable;
	for (i=0; i< MAX_SECTIONS; i++) {
		*p = 0;
		p++;
	}
}
void SectionTableInit(void)
{
	SECTION **p;
	int i;
	sectiontable = AllocateMemory(sizeof(SECTION *) * MAX_SECTIONS);
	p = sectiontable;
	for (i=0; i < MAX_SECTIONS; i++) {
		*p = 0;
		p++;
	}
}
void SectionTableRundown(void)
{
	LIST *l = sectionlist;
	DeallocateMemory(sectiontable);
	while (l) {
		SECTION *p = l->data;
		LIST *q = l->link;
		while (p) {
			SECTION *r = p->link;
			DeallocateMemory(p->name);
			DeallocateMemory(p->modname);
			DeallocateMemory(p);
			p = r;
		}
		DeallocateMemory(l);
		l = q;
	}
}
void SectionInit(int mode)
{
	if (mode == RESOLVE) {
		LIST *p = sectionlist, *r;
		int i;
		numsections = 0;
		while (p) {
			numsections++;
			p = p->link;
		}
		p = sectionlist;
		section_ptrs = AllocateMemory(sizeof(SECTION *) * numsections);
		for (i=0; i < numsections; i++) {
			SECTION *q = p->data;
			q->section = i;
			section_ptrs[i] = p->data;
			p = p->link;
		}

		p = sectionlist;
		for (i=0; i < numsections; i++) {
			LIST *postpones = 0;
			SECTION *q = p->data;
			long address = 0;
			while (q) {
				if (q->flags & SECTION_POSTPONE) {
					r = AllocateMemory(sizeof(LIST));
					r->data = q;
					r->link = 0;
					AppendToList(&postpones,r);
				}
				else {
					long t = (address+q->align-1)%q->align;
					t = q->align -1 - t;
					q->base = address + t;
					address = address + q->size+ t;
					if ((q->align > q->parent->align && q->align % q->parent->align)
						    || (q->align < q->parent->align && q->parent->align % q->align))
						q->parent->align *= q->align;
					else
						if (q->align > q->parent->align)
							q->parent->align = q->align;
				}
				q = q->link;
			}
			r = postpones;
			while (r) {
				SECTION *q = postpones->data;
				long t = (address+q->align-1)%q->align;
				t = q->align -1 - t;
				q->base = address + t;
				address = address + q->size + t;
				if ((q->align > q->parent->align && q->align % q->parent->align)
					    || (q->align < q->parent->align && q->parent->align % q->align))
					q->parent->align *= q->align;
				else
					if (q->align > q->parent->align)
						q->parent->align = q->align;
				r = r->link;
				DeallocateMemory(postpones);
				postpones = r;
			}
			q = p->data;
			q->abssize = address;
			p = p->link;
		}
	}
}
void SectionRundown(void)
{
	LIST *p = sectionlist;
	DeallocateMemory(section_ptrs);
	while (p) {
		SECTION *r = p->data;
		LIST *f = r->fixups;
		while (f) {
			FIXUPS *t = f->data;
			LIST *l = f->link; 
			DeleteExpression(t->exp);
			DeallocateMemory(t);
			f = l;
		}
		DeallocateMemory(r->buffer);
		p = p->link;
	}
}