; xcode is the assembler optimized variable sized packed code
; extraction routines. Written by Tom Horsley
; (tahorsley@ssd.harris.com) Dec 1988.
;
; struct codebuf {
;    void near   (*codep)();
;    char near * bufp;
; };
;
; extern unsigned int near xcode(struct codebuf near * cbp)
;
; xcode extracts the next variable sized code from the buffer pointed
; at by bufp. The codep function pointer is a state variable recording
; the address of the routine that will be used to extract the next
; code byte.
;
; xcode works by jumping to the codep address. Each code extraction
; routine extracts the specific offset code it is designed for then
; sets codep to point to the next routine that will extract the next
; code. Eventually we sync up on a byte boundary and start over again
; with the original routine.
;
; Note: As you can see below, this uses the new .MODEL directive
; Microsoft introduced sometime around 5.0, so you need the latest
; version of the assembler to build this.

.MODEL	 SMALL
.CODE
	 PUBLIC	   _xcode
_xcode	 PROC
	 push	   bp
	 mov	   bp,sp     ; add code to push ss, ds, si, di
	 push	   si

	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,[bx]   ; cx points to routine
	 mov	   si,[bx+2] ; si points to data buffer
	 call	   cx	     ; go make state transition
	 mov	   [bx],cx   ; save new state routine
	 mov	   [bx+2],si ; save new buffer ptr

	 pop	   si
	 pop	   bp	     ; add code to pop di, si, ds, ss
	 ret
_xcode	 ENDP

; The following code implements all the code extraction routines.
; Because of the way that the net compress utility works there are
; eight routines for each size code from 9 to 15 bits. For 10, 12, 14,
; and 16 bits we could do with fewer routines since we sync up on a
; byte boundary before looking at n_bits bytes, but the compress.c
; code always flushes out to n_bits byte boundary when changing code
; size, so there are 8 routines not just to extract codes, but also to
; maintain unique state information so that a buffer flush can get to
; the correct byte boundary to find the next set of codes.
;
; A fanatic would probably say "My gosh! we are wasteing space!" and
; implement a new magic number and a new compression algorithm to
; recover the dozen or so bytes wasted, but since I am not (that sort
; of) a fanatic, and I like the idea of sticking to standards, I leave
; it the way it is. (P.S. The percentage of wasted space is somewhere
; around .000001, so there is no real reason to get upset).

; ********************************************** 9 bit code extraction routines

	 PUBLIC	   _init9
_init9	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g9c0 ; initialize state to g9c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init9	 ENDP

; g9c0 get 9 bit code 0: 11111111 -------1

g9c0	 PROC
	 mov	   ax,[si]   ; get low in al, high bit in ah
	 and	   ah,001h   ; mask off junk
	 inc	   si	     ; skip past first byte
	 mov	   cx,offset g9c1 ; point to next code routine
	 ret
g9c0	 ENDP

; g9c1 get 9 bit code 1: 1111111- ------11

g9c1	 PROC
	 mov	   ax,[si]   ; get data in ax << 1
	 shr	   ax,1	     ; position correctly
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c2 ; point to next code routine
	 ret
g9c1	 ENDP

; g9c2 get 9 bit code 2: 111111-- -----111

g9c2	 PROC
	 mov	   ax,[si]   ; get data in ax << 2
	 shr	   ax,1	     ; position correctly
	 shr	   ax,1
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c3 ; point to next code routine
	 ret
g9c2	 ENDP

; g9c3 get 9 bit code 3: 11111--- ----1111

g9c3	 PROC
	 mov	   ax,[si]   ; get data in ax << 3
	 shr	   ax,1	     ; position correctly
	 shr	   ax,1
	 shr	   ax,1
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c4 ; point to next code routine
	 ret
g9c3	 ENDP

; g9c4 get 9 bit code 4: 1111---- ---11111

g9c4	 PROC
	 mov	   ax,[si]   ; get data in ax << 4
	 shr	   ax,1	     ; position correctly
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c5 ; point to next code routine
	 ret
g9c4	 ENDP

; g9c5 get 9 bit code 5: 111----- --111111

g9c5	 PROC
	 mov	   ax,[si]   ; ax = --123456 789-----
	 xchg	   ah,al     ; ax = 789----- --123456
	 rol	   ax,1	     ; ax = 89------ -1234567
	 rol	   ax,1	     ; ax = 9------- 12345678
	 rol	   ax,1	     ; ax = -------1 23456789
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c6 ; point to next code routine
	 ret
g9c5	 ENDP

; g9c6 get 9 bit code 6: 11------ -1111111

g9c6	 PROC
	 mov	   ax,[si]   ; ax = -1234567 89------
	 xchg	   ah,al     ; ax = 89------ -1234567
	 rol	   ax,1	     ; ax = 9------- 12345678
	 rol	   ax,1	     ; ax = -------1 23456789
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c7 ; point to next code routine
	 ret
g9c6	 ENDP

; g9c7 get 9 bit code 7: 1------- 11111111

g9c7	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9-------
	 xchg	   ah,al     ; ax = 9------- 12345678
	 rol	   ax,1	     ; ax = -------1 23456789
	 and	   ah,001h   ; mask junk
	 inc	   si	     ; skip to next
	 inc	   si	     ; skip to next
	 mov	   cx,offset g9c0 ; synced, point back to code 0 routine
	 ret
g9c7	 ENDP

; ********************************************* 10 bit code extraction routines

	 PUBLIC	   _init10
_init10	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g10c0 ; initialize state to g10c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init10	 ENDP

; g10c0 get 10 bit code 0: 11111111 ------11

g10c0	 PROC
	 mov	   ax,[si]   ; code in ax
	 and	   ah,003h   ; mask off junk
	 inc	   si	     ; point to next byte
	 mov	   cx,offset g10c1 ; point to next code routine
	 ret
g10c0	 ENDP

; g10c1 get 10 bit code 1: 111111-- ----1111

g10c1	 PROC
	 mov	   ax,[si]   ; ax = ----1234 56789a--
	 shr	   ax,1	     ; ax = -----123 456789a-
	 shr	   ax,1	     ; ax = ------12 3456789a
	 and	   ah,003h   ; mask off junk
	 inc	   si	     ; point to next byte
	 mov	   cx,offset g10c2 ; point to next code routine
	 ret
g10c1	 ENDP

; g10c2 get 10 bit code 2: 1111---- --111111

g10c2	 PROC
	 mov	   ax,[si]   ; ax = --123456 789a----
	 shr	   ax,1	     ; ax = ---12345 6789a---
	 shr	   ax,1	     ; ax = ----1234 56789a--
	 shr	   ax,1	     ; ax = -----123 456789a-
	 shr	   ax,1	     ; ax = ------12 3456789a
	 and	   ah,003h   ; mask off junk
	 inc	   si	     ; point to next byte
	 mov	   cx,offset g10c3 ; point to next code routine
	 ret
g10c2	 ENDP

; g10c3 get 10 bit code 3: 11------ 11111111

g10c3	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9a------
	 xchg	   ah,al     ; ax = 9a------ 12345678
	 rol	   ax,1	     ; ax = a------1 23456789
	 rol	   ax,1	     ; ax = ------12 3456789a
	 and	   ah,003h   ; mask off junk
	 inc	   si	     ; inc past code
	 inc	   si
	 mov	   cx,offset g10c4 ; point to next code routine
	 ret
g10c3	 ENDP

; g10c4 get 10 bit code 4: 11111111 ------11

g10c4	 PROC		     ; same as c0
	 mov	   ax,[si]
	 and	   ah,003h
	 inc	   si
	 mov	   cx,offset g10c5
	 ret
g10c4	 ENDP

; g10c5 get 10 bit code 5: 111111-- ----1111

g10c5	 PROC		     ; same as c1
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 and	   ah,003h
	 inc	   si
	 mov	   cx,offset g10c6
	 ret
g10c5	 ENDP

; g10c6 get 10 bit code 6: 1111---- --111111

g10c6	 PROC		     ; same as c2
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 and	   ah,003h
	 inc	   si
	 mov	   cx,offset g10c7
	 ret
g10c6	 ENDP

; g10c7 get 10 bit code 3: 11------ 11111111

g10c7	 PROC		     ; same as c3
	 mov	   ax,[si]
	 xchg	   ah,al
	 rol	   ax,1
	 rol	   ax,1
	 and	   ah,003h
	 inc	   si
	 inc	   si
	 mov	   cx,offset g10c0 ; synced, wrap back around
	 ret
g10c7	 ENDP

; ********************************************* 11 bit code extraction routines

	 PUBLIC	   _init11
_init11	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g11c0 ; initialize state to g11c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init11	 ENDP

; g11c0 get 11 bit code 0: 11111111 -----111

g11c0	 PROC
	 mov	   ax,[si]   ; code in ax
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g11c1 ; point to next code routine
	 ret
g11c0	 ENDP

; g11c1 get 11 bit code 1: 11111--- --111111

g11c1	 PROC
	 mov	   ax,[si]   ; ax = --123456 789ab---
	 shr	   ax,1	     ; ax = ---12345 6789ab--
	 shr	   ax,1	     ; ax = ----1234 56789ab-
	 shr	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g11c2 ; point to next code routine
	 ret
g11c1	 ENDP

; g11c2 get 11 bit code 2: 11------ 11111111 -------1

g11c2	 PROC
	 mov	   ax,[si]   ; ax = 23456789 ab------
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cl,[si]   ; cl = -------1
	 rcr	   cl,1	     ; carry bit = 1
	 rcr	   ax,1	     ; ax = 12345678 9ab-----
	 xchg	   ah,al     ; ax = 9ab----- 12345678
	 rol	   ax,1	     ; ax = ab-----1 23456789
	 rol	   ax,1	     ; ax = b-----12 3456789a
	 rol	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 mov	   cx,offset g11c3 ; point to next code routine
	 ret
g11c2	 ENDP

; g11c3 get 11 bit code 3: 1111111- ----1111

g11c3	 PROC
	 mov	   ax,[si]   ; ax = ----1234 56789ab-
	 shr	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; next byte
	 mov	   cx,offset g11c4 ; point to next code routine
	 ret
g11c3	 ENDP

; g11c4 get 11 bit code 4: 1111---- -1111111

g11c4	 PROC
	 mov	   ax,[si]   ; ax = -1234567 89ab----
	 shr	   ax,1	     ; ax = --123456 789ab---
	 shr	   ax,1	     ; ax = ---12345 6789ab--
	 shr	   ax,1	     ; ax = ----1234 56789ab-
	 shr	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; next byte
	 mov	   cx,offset g11c5 ; point to next code routine
	 ret
g11c4	 ENDP

; g11c5 get 11 bit code 5: 1------- 11111111 ------11

g11c5	 PROC
	 mov	   ax,[si]   ; ax = 3456789a b-------
	 and	   al,080h   ; mask off junk
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cl,[si]   ; cl = ------12
	 and	   cl,003h   ; mask off junk
	 or	   al,cl     ; ax = 3456789a b-----12
	 rol	   ax,1	     ; ax = 456789ab -----123
	 xchg	   ah,al     ; ax = -----123 456789ab
	 mov	   cx,offset g11c6 ; point to next code routine
	 ret
g11c5	 ENDP

; g11c6 get 11 bit code 6: 111111-- ---11111

g11c6	 PROC
	 mov	   ax,[si]   ; ax = ---12345 6789ab--
	 shr	   ax,1	     ; ax = ----1234 56789ab-
	 shr	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g11c7 ; point to next code routine
	 ret
g11c6	 ENDP

; g11c7 get 11 bit code 7: 111----- 11111111

g11c7	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9ab-----
	 xchg	   ah,al     ; ax = 9ab----- 12345678
	 rol	   ax,1	     ; ax = ab-----1 23456789
	 rol	   ax,1	     ; ax = b-----12 3456789a
	 rol	   ax,1	     ; ax = -----123 456789ab
	 and	   ah,007h   ; mask off junk
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cx,offset g11c0 ; synced, wrap around
	 ret
g11c7	 ENDP

; ********************************************* 12 bit code extraction routines

	 PUBLIC	   _init12
_init12	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g12c0 ; initialize state to g12c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init12	 ENDP

; g12c0 get 12 bit code 0: 11111111 ----1111

g12c0	 PROC
	 mov	   ax,[si]   ; code in ax
	 and	   ah,00fh   ; mask off junk
	 inc	   si	     ; next byte
	 mov	   cx,offset g12c1 ; point to next routine
	 ret
g12c0	 ENDP

; g12c1 get 12 bit code 1: 1111---- 11111111

g12c1	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9abc----
	 shr	   ax,1	     ; ax = -1234567 89abc---
	 shr	   ax,1	     ; ax = --123456 789abc--
	 shr	   ax,1	     ; ax = ---12345 6789abc-
	 shr	   ax,1	     ; ax = ----1234 56789abc
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cx,offset g12c2 ; point to next routine
	 ret
g12c1	 ENDP

; g12c2 get 12 bit code 2: 11111111 ----1111

g12c2	 PROC		     ; same as 0
	 mov	   ax,[si]
	 and	   ah,00fh
	 inc	   si
	 mov	   cx,offset g12c3
	 ret
g12c2	 ENDP

; g12c3 get 12 bit code 3: 1111---- 11111111

g12c3	 PROC		     ; same as 1
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 inc	   si
	 inc	   si
	 mov	   cx,offset g12c4
	 ret
g12c3	 ENDP

; g12c4 get 12 bit code 4: 11111111 ----1111

g12c4	 PROC		     ; same as 0
	 mov	   ax,[si]
	 and	   ah,00fh
	 inc	   si
	 mov	   cx,offset g12c5
	 ret
g12c4	 ENDP

; g12c5 get 12 bit code 5: 1111---- 11111111

g12c5	 PROC		     ; same as 1
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 inc	   si
	 inc	   si
	 mov	   cx,offset g12c6
	 ret
g12c5	 ENDP

; g12c6 get 12 bit code 6: 11111111 ----1111

g12c6	 PROC		     ; same as 0
	 mov	   ax,[si]
	 and	   ah,00fh
	 inc	   si
	 mov	   cx,offset g12c7
	 ret
g12c6	 ENDP

; g12c7 get 12 bit code 7: 1111---- 11111111

g12c7	 PROC		     ; same as 1
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 inc	   si
	 inc	   si
	 mov	   cx,offset g12c0 ; synced, wrap around
	 ret
g12c7	 ENDP

; ********************************************* 13 bit code extraction routines

	 PUBLIC	   _init13
_init13	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g13c0 ; initialize state to g13c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init13	 ENDP

; g13c0 get 13 bit code 0: 11111111 ---11111

g13c0	 PROC
	 mov	   ax,[si]   ; ax = 12345 6789abcd
	 and	   ah,01fh   ; mask off junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g13c1 ; point to next routine
	 ret
g13c0	 ENDP

; g13c1 get 13 bit code 1: 111----- 11111111 ------11

g13c1	 PROC
	 mov	   ax,[si]   ; ax = 3456789a bcd-----
	 and	   al,0e0h   ; mask off junk
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cl,[si]   ; cl = ------12
	 and	   cl,003h   ; mask off junk
	 or	   al,cl     ; ax = 3456789a bcd---12
	 rol	   ax,1	     ; ax = 456789ab cd---123
	 rol	   ax,1	     ; ax = 56789abc d---1234
	 rol	   ax,1	     ; ax = 6789abcd ---12345
	 xchg	   ah,al     ; ax = ---12345 6789abcd
	 mov	   cx,offset g13c2 ; point to next routine
	 ret
g13c1	 ENDP

; g13c2 get 13 bit code 2: 111111-- -1111111

g13c2	 PROC
	 mov	   ax,[si]   ; ax = -1234567 89abcd--
	 shr	   ax,1	     ; ax = --123456 789abcd-
	 shr	   ax,1	     ; ax = ---12345 6789abcd
	 and	   ah,01fh   ; mask junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g13c3 ; point to next routine
	 ret
g13c2	 ENDP

; g13c3 get 13 bit code 3: 1------- 11111111 ----1111

g13c3	 PROC
	 mov	   ax,[si]   ; ax = 56789abc d-------
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cl,[si]   ; cl = ----1234
	 and	   cl,00fh   ; mask junk
	 and	   al,080h   ; mask junk
	 or	   al,cl     ; ax = 56789abc d---1234
	 rol	   ax,1	     ; ax = 6789abcd ---12345
	 xchg	   ah,al     ; ax = ---12345 6789abcd
	 mov	   cx,offset g13c4 ; point to next routine
	 ret
g13c3	 ENDP

; g13c4 get 13 bit code 4: 1111---- 11111111 -------1

g13c4	 PROC
	 mov	   ax,[si]   ; ax = 23456789 abcd----
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cl,[si]   ; cl = ------1
	 rcr	   cl,1	     ; carry bit = 1
	 rcr	   ax,1	     ; ax = 12345678 9abcd---
	 shr	   ax,1	     ; ax = -1234567 89abcd--
	 shr	   ax,1	     ; ax = --123456 789abcd-
	 shr	   ax,1	     ; ax = ---12345 6789abcd
	 mov	   cx,offset g13c5 ; point to next routine
	 ret
g13c4	 ENDP

; g13c5 get 13 bit code 5: 1111111- --111111

g13c5	 PROC
	 mov	   ax,[si]   ; ax = --123456 789abcd-
	 shr	   ax,1	     ; ax = ---12345 6789abcd
	 and	   ah,01fh   ; mask junk
	 inc	   si
	 mov	   cx,offset g13c6 ; point to next routine
	 ret
g13c5	 ENDP

; g13c6 get 13 bit code 6: 11------ 11111111 -----111

g13c6	 PROC
	 mov	   ax,[si]   ; ax = 456789ab cd------
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = -----123
	 and	   al,0c0h   ; mask junk
	 and	   cl,007h   ; mask junk
	 or	   al,cl     ; ax = 456789ab cd---123
	 rol	   ax,1	     ; ax = 56789abc d---1234
	 rol	   ax,1	     ; ax = 6789abcd ---12345
	 xchg	   ah,al     ; ax = ---12345 6789abcd
	 mov	   cx,offset g13c7 ; point to next routine
	 ret
g13c6	 ENDP

; g13c7 get 13 bit code 7: 11111--- 11111111

g13c7	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9abcd---
	 shr	   ax,1	     ; ax = -1234567 89abcd--
	 shr	   ax,1	     ; ax = --123456 789abcd-
	 shr	   ax,1	     ; ax = ---12345 6789abcd
	 inc	   si	     ; bump pointer
	 inc	   si
	 mov	   cx,offset g13c0 ; synced, wrap around
	 ret
g13c7	 ENDP

; ********************************************* 14 bit code extraction routines

	 PUBLIC	   _init14
_init14	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g14c0 ; initialize state to g14c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init14	 ENDP

; g14c0 get 14 bit code 0: 11111111 --111111

g14c0	 PROC
	 mov	   ax,[si]   ; ax = --123456 789abcde
	 and	   ah,03fh   ; mask junk
	 inc	   si	     ; bump pointer
	 mov	   cx,offset g14c1 ; point to next routine
	 ret
g14c0	 ENDP

; g14c1 get 14 bit code 1: 11------ 11111111 ----1111

g14c1	 PROC
	 mov	   ax,[si]   ; ax = 56789abc de------
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = ----1234
	 and	   al,0c0h   ; mask junk
	 and	   cl,00fh   ; mask junk
	 or	   al,cl     ; ax = 56789abc de--1234
	 rol	   ax,1	     ; ax = 6789abcd e--12345
	 rol	   ax,1	     ; ax = 789abcde --123456
	 xchg	   ah,al     ; ax = --123456 789abcde
	 mov	   cx,offset g14c2 ; point to next routine
	 ret
g14c1	 ENDP

; g14c2 get 14 bit code 2: 1111---- 11111111 ------11

g14c2	 PROC
	 mov	   ax,[si]   ; ax = 3456789a bcde----
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = ------12
	 and	   al,0f0h   ; mask junk
	 and	   cl,003h   ; mask junk
	 or	   al,cl     ; ax = 3456789a bcde--12
	 ror	   ax,1	     ; ax = 23456789 abcde--1
	 ror	   ax,1	     ; ax = 12345678 9abcde--
	 shr	   ax,1	     ; ax = -1234567 89abcde-
	 shr	   ax,1	     ; ax = --123456 789abcde
	 mov	   cx,offset g14c3 ; point to next routine
	 ret
g14c2	 ENDP

; g14c3 get 14 bit code 3: 111111-- 11111111

g14c3	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9abcde--
	 shr	   ax,1	     ; ax = -1234567 89abcde-
	 shr	   ax,1	     ; ax = --123456 789abcde
	 inc	   si
	 inc	   si
	 mov	   cx,offset g14c4 ; point to next routine
	 ret
g14c3	 ENDP

; g14c4 get 14 bit code 4: 11111111 --111111

g14c4	 PROC		     ; same as 0
	 mov	   ax,[si]
	 and	   ah,03fh
	 inc	   si
	 mov	   cx,offset g14c5
	 ret
g14c4	 ENDP

; g14c5 get 14 bit code 5: 11------ 11111111 ----1111

g14c5	 PROC		     ; same as 1
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cl,[si]
	 and	   al,0c0h
	 and	   cl,00fh
	 or	   al,cl
	 rol	   ax,1
	 rol	   ax,1
	 xchg	   ah,al
	 mov	   cx,offset g14c6
	 ret
g14c5	 ENDP

; g14c6 get 14 bit code 6: 1111---- 11111111 ------11

g14c6	 PROC		     ; same as 2
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cl,[si]
	 and	   al,0f0h
	 and	   cl,003h
	 or	   al,cl
	 ror	   ax,1
	 ror	   ax,1
	 shr	   ax,1
	 shr	   ax,1
	 mov	   cx,offset g14c7
	 ret
g14c6	 ENDP

; g14c7 get 14 bit code 7: 111111-- 11111111

g14c7	 PROC		     ; same as 3
	 mov	   ax,[si]
	 shr	   ax,1
	 shr	   ax,1
	 inc	   si
	 inc	   si
	 mov	   cx,offset g14c0 ; synced, wrap around
	 ret
g14c7	 ENDP

; ********************************************* 15 bit code extraction routines

	 PUBLIC	   _init15
_init15	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g15c0 ; initialize state to g15c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init15	 ENDP

; g15c0 get 15 bit code 0: 11111111 -1111111

g15c0	 PROC
	 mov	   ax,[si]   ; ax = -1234567 89abcdef
	 and	   ah,07fh   ; mask junk
	 inc	   si
	 mov	   cx,offset g15c1 ; point to next routine
	 ret
g15c0	 ENDP

; g15c1 get 15 bit code 1: 1------- 11111111 --111111

g15c1	 PROC
	 mov	   ax,[si]   ; ax = 789abcde f-------
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; ax = --123456
	 and	   al,080h   ; mask junk
	 and	   cl,03fh   ; mask junk
	 or	   al,cl     ; ax = 789abcde f-123456
	 rol	   ax,1	     ; ax = 89abcdef -1234567
	 xchg	   ah,al     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c2 ; point to next routine
	 ret
g15c1	 ENDP

; g15c2 get 15 bit code 2: 11------ 11111111 ---11111

g15c2	 PROC
	 mov	   ax,[si]   ; ax = 6789abcd ef------
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = ---12345
	 and	   al,0c0h   ; mask junk
	 and	   cl,01fh   ; mask junk
	 or	   al,cl     ; ax = 6789abcd ef-12345
	 rol	   ax,1	     ; ax = 789abcde f-123456
	 rol	   ax,1	     ; ax = 89abcdef -1234567
	 xchg	   ah,al     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c3 ; point to next routine
	 ret
g15c2	 ENDP

; g15c3 get 15 bit code 3: 111----- 11111111 ----1111

g15c3	 PROC
	 mov	   ax,[si]   ; ax = 56789abc def-----
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = ----1234
	 and	   al,0e0h
	 and	   cl,00fh
	 or	   al,cl     ; ax = 56789abc def-1234
	 rol	   ax,1	     ; ax = 6789abcd ef-12345
	 rol	   ax,1	     ; ax = 789abcde f-123456
	 rol	   ax,1	     ; ax = 89abcdef -1234567
	 xchg	   ah,al     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c4 ; point to next routine
	 ret
g15c3	 ENDP

; g15c4 get 15 bit code 4: 1111---- 11111111 -----111

g15c4	 PROC
	 mov	   ax,[si]   ; ax = 456789ab cdef----
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = -----123
	 and	   cl,007h
	 and	   al,0f0h
	 or	   al,cl     ; ax = 456789ab cdef-123
	 ror	   ax,1	     ; ax = 3456789a bcdef-12
	 ror	   ax,1	     ; ax = 23456789 abcdef-1
	 ror	   ax,1	     ; ax = 12345678 9abcdef-
	 shr	   ax,1	     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c5 ; point to next routine
	 ret
g15c4	 ENDP

; g15c5 get 15 bit code 5: 11111--- 11111111 ------11

g15c5	 PROC
	 mov	   ax,[si]   ; ax = 3456789a bcdef---
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = ------12
	 and	   al,0f8h
	 and	   cl,003h
	 or	   al,cl     ; ax = 3456789a bcdef-12
	 ror	   ax,1	     ; ax = 23456789 abcdef-1
	 ror	   ax,1	     ; ax = 12345678 9abcdef-
	 shr	   ax,1	     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c6 ; point to next routine
	 ret
g15c5	 ENDP

; g15c6 get 15 bit code 6: 111111-- 11111111 -------1

g15c6	 PROC
	 mov	   ax,[si]   ; ax = 23456789 abcdef--
	 inc	   si
	 inc	   si
	 mov	   cl,[si]   ; cl = -------1
	 rcr	   cl,1	     ; carry bit = 1
	 rcr	   ax,1	     ; ax = 12345678 9abcdef-
	 shr	   ax,1	     ; ax = -1234567 89abcdef
	 mov	   cx,offset g15c7 ; point to next routine
	 ret
g15c6	 ENDP

; g15c7 get 15 bit code 7: 1111111- 11111111

g15c7	 PROC
	 mov	   ax,[si]   ; ax = 12345678 9abcdef-
	 shr	   ax,1	     ; ax = -1234567 89abcdef
	 inc	   si
	 inc	   si
	 mov	   cx,offset g15c0 ; synced, wrap around
	 ret
g15c7	 ENDP

; ********************************************** 16 bit code extraction routine

	 PUBLIC	   _init16
_init16	 PROC
	 push	   bp
	 mov	   bp,sp
	 mov	   bx,[bp+4] ; bx points to codebuf struct
	 mov	   cx,offset g16c0 ; initialize state to g16c0
	 mov	   [bx],cx   ; save new state routine
	 pop	   bp
	 ret
_init16	 ENDP

; g16c0 get 16 bit code 0: 11111111 11111111

g16c0	 PROC
	 mov	   ax,[si]   ; get low & high bytes
	 inc	   si	     ; skip past them
	 inc	   si
	 mov	   cx,offset g16c1 ; next routine
	 ret
g16c0	 ENDP

g16c1	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c2
	 ret
g16c1	 ENDP

g16c2	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c3
	 ret
g16c2	 ENDP

g16c3	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c4
	 ret
g16c3	 ENDP

g16c4	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c5
	 ret
g16c4	 ENDP

g16c5	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c6
	 ret
g16c5	 ENDP

g16c6	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c7
	 ret
g16c6	 ENDP

g16c7	 PROC		     ; same as 0
	 mov	   ax,[si]
	 inc	   si
	 inc	   si
	 mov	   cx,offset g16c0 ; wrap back around
	 ret
g16c7	 ENDP

	 END
