/**************************************
 * calendar.h
 * A collection of functions for calendrical computations v. 0
 * Borland C 1.5, 2.0, 3.0 and Power C. ISO/ANSI C.
 * By Joao C. de Magalhaes, 1994-06-08
 */

#ifndef _math.h_
#include <math.h>
#endif

#define _CALENDAR_ 1


/**************************************
* Offsets for Julian day number JDN,
* gregorian system origin GREG,
* modified Julian day number MJDN and
* J2000 day number J2000DN
*/

/** JDN:        day 0: Monday -4713 (4714 BC) November 24 at 00:00 */
/**                    Astronomer's count begins at noon           */
#define         JDN             0L

/** GREG        day 0: (Gregorian) Friday 1582 October (16-1)      */
/**                    Not an astronomer's counting origin         */
#define         GREG            2299161L

/** MJDN:       day 0: Tuesday 1858 November 16 at 00:00           */
/**                    Astronomer's count also begins at 00:00 GMT */
#define         MJDN            2400000L

/** J2000DN:    day 0: Saturday 2000 January 1 at 00:00       */
/**                    Astronomer's count begins at noon UTC  */
#define         J2000DN         2451545L

/**************************************
 * structure to pass date results
 */
struct cal_date
  {
  int year, month, day;
  };


#ifndef  CALENDAR_ORIGIN
#define  CALENDAR_ORIGIN        JDN
#endif

/**************************************
 * Is Y a Gregorian leap year?
 */
#define leapyear(Y)     ( !(Y %4) && (Y %100) || !(Y %400) )


/**************************************
 * Is Date a valid Gregorian date format?
 */
int vdate(int Year, int Month, int Day)
   {
   int Days[] = { 0,
		  31, 28, 31, 30, 31, 30,
		  31, 31, 30, 31, 30, 31};

   if ((Month < 1) || (Day < 1))
      return 0;
   if ((Month > 12) || (Day > 31))
      return 0;

   if (Month != 2)
      if (Day > Days[Month])
	 return 0;
      else return 1;
   else
      if (leapyear(Year))
	 if (Day > 29)
	    return 0;
	 else return 1;
      else
	 if (Day > 28)
	    return 0;
	 else return 1;
   }


/**************************************
 * Is Date a valid Julian date format?
 */
int vjdate(int Year, int Month, int Day)
   {
   int Days[] = { 31,
		  31, 28, 31, 30, 31, 30,
		  31, 31, 30, 31, 30, 31};

   if ((Month < 1) || (Day < 1))
      return 0;
   if ((Month > 12) || (Day > 31))
      return 0;

   if (Month != 2)
      if (Day > Days[Month])
	 return 0;
      else return 1;
   else
      if (!(Year %4))
	 if (Day > 29)
	    return 0;
	 else return 1;
      else
	 if (Day > 28)
	    return 0;
	 else return 1;
   }


/**************************************
 * Date To Day Number
 * Modified from Fliegel and van Flandern (1968)
 * as in: Explanatory Supplement to the Astronomical Almanac
 * University Science Books, Mill Valley, 1992
 * ISBN 0-935702-68-7. Chapter 12, Section 92
 */
long datetodn(int Year, int Month, int Day)
  {
  long A, B;

  A = (Year+4800);
  B = (Month-14) /12;
  return ((1461 * (A + B)) >> 2)
	 +((367 *(Month -2 -12*B)) /12 )
	 -(3 *((A +100 + B) /100) >> 2)
	 +Day - 32075L -CALENDAR_ORIGIN ;
  }


/**************************************
 * Day Number to Date
 * Modified from Fliegel and van Flandern (1968)
 * as in: Explanatory Supplement to the Astronomical Almanac
 * University Science Books, Mill Valley, 1992
 * ISBN 0-935702-68-7. Chapter 12, Section 92
 */
struct cal_date dntodate(long DN)
  {
  long A, B, C, D, Day, Month, Year= 0;
  struct cal_date Result;

  A= DN + 68569L + CALENDAR_ORIGIN;
  B= (A << 2 ) /146097L;
  A -= ((146097L *B +3) >> 2);
  C= (4000 *(A +1)) /1461001L;
  A -= ((1461 *C) >> 2);
  A +=31;
  D= (80 *A)/2447;
  Day= A- (2447 *D) /80;
  A= D/11;
  /* if above D<11, A is set to 0 causing divide error: */
	if (A>0) Month= D +2 -(12 / A);
	else Month= D +2;

  Year= 100 *(B -49) +C +A;

  Result.year   = (int) Year;
  Result.month  = (int) Month;
  Result.day    = (int) Day;

  return Result;
  }


/**************************************
 * Days between dates
 */
long datediff(int Y1, int M1, int D1,
	      int Y2, int M2, int D2)
   {
   return (datetodn(Y2, M2, D2) - datetodn(Y1, M1, D1));
   }


/**************************************
 * New date given starting date and number of days
 */
struct cal_date newdate(int Y, int M, int D, long Dif)
   {
   return dntodate(datetodn(Y, M, D) +Dif);
   }


/**************************************
 * Day Number To Week Day (Monday =0)
 */
int dntowd(long DN)
   {
   int Temp;

   Temp = (int) ((DN %7 + (CALENDAR_ORIGIN) %7 ) %7);
   if (Temp < 0) Temp += 7;

   return Temp;
   }


/**************************************
 * ISO date format
 */
long ISOdate(int Year, int Month, int Day)
   {
   long Temp = Year;
   if (Year < 0) Temp *= (-10000);
   else Temp *= 10000;
   Temp += Month * 100;
   Temp += Day;
   if (Year < 0) Temp *= (-1);
   return Temp;
   }


/**************************************
 * Gauss' Easter Sunday - Modification from COBOL source
 * as reported by Constance Veening, Almere, The Netherlands
 */
int easter(int Year)
    {
    int HundredY,
    Month,
    Day,
    A, B, C, D;

    /* Check for domain error */
    if ((Year < 1583) || (Year > 2199))
	       return 0;                /* exit with error */

    HundredY = Year / 100;

    switch (HundredY) {
      case 15 : A = 22; B = 2; break;
      case 16 : A = 22; B = 2; break;
      case 17 : A = 23; B = 3; break;
      case 18 : A = 23; B = 4; break;
      case 19 : A = 24; B = 5; break;
      case 20 : A = 24; B = 5; break;
      case 21 : A = 24; B = 6; break;
      }

    C = (19 * (Year % 19) + A) % 30;
    D = (2 * (Year % 4) + 4 * (Year % 7) + 6 * C + B) % 7;

    Day = 22 + C + D;

    if (Day > 31)
       { Day = C + D -9; Month = 4; }
    else Month = 3;

    if ((Month == 4) && (Day == 26))
       { Day = 19; Month = 4; }

    if ((C == 28) && ((Year % 19) > 10)
       && (Month == 4) && (Day == 25))
		{ Day = 18; Month = 4; }

    return 100 * Month + Day;
    }


/******************
 * Oudin's Easter
 * Modified from Oudin J.-M.: "Etude sur la Date de Paques",
 * Bulletin Astronomique, (1940) 2-12:391-410
 * as in: Explanatory Supplement to the Astronomical Almanac
 * University Science Books, Mill Valley, 1992
 * ISBN 0-935702-68-7. Chapter 12, Section 22
 */
int oeaster(int Year)
   {
   long A, B, C, D, E, F, Month, Day;

   if (Year < 1583) return 0;   /* domain error */

   A = Year / 100;
   B = Year - 19 * (Year / 19);
   C = (A -17) /25;
   D = A -A/4 -(A-C)/3 +19*B +15;
   D = D -30*(D/30);
   D = D -(D/28)*(1-(D/28)*(29/(D+1))*((21-B)/11));
   E = Year +Year/4 +D +2 -A +A/4;
   E = E -7*(E/7);
   F = D -E;
   Month = 3 +(F +40) /44;
   Day   = F +28 -31*(Month/4);

   return (int) (Month*100 +Day);
   }



/******************
 * MardiGras
 */
int mardigras(int Y)
   {
   long Temp;
   struct cal_date D;

   if ((Y<1583) || (Y> 2199)) return 0;
   Temp = easter(Y);
   Temp = datetodn(Y, (int) Temp / 100, (int) Temp % 100) -47;
   D    = dntodate(Temp);
   Temp = D.month * 100 + D.day;

   return (int) Temp;
   }


/******************
 * Corpo de Deus, Portugal (France: Fte-Dieu, Sunday)
 * mobile holiday on a Thursday
 */
int corpodedeus(int Y)
   {
   long Temp;
   struct cal_date D;

   if ((Y<1583) || (Y> 2199)) return 0;
   Temp = easter(Y);
   Temp = datetodn(Y, (int) Temp / 100, (int) Temp % 100) +60;
   D    = dntodate(Temp);
   Temp = D.month * 100 + D.day;

   return (int) Temp;
   }


/******************
 * Month Names
 */
char * mname(int Locus, int N, char * S)
   {
   char Names[][13][15] = {
   /* 0= Cantonese */    {"Iat Yut", "Yee Yut", "Sam Yut", "Sei Yut",
			  "Ng Yut", "Loc Yut", "Chet Yut", "Pat Yut",
			  "Kau Yut", "Sap Yut", "Sap Iat Yut",
			  "Sap Yee Yut"},
   /* 1= English */      {"January", "February", "March",
			  "April", "May", "June", "July", "August",
			  "September", "October", "November", "December"},
   /* 2= Spanish */      {"Enero", "Febrero", "Marzo",
			  "Abril", "Mayo", "Junio", "Julio", "Agosto",
			  "Septiembre", "Octubre", "Noviembre", "Diciembre"},
   /* 3= Portuguese */   {"Janeiro", "Fevereiro", "Maro",
			  "Abril", "Maio", "Junho", "Julho", "Agosto",
			  "Setembro", "Outubro", "Novembro", "Dezembro"},
   /* 4= French  */      {"Janvier", "Fvrier", "Mars",
			  "Avril", "Mai", "Juin", "Julliet", "Aout",
			  "Septembre", "Octobre", "Novembre", "Dcembre"},
   /* 5= German  */      {"Januar", "Februar", "Marz",
			  "April", "Mai", "Juni", "Juli", "August",
			  "September", "Oktober", "November", "Dezember"},
   /* 6= Italian */      {"Gennaio", "Febbraio", "Marzo",
			  "Aprile", "Maggio", "Giugno", "Luglio", "Agosto",
			  "Settembre", "Ottobre", "Novembre", "Dicembre"},
			  };
   char * P = Names[0][0];
   char * Q = S;

   N --;
   if ((Locus>6) || (Locus<0)) Locus = 0;
   if ((N<0) || (N>12))        N = 0;
   P = Names[Locus][N];
   while(*P)  *S++ = *P++;
   *S++ = (char) 0x00;
   return (char *) Q;
   }


/******************
 * Week Day Names
 */
char * wdname(int Locus, int N, char * S)
   {
   char Names[][7][15] ={
   /* 0= Cantonese */   {"Sen Kei Iut", "Sen Kei Yee", "Sen Kei Sam",
			 "Sen Kei Sei", "Sen Kei Ng", "Sen Kei Lok",
			 "Sen Kei Iat"},
   /* 1= English */     {"Monday", "Tuesday", "Wednesday", "Thursday",
			 "Friday", "Saturday", "Sunday"},
   /* 2= Spanish */     {"Lunes", "Martes", "Mircoles", "Jueves",
			 "Viernes", "Sbado", "Domingo"},
   /* 3= Portuguese */  {"Segunda-feira", "Tera-feira", "Quarta-feira",
			 "Quinta-feira", "Sexta-feira", "Sbado",
			 "Domingo"},
   /* 4= French */      {"Lundi", "Mardi", "Mercredi", "Jeudi",
			 "Vendredi", "Samedi", "Dimanche"},
   /* 5= German */      {"Montag", "Dienstag", "Mittwoch", "Donnerstag",
			 "Freitag", "Samstag", "Sonntag"},
   /* 6= Italian */     {"Lunedi", "Martedi", "Mercoledi", "Giovedi",
			 "Venerdi", "Sabato", "Domenica"},
			 };
   char * P = Names[0][0];
   char * Q = S;

   if ((Locus>6) || (Locus<0)) Locus = 0;
   if ((N<0) || (N>6))         N = 0;
   P = Names[Locus][N];
   while(*P)  *S++ = *P++;
   *S++ = (char) 0x00;
   return (char *) Q;
   }
