/*

Micro BASIC Interpeter (support.c)

All the internal support functions used by the interpreter are declared
in this module. Including the list structures routines used to handle
arrays and DATA/READ data.

*** Copyleft - Andre Murta - August 1992/December 2002 ***

Adapted from Herbert Schildt Little C (Dr. Dobbs Journal, August 1989)

*/

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <getargs.h>
#include <stdarg.h>
#include <conio.h>
#include "mbasic.h"
#include "structures.h"

/* user defined functions */
extern struct user_function user_func_table[NUM_FUNCS];

/* array support functions*/
struct node * InsertAtEnd(struct node newVal, struct node **array_items);
void deleteNodeValue(struct node delVal, short dim, struct node ** array_items);
struct node * search(struct node data, short dim, struct node *pstack);
struct node * getMemory(void);

/* DATA & READ support functions */
struct data_read * InsertData(struct data_read newVal, struct data_read **items);
void deleteItemAtPos(int pos, struct data_read **items);
struct data_read * returnItemAtPos(int pos, struct data_read *items);
struct data_read * getDataMemory(void);

/* string memory functions */
char *getmem(int size, unsigned short clean);
char *allocateString(char *s);



/* Dynamic list in memory functions - these are used to support array operations in MBasic */
/* ----------------------------------------------------------------------- */
/* BEGIN array support functions */
/* ----------------------------------------------------------------------- */
/*
    This function will insert a node at the end of the dynamic list. The node is an array
    node and keeps inside its structure, the information about how to recover this information
    (array indexes, associated var index).
*/
struct node * InsertAtEnd(struct node newVal, struct node **array_items)
{
    struct node *newNode;
    struct node *ptemp;

    ptemp = *array_items;
    newNode = getMemory(); /* call getMemory to allocate and test */

    newNode->var_index = newVal.var_index;
    newNode->type = newVal.type;
    newNode->index.x = newVal.index.x;
    newNode->index.y = newVal.index.y;
    newNode->index.z = newVal.index.z;
    newNode->type = newVal.type;
    if(newVal.type == NUMERIC_VAR)
        newNode->value.dv = newVal.value.dv;
    else
        strcpy(newNode->value.sv, newVal.value.sv);

    while(ptemp != NULL && ptemp->next != NULL)
        ptemp = ptemp->next;
    
    if(ptemp == NULL) /* if list is empty */
    {
        newNode->next = NULL;
        newNode->previous = NULL;
        *array_items = newNode;
    }
    else /* Insert at end of list */
    {
      newNode->previous = ptemp;
      newNode->next = ptemp->next;
      ptemp->next = newNode;
    }
    
    return(*array_items);   /* pass back list pointer */
}

/*
    This function is used internally by the interpreter to replace an array item value.
    Basically, this function removes the old node that should be replaced by the new one.
*/
void deleteNodeValue(struct node delVal, short dim, struct node ** array_items)  /* deletes node from list */
{
    struct node *temp = NULL;
    struct node *next1 = NULL;

    next1 = *array_items;     /* set pointer to list head */
    while(next1 != NULL)     /* search through whole list */
    {
        /* The test vary according to the array dimension */
        if(dim == 1) {
            if(next1->var_index == delVal.var_index && next1->index.x == delVal.index.x) /* if delVal is found */
            {
                /* if deleting only node */
                if(next1->next == NULL && next1->previous == NULL)
                {   /* deleting only node.  Point array_items to NULL */
                    free(next1);				
                    *array_items = NULL;
                }
                /* if deleting first node */
                else if(next1->previous == NULL)
                {
                    /* skip next1 to next node */
                    next1 = next1->next;
                    /* since first node assign previous to NULL */
                    next1->previous = NULL;
                    /* point the list pointer to the new first node */
                    *array_items = next1;
                    /* free old first node */
                    free(temp);
                    /* point temp to list */
                    temp = *array_items;
                }
                /* if deleting last node */
                else if(next1->next == NULL)
                {
                    /* set temp as last node */
                    temp->next = NULL;
                    /* free last node in list */
                    free(next1);
                    /* point next1 to NULL; */
                    next1 = temp->next;
                }
                else
                {
                    /* shift pointer to skip over node to be deleted */
                    temp->next = next1->next;
                    /* point previous element of 1st node after deleted node to temp */
                    (next1->next)->previous = temp;
                    free(next1);     /* free node */
                    next1 = temp->next;     /* point next1 to next node */
                }
            }
            else
            {
                temp = next1;
                /* go through list */
                next1 = next1->next;
            }
        } else if(dim == 2) {
            if(next1->var_index == delVal.var_index && next1->index.x == delVal.index.x &&
                                                       next1->index.y == delVal.index.y)     /* if delVal is found */
            {
                /* if deleting only node */
                if(next1->next == NULL && next1->previous == NULL)
                {   /* deleting only node.  Point array_items to NULL */
                    free(next1);				
                    *array_items = NULL;
                }
                /* if deleting first node */
                else if(next1->previous == NULL)
                {
                    /* skip next1 to next node */
                    next1 = next1->next;
                    /* since first node assign previous to NULL */
                    next1->previous = NULL;
                    /* point the list pointer to the new first node */
                    *array_items = next1;
                    /* free old first node */
                    free(temp);
                    /* point temp to list */
                    temp = *array_items;
                }
                /* if deleting last node */
                else if(next1->next == NULL)
                {
                    /* set temp as last node */
                    temp->next = NULL;
                    /* free last node in list */
                    free(next1);
                    /* point next1 to NULL; */
                    next1 = temp->next;
                }
                else     /* else deleting a node between nodes */
                {
                    /* shift pointer to skip over node to be deleted */
                    temp->next = next1->next;
                    /* point previous element of 1st node after deleted node to temp */
                    (next1->next)->previous = temp;
                    free(next1);     /* free node */
                    next1 = temp->next;     /* point next1 to next node */
                }
            }
            else
            {
                temp = next1;
                /* go through list */
                next1 = next1->next;
            }   /* end else */
        } else if(dim == 3) {
            if(next1->var_index == delVal.var_index && next1->index.x == delVal.index.x &&
                                                       next1->index.y == delVal.index.y &&
                                                       next1->index.z == delVal.index.z)     /* if delVal is found */
            {
                /* if deleting only node */
                if(next1->next == NULL && next1->previous == NULL)
                {   /* deleting only node.  Point array_items to NULL */
                    free(next1);				
                    *array_items = NULL;
                }
                /* if deleting first node */
                else if(next1->previous == NULL)
                {
                    /* skip next1 to next node */
                    next1 = next1->next;
                    /* since first node assign previous to NULL */
                    next1->previous = NULL;
                    /* point the list pointer to the new first node */
                    *array_items = next1;
                    /* free old first node */
                    free(temp);
                    /* point temp to list */
                    temp = *array_items;
                }
                /* if deleting last node */
                else if(next1->next == NULL)
                {
                    /* set temp as last node */
                    temp->next = NULL;
                    /* free last node in list */
                    free(next1);
                    /* point next1 to NULL; */
                    next1 = temp->next;
                }
                else     /* else deleting a node between nodes */
                {
                    /* shift pointer to skip over node to be deleted */
                    temp->next = next1->next;
                    /* point previous element of 1st node after deleted node to temp */
                    (next1->next)->previous = temp;
                    free(next1);     /* free node */
                    next1 = temp->next;     /* point next1 to next node */
                }   /* end else */
            }   /* end if */
            else
            {
                temp = next1;
                /* go through list */
                next1 = next1->next;
            }   /* end else */
        }
    }   /* end while */
}

/*
    This function searches for an item inside the array "list". The search is based on the
    item indexes and the associated var index.
*/
struct node * search(struct node data, short dim, struct node *pstack)
{
    struct node *ptemp=NULL;

    ptemp = pstack;
    if(ptemp != NULL) {
        while(ptemp != NULL) {
            if(dim == 1) {
                if(ptemp->var_index == data.var_index && ptemp->index.x == data.index.x)
                    return ptemp;
            } else if(dim == 2) {
                if(ptemp->var_index == data.var_index && ptemp->index.x == data.index.x &&
                                                         ptemp->index.y == data.index.y)
                    return ptemp;
            } else if(dim == 3) {
                if(ptemp->var_index == data.var_index && ptemp->index.x == data.index.x &&
                                                         ptemp->index.y == data.index.y &&
                                                         ptemp->index.z == data.index.z)
                    return ptemp;
            }

            ptemp = ptemp->next;
        }
    }

    return NULL;
}

/*
    This function returns the exact ammount of memory necessary to store an array item.
*/
struct node * getMemory(void)     /* allocates & tests return value  */
{
   struct node * newPtr = NULL;

   if((newPtr = (struct node *) malloc(sizeof(struct node))) == NULL)
   {     /* if malloc returns a NULL pointer */
      printf("ERROR!  Unable to allocate memory - Abort\n");
      abort();     /* print error msg & abort function */
      return(0);   /* make the compiler happy :>) */
   } 
   else return newPtr;
}
/* ----------------------------------------------------------------------- */
/* END array support functions */
/* ----------------------------------------------------------------------- */



/* Dynamic list in memory functions - these are used to support the commands DATA and READ in MBasic */
/* ----------------------------------------------------------------------- */
/* BEGIN DATA/READ support functions */
/* ----------------------------------------------------------------------- */
/*
    This function will insert a value into a specific position in memory. It is called by the command DATA
*/
struct data_read * InsertData(struct data_read newVal, struct data_read **items)
{
    struct data_read *newNode;
    struct data_read *ptemp;

    ptemp = *items;
    newNode = getDataMemory();

    newNode->type = newVal.type;
    if(newVal.type == NUMERIC_VAR)
        newNode->value.dv = newVal.value.dv;
    else
        strcpy(newNode->value.sv, newVal.value.sv);

    while(ptemp != NULL && ptemp->next != NULL)
        ptemp = ptemp->next;
    
    if(ptemp == NULL) {
        newNode->next = NULL;
        newNode->previous = NULL;
        *items = newNode;
    } else {
      newNode->previous = ptemp;
      newNode->next = ptemp->next;
      ptemp->next = newNode;
    }
    
    return(*items);
}

/*
    This function deletes a value stored in a specific position of the memory.
    It is used  by the READ command.
*/
void deleteItemAtPos(int pos, struct data_read **items)
{
   struct data_read *tempList = NULL;
   struct data_read *temp = NULL;
   struct data_read *next1 = NULL;
   int count = 0;
   int listsize;

   tempList = *items;
   listsize = getItemListSize(tempList);

   if(pos < 0) pos = 0;
   if(pos >= listsize) pos = (listsize - 1);

   next1 = *items;     /* set pointer to list head */
   while(next1 != NULL)     /* search through whole list */
   {
     if(count == pos)
     {
        /* if deleting only node */
        if(next1->next == NULL && next1->previous == NULL)
        {   /* deleting only node.  Point plist to NULL */
           free(next1);				
           *items = NULL;
        }
        /* if deleting first node */
        else if(next1->previous == NULL)
        {
           /* skip next1 to next node */
           next1 = next1->next;
           /* since first node assign previous to NULL */
           next1->previous = NULL;
           /* point the list pointer to the new first node */
           *items = next1;
           /* free old first node */
           free(temp);
           /* point temp to list */
           temp = *items;
        }
        /* if deleting last node */
        else if(next1->next == NULL)
        {
           /* set temp as last node */
           temp->next = NULL;
           /* free last node in list */
           free(next1);
           /* point next1 to NULL; */
           next1 = temp->next;
        }
        else     /* else deleting a node between nodes */
        {
           /* shift pointer to skip over node to be deleted */
           temp->next = next1->next;
           /* point previous element of 1st node after deleted node to temp */
           (next1->next)->previous = temp;
           free(next1);     /* free node */
           next1 = temp->next;     /* point next1 to next node */
        }

        /* Nothing more to do */
        return;
     }

     count++;
     temp = next1;
     /* go through list */
     next1 = next1->next;
   }
}

/*
    This function returns a value stored in a specific position of memory.
    It is used  by the READ command.
*/
struct data_read * returnItemAtPos(int pos, struct data_read *items)
{
   int listsize;
   int count = 0;

   listsize = getItemListSize(items);
   if(pos < 0) pos = 0;
   if(pos >= listsize) pos = listsize-1;

   while(items != NULL)
   {
      if(count == pos) return items;
      count++;
      items = items->next;   /* point plist to next node */
   }
}

/*
    This function returns the ammount of values stored by a DATA command
*/
int getItemListSize(struct data_read *items)
{
   int count = 0;
   if(items != NULL)
   {
      while(items != NULL)
      {
         count++;
         items = items->next;   /* point plist to next node */
      }
   }
   return count;
}

/*
    ***** DEBUG FUNCTION *****
    ***** not really used *****
*/
int PrintItemList(struct data_read *items)
{
    if(items != NULL)
    {
        while(items != NULL)
        {
            printf("Type: %d\n", items->type);
            switch(items->type) {
                case NUMERIC_VAR: printf("Value: %g\n", items->value.dv);
                     break;
                case STRING_VAR: printf("Value: %s\n", items->value.sv);
                     break;
            }
            items = items->next;
        }
    }

    return 0;
}

/*
    This function allocates and returns the ammount of memory necessary to store a value.
    It is used  by the DATA command.
*/
struct data_read * getDataMemory(void)
{
   struct data_read * newPtr = NULL;

   if((newPtr = (struct data_read *) malloc(sizeof(struct data_read))) == NULL)
   {     /* if malloc returns a NULL pointer */
      printf("ERROR!  Unable to allocate memory - Abort\n");
      exit(1);
   } 
   else return newPtr;
}
/* ----------------------------------------------------------------------- */
/* END DATA/READ support functions */
/* ----------------------------------------------------------------------- */



/* String handling functions - these are used to allocate space for strings  */
/* ------------------------------------------------------------------------- */
/* BEGIN string handling functions                                           */
/* ------------------------------------------------------------------------- */
char *getmem(int size, unsigned short clean) {
    char *p;

    p = malloc(size);
    if (p == NULL) {
        printf("could not allocate memory.\n");
        exit(1);
    }

    if(clean) memset(p, '\0', size);
    return(p);
}

char *allocateString(char *s) {
    char *p;

    if (s == NULL) return(NULL);
    /* p = getmem(strlen(s) + 1, TRUE); */
    p = getmem(255, TRUE);
    strcpy(p,s);

    return(p);
}
/* ----------------------------------------------------------------------- */
/* END string handling functions */
/* ----------------------------------------------------------------------- */

/*
    ***** DEBUG FUNCTION *****
    ***** not really used *****
*/
int print_func_stack() {
    int i,f;
    for(i=0; i<NUM_FUNCS; ++i)
        if(user_func_table[i].name[0] != '\0') {
            printf("Function: %s, address: %d\n",user_func_table[i].name, user_func_table[i].loc);
            for(f = 0; f < user_func_table[i].argc; ++f) {
                printf("\tArg[%d].name: %s\n",f,user_func_table[i].args[f].name);
                switch(user_func_table[i].args[f].type) {
                    case NUMERIC_VAR: printf("\tArg[%d].type: NUMERIC\n",f);
                                      printf("\tArg[%d].value: %g\n",f,user_func_table[i].args[f].value.dv);
                                      break;
                    case STRING_VAR: printf("\tArg[%d].type: STRING\n",f);
                                     printf("\tArg[%d].value: %s\n",f,user_func_table[i].args[f].value.sv);
                                     break;
                }
            }
        }
}
