/*

Micro BASIC Interpeter (exparse.c)

This is the parser module. It will compute math or string expression and return
results to the interpreter engine.

*** Copyleft - Andre Murta - August 1992/December 2002 ***

Adapted from Herbert Schildt Little C (Dr. Dobbs Journal, August 1989)

*/

#include <setjmp.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include "mbasic.h"
#include "structures.h"

extern char *prog;	/* Expression to be analyzed */
extern char *p_buf; /* program buffer */
extern jmp_buf e_buf; /* used by long_jmp() */
extern struct node *array_items; /* pointer for first list */

/* array support functions*/
extern struct node * InsertAtEnd(struct node newVal, struct node **array_items);
extern void deleteNodeValue(struct node delVal, short dim, struct node ** array_items);
extern struct node * search(struct node data, short dim, struct node *pstack);
extern struct node * getMemory(void);

/* string memory functions */
extern char *getmem(int size, unsigned short clean);
extern char *allocateString(char *s);

/* Variables structure */
extern struct variables vars[NUM_VARS];

/* user defined functions */
extern struct user_function user_func_table[NUM_FUNCS];

/* Commands table */
extern struct commands table[];

/* --------------------------------- */
/* MBASIC internal functions - BEGIN */
/* --------------------------------- */
extern char *func_chr();
extern double func_asc();
extern double func_len();
extern double func_inkey();
extern char *func_left();
extern char *func_mid();
extern char *func_right();
extern double func_sqr();
extern double func_log();
extern double func_exp();
extern double func_sin();
extern double func_cos();
extern double func_tan();
extern double func_atn();
extern double func_pi();
extern double func_abs();
extern double func_sgn();
extern double func_int();
extern double func_timer();
extern double func_val();
extern char *func_str();
extern char *func_lcase();
extern char *func_ucase();
extern char *func_time();
extern char *func_date();
extern double func_asin();
extern double func_acos();
extern double func_sinh();
extern double func_cosh();
extern double func_tanh();
extern double func_log10();
extern double func_floor();
extern double func_rnd();
extern double func_etype();
extern double func_chdir();
extern double func_rmdir();
extern double func_mkdir();
extern double func_col();
extern double func_row();
extern double func_cols();
extern double func_rows();
extern double func_stack();
extern double func_copytext();
extern double func_rename();
extern double func_kill();
extern double func_loc();
extern double func_lof();
extern double func_eof();
extern double func_instr();
extern char *func_spc();
extern char *func_tab();
extern double func_asinh();
extern double func_acosh();
extern double func_atnh();
extern double func_atn2();
extern double func_log2();
extern double func_ceil();
extern double func_fmod();
extern double func_hypot();
extern double func_pow();
extern double func_pow10();
extern double func_pow2();
extern char *func_gettext();
/* ------------------------------- */
/* MBASIC internal functions - END */
/* ------------------------------- */

/* used to find and execute internal function
   during program execution */
struct internal_functions
{
    char funcname[VAR_LEN];
    double (*pd)();
    char *(*ps)();
    char rtype;
} func_table[] = {
    "chr", NULL, func_chr, STRING_VAR,
    "asc", func_asc, NULL, NUMERIC_VAR,
    "len", func_len, NULL, NUMERIC_VAR,
    "inkey", func_inkey, NULL, NUMERIC_VAR, /* HW dependent*/
    "left", NULL, func_left, STRING_VAR,
    "mid", NULL, func_mid, STRING_VAR,
    "right", NULL, func_right, STRING_VAR,
    "sqr", func_sqr, NULL, NUMERIC_VAR,
    "log", func_log, NULL, NUMERIC_VAR,
    "exp", func_exp, NULL, NUMERIC_VAR,
    "sin", func_sin, NULL, NUMERIC_VAR,
    "cos", func_cos, NULL, NUMERIC_VAR,
    "tan", func_tan, NULL, NUMERIC_VAR,
    "atn", func_atn, NULL, NUMERIC_VAR,
    "pi", func_pi, NULL, NUMERIC_VAR,
    "abs", func_abs, NULL, NUMERIC_VAR,
    "sgn", func_sgn, NULL, NUMERIC_VAR,
    "int", func_int, NULL, NUMERIC_VAR,
    "timer", func_timer, NULL, NUMERIC_VAR, /* HW dependent */
    "val", func_val, NULL, NUMERIC_VAR,
    "str", NULL, func_str, STRING_VAR,
    "lcase", NULL, func_lcase, STRING_VAR,
    "ucase", NULL, func_ucase, STRING_VAR,
    "time", NULL, func_time, STRING_VAR,
    "date", NULL, func_date, STRING_VAR,
    "asin", func_asin, NULL, NUMERIC_VAR,
    "acos", func_acos, NULL, NUMERIC_VAR,
    "sinh", func_sinh, NULL, NUMERIC_VAR,
    "cosh", func_cosh, NULL, NUMERIC_VAR,
    "tanh", func_tanh, NULL, NUMERIC_VAR,
    "log10", func_log10, NULL, NUMERIC_VAR,
    "floor", func_floor, NULL, NUMERIC_VAR,
    "rnd", func_rnd, NULL, NUMERIC_VAR,
    "etype", func_etype, NULL, NUMERIC_VAR,
    "chdir", func_chdir, NULL, NUMERIC_VAR,
    "rmdir", func_rmdir, NULL, NUMERIC_VAR,
    "mkdir", func_mkdir, NULL, NUMERIC_VAR,
    "col", func_col, NULL, NUMERIC_VAR, /* HW dependent */
    "row", func_row, NULL, NUMERIC_VAR, /* HW dependent */
    "cols", func_cols, NULL, NUMERIC_VAR, /* HW dependent */
    "rows", func_rows, NULL, NUMERIC_VAR, /* HW dependent */
    "stack", func_stack, NULL, NUMERIC_VAR, /* HW dependent */
    "copytext", func_copytext, NULL, NUMERIC_VAR, /* HW dependent */
    "rename", func_rename, NULL, NUMERIC_VAR,
    "kill", func_kill, NULL, NUMERIC_VAR,
    "loc", func_loc, NULL, NUMERIC_VAR,
    "lof", func_lof, NULL, NUMERIC_VAR,
    "eof", func_eof, NULL, NUMERIC_VAR,
    "instr", func_instr, NULL, NUMERIC_VAR,
    "spc", NULL, func_spc, STRING_VAR,
    "tab", NULL, func_tab, STRING_VAR, /* HW dependent */
    "asinh", func_asinh, NULL, NUMERIC_VAR,
    "acosh", func_acosh, NULL, NUMERIC_VAR,
    "atnh", func_atnh, NULL, NUMERIC_VAR,
    "atn2", func_atn2, NULL, NUMERIC_VAR,
    "log2", func_log2, NULL, NUMERIC_VAR,
    "ceil", func_ceil, NULL, NUMERIC_VAR,
    "fmod", func_fmod, NULL, NUMERIC_VAR,
    "hypot", func_hypot, NULL, NUMERIC_VAR,
    "pow", func_pow, NULL, NUMERIC_VAR,
    "pow10", func_pow10, NULL, NUMERIC_VAR,
    "pow2", func_pow2, NULL, NUMERIC_VAR,
    "gettext", NULL, func_gettext, STRING_VAR,  /* HW dependent */

    "", NULL, NULL, 0,
};

/* functions - END */

extern char token[];	/* Hold string representation of token */
extern char token_type;
extern char tok;
extern short local_exp; /* Hold if expression call is inside user function */
extern int current_func;  /* Hold index of function being executed, -1 main program */

/* headers */
void get_exp(struct variables *result);
void level1(struct variables *result);
void level2(struct variables *result);
void level3(struct variables *result);
void level4(struct variables *result);
void level5(struct variables *result);
void level6(struct variables *result);
void level7(struct variables *result);
void primitive(struct variables *result);
void arith(char o, struct variables *r, struct variables *h);
void unary(char o, struct variables *r);
void serror(int error); 
void putback();
int  get_token();
int  look_up(char *s);
int  isdelim(char c);
int  iswhite(char c);
double find_var_value(char *s);
char *find_var_string(char *s);
struct variables find_var_contents(char *s);
struct node *find_vec_contents(int var_index, int dimension);
struct user_function *find_user_func(char *name);
int find_user_func_index(char *name);
struct variables call(int func_addr);

/* Main entry point into parser */
void get_exp(struct variables *result)
{
    get_token();

    if(!*token)
    {
        serror(2);
        return;
    }

    level1(result);
    putback();
}

/* AND, OR*/
void level1(struct variables *result)
{
    register char op;
    struct variables partial_result;

    /* puts("level1"); */

    level2(result);
    op = tok;
    if(op == AND || op == OR) {
        get_token();
        level2(&partial_result);

        /* AND and OR only make sense to numeric values */
        if(result->type != NUMERIC_VAR && partial_result.type != NUMERIC_VAR) serror(18);

        switch(op) {
            case AND: result->value.dv = result->value.dv && partial_result.value.dv;
                      break;
            case OR:  result->value.dv = result->value.dv || partial_result.value.dv;
                      break;
        }
    }
}

/* >, <, >=, <=, =, != */
void level2(struct variables *result)
{
    register char op;
    struct variables partial_result;

    /* puts("level2"); */

    level3(result);
    op = tok;
    if(op==LT || op==LE || op==GT || op==GE || op==EQ || op==NE) {
        get_token();
        level3(&partial_result);

        /* MUST be the same type */
        if(result->type != partial_result.type) serror(18);

        switch(op) {
              case LT:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv < partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv < 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
              case LE:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv <= partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv < 0) result->value.dv = 1;
                    else if(result->value.dv == 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
              case GT:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv > partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv > 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
              case GE:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv >= partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv > 0) result->value.dv = 1;
                    else if(result->value.dv == 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
              case EQ:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv == partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv == 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
              case NE:
                if(result->type == NUMERIC_VAR && partial_result.type == NUMERIC_VAR)
                    result->value.dv = result->value.dv != partial_result.value.dv;
                else {
                    result->type = NUMERIC_VAR; /* we MUST setup to numeric type */
                    result->value.dv = strcmp(result->value.sv, partial_result.value.sv);
                    if(result->value.dv != 0) result->value.dv = 1;
                    else result->value.dv = 0;
                }
                break;
        }
    }
}

/* +- */
void level3(struct variables *result)
{
    register char op;
    struct variables partial_result;

    /* puts("level3"); */

    level4(result);
    while((op = tok) == ADD || op == SUB) {
        get_token();
        level4(&partial_result);

        if(result->type != partial_result.type) serror(18);

        arith(op, result, &partial_result);
    }
}

/* /%* */
void level4(struct variables *result)
{
	register char op;
	struct variables partial_result;

        /* puts("level4"); */

	level5(result);
	while((op = tok) == MUL || op == DIV || op == MOD)
	{
		get_token();
		level5(&partial_result);

        if(result->type != partial_result.type) serror(18);

		arith(op, result, &partial_result);
	}
}

/* ^ */
void level5(struct variables *result)
{
    struct variables partial_result;

    /* puts("level5"); */

    level6(result);
    while(tok == EXP)
    {
        get_token();
        level5(&partial_result);

        if(result->type != partial_result.type) serror(18);

        arith(EXP, result, &partial_result);
    }
}

/* unary minus */
void level6(struct variables *result)
{
    register char op;

    /* puts("level6"); */

    op = 0;
    if((token_type == DELIMITER) && (tok == ADD || tok == SUB))
    {
        op = tok;
        get_token();
    }
    level7(result);
    if(op) {
        if(result->type != NUMERIC_VAR) serror(18);

        unary(op, result);
    }
}

/* (, ) */
void level7(struct variables *result)
{
    /* puts("level7"); */

    if((tok == LPAREN) && (token_type == DELIMITER))
    {
        get_token();

        level1(result);
        if(tok != RPAREN) serror(1);
        get_token();
    }
    else primitive(result);
}

/* get the value of a local/global variable, internal/user defined function */
void primitive(struct variables *result)
{
    int i,f;
    struct node *search_vec;
    struct variables search_var, user_func_result;
    struct user_function *func;

    /* puts("primitive"); */

    switch(token_type)
    {
        case VARIABLE:
            to_upper(&token);

            i = look_up_internal_func(token);
            if(i != -1) { /* It is an internal function */
                if(func_table[i].rtype == NUMERIC_VAR) {
                    result->type = NUMERIC_VAR;
                    result->value.dv = (func_table[i].pd)();
                } else {
                    result->type = STRING_VAR;
                    strcpy(result->value.sv, (func_table[i].ps)());
                }
                /* get next token and returns */
                get_token();
                return;
            } 

            i = look_up_user_func(token);
            if (i != -1)
            {
                func = find_user_func(token);
                if(func) {
                    user_func_result = call(find_user_func_index(func->name));
                    result->type = user_func_result.type;
                    switch(result->type) {
                        case NUMERIC_VAR:
                            result->value.dv = user_func_result.value.dv;
                            break;
                        case STRING_VAR:
                            strcpy(result->value.sv, user_func_result.value.sv);
                            break;
                    }
                }

                /* get next token and returns */
                get_token();
                return;
            }

            if(local_exp) {
                /* it's a user function, search in local vars first */
                for(f=0; f<user_func_table[current_func].argc; ++f) {
                    /* if var is found, process it */
                    if(!strcmp(user_func_table[current_func].args[f].name, token)) {
                        result->type = user_func_table[current_func].args[f].type;
                        switch(result->type) {
                            case NUMERIC_VAR: result->value.dv = user_func_table[current_func].args[f].value.dv;
                                              break;
                            case STRING_VAR: strcpy(result->value.sv, user_func_table[current_func].args[f].value.sv);
                                             break;
                        }
                        /* get next token and returns */
                        get_token();
                        return;
                    }
                }

                /* If not found in local vars, try global */
                to_upper(&token); /* due to the call to look_up_*_func() */
    
                /* If we are here it's a global var, otherwise it will be an error */
                if(!is_dimvar(token)) {
                    result->type = find_var_type(token);
                    switch(result->type) {
                        case NUMERIC_VAR: result->value.dv = find_var_value(token);
                                          break;
                        case STRING_VAR: strcpy(result->value.sv, find_var_string(token));
                                         break;
                    }
                } else {
                    search_vec = find_vec_contents(find_var_index(token), find_var_dim(token));
                    result->type = search_vec->type;
                    switch(search_vec->type) {
                        case NUMERIC_VAR: result->value.dv = search_vec->value.dv;
                                          break;
                        case STRING_VAR: strcpy(result->value.sv, search_vec->value.sv);
                                         break;
                    }
                }
                /* get next token and returns */
                get_token();
                return;
            }

            /* It is a variable */
            to_upper(&token); /* due to the call to look_up_*_func() */

            /* If we are here it's a global var, otherwise it will be an error */
            if(!is_dimvar(token)) {
                result->type = find_var_type(token);
                switch(result->type) {
                    case NUMERIC_VAR: result->value.dv = find_var_value(token);
                                      break;
                    case STRING_VAR: strcpy(result->value.sv, find_var_string(token));
                                     break;
                }
            } else {
                search_vec = find_vec_contents(find_var_index(token), find_var_dim(token));
                result->type = search_vec->type;
                switch(search_vec->type) {
                    case NUMERIC_VAR: result->value.dv = search_vec->value.dv;
                                      break;
                    case STRING_VAR: strcpy(result->value.sv, search_vec->value.sv);
                                     break;
                }
            }
            get_token();
            return;

        case NUMBER:
            result->type = NUMERIC_VAR;
            result->value.dv = atof(token);
            get_token();
            return;

        case QUOTE:
            result->type = STRING_VAR;
            strcpy(result->value.sv, token);
            get_token();
            return;

        default:
            serror(0);
    }
}

/* perform arith operations */
void arith(char o, struct variables *r, struct variables *h)
{
    double t, ex;
    char *str_result;

    switch(o)
    {
        case SUB:
            if(r->type == NUMERIC_VAR && h->type == NUMERIC_VAR)
                r->value.dv = r->value.dv - h->value.dv;
            else serror(18);
            break;

        case ADD:
            if(r->type == NUMERIC_VAR && h->type == NUMERIC_VAR)
                r->value.dv = r->value.dv + h->value.dv;
            else if(r->type == STRING_VAR && h->type == STRING_VAR)
                sprintf(r->value.sv, "%s%s", r->value.sv, h->value.sv);
            else serror(18);
            break;

        case MUL:
            if(r->type == NUMERIC_VAR && h->type == NUMERIC_VAR)
                r->value.dv = r->value.dv * h->value.dv;
            else serror(18);
            break;

        case DIV:
            if(r->type == NUMERIC_VAR && h->type == NUMERIC_VAR)
                r->value.dv = (r->value.dv) / (h->value.dv);
            else serror(18);
            break;

        case MOD:
            if(r->type != NUMERIC_VAR && h->type != NUMERIC_VAR) serror(18);
            t = (r->value.dv) / (h->value.dv);
            r->value.dv = r->value.dv - (t * (h->value.dv));
            break;

        case EXP:
            if(r->type != NUMERIC_VAR && h->type != NUMERIC_VAR) serror(18);
            ex = r->value.dv;
            if(h->value.dv == 0)
            {
                r->value.dv = 1;
                break;
            }
            for(t = h->value.dv-1; t > 0; --t) r->value.dv = (r->value.dv) * ex;
            break;
    }
}

void unary(char o, struct variables *r)
{
    if(o == SUB) r->value.dv = -(r->value.dv);
}

/* returns the numeric value of a global var */
double find_var_value(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i].value.dv;

    serror(14); /* Variable not found */
}

/* returns the alphanumeric value of a global var */
char *find_var_string(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i].value.sv;

    serror(14); /* Variable not found */
}

/* well... returns the variable type */
int find_var_type(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i].type;

    serror(14); /* Variable not found */
}

/* returns a 'node' struct with a specific array index value */
struct node *find_vec_contents(int var_index, int dimension)
{
    struct variables val1, val2, val3;
    struct node vec;
    struct node *result = NULL;

    get_token();
    if(tok != LPAREN) serror(19);

    get_exp(&val1);
    if(val1.type != NUMERIC_VAR) serror(18);

    get_token();
    if(tok == COMMA) {
        if(dimension < 2) serror(21);

        get_exp(&val2);
        if(val2.type != NUMERIC_VAR) serror(18);

        get_token();
        if(tok == COMMA) {
            if(dimension < 3) serror(21);

            get_exp(&val3);
            if(val3.type != NUMERIC_VAR) serror(18);

            get_token();
            if(tok != RPAREN) serror(0);
        } else if(tok != RPAREN) serror(0);
    } else if(tok != RPAREN) serror(0);

    vec.var_index = var_index;
    vec.index.x = (int)(val1.value.dv);
    vec.index.y = (int)(val2.value.dv);
    vec.index.z = (int)(val3.value.dv);

    result = search(vec, dimension, array_items);
    if(result == NULL) serror(24);

    return result;
}

/* returns TRUE or FALSE depending on the variable is or not an array var */
int is_dimvar(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i].isdim;

    serror(14); /* Variable not found */
}

/* returns 1,2 or 3 */
int find_var_dim(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i].dim;

    serror(14); /* Variable not found */
}

/* returns the index of a variable in the var table */
int find_var_index(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return i;

    serror(14); /* Variable not found */
}

/* returns the entire structure of a variable */
struct variables find_var_contents(char *s)
{
    register int i;
    
    /* if variable already exists, return it's pointer */
    for (i = 0; i < NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return vars[i];

    serror(14); /* Variable not found */
}

void serror(int error)
{
    static char *e[] = {
        /*  0 */ "Syntax error",
        /*  1 */ "Unbalanced parens",
        /*  2 */ "No expression present",
        /*  3 */ "Expected: =",
        /*  4 */ "Not a variable",
        /*  5 */ "Lable table overflow",
        /*  6 */ "Duplicate label",
        /*  7 */ "Undefined label",
        /*  8 */ "THEN expected",
        /*  9 */ "TO expected",
        /* 10 */ "Too many nested FOR loops",
        /* 11 */ "NEXT without FOR",
        /* 12 */ "Too many nested GOSUBs",
        /* 13 */ "RETURN without GOSUB",
        /* 14 */ "Undeclared variable",
        /* 15 */ "Variables stack overflow",
        /* 16 */ "Too many nested WHILE",
        /* 17 */ "WEND without WHILE",
        /* 18 */ "Type mismatch",
        /* 19 */ "Left paren expected",
        /* 20 */ "Right paren expected",
        /* 21 */ "Invalid range",
        /* 22 */ "Arrays can have a maximum of three dimensions",
        /* 23 */ "Variable has no dimensions",
        /* 24 */ "Index not found",
        /* 25 */ "Index out of bounds",
        /* 26 */ "Comma expected",
        /* 27 */ "Expression must return a positive value",
        /* 28 */ "No data in memory",
        /* 29 */ "Function stack overflow",
        /* 30 */ "Duplicated function definition",
        /* 31 */ "Expected FOR",
        /* 32 */ "File opening mode expected (READING, WRITING, APPENDING)",
        /* 33 */ "Expected AS",
        /* 34 */ "Expected file pointer identifier (#)",
        /* 35 */ "Expected numeric expression",
        /* 36 */ "Expected alphanumeric expression",
        /* 37 */ "Data file space is already in use",
        /* 38 */ "File space is not in use",
        /* 39 */ "File opening mode expected (RANDOM, SEQUENTIAL)",
        /* 40 */ "File note opened in RANDOM mode",
        /* 41 */ "File note opened in SEQUENTIAL mode",
        /* 42 */ "String identifier ($) expected",
        0
    };
    char *p, *temp;
    int linecount, linestart;
    register int i;

    p = p_buf;

    linestart = linecount = 0;
    while (p != prog) {     /* find line number of error */
        p++;
        if (*p == '\n') {
            linestart = (int)(p-p_buf);
            linecount++;
        }
    }

    puts("");
	printf("Error in line [%d]: %s\n", linecount, e[error]);
    temp = p;
    printf("'");
    for(i=0; i < 20 && p > p_buf && *p != '\n'; i++, p--);
    for(i=0; i < 30 && p <= temp; i++, p++) printf("%c", *p);
    printf("'");
    puts("");

	exit(-1);
}

/*
    this function was changed to be able to handle
    more error possibilities.
*/
int get_token()
{
	register char *temp;

	token_type = tok = 0;
	temp = token;

	if(*prog == EOF)
	{
		*token = EOF;
		tok = FINISHED;
		return(token_type = DELIMITER);
	}

	while(iswhite(*prog)) prog++;

        /* Is it a QUOTE? */
	if(*prog == '"')
	{
		prog++;
		while(*prog != '"' && *prog != '\r') *temp++ = *prog++;
		if(*prog == '\n') serror(1);
		prog++;
		*temp = 0;
		return(token_type = QUOTE);
	}

        /* Is it an CR-LF? */
	if(*prog == '\n' || *prog == ':')
	{
		prog++;
		tok = EOL;
		*token = '\n';
		token[1] = 0;
		return(token_type = DELIMITER);
	}

        /* special chars */
	if(strchr("+-*/^%;,()$=[]#", *prog))
	{
        switch(*prog) {
                case '+': tok = ADD; break;
                case '-': tok = SUB; break;
                case '*': tok = MUL; break;
                case '/': tok = DIV; break;
                case '^': tok = EXP; break;
                case '%': tok = MOD; break;
                case ';': tok = DCOMMA; break;
                case ',': tok = COMMA; break;
                case '(': tok = LPAREN; break;
                case ')': tok = RPAREN; break;
                case '$': tok = DOLLAR; break;
                case '=': tok = EQ; break;
                case '[': tok = LBRACK; break;
                case ']': tok = RBRACK; break;
                case '#': tok = SHARP; break;
        }

		*temp = *prog;
		prog++;
		temp++;
		*temp = 0;

		return(token_type = DELIMITER);
	}

    if(strchr("!<>=&|", *prog)) { /* is or might be a relational operator */
            switch(*prog) {
              case '&': if(*(prog+1) == '&') {
                            prog++; prog++;
                            *temp = AND;
                            temp++; *temp = AND; temp++;
                            *temp = '\0';
                            tok = AND;
                        }
                        break;
              case '|': if(*(prog+1) == '|') {
                            prog++; prog++;
                            *temp = OR;
                            temp++; *temp = OR; temp++;
                            *temp = '\0';
                            tok = OR;
                        }
                        break;
              case '<': if(*(prog+1) == '=') {
                            prog++; prog++;
                            *temp = LE; temp++; *temp = LE;
                            tok = LE;
                        } else if(*(prog+1) == '>') {
                            prog++; prog++;
                            *temp = NE; temp++; *temp = NE;
                            temp++;
                            *temp = '\0';
                            tok = NE;
                        } else {
                            prog++;
                            *temp = LT;
                            tok = LT;
                        }
                        temp++;
                        *temp = '\0';
                        break;
              case '>': if(*(prog+1) == '=') {
                            prog++; prog++;
                            *temp = GE; temp++; *temp = GE;
                            tok = GE;
                        } else {
                            prog++;
                            *temp = GT;
                            tok = GT;
                        }
                        temp++;
                        *temp = '\0';
                        break;
            }
            return(token_type = DELIMITER);
    }

    /*
        test if it is a number. Could be improved to handle also
        scientifical notation.
    */
    if(isdigit(*prog))
    {
            while(isdigit(*prog)) *temp++ = *prog++;
            if(*prog == '.') {
                *temp++ = *prog++;
                if(!isdigit(*prog)) {
                    *temp = 0;
                    return(token_type = UNKNOW);
                }
                while(isdigit(*prog)) *temp++ = *prog++;
                *temp = 0;
                return(token_type = NUMBER);
            } else if(isdelim(*prog)) {
                *temp = 0;
                return(token_type = NUMBER);
            } else {
                return(token_type = UNKNOW);
            }
    }

    /* variable or command? */
    if(isalpha(*prog))
    {
            while(!isdelim(*prog)) *temp++ = *prog++;
            *temp = '\0';
            tok = look_up(token);        
            if(!tok) return(token_type = VARIABLE);
            else return(token_type = COMMAND);
    }
}

void putback()
{
	char *t;

	t = token;
	for(; *t; t++) prog--;
}

int look_up(char *s)
{
	register i, j;
	char *p;

	p = s;
	while(*p) { *p = tolower(*p); p++; }
	for(i = 0; *table[i].command; i++)
		if(!strcmp(table[i].command, s)) return table[i].tok;
	return 0;
}

int look_up_user_func(char *s)
{
    register i;
    char *p;

    p = s;
    while(*p) { *p = tolower(*p); p++; }
    for(i = 0; i < NUM_FUNCS; i++)
       if(!strcmp(user_func_table[i].name, s)) return i;
    return -1;
}

struct user_function *find_user_func(char *name)
{
    register int i;

    for (i = 0; i < NUM_FUNCS; i++)
        if (!strcmp(name, user_func_table[i].name))
            return &user_func_table[i];

    return NULL;
}

int find_user_func_index(char *name)
{
    register int i;

    for (i = 0; i < NUM_FUNCS; i++)
        if (!strcmp(name, user_func_table[i].name))
            return i;

    return NULL;
}

int look_up_internal_func(char *s)
{
    register i;
    char *p;

    p = s;
    while(*p) { *p = tolower(*p); p++; }
    for(i = 0; *func_table[i].funcname; i++)
       if(!strcmp(func_table[i].funcname, s)) return i;
    return -1;
}

int isdelim(char c)
{
	if(strchr(" ;,+-<>/*%^=()|&$[]#", c) || c == 9 || c == '\n' || c == 0)
		return 1;
	return 0;
}

int iswhite(char c)
{
	if(c == ' ' || c == '\t') return 1;
	else return 0;
}

/*
   this function is executed when the parser finds a user defined function call
   it will find the function definition, associate the arguments, execute the
   expression and return the result.
*/
struct variables call(int func_addr)
{
    int arg_index;
    struct variables parameter, return_val;
    char *temp;

    get_token();
    if(tok == LPAREN) {
        arg_index = 0;
        do {
            get_exp(&parameter);
            user_func_table[func_addr].args[arg_index].type = parameter.type;
            switch(parameter.type) {
                case NUMERIC_VAR:
                    user_func_table[func_addr].args[arg_index].value.dv = parameter.value.dv;
                    break;
                case STRING_VAR:
                    strcpy(user_func_table[func_addr].args[arg_index].value.sv, parameter.value.sv);
                    break;
            }
            arg_index++;

            get_token();
            if(tok != COMMA && tok != RPAREN) serror(26);
        } while(tok != RPAREN);
    }

    temp = prog; /* save program point*/
    prog = user_func_table[func_addr].loc; /* move to function */

    local_exp = TRUE; /* search for local vars first */
    current_func = func_addr; /* identify current function being executed */
    get_exp(&return_val);
    local_exp = FALSE; /* back to main body */
    current_func = -1; /* we are back to main body */

    prog = temp; /* restore program point */

    return(return_val);
}
