/************************************************************************
 *                                                                      *
 *                  Copyright (c) 1992, Frank van der Hulst             *
 *                          All Rights Reserved                         *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 * Authors:                                                             *
 *        FvdH - Frank van der Hulst (Wellington, NZ)                   *
 *                                                                      *
 * Versions:                                                            *
 *    V1.0 911031 FvdH - Released as part of PVQUAN13.ZIP               *
 *    V1.4 920303 FvdH - Ported to gcc (PVQUAN14.ZIP)                   *
 *    V1.5 920401 FvdH - Improved usage                                 *
 *		     920406 FvdH - Fixed bug which created files with leading     *
 *								 spaces in their names                          *
 *                                                                      *
 ************************************************************************/

/*
	This program reads an animation source file (*.ANM), and uses it to
	generate data files for the PoV ray tracer. An initial data file
	must also be given.

	Command format:
		 MAKEANIM data animation

	The animation file has the following format:
	Line 1: name1  [,name2]...
	Line 2: value1 [,value2]...

	name1, name2, etc. are the names of items #declared in the initial data
	file, separated by commas. The names must be in the same order as they
	are #defined in the data file.

	value1, value2, etc. are the values to be assigned to those #defined
	items, separated by commas.
	Each line represents one screen.

	This program reads the initial data file, searching for the lines where
	the items named in the animation file are defined.

	Data files are then generated, replacing the lines containing the
	#defines identified with new values.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define FALSE 0
#define TRUE 1

#ifdef __GNUC__
#define SEEK_SET 0
#define cdecl
#endif

#define MAX_ITEMS 20

struct ITEM {
	char		name[256];
	int		line;
} ;

struct ITEM item[MAX_ITEMS];
int num_items;

FILE *anim_file, *data_file;

#ifndef __TURBOC__
void strlwr(char *s)
{
	while (*s) {
		*s = tolower(*s);
		s++;
	}
}
#endif

int get_field_names(char *field_buff, struct ITEM *field)
{
	int i;
	char *start = field_buff;
	char *dest;

	strlwr(field_buff);
	for (i = 0; i < MAX_ITEMS; i++) {
		dest = field[i].name;
		while (*start == ' ' || *start == '\t' && *start != '\n') start++;
		while (*start != ',' && *start != '\n') {
			if (*start != ' ' && *start != '\t')	*dest++ = *start;
			start++;
		}
		if (*start == '\n') break;
		start++;
	}
	return i+1;
}

void split_fields(char *anim_buff, char name[256][MAX_ITEMS], int num_items)
{
	int i;
	char *end;

	for (i = 0; i < num_items; i++) {
		end = strpbrk(anim_buff, ",\n");
		*end = '\0';
		strcpy(name[i], anim_buff);
		anim_buff = end + 1;
	}
}

void scan_data_file(struct ITEM *field, int num_items)
{
	char data_buff[256];
	char *name;
	char *end;
	int line, i;

	for (line = 0; !feof(data_file); line++) {
		fgets(data_buff, sizeof(data_buff), data_file);
		strlwr(data_buff);
		if (strncmp("#declare", data_buff, 8) != 0) continue;
		name = data_buff+8;
		while (*name == ' ' || *name == '\t') name++;
		end = name;
		while (*end != ' ' && *end != '\t') end++;
		*end = '\0';
		for (i = 0; i < num_items; i++)
			if (strcmp(name, field[i].name) == 0) {
				field[i].line = line;
				break;
			}
	}
}

void generate_file(char *fname, struct ITEM *field, int num_fields, char *anim_line)
{
	int line, i;
	FILE *out;
	static char value[256][MAX_ITEMS];
	char data_buff[256];

	fseek(data_file, 0L, SEEK_SET);
	if ((out = fopen(fname, "wt")) == NULL) {
		printf("Couldn't open %s\n", fname);
		exit(1);
	}

	line = 0;
	split_fields(anim_line, value, num_fields);
	for (i = 0; i < num_fields; i++) {
		for ( ; line < field[i].line; line++) {
			fgets(data_buff, sizeof(data_buff), data_file);
			fputs(data_buff, out);
		}
		fgets(data_buff, sizeof(data_buff), data_file);
		fprintf(out, "#declare %s = %s\n", field[i].name, value[i]);
		line++;
	}
	fgets(data_buff, sizeof(data_buff), data_file);
	while (!feof(data_file)) {
		fputs(data_buff, out);
		fgets(data_buff, sizeof(data_buff), data_file);
	}
	fclose(out);
}

void cdecl main(int argc, char *argv[])
{
	char fname[256];
	char anim_line[256];
	int file_no;

    printf("MAKEANIM v1.50 -- Create a series of PVRay data files.\n");
	printf("By F van der Hulst. Copyright 1992\n\n");

	if (argc != 3) {
        printf("Usage: %s PVRay_data_file Animation_script_file\n", argv[0]);
		exit(1);
	}

	strcpy(fname, argv[1]);
	strcat(fname, ".dat");
	if ((data_file = fopen(fname, "rt")) == NULL) {
		printf("Couldn't open %s\n", fname);
		exit(1);
	}

	strcpy(fname, argv[2]);
	strcat(fname, ".anm");
	if ((anim_file = fopen(fname, "rt")) == NULL) {
		printf("Couldn't open %s\n", fname);
		exit(1);
	}

	fgets(anim_line, sizeof(anim_line), anim_file);
	num_items = get_field_names(anim_line, item);
	if (num_items == 0) {
		printf("No items specified\n");
		exit(1);
	}

	printf("%12.12s  %65.65s\n", " ", anim_line);
	scan_data_file(item, num_items);

	fgets(anim_line, sizeof(anim_line), anim_file);
	for (file_no = 0; !feof(anim_file); file_no++) {
		sprintf(fname, "%s_%d.dat", argv[2], file_no);
		printf("%12.12s: %65.65s\n", fname, anim_line);
		generate_file(fname, item, num_items, anim_line);
		fgets(anim_line, sizeof(anim_line), anim_file);
	}
}
