#define COPYRIGHT_NOTICE "Copyright (C) 1999 Free Software Foundation, Inc."
#define BUG_ADDRESS "bug-gnu-utils@gnu.org"

/*  GNU SED, a batch stream editor.
    Copyright (C) 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1998, 1999 \
    Free Software Foundation, Inc."

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */


#include "config.h"
#include <stdio.h>

#ifndef HAVE_STRING_H
# include <strings.h>
#else
# include <string.h>
#endif

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifndef EXIT_SUCCESS
# define EXIT_SUCCESS 0
#endif

#ifdef HAVE_MMAP
# ifdef HAVE_UNISTD_H
#  include <unistd.h>
# endif
# include <sys/types.h>
# include <sys/mman.h>
# ifndef MAP_FAILED
#  define MAP_FAILED	((char *)-1)	/* what a stupid concept... */
# endif
# include <sys/stat.h>
# ifndef S_ISREG
#  define S_ISREG(m)	(((m)&S_IFMT) == S_IFMT)
# endif
#endif /* HAVE_MMAP */

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_LOCALE_H
# include <locale.h>
#endif
#include "getopt.h"
#include "basicdefs.h"
#include "utils.h"
#include "sed.h"

#ifndef HAVE_STDLIB_H
 extern char *getenv P_((const char *));
#endif

#ifndef HAVE_STRTOUL
# define ATOI(x)	atoi(x)
#else
# define ATOI(x)	strtoul(x, NULL, 0)
#endif

flagT use_extended_syntax_p = 0;

/* If set, buffer input as minimally as practical,
   and fflush(stdout) more often. */
flagT force_unbuffered = 0;

/* If set, don't write out the line unless explicitly told to */
flagT no_default_output = 0;

/* Do we need to be pedantically POSIX compliant? */
flagT POSIXLY_CORRECT;

/* How long should the `l' command's output line be? */
countT lcmd_out_line_len = 70;

static void usage P_((int));
static void
usage(status)
  int status;
{
  FILE *out = status ? stderr : stdout;

  fprintf(out, _("\
Usage: %s [OPTION]... {script-only-if-no-other-script} [input-file]...\n\
\n\
  -n, --quiet, --silent\n\
                 suppress automatic printing of pattern space\n\
  -e script, --expression=script\n\
                 add the script to the commands to be executed\n\
  -f script-file, --file=script-file\n\
                 add the contents of script-file to the commands to be executed\n\
  -l N, --line-length=N\n\
                 specify the desired line-wrap length for the `l' command\n\
  -u, --unbuffered\n\
\n\
      --help     display this help and exit\n\
  -V, --version  output version information and exit\n\
\n\
If no -e, --expression, -f, or --file option is given, then the first\n\
non-option argument is taken as the sed script to interpret.  All\n\
remaining arguments are names of input files; if no input files are\n\
specified, then the standard input is read.\n\
\n"), myname);
  fprintf(out, _("E-mail bug reports to: %s .\n\
Be sure to include the word ``%s'' somewhere in the ``Subject:'' field.\n"),
	  BUG_ADDRESS, PACKAGE);
  exit(status);
}

int
main(argc, argv)
  int argc;
  char **argv;
{
  static struct option longopts[] = {
    {"rxtest", 0, NULL, 'r'},
    {"expression", 1, NULL, 'e'},
    {"file", 1, NULL, 'f'},
    {"line-length", 1, NULL, 'l'},
    {"quiet", 0, NULL, 'n'},
    {"silent", 0, NULL, 'n'},
    {"unbuffered", 0, NULL, 'u'},
    {"version", 0, NULL, 'V'},
    {"help", 0, NULL, 'h'},
    {NULL, 0, NULL, 0}
  };

  /* The complete compiled SED program that we are going to run: */
  struct vector *the_program = NULL;
  int opt;
  flagT bad_input;	/* If this variable is non-zero at exit, one or
			   more of the input files couldn't be opened. */

#if HAVE_SETLOCALE
  /* Set locale according to user's wishes.  */
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  /* Tell program which translations to use and where to find.  */
  /*  bindtextdomain (PACKAGE, LOCALEDIR); */
  textdomain (PACKAGE);
#endif

  POSIXLY_CORRECT = (getenv("POSIXLY_CORRECT") != NULL);

  /* If environment variable `COLS' is set, use its value for
     the baseline setting of `lcmd_out_line_len'.  The "-1"
     is to avoid gratuitous auto-line-wrap on ttys.
   */
  {
    const char *p = getenv("COLS");
    if (p)
      {
	countT t = ATOI(p);
	if (1 < t)
	  lcmd_out_line_len = t-1;
      }
  }

  myname = *argv;
  while ((opt = getopt_long(argc, argv, "hnruVe:f:l:", longopts, NULL)) != EOF)
    {
      switch (opt)
	{
	case 'n':
	  no_default_output = 1;
	  break;
	case 'e':
	  the_program = compile_string(the_program, optarg, strlen(optarg));
	  break;
	case 'f':
	  the_program = compile_file(the_program, optarg);
	  break;

	case 'l':
	  lcmd_out_line_len = ATOI(optarg);
	  break;
	case 'r':
	  use_extended_syntax_p = 1;
	  break;
	case 'u':
	  force_unbuffered = 1;
	  break;

	case 'V':
	  fprintf(stdout, "GNU %s version %s\n\n", PACKAGE, VERSION);
	  fprintf(stdout, _("%s\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE,\n\
to the extent permitted by law.\n\
"), COPYRIGHT_NOTICE);
	  exit(EXIT_SUCCESS);
	case 'h':
	  usage(EXIT_SUCCESS);
	default:
	  usage(4);
	}
    }

  if (!the_program)
    {
      if (optind < argc)
	{
	  char *arg = argv[optind++];
	  the_program = compile_string(the_program, arg, strlen(arg));
	}
      else
	usage(4);
    }
  check_final_program(the_program);

  bad_input = process_files(the_program, argv+optind);

  finish_program(the_program);
  ck_fclose(stdout);
  if (bad_input)
    exit(2);
  return EXIT_SUCCESS;
}


/* Attempt to mmap() a file.  On failure, just return a zero,
   otherwise set *base and *len and return non-zero.  */
flagT
map_file(fp, base, len)
  FILE *fp;
  VOID **base;
  size_t *len;
{
#ifdef HAVE_MMAP
  struct stat s;
  VOID *nbase;
  size_t size = 0;

  if (fstat(fileno(fp), &s) == 0)
    size = CAST(size_t)s.st_size;
  if (0 < size && s.st_size == CAST(off_t)size)
    {
      /* "As if" the whole file was read into memory at this moment... */
      nbase = VCAST(VOID *)mmap(NULL, size, PROT_READ,
				MAP_PRIVATE, fileno(fp), CAST(off_t)0);
      if (nbase != MAP_FAILED)
	{
	  *base = nbase;
	  *len =  size;
	  return 1;
	}
    }
#endif /* HAVE_MMAP */

  return 0;
}

/* Attempt to munmap() a memory region. */
void
unmap_file(base, len)
  VOID *base;
  size_t len;
{
#ifdef HAVE_MMAP
  if (base)
    munmap(VCAST(caddr_t)base, len);
#endif
}
