/* $Id: initdb.c,v 1.11 2001/07/29 18:00:17 richdawe Exp $ */

/*
 *  initdb.c - Database initialisation routines for zippo
 *  Copyright (C) 1999-2001 by Richard Dawe
 *      
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef __DJGPP__
/* Added for the os detection routines */
#include <dpmi.h>
#include <go32.h>
#include <dos.h>
#include <sys/movedata.h>
#endif /* __DJGPP__ */

/* libzippo includes */
#include <libzippo/package.h>
#include <libzippo/packlist.h>
#include <libzippo/util.h>

#include "zippo.h"
#include "rcfile.h"
#include "initdb.h"

/* Default DSMs for different platforms */
extern char *platform_dsm_table[];
extern char *platform_dsm_name_table[];

extern char zipporc_default[];     /* Default zipporc configuration file */
extern char simtelrc_default[];    /* Default simtelrc configuration file */
extern char zippo_dsm[];           /* DSM for zippo */
extern char zippo_dsm_filename[];  /* Filename for DSM for zippo */

/* structs and enums which are needed for the OS autodetection part */

typedef enum {
  PLATFORM_NONE=0,
  PLATFORM_MSDOS,
  PLATFORM_WIN3,
  PLATFORM_WIN95,
  PLATFORM_WIN98,
  PLATFORM_WINME,
  PLATFORM_WINNT4,
  PLATFORM_WIN2K,
  PLATFORM_DOSEMU
} platform_t;

typedef struct {
  platform_t platform;
  /* int major_version;
     int minor_version; */
  char dsm_name[PACKAGE_MAXNAMELEN]; /* DSM name for this platform */
  char name[PACKAGE_MAXNAMELEN];     /* More readable name */
} platform_mapping_t;

platform_mapping_t platform_mapping_table[]={
  { PLATFORM_NONE,   "none",   "None"         },
  { PLATFORM_MSDOS,  "msdos",  "MS-DOS"       },
  { PLATFORM_WIN3,   "win3",   "Windows 3"    },
  { PLATFORM_WIN95,  "win95",  "Windows '95"  },
  { PLATFORM_WIN98,  "win98",  "Windows '98"  },
  { PLATFORM_WINME,  "winme",  "Windows ME"   },
  { PLATFORM_WINNT4, "winnt4", "Windows NT 4" },
  { PLATFORM_WIN2K,  "win2k",  "Windows 2000" },
  { PLATFORM_DOSEMU, "dosemu", "dosemu"       },
  { 0,               "",       ""             }
};

/* -------------
 * - detect_os -
 * ------------- */

/* lets attempt to determine the os zippo is running under.. */
/* NOTE -: This uses DJGPP specific routines...hence the #ifdef */

static platform_t
detect_os (void)
{
#ifdef __DJGPP__
  __dpmi_regs r;
  union REGS regs;
  char buf[16];
#endif /* __DJGPP__ */

  /* For non-DJGPP platforms, display a warning saying that zippo doesn't
   * support OS detection on this platform (e.g. Linux). */
#ifndef __DJGPP__
  warn("No support for OS detection on this platform");
  return(PLATFORM_NONE);
#endif /* __DJGPP__ */

#ifdef __DJGPP__
  /* Check which OS we are running under */
  r.x.ax = 0x1600; 
  __dpmi_int(0x2F, &r);
  if (   (r.h.al != 0)    && (r.h.al != 1)
      && (r.h.al != 0x80) && (r.h.al != 0xFF)) {
    /* Windows 3.1, '95, '98 or ME */
    if (r.h.al == 4) {
      if (r.h.ah < 10) 
	return PLATFORM_WIN95; /* Windoze 95 */
      else if (r.h.ah < 90)
	return PLATFORM_WIN98; /* Windoze 98 */
      else
	return PLATFORM_WINME; /* Windows ME */
    } else {
      return PLATFORM_WIN3; /* Windows 3.x */
    }
  } else {
    if (_get_dos_version(1) == 0x0532) {
      return PLATFORM_WINNT4;				    
    } else {	
      /*
       * Check for Linux DOSEMU using the method described in Ralph Brown's
       * Interrupt List:
       *
       * INT E6 - Linux DOSEMU - INSTALLATION CHECK
       * AH = 00h
       * Return: AX = AA55h if installed
       * BH = major version number
       * BL = minor version number
       * CX = patchlevel
       *
       * Notes: check for the BIOS date string "02/25/93" at
       * F000:FFF5 before calling this function. In addition,
       * the segment address of this vector should be F000h
       * (for existing versions of DOSemu, the vector is
       * F000h:0E60h
       */
      dosmemget(0xFFFF5, 10, buf); 
      buf[8] = 0;
      if (!strcmp(buf, "02/25/93")) {
	regs.x.ax = 0;
	int86(0xE6, &regs, &regs);
	if (regs.x.ax == 0xAA55) {
	  return PLATFORM_DOSEMU;
	}
      }
    }
  }
	
  /* Well I think it is now safe to assume that we are running under
   * Normal DOS */
  return PLATFORM_MSDOS;
#endif /* __DJGPP__ */
}
	  

/* ------------------
 * - perform_initdb -
 * ------------------ */

int
perform_initdb (ZIPPO_INITDB *req)
{
  FILE *fp = NULL;
  char db_path[PATH_MAX], db_avail_path[PATH_MAX], share_path[PATH_MAX];
  char path[PATH_MAX];
  int i, j, found, ret;
  mode_t mode;

  /* for the new os detection and adding of the platform to the
   * req->platforms array */
  platform_t   detected_platform;
  char        *detected_platform_dsm_name = NULL;
  char        *detected_platform_name     = NULL;
  char       **platforms = NULL;
  int          detected_platform_already_specified = 0;

  /* Construct share, db & db-avail paths. */
  strcpy(share_path, req->root);
  addforwardslash(share_path);
  strcat(share_path, ZIPPO_SHARE_PREFIX);
  addforwardslash(share_path);

  strcpy(db_path, req->root);
  addforwardslash(db_path);
  strcat(db_path, ZIPPO_DB_PREFIX);
  addforwardslash(db_path);

  strcpy(db_avail_path, req->root);
  addforwardslash(db_avail_path);
  strcat(db_avail_path, ZIPPO_DB_AVAIL_PREFIX);
  addforwardslash(db_avail_path);

  /* Create the root & share/zippo/db directories */
  strcpy(path, share_path);
  printf(". Creating database structure in '%s'...", path);

  mode = S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;
  ret =               recursive_mkdir(path, mode);
  ret = (ret == 0) && recursive_mkdir(db_path, mode);
  ret = (ret == 0) && recursive_mkdir(db_avail_path, mode);

  if ((ret != 0) && (errno != EEXIST)) {
    printf("\n");
    perror("zippo");
    return(EXIT_FAILURE);
  }

  printf("done\n");

  /* Platform detection code added by Kalum (thanks!). */
  detected_platform = detect_os();

  /* Display the OS detected */
  detected_platform_name = platform_mapping_table[detected_platform].name;

  detected_platform_dsm_name
    = platform_mapping_table[detected_platform].dsm_name;

  printf(". Detected platform: %s\n", detected_platform_name);

  if (req->platforms == NULL) {
    platforms = (char **) malloc(sizeof(char *) * 2);

    if (platforms != NULL) {
      platforms[0] = strdup(detected_platform_dsm_name);
      platforms[1] = NULL;
    }

    /* Handle memory alloc failure */
    if ((platforms == NULL) || (platforms[0] == NULL)) {
      if (platforms != NULL)
	free(platforms);

      warnf("Unable to allocate memory - %s:%d", __FILE__, __LINE__);
      return(EXIT_FAILURE);
    }

    /* Assign the array to req->platforms */
    req->platforms = platforms;
  } else  {
    /* the array has already been allocated so search and see
     * whether the user has already specified the detected
     * platform using --with-zippo option..if not we will use
     * realloc to increase the size by 1 (i+2) */
    for (i = 0, detected_platform_already_specified = 0;
	 req->platforms[i] != NULL;
	 i++) {
      /* find whether the detected platform has already been
       * specified by the user */
      if (strcmp(req->platforms[i], detected_platform_dsm_name) == 0) {
	/* No need to do anything - set 'already specified' flag, so
	 * that the platform is not added twice to the platform list. */
	detected_platform_already_specified = 1;
	break;
      }     
    }

    /* If the detected platform has not been specified, then use
     * realloc() to increase the array size and add it. */
    if (!detected_platform_already_specified) {
      platforms = realloc(req->platforms, sizeof(char *) * (i + 2));

      if (platforms == NULL) {
	/* Memory allocation failed => issue warning and keep old table for
	 * now. */
	warnf("Unable to reallocate memory - %s:%d", __FILE__,__LINE__);
      } else {
	/* Success - add new entry. */
	req->platforms        = platforms;
	req->platforms[i]     = strdup(detected_platform_dsm_name);
	req->platforms[i + 1] = NULL;
      }
    }
  }

  /* Create any platform DSMs necessary. */
  if (req->platforms != NULL)
    for (i = 0; req->platforms[i] != NULL; i++) {
      /* Find the DSM name in the table. */
      for (found = -1, j = 0; platform_dsm_name_table[j] != NULL; j++) {
	if (strcmp(req->platforms[i], platform_dsm_name_table[j]) == 0) {
	  found = j;
	  break;
	}
      }

      if (found == -1) {
	warnf("Unable to find built-in DSM "
	      "for platform '%s'!",
	      req->platforms[i]);
	continue;
      }

      /* Create a DSM file. */
      strcpy(path, db_path);
      strcat(path, req->platforms[i]);
      strcat(path, ".dsm");
      printf(". Creating plaform DSM '%s'...", path);

      fp = fopen(path, "wt");
      if (fp == NULL) {
	printf("\n");
	warnf("Unable to write DSM for platform '%s'!",
	      req->platforms[i]);
	continue;
      }
      fputs(platform_dsm_table[found], fp);
      fclose(fp);
      printf("done\n");
    }

  /* - Create a default zipporc, if necessary. - */

  /* Check for a configuration file. */
  strcpy(path, share_path);
  strcat(path, RCFILE_DEFAULT);

  if (access(path, R_OK) != 0) {
    /* Need to create default configuration file. */
    printf(". Creating default configuration file '%s'...",
	   path);

    fp = fopen(path, "wt");
    if (fp == NULL) {
      printf("\n");
      /* TODO: Should this tidy up first? */
      dief("Unable to create configuration file '%s'!\n", path);
    }

    fputs(zipporc_default, fp);
    fclose(fp);
    printf("done\n");
  } else {
    infof("Configuration file '%s' found & kept!", path);
  }

  /* - Create a default simtelrc, if necessary. - */

  /* Check for a configuration file. */
  strcpy(path, share_path);
  strcat(path, "simtelrc"); /* TODO: Put this in a constant somewhere. */

  if (access(path, R_OK) != 0) {
    /* Need to create default configuration file. */
    printf(". Creating default configuration file '%s'...", path);

    fp = fopen(path, "wt");
    if (fp == NULL) {
      printf("\n");
      /* TODO: Should this tidy up first? */
      dief("Unable to create configuration file '%s'!\n", path);
    }

    fputs(simtelrc_default, fp);
    fclose(fp);
    printf("done\n");
  } else {
    infof("Configuration file '%s' found & kept!", path);
  }

  /* Copy the zippo binary and DSM, if necessary. */
  if (req->zippo != NULL) {
    /* Copy the binary */
    strcpy(path, req->root);
    strcat(path, "bin/");
	
    ret = recursive_mkdir(path, mode);

    if ((ret != 0) && (errno != EEXIST)) {
      warnf("Unable to create program directory '%s'", path);
      perror("zippo");
      return(EXIT_FAILURE);
    }

#ifdef __DJGPP__
    strcat(path, "zippo.exe");
#else
    strcat(path, "zippo");
#endif

    if (strcmp(req->zippo, path) == 0) {
      printf(". Cannot install zippo binary '%s' onto itself - skipping\n",
	     path);
    } else {
      if (!copy_file(req->zippo, path)) {
	warnf("Unable to copy zippo binary '%s' -> '%s'", req->zippo, path);
	return(EXIT_FAILURE);
      } else {
	printf(". Copied zippo binary: '%s' -> '%s'\n",
	       req->zippo, path);
      }
    }

    /* Install the DSM */
    strcpy(path, db_path);
    strcat(path, zippo_dsm_filename);

    printf(". Installing zippo's DSM '%s'...", path);

    fp = fopen(path, "wt");
    if (fp == NULL) {
      printf("\n");
      /* TODO: Should this tidy up first? */
      dief("Unable to create zippo's DSM '%s'!\n", path);
    }

    fputs(zippo_dsm, fp);
    fclose(fp);
    printf("done\n");
  }

  return(EXIT_SUCCESS);
}
