/**************************************************************************
 *
 * FILENAME:    xprint.c
 *
 * DESCRIPTION:
 *  Program to demonstrate the use of X Printing as implemented in
 *  DESQview/X.
 *
 * PUBLIC FUNCTIONS:
 *  None
 *
 * NOTES:
 *  Syntax: xprint [-printer <printer>] [-size <font size>] filename.ext
 *
 *  This version does not take wildcards or multiple filenames.
 *  This version does not handle formfeed characters as such.
 *
 *  Copyright (c) Quarterdeck Office Systems, Inc. 1992
 *
 * PVCS INFO:
 *  $Header::   C:/wip/xprint/vcs/xprint.c_v   1.0   19 May 1992 11:18:52 $
 *
 * CHANGES:
 *  $Log::   C:/wip/xprint/vcs/xprint.c_v                                 $
 *
 *     Rev 1.0   19 May 1992 11:18:52   BRIAN
 *  Initial revision.
 *
 **************************************************************************

static char id_xprint_c[3][] =  {
                                "$Workfile::   xprint.c                 $",
                                "$Revision::   1.0                      $",
                                "    $Date::   19 May 1992 11:18:52     $"
                                };

/*----- compilation and control switches --------------------------------*/


/**************************************************************************
 *  INCLUDE FILES
 *************************************************************************/

/*----- system and platform files ---------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>

/*----- program files ---------------------------------------------------*/


/**************************************************************************
 *  EXTERNAL REFERENCES
 *************************************************************************/

/*----- data declarations -----------------------------------------------*/

/*----- function prototypes ---------------------------------------------*/


/**************************************************************************
 *  PUBLIC DECLARATIONS
 *************************************************************************/

/*----- context ---------------------------------------------------------*/

/*----- data declarations -----------------------------------------------*/

/*----- function prototypes ---------------------------------------------*/


/**************************************************************************
 *  PRIVATE DECLARATIONS
 *************************************************************************/

/*----- context ---------------------------------------------------------*/

#define LOCAL_PRINTER       ":7"
#define DOS_FONT_NAME    "-adobe-lettergothic-Bold-r-normal--0-%d-%d-%d-c-0-iso8859-1"
#define DEFAULT_FONT_SIZE   10.0
#define INCHES_PER_MM       25.4
#define INPUT_BUFFER_LENGTH 512
#define OUTPUT_BUFFER_LENGTH    512
#define TAB_SIZE            8

typedef enum
{
	FALSE = 0, TRUE
}               BOOL;

/*----- data declarations -----------------------------------------------*/

/*----- Set as a result of command line options. ------------------------*/

char           *pPrinterName = (char *) NULL;	/* Display name for prtr */
double          FontSize = DEFAULT_FONT_SIZE;	/* Font size in points.  */
char           *pFilename = (char *) NULL;	/* Filename to print. */

/*----- X data structures for printing. ---------------------------------*/

Display        *pPrinter;		/* X display for printer. */
Window          Page;			/* X window for printer. */
XFontStruct    *pFontInfo;		/* X font info struct for printer font. */
GC              PrintContext;	/* X graphics context for printer. */

/*----- Page size information. ------------------------------------------*/

unsigned int    PageWidthInPixels;	/* Width of printer page in pixels. */
unsigned int    PageHeightInPixels;	/* Height of printer page in pixels. */

/*----- function prototypes ---------------------------------------------*/

int             main(int argc, char **argv);
void            ParseArguments(int argc, char **argv);
void            Usage(void);
void            PrintFile(char *pFilename);
void            StartPrintJob(void);
BOOL            PrintPage(FILE * pFile);
int             PrepareLine(char *pInput, char *pOutput);
void            EjectPage(void);
void            FinishPrintJob(void);

/*----- macros ----------------------------------------------------------*/



/**************************************************************************
 *  PUBLIC FUNCTION DECLARATIONS
 *************************************************************************/


/**************************************************************************
 *  PRIVATE FUNCTION DECLARATIONS
 *************************************************************************/


/**************************************************************************
 * NAME:
 *  main(int argc, char **argv)
 *
 * DESCRIPTION:
 *  Program entry point.
 *
 * INPUTS:
 *  PARAMATERS:
 *      int    argc     Count of command line arguments
 *      char **argv     Array of pointers to command line argument strings
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      Exits to DOS on completion.
 *
 * NOTES:
 *  Proposed enhancements
 *      Allow for printing of multiple files.
 *
 * CHANGES:
 *  None
 */

int
main(int argc, char **argv)
{

	/*********************************************************************/


	/****** CODE *********************************************************/

	ParseArguments(argc, argv);

	PrintFile(pFilename);

	exit(0);
}


/**************************************************************************
 * NAME:
 *  ParseArguments(int argc, char ** argv)
 *
 * DESCRIPTION:
 *  Parse command line arguments and set corresponding global variables.
 *
 * INPUTS:
 *  PARAMATERS:
 *      int    argc     Count of command line arguments
 *      char **argv     Array of pointers to command line argument strings
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      char  *pPrinterName     Pointer to display name for printer.
 *      double FontSize         Font size to use when printing in points
 *      char  *pFilename        Pointer to name of file to print.
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Proposed enhancements
 *      Allow for abbreviations of command line switches.
 *      Make command line switches case insensitive.
 *      Allow for printing of multiple files.
 *      Allow for wildcarding of filenames.
 *
 * CHANGES:
 *  None
 */

void
ParseArguments(int argc, char **argv)
{

	/*********************************************************************/

	int             ArgumentIndex;

	/****** CODE *********************************************************/

	for (ArgumentIndex = 1; ArgumentIndex < argc; ArgumentIndex++)
		{
		if (strcmp(argv[ArgumentIndex], "-printer") == 0)
			{
			if (++ArgumentIndex >= argc)
				Usage();

			pPrinterName = argv[ArgumentIndex];

			continue;
			}

		if (strcmp(argv[ArgumentIndex], "-size") == 0)
			{
			if (++ArgumentIndex >= argc)
				Usage();

			FontSize = atof(argv[ArgumentIndex]);

			continue;
			}

		pFilename = argv[ArgumentIndex];
		}

	if (pFilename == (char *) NULL)
		Usage();
}


/**************************************************************************
 * NAME:
 *  Usage(void)
 *
 * DESCRIPTION:
 *  Prints usage message and then exits to DOS.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  This should be updated when command line switches are added.
 *
 * CHANGES:
 *  None
 */

void
Usage(void)
{

	/*********************************************************************/


	/****** CODE *********************************************************/

	fprintf(stderr, "usage: xprint [-printer <printer>] [-size <font size in points>] filename.ext\n");

	exit(1);
}


/**************************************************************************
 * NAME:
 *  PrintFile(char *pFilename)
 *
 * DESCRIPTION:
 *  Print the file whose name is pointed at by pFilename to the X printer
 *
 * INPUTS:
 *  PARAMATERS:
 *      char  *pFilename        Pointer to name of file to print.
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  This is the topmost routine that actually handles the printing of
 *  files.  It's purpose is to start the print job, print and eject each
 *  page in turn.  After we print the last page, instead of ejecting it
 *  explicitly, we finish the print job which will automatically eject the
 *  final page.
 *
 * CHANGES:
 *  None
 */

void
PrintFile(char *pFilename)
{

	/*********************************************************************/

	FILE           *pFile;		/* FILE * for file being printed. */
	BOOL            LastPage;	/* Flag indicating that the file is done */

	/****** CODE *********************************************************/

	/*----- Open file for printing. -------------------------------------*/

	pFile = fopen(pFilename, "rb");

	if (pFile == (FILE *) NULL)
		{
		fprintf(stderr, "Unable to open file: %s\n", pFilename);
		return;
		}

	/*----- Print File. -------------------------------------------------*/

	fprintf(stdout, "Printing %s\n", pFilename);

	StartPrintJob();

	do
		{
		LastPage = PrintPage(pFile);

		if (LastPage == FALSE)
			EjectPage();

		} while (LastPage == FALSE);

	FinishPrintJob();

	fclose(pFile);
}


/**************************************************************************
 * NAME:
 *  StartPrintJob(void)
 *
 * DESCRIPTION:
 *  Connect to X Print Server and setup up printing environment.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      char  *pPrinterName     Pointer to display name for printer.
 *      double FontSize         Font size to use when printing (in points).
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      Display     *pPrinter;              X display for printer.
 *      Window       Page;                  X window for printer.
 *      XFontStruct *pFontInfo;             X font info for printer font.
 *      GC           PrintContext;          X graphics context for printer.
 *      unsigned int PageWidthInPixels      Width of print page in pixels.
 *      unsigned int PageHeightInPixels     Height of print page in pixels.
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  Basic steps for starting print job.
 *      Connect to X Print Server.
 *      Query for page size and resolution.
 *      Create window that will represent the printed page.
 *      Create X font string and load printer font.
 *      Create graphics context for printing.
 *
 * CHANGES:
 */

void
StartPrintJob(void)
{

	/*********************************************************************/

	unsigned int    XResolution;/* X resolution of printer. */
	unsigned int    YResolution;/* Y resolution of printer. */

	char            FontNameBuffer[128];	/* X font string. */
	int             ScreenNumber;	/* X screen number for printer. */
	XGCValues       GCValues;	/* Initial values for printer GC when it is
								 * created. */

	/****** CODE *********************************************************/

	/*----- Establish connection to X Print Server ----------------------*/

	if (pPrinterName == (char *) NULL)
		pPrinterName = LOCAL_PRINTER;

	pPrinter = XOpenDisplay(pPrinterName);

	if (pPrinter == NULL)
		{
		fprintf(stderr,
				"xprint: cannot connect to X printer %s\n",
				pPrinterName);

		exit(-1);
		}


	/*----- Query information from XLib ---------------------------------*/

	ScreenNumber = DefaultScreen(pPrinter);

	PageWidthInPixels = DisplayWidth(pPrinter, ScreenNumber);

	PageHeightInPixels = DisplayHeight(pPrinter, ScreenNumber);

	XResolution = (unsigned int) ((PageWidthInPixels * INCHES_PER_MM)
								  / DisplayWidthMM(pPrinter, ScreenNumber));

	YResolution = (unsigned int) ((PageHeightInPixels * INCHES_PER_MM)
								  / DisplayHeightMM(pPrinter, ScreenNumber));


	/*----- Create window that will represent the printed page. ---------*/

	Page = XCreateSimpleWindow(pPrinter,
							   RootWindow(pPrinter, ScreenNumber),
							   0,
							   0,
							   PageWidthInPixels,
							   PageHeightInPixels,
							   0,
							   BlackPixel(pPrinter, ScreenNumber),
							   WhitePixel(pPrinter, ScreenNumber));


	/*----- Create X font string and load printer font. -----------------*/

	sprintf(FontNameBuffer, DOS_FONT_NAME, (int) (FontSize * 10),
			XResolution, YResolution);

	pFontInfo = XLoadQueryFont(pPrinter, FontNameBuffer);

	if (pFontInfo == (XFontStruct *) NULL)
		{
		fprintf(stderr, "xprint: Unable to load font: %s\n", FontNameBuffer);
		exit(-1);
		}


	/*----- Create printer graphics context. ----------------------------*/

	GCValues.font = pFontInfo->fid;

	PrintContext = XCreateGC(pPrinter, Page, GCFont, &GCValues);
}


/**************************************************************************
 * NAME:
 *  PrintPage(FILE *pFile)
 *
 * DESCRIPTION:
 *  Output text to the printer for the current page.
 *
 * INPUTS:
 *  PARAMATERS:
 *      FILE *pFile;        FILE * for file currently being printed.
 *
 *  GLOBALS:
 *      Display     *pPrinter;              X display for printer.
 *      Window       Page;                  X window for printer.
 *      XFontStruct *pFontInfo;             X font info for printer font.
 *      GC           PrintContext;          X graphics context for printer.
 *      unsigned int PageHeightInPixels     Height of print page in pixels.
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      pFile is advanced to the beginning of the next page or the end
 *      of the file.
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      Boolean value indicating that the last page has been printed.
 *
 * NOTES:
 *  This function calls PrepareLine() in order to exapand tabs into spaces
 *  and strips linefeed characters so that we don't end up using the X
 *  glyphs for these characters.
 *
 *  This function uses the XFontInfo structure elements ascent and descent
 *  which added together make up the height of the font bounding box in
 *  order to determine the vertical positioning of the text which is output
 *  using XDrawString().
 *
 * CHANGES:
 *  None
 */

BOOL
PrintPage(FILE * pFile)
{

	/*********************************************************************/

	unsigned int    XPos;		/* Current logical cursor X position. */
	unsigned int    YPos;		/* Current logical cursor Y position. */

	int             StringLength;	/* length of string being output. */

	char            InputBuffer[INPUT_BUFFER_LENGTH];
	char            OutputBuffer[OUTPUT_BUFFER_LENGTH];

	/****** CODE *********************************************************/

	for (XPos = 0, YPos = pFontInfo->ascent;
		 (YPos + pFontInfo->descent) < PageHeightInPixels;
		 YPos += (pFontInfo->ascent + pFontInfo->descent))
		{
		if (fgets(InputBuffer, INPUT_BUFFER_LENGTH, pFile) == (char *) NULL)
			return (TRUE);

		StringLength = PrepareLine(InputBuffer, OutputBuffer);

		XDrawString(pPrinter,
					Page,
					PrintContext,
					XPos,
					YPos,
					OutputBuffer,
					StringLength);
		}

	return (FALSE);
}


/**************************************************************************
 * NAME:
 *  PrepareLine(char *pInput, char *pOutput)
 *
 * DESCRIPTION:
 *  Prepares line for X printing by stripping out nonprinting positioning
 *  characters like newlines and tabs.
 *
 * INPUTS:
 *  PARAMATERS:
 *      char *pInput        Pointer to input string.
 *      char *pOutput       Pointer to output buffer.
 *
 *  GLOBALS:
 *      None
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      pOutput buffer contains the modified version of the pInput string.
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      Number of characters in output buffer.
 *
 * NOTES:
 *  Certain characters that are used for positioning text in a text file
 *  can have X glyphs associated with them.  Some of these characters
 *  include tabs and linefeeds.  Since we are using the XLib functions for
 *  positioning and displaying text we will strip these characters out of
 *  the text so that we don't get unexpected characters in our printout.
 *
 *  We convert tabs into spaces and strip the linefeeds.
 *
 * CHANGES:
 *  None
 */

int
PrepareLine(char *pInput, char *pOutput)
{

	/*********************************************************************/

	int             CharCount;	/* Count of characters in output buffer. */
	char            CurrChar;	/* Current input character. */

	/****** CODE *********************************************************/

	for (CharCount = 0; CharCount < OUTPUT_BUFFER_LENGTH - 1;)
		{
		CurrChar = *pInput++;

		switch (CurrChar)
			{
		  case '\0':
		  case '\n':
		  case '\r':

			goto line_ready;

		  case '\t':

			do
				{

				*pOutput++ = ' ';

				} while ((++CharCount % TAB_SIZE) != 0);

			break;

		  default:

			*pOutput++ = CurrChar;
			CharCount++;
			}
		}

line_ready:

	*pOutput = '\0';

	return (CharCount);
}


/**************************************************************************
 * NAME:
 *  EjectPage(void)
 *
 * DESCRIPTION:
 *  Eject the current page.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      Display *pPrinter       X display for printer.
 *      Window   Page           X window for printer.
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  In order to eject a page and begin a new one we call XMapWindow().
 *  In effect, this maps the window to the printer.
 *
 * CHANGES:
 *  None
 */

void
EjectPage(void)
{

	/*********************************************************************/


	/****** CODE *********************************************************/

	XMapWindow(pPrinter, Page);
}


/**************************************************************************
 * NAME:
 *  FinishPrintJob(void)
 *
 * DESCRIPTION:
 *  Closes connection to X Print Server.
 *
 * INPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      Display     *pPrinter               X display for printer
 *      Window       Page;                  X window for printer.
 *      XFontStruct *pFontInfo;             X font info for printer font.
 *      GC           PrintContext;          X graphics context for printer.
 *
 * OUTPUTS:
 *  PARAMATERS:
 *      None
 *
 *  GLOBALS:
 *      None
 *
 *  RETURN:
 *      None
 *
 * NOTES:
 *  An X print job is finished when the connection to the X Print Server is
 *  closed.  A side effect of this is that the last page is automatically
 *  ejected.
 *
 * CHANGES:
 *  None
 */

void
FinishPrintJob(void)
{

	/*********************************************************************/


	/****** CODE *********************************************************/

	/*----- Free graphics context ---------------------------------------*/

	XFreeGC(pPrinter, PrintContext);

	/*----- Free font ---------------------------------------------------*/

	XFreeFont(pPrinter, pFontInfo);

	/*----- Free drawing window -----------------------------------------*/

	XDestroyWindow(pPrinter, Page);

	/*----- Close Display ending print jobs -----------------------------*/

	XCloseDisplay(pPrinter);
}
