/**********************************************************************
 *  
 *  NAME:           keybd.cpp
 *  
 *  DESCRIPTION:    keyboard interface
 *  
 *  copyright (c) 1990 J. Alan Eldridge
 * 
 *  M O D I F I C A T I O N   H I S T O R Y
 *
 *  when        who                 what
 *  -------------------------------------------------------------------
 *  08/14/90    J. Alan Eldridge    created
 *  
 *********************************************************************/

#include    "w.h"

//  defines for PC keyboard BIOS return values

#define FUNC_1      59
#define FUNC_1S     84
#define FUNC_1C     94
#define FUNC_1A     104
#define FUNC_11     133
#define FUNC_11S    135
#define FUNC_11C    137
#define FUNC_11A    139

#define NULL_CHAR   3

#define SHIFT_TAB   15

#define ALT_Q       16
#define ALT_W       17
#define ALT_E       18
#define ALT_R       19
#define ALT_T       20
#define ALT_Y       21
#define ALT_U       22
#define ALT_I       23
#define ALT_O       24
#define ALT_P       25

#define ALT_A       30
#define ALT_S       31
#define ALT_D       32
#define ALT_F       33
#define ALT_G       34
#define ALT_H       35
#define ALT_J       36
#define ALT_K       37
#define ALT_L       38

#define ALT_Z       44
#define ALT_X       45
#define ALT_C       46
#define ALT_V       47
#define ALT_B       48
#define ALT_N       49
#define ALT_M       50

#define HOME        71
#define UP          72
#define PGUP        73
#define LEFT        75
#define RIGHT       77
#define ENDKEY      79
#define DOWN        80
#define PGDN        81
#define INSRT       82
#define DEL         83

#define C_PRTSC     114
#define C_LEFT      115
#define C_RIGHT     116
#define C_END       117
#define C_PGDN      118
#define C_HOME      119

#define ALT_1       120

#define ALT_MINUS   130
#define ALT_EQUAL   131
#define C_PGUP      132

//  lookup table to map from BIOS to my key representation

typedef struct {
    uchar   biosval;
    int     newval;
} KEY_DEF;

#define NELEMS(arr) (sizeof(arr)/sizeof(arr[0]))

static KEY_DEF  keytable[] = {
    //  unmodified keys
    { NULL_CHAR,    K_NUL   },
    { FUNC_1,       K_F1    },
    { FUNC_1 + 1,   K_F2    },
    { FUNC_1 + 2,   K_F3    },
    { FUNC_1 + 3,   K_F4    },
    { FUNC_1 + 4,   K_F5    },
    { FUNC_1 + 5,   K_F6    },
    { FUNC_1 + 6,   K_F7    },
    { FUNC_1 + 7,   K_F8    },
    { FUNC_1 + 8,   K_F9    },
    { FUNC_1 + 9,   K_F10   },
    { FUNC_11,      K_F11   },
    { FUNC_11 + 1,  K_F12   },
    { HOME,         K_HOME  },
    { UP,           K_UP    },
    { PGUP,         K_PGUP  },
    { LEFT,         K_LEFT  },
    { RIGHT,        K_RIGHT },
    { ENDKEY,       K_END   },
    { DOWN,         K_DOWN  },
    { PGDN,         K_PGDN  },
    { INSRT,        K_INS   },
    { DEL,          K_DEL   },
    //  shift key down
    { FUNC_1S,      K_F1 | K_SHIFT  },
    { FUNC_1S + 1,  K_F2 | K_SHIFT  },
    { FUNC_1S + 2,  K_F3 | K_SHIFT  },
    { FUNC_1S + 3,  K_F4 | K_SHIFT  },
    { FUNC_1S + 4,  K_F5 | K_SHIFT  },
    { FUNC_1S + 5,  K_F6 | K_SHIFT  },
    { FUNC_1S + 6,  K_F7 | K_SHIFT  },
    { FUNC_1S + 7,  K_F8 | K_SHIFT  },
    { FUNC_1S + 8,  K_F9 | K_SHIFT  },
    { FUNC_1S + 9,  K_F10 | K_SHIFT },
    { FUNC_11S,     K_F11 | K_SHIFT },
    { FUNC_11S + 1, K_F12 | K_SHIFT },
    { SHIFT_TAB,    K_BACKTAB       },
    //  alt key down
    { FUNC_1A,      K_F1 | K_ALT    },
    { FUNC_1A + 1,  K_F2 | K_ALT    },
    { FUNC_1A + 2,  K_F3 | K_ALT    },
    { FUNC_1A + 3,  K_F4 | K_ALT    },
    { FUNC_1A + 4,  K_F5 | K_ALT    },
    { FUNC_1A + 5,  K_F6 | K_ALT    },
    { FUNC_1A + 6,  K_F7 | K_ALT    },
    { FUNC_1A + 7,  K_F8 | K_ALT    },
    { FUNC_1A + 8,  K_F9 | K_ALT    },
    { FUNC_1A + 9,  K_F10 | K_ALT   },
    { FUNC_11A,     K_F11 | K_ALT   },
    { FUNC_11A + 1, K_F12 | K_ALT   },
    { ALT_A,        'a' | K_ALT     },
    { ALT_B,        'b' | K_ALT     },
    { ALT_C,        'c' | K_ALT     },
    { ALT_D,        'd' | K_ALT     },
    { ALT_E,        'e' | K_ALT     },
    { ALT_F,        'f' | K_ALT     },
    { ALT_G,        'g' | K_ALT     },
    { ALT_H,        'h' | K_ALT     },
    { ALT_I,        'i' | K_ALT     },
    { ALT_J,        'j' | K_ALT     },
    { ALT_K,        'k' | K_ALT     },
    { ALT_L,        'l' | K_ALT     },
    { ALT_M,        'm' | K_ALT     },
    { ALT_N,        'n' | K_ALT     },
    { ALT_O,        'o' | K_ALT     },
    { ALT_P,        'p' | K_ALT     },
    { ALT_Q,        'q' | K_ALT     },
    { ALT_R,        'r' | K_ALT     },
    { ALT_S,        's' | K_ALT     },
    { ALT_T,        't' | K_ALT     },
    { ALT_U,        'u' | K_ALT     },
    { ALT_V,        'v' | K_ALT     },
    { ALT_W,        'w' | K_ALT     },
    { ALT_X,        'x' | K_ALT     },
    { ALT_Y,        'y' | K_ALT     },
    { ALT_Z,        'z' | K_ALT     },
    { ALT_1,        '1' | K_ALT     },
    { ALT_1 + 1,    '2' | K_ALT     },
    { ALT_1 + 2,    '3' | K_ALT     },
    { ALT_1 + 3,    '4' | K_ALT     },
    { ALT_1 + 4,    '5' | K_ALT     },
    { ALT_1 + 5,    '6' | K_ALT     },
    { ALT_1 + 6,    '7' | K_ALT     },
    { ALT_1 + 7,    '8' | K_ALT     },
    { ALT_1 + 8,    '9' | K_ALT     },
    { ALT_1 + 9,    '0' | K_ALT     },
    { ALT_MINUS,    '-' | K_ALT     },
    { ALT_EQUAL,    '=' | K_ALT     },
    //  control key down
    { FUNC_1C,      K_F1 | K_CTL    },
    { FUNC_1C + 1,  K_F2 | K_CTL    },
    { FUNC_1C + 2,  K_F3 | K_CTL    },
    { FUNC_1C + 3,  K_F4 | K_CTL    },
    { FUNC_1C + 4,  K_F5 | K_CTL    },
    { FUNC_1C + 5,  K_F6 | K_CTL    },
    { FUNC_1C + 6,  K_F7 | K_CTL    },
    { FUNC_1C + 7,  K_F8 | K_CTL    },
    { FUNC_1C + 8,  K_F9 | K_CTL    },
    { FUNC_1C + 9,  K_F10 | K_CTL   },
    { FUNC_11C,     K_F11 | K_CTL   },
    { FUNC_11C + 1, K_F12 | K_CTL   },
    { C_PRTSC,      K_PRTSC | K_CTL },
    { C_LEFT,       K_LEFT  | K_CTL },
    { C_RIGHT,      K_RIGHT | K_CTL },
    { C_END,        K_END   | K_CTL },
    { C_PGDN,       K_PGDN  | K_CTL },
    { C_HOME,       K_HOME  | K_CTL },
    { C_PGUP,       K_PGUP  | K_CTL }
};
    
//  get a character and return virtual keycode

#define IsExtendedKey(c)    (!(c)||(c)==0xE0)

int
kbd_getkey()
{
    uchar   c;
    
    c = getch();
    if (IsExtendedKey(c)) {
        c = getch();
        for (int n = 0; n < NELEMS(keytable); n++)
            if (keytable[n].biosval == c)
                return keytable[n].newval;

        return K_UNKNOWN;
    } else
        return c == K_CR ? K_NL : c;
}

//  is a particular key in an array?

int
findkey(
    int *arr,
    int key)
{
    for (int n = 0; arr[n] != K_ILLEGAL; n++)
        if (arr[n] == key) 
            return TRUE;

    return FALSE;
}
    
