/**********************************************************************
 *  
 *  cvidintf.c
 *  
 *  copyright (c) 1988,89,90,91 J. Alan Eldridge
 *
 *  video interface functions written in C
 *  
 *  11/02/91 JAE modified to work with GNU C++ (DJ Delorie's port)
 *
 *********************************************************************/

#include "curses.h"

#ifdef  __GNUC__
#include    "aedef.h"
#include    <pc.h>
#define MSC50 0
#define TURBOC 0
#define AZTEC 0
#endif

#if (MSC50 | TURBOC)
#include <dos.h>
#endif  /* (MSC50 | TURBOC) */

#if AZTEC
#include "aztecdos.h"
#endif  /* AZTEC */

#define  USE_STR_FUNCS  0  /* set to 1 to use AT video string funcs */

/**********************************************************************
 *  
 *  the cursor functions implement the concept of a cursor level:
 *  if the cursor level is 0, the cursor is visible
 *
 *  each call to hidecursor() increments the level, and each
 *  call to showcursor() decrements the level
 *  
 *  these static variable keep track of the cursor info when it is hidden
 *  
 *********************************************************************/

static  int hide_cnt    =   0;  /* # times cursor has been hidden ... */
                                /* if > 0, cursor is not visible */
static  VID_CURS_SIZE   c_size; /* start/end line of current cursor shape */    

/**********************************************************************
 *  
 *  vid_mov_curs(y, x)
 *  
 *  move the cursor to the indicated screen coordinates
 *  
 *********************************************************************/

void
vid_mov_curs(
    int y,
    int x)
{
#ifdef  __GNUC__
    ScreenSetCursor(y, x);
#else
    union REGS  reg_st;
  
    reg_st.h.ah = VB_SET_CURS_POSN;
    reg_st.h.bh = 0;
    reg_st.h.dh = y;
    reg_st.h.dl = x;

    int86(VID_INTR, &reg_st, &reg_st);
#endif
}

/**********************************************************************
 *  
 *  hidecursor()
 *  
 *  increment the cursor level, making it invisible and saving the
 *  information necessary to restore it
 *  
 *********************************************************************/

void
hidecursor(void)
{
    if (hide_cnt++ == 0) {
#ifndef __GNUC__
        union REGS  reg_st;

        reg_st.h.ah = VB_GET_CURS_INFO;
        reg_st.h.bh = 0;
        int86(VID_INTR, &reg_st, &reg_st);
        *(short *)(&c_size) = reg_st.x.cx;
        reg_st.x.cx = VB_NO_CURSOR;
        reg_st.h.ah = VB_SET_CURS_SIZE;
        int86(VID_INTR, &reg_st, &reg_st);
#endif
    }
}

/**********************************************************************
 *  
 *  showcursor()
 *  
 *  decrement the cursor level, making it visible using the saved
 *  information if the level becomes 0
 *  
 *********************************************************************/

void
showcursor(void)
{
    if (hide_cnt > 0)
        if (--hide_cnt == 0) {
#ifndef __GNUC__
            union REGS  reg_st;

            reg_st.h.ah = VB_SET_CURS_SIZE;
            reg_st.x.cx = *(short *)(&c_size);
            int86(VID_INTR, &reg_st, &reg_st);
#endif
        }
}

/**********************************************************************
 *  
 *  sizecursor()
 *  
 *  set the size of the cursor: if it is invisible, save the
 *  information for later
 *  
 *********************************************************************/

void
sizecursor(
    int start_line,
    int end_line)
{
#ifndef __GNUC__
    union REGS  reg_st;

    if (hide_cnt > 0) {
        c_size.top = start_line;
        c_size.bot = end_line;
    } else {
        reg_st.h.ah = VB_SET_CURS_SIZE;
        reg_st.h.ch = start_line;
        reg_st.h.cl = end_line;
        int86(VID_INTR, &reg_st, &reg_st);
    }
#endif
}

/**********************************************************************
 *  
 *  int
 *  vid_get_mode()
 *  
 *  return the current video mode as known by the BIOS
 *  
 *********************************************************************/

int
vid_get_mode(void)
{
#ifndef __GNUC__
    union REGS  reg_st;
    
    reg_st.h.ah = VB_GET_VID_MODE;
    int86(VID_INTR, &reg_st, &reg_st);
    return reg_st.h.al;
#else
    return ScreenMode();
#endif
}
    
/**********************************************************************
 *  
 *  long vid_buf_addr(mode)
 *  
 *  return buffer address based on video mode
 *  
 *********************************************************************/

long
vid_buf_addr(int mode)
{
#ifndef __GNUC__
    long    addr;
    
    switch (mode) {
    case VID_BandW_80x25:
    case VID_COLOR_80x25:
        addr = VID_COLOR_ADDR;
        break;
    case VID_MONO_80x25:
        addr = VID_MONO_ADDR;
        break;
    default:
        addr = 0L;
        break;
    }

    return addr;
#else
    return (long)ScreenPrimary;
#endif
}

/**********************************************************************
 *  
 *  vid_get_curs_info(), vid_set_curs_info()
 *  
 *  routines to get/set cursor size, posn
 *  
 *********************************************************************/

void
vid_get_curs_info(
    void    *posn,
    void    *size)
{
#ifndef __GNUC__
    union REGS  reg_st;
    
    reg_st.h.ah = VB_GET_CURS_INFO;
    reg_st.h.bh = VB_DISP_PAGE_0;
    int86(VID_INTR, &reg_st, &reg_st);
    if (posn)
        *(short *)posn = reg_st.x.dx;
    if (size)
        *(short *)size = reg_st.x.cx;
#else
    int row, col;

    ScreenGetCursor(&row, &col);
    if (posn)
        *(short *)posn = (row << 8) + col;
    if (size)
        *(short *)size = 7;
#endif    
}

void
vid_set_curs_info(
    void    *posn,
    void    *size)
{
#ifndef __GNUC__
    union REGS  reg_st;
    
    if (size) {
        reg_st.h.ah = VB_SET_CURS_SIZE;
        reg_st.h.bh = VB_DISP_PAGE_0;
        reg_st.x.cx = *(short *)size;
        int86(VID_INTR, &reg_st, &reg_st);
    }
#endif

    if (posn) {
#ifndef __GNUC__
        reg_st.h.ah = VB_SET_CURS_POSN;
        reg_st.h.bh = VB_DISP_PAGE_0;
        reg_st.x.dx = *(short *)posn;
        int86(VID_INTR, &reg_st, &reg_st);
#else
        short   pos = *(short*)posn;

        ScreenSetCursor(pos >> 8, pos & 0xff);
#endif    
    }
}

/**********************************************************************
 *  
 *  vid_save_scr(buffp)
 *  
 *  save the video screen image and cursor info
 *  
 *********************************************************************/

void
vid_save_scr(VID_SCR_BUFF *buffp)
{
    void far    *savep;

    savep = (void far *)(buffp->v_buff);
#if (MSC50 | TURBOC)
    movedata(__VIDADDR.segoffs.seg, 0, 
        FP_SEG(savep), FP_OFF(savep), VID_BUFF_SIZE);
#endif  /* MSC50 | TURBOC */
#if AZTEC
    movblock(__VIDADDR.farptr, savep, VID_BUFF_SIZE);
#endif  /* AZTEC */
#ifdef __GNUC__
    gettext(1, 1, 80, 25, savep);
#endif
    vid_get_curs_info(&buffp->c_posn, &buffp->c_size);
}

/**********************************************************************
 *  
 *  vid_rest_scr(buffp)
 *  
 *  undoes the effect of a call to vidsave()
 *  
 *********************************************************************/

void
vid_rest_scr(VID_SCR_BUFF *buffp)
{
    void far    *savep;
                
    savep = (void far *)(buffp->v_buff);
#if (MSC50 | TURBOC)
    movedata(FP_SEG(savep), FP_OFF(savep), 
        __VIDADDR.segoffs.seg, 0, VID_BUFF_SIZE);
#endif  /* MSC50 | TURBOC */
#if AZTEC
    movblock(savep, __VIDADDR.farptr, VID_BUFF_SIZE);
#endif  /* AZTEC */
#ifdef __GNUC__
    puttext(1, 1, 80, 25, savep);
#endif
    vid_set_curs_info(&buffp->c_posn, &buffp->c_size);
}

/**********************************************************************
 *  
 *  vid_clr_scr(int attrib)
 *  
 *  clears the PC screen using the given video attribute
 *  
 *********************************************************************/

void
vid_clr_scr(int attrib)
{
#ifndef __GNUC__
    if (__USE_BIOS) {
        union REGS  reg_st;
        
        reg_st.h.ah = VB_SCROLL_WIN_UP;
        reg_st.h.al = 0;    /* # lines to scroll (0 = erase) */
        reg_st.h.ch =       /* row # of upper left */
        reg_st.h.cl = 0;    /* col # of upper left */
        reg_st.h.dh = VID_MAX_ROWS - 1;    /* row # of lower right */
        reg_st.h.dl = VID_MAX_COLS - 1;    /* col # of lower right */
        reg_st.h.bh = attrib;
        int86(VID_INTR, &reg_st, &reg_st);
    } else {
        VIDCHR  fillch;
        
        fillch.chr = ' ';
        fillch.att = attrib;
        memsetw(__VIDADDR.farptr, &fillch, VID_BUFF_SIZE);
    }
#else
    int attr = ScreenAttrib;

    ScreenAttrib = attrib;
    ScreenClear();
    ScreenAttrib = attr;
#endif    
}

/**********************************************************************
 *  
 *  vid_upd_scr(row, col, src, cnt)
 *  
 *  update the video RAM at screen position (row, col) from
 *  the video buffer pointed to by src, transferring cnt words
 *  
 *********************************************************************/

void
vid_upd_scr(
    int     row,
    int     col,
    VIDCHR  *src,
    int     cnt)
{
#ifndef __GNUC__
    if (__USE_BIOS) {
        int         pos;

        hidecursor();
#if (TURBOC & USE_STR_FUNCS)
        if ((__MACHID.model == 0xfc) {
            /* if it's an AT ... */
            void far        *srcptr;
            struct REGPACK  regpk_st;
            
            srcptr = (void far *)src;
            regpk_st.r_dx = ((row << 8) | col);
            regpk_st.r_bp = FP_OFF(srcptr);
            regpk_st.r_es = FP_SEG(srcptr);
            regpk_st.r_ax = (VB_WRITE_CHRATT_STR << 8) | VB_NOFIX_CURSOR;
            regpk_st.r_cx = cnt;
            intr(VID_INTR, &regpk_st);
            return;
        }
#endif  /* TURBOC & USE_STR_FUNCS */
        for (pos = 0; pos < cnt; pos++) {        
            union REGS      reg_st;

            reg_st.h.ah = VB_SET_CURS_POSN;
            reg_st.h.bh = VB_DISP_PAGE_0;
            reg_st.h.dh = row;
            reg_st.h.dl = col++;
            int86(VID_INTR, &reg_st, &reg_st);
            
            reg_st.h.ah = VB_WRITE_CHR_ATT;            
            reg_st.h.bh = VB_DISP_PAGE_0;
            reg_st.h.al = src->chr;
            reg_st.h.bl = src++->att;
            reg_st.x.cx = 1;
            int86(VID_INTR, &reg_st, &reg_st);
        }
        showcursor();
    } else {
        void far    *srcp = (void far *)src;

#if (MSC50 | TURBOC)
        movedata(FP_SEG(srcp), FP_OFF(srcp), __VIDADDR.segoffs.seg, 
            ((row * VID_MAX_COLS) + col) * 2, cnt * 2);
#endif  /* MSC50 | TURBOC */
#if AZTEC
        movblock(srcp, (VIDCHR far *)(__VIDADDR.farptr) 
            + (row * VID_MAX_COLS) + col, cnt * 2);
#endif  /* AZTEC */
    }
#else
    puttext(col + 1, row + 1, col + cnt, row + 1, src);
#endif
}
