
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBCD      : Enthlt Routinen zum Ansteuerung eines CD-ROM-Laufwer-   }
{                  kes, das an der Sound-Blaster-Karte angeschlossen ist    }
{                  (in erster Linie Abspielen von Audio-CDs).               }
{                  Die Routinen greifen auf MSCDEX-Funktionen ber den Mul- }
{                  tiplex-Interrupt 2Fh sowie auf Funktionen des DOS-CD-    }
{                  ROM-Treibers zurck. Es sind nicht alle vorhandenen      }
{                  Funktionen bercksichtigt, sondern nur die, die fr das  }
{                  Abspielen von Audio-CDs von Bedeutung sind.              }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBCD.PAS                                                 }
{ entwickelt am  : 21.08.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.02                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBCD;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

USES DOS;                                              { DOS-Unit einbinden }

CONST

  {-------------------------------------------------------------------------}
  { Statuskonstanten (Zustand, in dem sich das CD-ROM-Laufwerk befindet)    }
  {-------------------------------------------------------------------------}

  cdstat_Error = $8000;                                            { Fehler }
  cdstat_Busy  = $0200;              { Busy (z.B. Abspielen einer Audio-CD) }
  cdstat_Done  = $0100;                                              { Done }

  {-------------------------------------------------------------------------}
  { Fehlerkonstanten                                                        }
  {-------------------------------------------------------------------------}

  cderr_WriteProtected      = $00;            { Medium ist schreibgeschtzt }
  cderr_UnknownDrive        = $01;                   { Unbekanntes Laufwerk }
  cderr_DriveNotReady       = $02;                  { Laufwerk nicht bereit }
  cderr_InvalidCommand      = $03;                      { Ungltiger Befehl }
  cderr_CRCError            = $04;                             { CRC-Fehler }
  cderr_InvalidCmdBlockSize = $05;          { Ungltige Kommando-Blockgre }
  cderr_SeekError           = $06;                            { Such-Fehler }
  cderr_UnknownMedia        = $07;                     { Unbekanntes Medium }
  cderr_SectorNotFound      = $08;                  { Sektor nicht gefunden }
  cderr_ReadError           = $0B;                             { Lesefehler }
  cderr_GeneralError        = $0C;                      { Genereller Fehler }
  cderr_NoMediaInDrive      = $0E;                  { Kein Medium eingelegt }
  cderr_InvalidMediaChange  = $0F;               { Ungltiger Medienwechsel }

  {-------------------------------------------------------------------------}
  { Konstanten zum Setzen des Adressiermodus                                }
  {-------------------------------------------------------------------------}

  cdmode_HSG     = 0;                           { HSG-Modus (BCD-Kodierung) }
  cdmode_RedBook = 1;                           { Red Book (Binrkodierung) }

  {-------------------------------------------------------------------------}
  { Befehlscodes                                                            }
  {-------------------------------------------------------------------------}

  cdcode_Init          = 0;
  cdcode_IOCTLInput    = 3;
  cdcode_IOCTLOutput   = 12;
  cdcode_Seek          = 131;
  cdcode_PlayAudio     = 132;
  cdcode_StopAudio     = 133;
  cdcode_ContinueAudio = 136;
  cdcode_InputFlush    = 7;

  {-------------------------------------------------------------------------}
  { Gertestatus- und Eigenschaftskonstanten (zu unterscheiden von den an-  }
  { deren Statuskonstanten)                                                 }
  {-------------------------------------------------------------------------}

  { Bit 0: gesetzt  = Laufwerkstr offen       }
  {        gelscht = Laufwerkstr geschlossen }

  devstat_DoorOpen   = $0001;
  devstat_DoorClosed = $0000;

  { Bit 1: gesetzt  = Tr unverriegelt }
  {        gelscht = Tr verriegelt   }

  devstat_DoorNotBolted = $0002;
  devstat_DoorBolted    = $0000;

  { Bit 2: im Zusammenhang mit Audio-CDs ohne Bedeutung }

  { Bit 3: gesetzt  = Schreiben und Lesen mglich }
  {        gelscht = nur Lesen mglich           }

  devstat_ReadAndWrite = $0008;
  devstat_ReadOnly     = $0000;

  { Bit 4: gesetzt  = Bearbeitung von Daten, Audio und Video mglich }
  {        gelscht = nur Bearbeitung von Daten mglich              }

  devstat_DataAudioVideo = $0010;
  devstat_DataOnly       = $0000;

  { Bit 5: im Zusammenhang mit Audio-CDs ohne Bedeutung }

  { Bit 6: reserviert }

  { Bit 7: im Zusammenhang mit Musik-CDs ohne Bedeutung }

  { Bit 8: gesetzt  = Manipulation des Audio-Kanals       }
  {        gelscht = keine Manipulation des Audio-Kanals }

  devstat_AudioManipulation   = $0100;
  devstat_NoAudioManipulation = $0000;

  { Bit 9: gesetzt  = HSG- (BCD-Kodierung) und RedBook- }
  {                   Adressierung (Binrkodierung)     }
  {        gelscht = HSG-Adressierung                  }

  devstat_HSGAndRedBook = $0200;
  devstat_HSGOnly       = $0000;

  { Bit 10: reserviert }

  { Bit 11: gesetzt  = keine CD im Laufwerk }
  {         gelscht = CD eingelegt         }

  devstat_NoCDInDrive = $0800;
  devstat_CDInDrive   = $0000;

  { Bit 12: im Zusammenhang mit Audio-CDs ohne besondere Bedeutung }

  { Bits 13-31: reserviert }

TYPE

  {=========================================================================}
  { DWord: Doppelwort                                                       }
  {=========================================================================}

  DWord = RECORD
    CASE Boolean OF
      TRUE :
        (LoWord, HiWord : Word);
      FALSE :
        (LI : LongInt);
  END;

  {=========================================================================}
  { PReqHeader: Zeiger auf TReqHeader                                       }
  {=========================================================================}
  { TReqHeader: Request Header                                              }
  {=========================================================================}

  PReqHeader = ^TReqHeader;
  TReqHeader = RECORD
    Len      : Byte;                                        { Lnge in Byte }
    SubUnit  : Byte;                                              { Subunit }
    CmdCode  : Byte;                                          { Befehlscode }
    Status   : Word;                           { Status (aktueller Zustand) }
    Reserved : ARRAY[1..8] OF Char;                            { reserviert }
  END;

  {=========================================================================}
  { PStruct_Init: Zeiger auf TStruct_Init                                   }
  {=========================================================================}
  { TStruct_Init: Struktur zum Aufruf der Initialisierungsroutine des DOS-  }
  {               CD-ROM-Treibers. Nach der Initialisierung wird immer ein  }
  {               Fehler gemeldet.                                          }
  {               Der Befehlscode der Routine ist 0.                        }
  {=========================================================================}

  PStruct_Init = ^TStruct_Init;
  TStruct_Init = RECORD
    ReqHeader   : TReqHeader;                              { Request Header }
    Drives      : Byte;                    { Anzahl der Laufwerke (immer 0) }
    EndAddress  : Pointer;                        { Endadresse des Treibers }
    BPBFieldPtr : Pointer;                            { Zeiger auf BPB-Feld }
    BlockDevNmb : Byte;                 { Block-Device-Nummer (ebenfalls 0) }
  END;

  {=========================================================================}
  { PStruct_IOCTLInput: Zeiger auf TStruct_IOCTLInput                       }
  {=========================================================================}
  { TStruct_IOCTLInput: Struktur zum Aufruf von Funktionen, die einer ei-   }
  {                     genstndigen Gruppe angehren, ber die man diverse }
  {                     Laufwerksparameter und -aktivitten ermitteln kann. }
  {                     Der Befehlscode ist 3.                              }
  {=========================================================================}

  PStruct_IOCTLInput = ^TStruct_IOCTLInput;
  TStruct_IOCTLInput = RECORD
    ReqHeader       : TReqHeader;                          { Request Header }
    MediaDescriptor : Byte;            { Media-Deskriptor aus BPB (immer 0) }
    BufPtr          : Pointer;                               { Pufferzeiger }
    BufSize         : Word;                        { Anzahl Bytes im Puffer }
    StartSector     : Word;                     { Startsektor (ebenfalls 0) }
    VolumePtr       : Pointer;         { Zeiger auf Volume (Fehler, auch 0) }
  END;

  {=========================================================================}
  { PStruct_IOCTLOutput: Zeiger auf TStruct_IOCTLOutput                     }
  {=========================================================================}
  { TStruct_IOCTLOutput: Struktur zum Aufruf von Funktionen, die einer ei-  }
  {                      genstndigen Gruppe angehren, ber die man diver- }
  {                      se Laufwerksparameter und -aktivitten (hnlich    }
  {                      wie mit TStruct_IOCTLInput) ermitteln kann.        }
  {                      Der Befehlscode ist 12.                            }
  {=========================================================================}

  PStruct_IOCTLOutput = ^TStruct_IOCTLOutput;
  TStruct_IOCTLOutput = RECORD
    ReqHeader       : TReqHeader;                          { Request Header }
    MediaDescriptor : Byte;            { Media-Deskriptor aus BPB (immer 0) }
    BufPtr          : Pointer;                               { Pufferzeiger }
    BufSize         : Word;                        { Anzahl Bytes im Puffer }
    StartSector     : Word;                     { Startsektor (ebenfalls 0) }
    VolumePtr       : Pointer;                 { Zeiger auf Volume (Fehler) }
  END;

  {=========================================================================}
  { PStruct_Seek: Zeiger auf TStruct_Seek                                   }
  {=========================================================================}
  { TStruct_Seek: Struktur zum Aufruf der Routine, mit deren Hilfe der      }
  {               Laufwerkskopf positioniert werden kann. Das ausfhrende   }
  {               Programm erhlt sofort die Kontrolle zurck.              }
  {               Der Befehlscode der Routine ist 131.                      }
  {=========================================================================}

  PStruct_Seek = ^TStruct_Seek;
  TStruct_Seek = RECORD
    ReqHeader   : TReqHeader;                              { Request Header }
    Mode        : Byte;            { Adressiermodus (0 = HSG, 1 = Red Book) }
    BufPtr      : Pointer;                  { Pufferzeiger (wird ignoriert) }
    Sectors     : Word;                   { Sektorenanzahl (wird ignoriert) }
    StartSector : LongInt; { Startsektor (Frame:Minute:Sec.-Binrkodierung) }
  END;

  {=========================================================================}
  { PStruct_PlayAudio: Zeiger auf TStruct_PlayAudio                         }
  {=========================================================================}
  { TStruct_PlayAudio: Struktur zum Aufruf der Routine, die das Abspielen   }
  {                    von Audio-Daten veranlat. Derartige Aktivitt wird  }
  {                    durch das Busy-Bit in den Status-Flags angezeigt.    }
  {                    Der Befehlscode der Routine ist 132.                 }
  {=========================================================================}

  PStruct_PlayAudio = ^TStruct_PlayAudio;
  TStruct_PlayAudio = RECORD
    ReqHeader   : TReqHeader;                              { Request Header }
    Mode        : Byte;            { Adressiermodus (0 = HSG, 1 = Red Book) }
    StartSector : LongInt; { Startsektor (Frame:Minute:Sec.-Binrkodierung) }
    Sectors     : LongInt;             { Anzahl der abzuspielenden Sektoren }
  END;

  {=========================================================================}
  { PStruct_StopAudio: Zeiger auf TStruct_StopAudio                         }
  {=========================================================================}
  { TStruct_StopAudio: Struktur zum Aufruf der Routine, die eine Unterbre-  }
  {                    chung (Stop) des Abspielvorgangs (Abspielen von Au-  }
  {                    dio-Daten) veranlat. Ein zweiter Aufruf der Routine }
  {                    stoppt das CD-ROM-Laufwerk endgltig.                }
  {                    Der Befehlscode der Routine ist 133.                 }
  {=========================================================================}

  PStruct_StopAudio = ^TStruct_StopAudio;
  TStruct_StopAudio = RECORD
    ReqHeader : TReqHeader;                                { Request Header }
  END;

  {=========================================================================}
  { PStruct_ContinueAudio: Zeiger auf TStruct_ContinueAudio                 }
  {=========================================================================}
  { TStruct_ContinueAudio: Struktur zum Aufruf der Routine, die die Fort-   }
  {                        setzung eines zuvor unterbrochenen Abspielvor-   }
  {                        gangs veranlat.                                 }
  {                        Der Befehlscode der Routine ist 136.             }
  {=========================================================================}

  PStruct_ContinueAudio = ^TStruct_ContinueAudio;
  TStruct_ContinueAudio = RECORD
    ReqHeader : TReqHeader;                                { Request Header }
  END;

  {=========================================================================}
  { PStruct_InputFlush: Zeiger auf TStruct_InputFlush                       }
  {=========================================================================}
  { TStruct_InputFlush: Struktur zum Aufruf der Routine, die den Treiber    }
  {                     auffordert, alle Input-Puffer und anliegenden An-   }
  {                     forderungen zu lschen.                             }
  {                     Der Befehlscode der Routine ist 7.                  }
  {=========================================================================}

  PStruct_InputFlush = ^TStruct_InputFlush;
  TStruct_InputFlush = RECORD
    ReqHeader : TReqHeader;                                { Request Header }
  END;

  {=========================================================================}
  { TCDInfo: Enthlt Felder zum Speichern von Informationen ber eine       }
  {          Audio-CD.                                                      }
  {=========================================================================}

  TCDInfo = RECORD
    MinTrack     : Byte;                             { kleinste Tracknummer }
    MaxTrack     : Byte;                               { grte Tracknummer }
    LeadOutTrack : LongInt;    { sog. Lead-Out-Tracknummer (letzter Sektor) }
  END;

  {=========================================================================}
  { TQChannelInfo: Enthlt Felder zum Speichern von Informationen ber den  }
  {                sog. Audio-Q-Channel.                                    }
  {=========================================================================}

  TQChannelInfo = RECORD
    CtrlAndAdrByte : Byte;                          { CONTROL- und ADR-Byte }
    Track          : Byte;                                    { Tracknummer }
    Point          : Byte;                                  { Index (POINT) }
    Min            : Byte;   { Minuten-Angabe (in bezug auf den akt. Track) }
    Sec            : Byte;  { Sekunden-Angabe (in bezug auf den akt. Track) }
    Frame          : Byte;                                          { FRAME }
    Zero           : Byte;                                           { ZERO }
    CDMin          : Byte;     { Minuten-Angabe (relativ zum Anfang der CD) }
    CDSec          : Byte;    { Sekunden-Angabe (relativ zum Anfang der CD) }
    CDFrame        : Byte;                                         { AFRAME }
  END;

  {=========================================================================}
  { TCDTime: Zeit (Minute, Sekunde, Millisekunde)                           }
  {=========================================================================}

  TCDTime = RECORD
    Min, Sec, Sec100 : Byte;
  END;

{===========================================================================}
{ Prozedur SBCDGetDriveNmb: Ermittelt die Anzahl der vorhandenen CD-ROM-    }
{                           Laufwerke und die dem ersten zugeordnete Lauf-  }
{                           werksnummer (A = 0, B = 1, C = 2, usw.). Diese  }
{                           Routine gilt als Installationsprfung fr das   }
{                           Programm MSCDEX.EXE.                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Drives = Anzahl der CD-ROM-Laufwerke                             }
{          Nmb    = Nummer des ersten Laufwerkes                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDGetDriveNmb(VAR Drives, Nmb : Word);

{===========================================================================}
{ Funktion SBCDCheckDrive: Prft, ob das angegebene CD-ROM-Laufwerk exis-   }
{                          tiert.                                           }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes                                      }
{ Ausgabe: TRUE, wenn das Laufwerk existiert, sonst FALSE                   }
{---------------------------------------------------------------------------}

FUNCTION SBCDCheckDrive(Nmb : Word) : Boolean;

{===========================================================================}
{ Prozedur SBCDGetMSCDEXVer: Ermittelt die Nummer der MSCDEX-Version        }
{                            (Haupt- und Unterversion). Ein Programm kann   }
{                            die Versionsnummer berprfen, um wissen zu    }
{                            knnen, welche Funktionen untersttzt werden.  }
{                            Die Funktion ist erst ab der Version 2.00 ver- }
{                            fgbar.                                        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Major = Hauptversion                                             }
{          Minor = Unterversion                                             }
{---------------------------------------------------------------------------}

PROCEDURE SBCDGetMSCDEXVer(VAR Major, Minor : Byte);

{===========================================================================}
{ Prozedur SBCDSendDriverReq: Ermglicht den Zugriff auf den eigentlichen   }
{                             DOS-CD-ROM-Treiber, der das Laufwerk direkt   }
{                             ansteuert. Die Selektion erfolgt ber die     }
{                             Laufwerksnummer.                              }
{===========================================================================}
{ Eingabe: Nmb    = Nummer des CD-ROM-Laufwerkes                            }
{          Struct = Zeiger auf die Struktur zum Aufruf einer Routine des    }
{                   DOS-CD-ROM-Treibers (CD Driver Request Header)          }
{ Ausgabe: Struct (Erluterung s. Eingabe)                                  }
{---------------------------------------------------------------------------}

PROCEDURE SBCDSendDriverReq(Nmb : Word; VAR Struct : Pointer);

{===========================================================================}
{ Funktion SBCDSeek: Positioniert den Kopf des CD-ROM-Laufwerkes.           }
{===========================================================================}
{ Eingabe: Nmb         = Nummer des Laufwerkes (mu vorher mit Hilfe von    }
{                        GetCDDriveNmb ermittelt werden)                    }
{          Mode        = Adressiermodus (0 = HSG, 1 = Red Book)             }
{          StartSector = Startsektor (Frame:Minute:Second-Binrkodierung)   }
{ Ausgabe: Status (aktueller Zustand) des Laufwerkes                        }
{---------------------------------------------------------------------------}

FUNCTION SBCDSeek(Nmb : Word; Mode : Byte; StartSector : LongInt) : Word;

{===========================================================================}
{ Funktion SBCDPlayAudio: Spielt die Audio-Daten ab.                        }
{===========================================================================}
{ Eingabe: Nmb         = Nummer des Laufwerkes (mu vorher mit Hilfe von    }
{                        GetCDDriveNmb ermittelt werden)                    }
{          Mode        = Adressiermodus (0 = HSG, 1 = Red Book)             }
{          StartSector = Startsektor (Frame:Minute:Second-Binrkodierung)   }
{          Sectors     = Anzahl der abzuspielenden Sektoren                 }
{ Ausgabe: Status (aktueller Zustand) des Laufwerkes                        }
{---------------------------------------------------------------------------}

FUNCTION SBCDPlayAudio(Nmb : Word; Mode : Byte; StartSector,
  Sectors : LongInt) : Word;

{===========================================================================}
{ Funktion SBCDStopAudio: Unterbricht den Abspielvorgang einer Audio-CD.    }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: Status (aktueller Zustand) des Laufwerkes                        }
{---------------------------------------------------------------------------}

FUNCTION SBCDStopAudio(Nmb : Word) : Word;

{===========================================================================}
{ Funktion SBCDContinueAudio: Setzt den zuvor unterbrochenen Abspielvorgang }
{                             einer Audio-CD wieder fort.                   }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: Status (aktueller Zustand) des Laufwerkes                        }
{---------------------------------------------------------------------------}

FUNCTION SBCDContinueAudio(Nmb : Word) : Word;

{===========================================================================}
{ Funktion SBCDHeadPos: Liefert die aktuelle Position des Kopfes in Abhn-  }
{                       gigkeit von dem Adressiermodus.                     }
{===========================================================================}
{ Eingabe: Nmb  = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-    }
{                 DriveNmb ermittelt werden)                                }
{          Mode = Adressiermodus (0 = HSG, 1 = Red Book)                    }
{ Ausgabe: Kopfposition                                                     }
{---------------------------------------------------------------------------}

FUNCTION SBCDHeadPos(Nmb : Word; Mode : Byte) : LongInt;

{===========================================================================}
{ Funktion SBCDDevStatus: Erfragt den aktuellen Gertestatus.               }
{===========================================================================}
{ Eingabe: Nmb    = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-  }
{                   DriveNmb ermittelt werden)                              }
{ Ausgabe: Status = Gertestatus                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDGetDevStatus(Nmb : Word; VAR Status : DWord);

{===========================================================================}
{ Funktion SBCDVolumeSize: Ermittelt die Gesamtanzahl der Sektoren einer    }
{                          CD (HSG-Adressiermodus).                         }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: Anzahl der Sektoren                                              }
{---------------------------------------------------------------------------}

FUNCTION SBCDVolumeSize(Nmb : Word) : LongInt;

{===========================================================================}
{ Funktion SBCDMediaChange: Liefert TRUE zurck, wenn zwischenzeitlich die  }
{                           CD gewechselt wurde, ansonsten FALSE.           }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBCDMediaChange(Nmb : Word) : Boolean;

{===========================================================================}
{ Prozedur SBCDGetInfo: Liefert Informationen ber die Audio-Daten.         }
{===========================================================================}
{ Eingabe: Nmb  = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-    }
{                 DriveNmb ermittelt werden)                                }
{ Ausgabe: Info = Audio-Informationen                                       }
{---------------------------------------------------------------------------}

PROCEDURE SBCDGetInfo(Nmb : Word; VAR Info : TCDInfo);

{===========================================================================}
{ Funktion SBCDTrackToSector: Rechnet die angegebene Tracknummer in eine    }
{                             Sektornummer (Startpunkt) im RedBook-Adres-   }
{                             siermodus um.                                 }
{===========================================================================}
{ Eingabe: Nmb   = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-   }
{                  DriveNmb ermittelt werden)                               }
{          Track = Tracknummer                                              }
{ Ausgabe: Nummer des Startsektors                                          }
{---------------------------------------------------------------------------}

FUNCTION SBCDTrackToSector(Nmb : Word; Track : Byte) : LongInt;

{===========================================================================}
{ Prozedur SBCDGetQChannelInfo: Liest die aktulle Position des Laufwerks-   }
{                               kopfes und gibt Informationen ber die      }
{                               aktuelle Tracknummer sowie ber Zeitangaben }
{                               (Zeit in bezug auf den aktuellen Track und  }
{                               relativ zum Anfang der CD) zurck.          }
{===========================================================================}
{ Eingabe: Nmb  = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-    }
{                 DriveNmb ermittelt werden)                                }
{ Ausgabe: Info = Informationen                                             }
{---------------------------------------------------------------------------}

PROCEDURE SBCDGetQChannelInfo(Nmb : Word; VAR Info : TQChannelInfo);

{===========================================================================}
{ Funktion SBCDPaused: Liefert TRUE zurck, wenn sich das CD-ROM-Laufwerk   }
{                      im Pause-Modus befindet, ansonsten FALSE.            }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SBCDPaused(Nmb : Word) : Boolean;

{===========================================================================}
{ Prozedur SBCDEjectDisk: Entriegelt die Laufwerkstr und wirft die CD aus. }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDEjectDisk(Nmb : Word);

{===========================================================================}
{ Prozedur SBCDLockDoor: Verriegelt die Laufwerkstr.                       }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDLockDoor(Nmb : Word);

{===========================================================================}
{ Prozedur SBCDUnlockDoor: Entriegelt die Laufwerkstr.                     }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDUnlockDoor(Nmb : Word);

{===========================================================================}
{ Prozedur SBCDResetDrive: Fhrt einen Reset (Zurcksetzen) des Gertetrei- }
{                          bers aus. Danach wird das CD-ROM-Laufwerk ini-   }
{                          tialisiert.                                      }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDResetDrive(Nmb : Word);

{===========================================================================}
{ Prozedur SBCDCloseTray: Gibt dem Laufwerk den Befehl, die Tr zu schlie-  }
{                         en und wenn vorhanden, die Schublade einzufah-   }
{                         ren.                                              }
{===========================================================================}
{ Eingabe: Nmb = Nummer des Laufwerkes (mu vorher mit Hilfe von GetCD-     }
{                DriveNmb ermittelt werden)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBCDCloseTray(Nmb : Word);

{===========================================================================}
{ Prozedur SBCDSectorToTime: Errechnet aus der vorgebenen Sektornummer die  }
{                            Zeit (Minute, Sekunde, Millisekunde). Ein Sek- }
{                            tor hat eine Lnge von 1/75 Sekunde.           }
{===========================================================================}
{ Eingabe: Sector = Nummer des Sektors (HSG-Adressiermodus)                 }
{ Ausgabe: Time   = Zeit                                                    }
{---------------------------------------------------------------------------}

PROCEDURE SBCDSectorToTime(Sector : LongInt; VAR Time : TCDTime);

{===========================================================================}
{ Funktion SBCDTimeToSector: Errechnet aus der vorgegebenen Zeit die Sek-   }
{                            tornummer (HSG-Adressiermodus).                }
{===========================================================================}
{ Eingabe: Time   = Zeit                                                    }
{ Ausgabe: Sector = Sektornummer                                            }
{---------------------------------------------------------------------------}

FUNCTION SBCDTimeToSector(Time : TCDTime) : LongInt;

IMPLEMENTATION

VAR
  Regs : Registers;                                    { Prozessor-Register }

PROCEDURE SBCDGetDriveNmb(VAR Drives, Nmb : Word);

BEGIN
  Regs.AX := $1500;                  { Funktionsnummer in AX-Register laden }
  Regs.BX := $0000;                              { BX-Register zurcksetzen }
  Intr($2F, Regs);                           { Multiplex-Interrupt auslsen }
  Drives := Regs.BX;                   { Anzahl der Laufwerke zurckliefern }
  Nmb := Regs.CX;              { Nummer des ersten Laufwerkes zurckliefern }
END;

FUNCTION SBCDCheckDrive(Nmb : Word) : Boolean;

BEGIN
  SBCDCheckDrive := FALSE;
  Regs.AX := $150B;                  { Funktionsnummer in AX-Register laden }
  Regs.CX := Nmb;                    { Laufwerksnummer in CX-Register laden }
  Intr($2F, Regs);                           { Multiplex-Interrupt auslsen }
  IF Regs.AX = $0000 THEN                  { Wird das Laufwerk untersttzt? }
    IF Regs.BX = $ADAD THEN                            { ja, Signatur o.k.? }
      SBCDCheckDrive := TRUE;             { ja, also existiert das Laufwerk }
END;

PROCEDURE SBCDGetMSCDEXVer(VAR Major, Minor : Byte);

BEGIN
  Regs.AX := $150C;                  { Funktionsnummer in AX-Register laden }
  Intr($2F, Regs);                           { Multiplex-Interrupt auslsen }
  Major := Regs.BH;                                  { Hauptversion liefern }
  Minor := Regs.BL;                                { Unterversion ermitteln }
END;

PROCEDURE SBCDSendDriverReq(Nmb : Word; VAR Struct : Pointer);

BEGIN
  Regs.AX := $1510;                  { Funktionsnummer in AX-Register laden }
  Regs.CX := Nmb;                                 { Laufwerksnummer nach CX }
  Regs.ES := Seg(Struct^);   { Segment-Adresse des CD Driver Request Header }
  Regs.BX := Ofs(Struct^);                                 { Offset-Adresse }
  Intr($2F, Regs);                           { Multiplex-Interrupt auslsen }
END;

FUNCTION SBCDSeek(Nmb : Word; Mode : Byte; StartSector : LongInt) : Word;

VAR
  P : PStruct_Seek; { Zeiger auf die Struktur zum Aufruf der Treiberroutine }

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_Seek;
  P^.Mode := Mode;
  P^.StartSector := StartSector;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDSeek := P^.ReqHeader.Status;               { Statuscode zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDPlayAudio(Nmb : Word; Mode : Byte; StartSector,
  Sectors : LongInt) : Word;

VAR
  P : PStruct_PlayAudio;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_PlayAudio;
  P^.Mode := Mode;
  P^.StartSector := StartSector;
  P^.Sectors := Sectors;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDPlayAudio := P^.ReqHeader.Status;          { Statuscode zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDStopAudio(Nmb : Word) : Word;

VAR
  P : PStruct_StopAudio;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_StopAudio;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDStopAudio := P^.ReqHeader.Status;          { Statuscode zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDContinueAudio(Nmb : Word) : Word;

VAR
  P : PStruct_ContinueAudio;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_ContinueAudio;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDContinueAudio := P^.ReqHeader.Status;      { Statuscode zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDHeadPos(Nmb : Word; Mode : Byte) : LongInt;

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
    Mode    : Byte;                                        { Adressiermodus }
    HeadPos : LongInt;                                       { Kopfposition }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 1;
  S.Mode := Mode;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDHeadPos := S.HeadPos;                    { Kopfposition zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDGetDevStatus(Nmb : Word; VAR Status : DWord);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
    Status  : DWord;                                         { Gertestatus }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 6;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Status := S.Status;                          { Gertestatus zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDVolumeSize(Nmb : Word) : LongInt;

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
    VolSize : LongInt;                                       { Volume-Gre }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 8;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDVolumeSize := S.VolSize;          { Anzahl der Sektoren zurckliefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDMediaChange(Nmb : Word) : Boolean;

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command   : Byte;                                            { Kommando }
    MediaByte : Byte; { 01h = kein CD-Wechsel, FFh = Wechsel, 00h = ungewi }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  SBCDMediaChange := FALSE;
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 9;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  IF S.MediaByte = $FF THEN                      { Wurde die CD gewechselt? }
    SBCDMediaChange := TRUE;                                           { ja }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDGetInfo(Nmb : Word; VAR Info : TCDInfo);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
    Info    : TCDInfo;                                { Audio-Informationen }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 10;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Info := S.Info;                             { Audio-Informationen liefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDTrackToSector(Nmb : Word; Track : Byte) : LongInt;

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command   : Byte;                                            { Kommando }
    Track     : Byte;                                         { Tracknummer }
    Sector    : LongInt;                             { Sektornummer (Start) }
    TrackInfo : Byte;                                          { Track-Info }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 11;
  S.Track := Track;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  SBCDTrackToSector := S.Sector;          { Nummer des Startsektors liefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDGetQChannelInfo(Nmb : Word; VAR Info : TQChannelInfo);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
    Info    : TQChannelInfo;                         { Audio-Q-Channel-Info }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 12;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }

  { Tracknummer-Korrektur }

  IF S.Info.Track >= 48 THEN
    Dec(S.Info.Track, 18)
  ELSE
    IF S.Info.Track >= 32 THEN
      Dec(S.Info.Track, 12)
    ELSE
      IF S.Info.Track >= 16 THEN
        Dec(S.Info.Track, 6);

  Info := S.Info;                            { Audio-Q-Channel-Info liefern }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

FUNCTION SBCDPaused(Nmb : Word) : Boolean;

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command       : Byte;                                        { Kommando }
    AudioStatus   : Word;              { Audio-Status (Bit 0 ist Pause-Bit) }
    StartLastPlay : LongInt;            { Start des letzten Abspielvorgangs }
    EndLastPlay   : LongInt;             { Ende des letzten Abspielvorgangs }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLInput;

BEGIN
  SBCDPaused := FALSE;
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLInput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 15;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  IF (S.AudioStatus AND 1) = 1 THEN                    { Pause-Bit gesetzt? }
    SBCDPaused := TRUE;                                                { ja }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDEjectDisk(Nmb : Word);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLOutput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLOutput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 0;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDLockDoor(Nmb : Word);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command    : Byte;                                           { Kommando }
    LockUnlock : Byte;                            { Ver-/Entriegeln der Tr }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLOutput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLOutput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 1;
  S.LockUnlock := 1;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDUnlockDoor(Nmb : Word);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command    : Byte;                                           { Kommando }
    LockUnlock : Byte;                            { Ver-/Entriegeln der Tr }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLOutput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }
  FillChar(P^, SizeOf(P^), 0);               { Puffer mit dem Wert 0 fllen }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLOutput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 1;
  S.LockUnlock := 0;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDResetDrive(Nmb : Word);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLOutput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLOutput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 2;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDCloseTray(Nmb : Word);

TYPE
  TStruct = RECORD                 { Struktur zum Aufruf der Treiberroutine }
    Command : Byte;                                              { Kommando }
  END;

VAR
  S : TStruct;
  P : PStruct_IOCTLOutput;

BEGIN
  New(P);                                       { Speicherplatz reservieren }

  { Aufrufparameter setzen }

  P^.ReqHeader.Len := SizeOf(P^);
  P^.ReqHeader.CmdCode := cdcode_IOCTLOutput;
  P^.BufPtr := @S;
  P^.BufSize := SizeOf(TStruct);
  S.Command := 5;

  SBCDSendDriverReq(Nmb, Pointer(P));             { Treiberroutine aufrufen }
  Dispose(P);                                     { Speicherplatz freigeben }
END;

PROCEDURE SBCDSectorToTime(Sector : LongInt; VAR Time : TCDTime);

VAR
  R : Real;                                                 { Hilfsvariable }

BEGIN
  R := Sector/75;
  WITH Time DO
    BEGIN
      Min := Trunc(R/60);
      Sec := Trunc(R)-(Min*60);
      Sec100 := Trunc(Frac(R)*100);
    END;
END;

FUNCTION SBCDTimeToSector(Time : TCDTime) : LongInt;

VAR
  R : Real;                                                 { Hilfsvariable }

BEGIN
  WITH Time DO
    BEGIN
      IF Sec > 59 THEN
        Sec := 59;
      IF Sec100 > 99 THEN
        Sec100 := 99;
      R := (Min*60)+Sec+(Sec100/100);
      SBCDTimeToSector := Round(R*75);
    END;
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

END.
