;****************************************************************************
;
;  File              : recset.asm
;  Date Created      : 3/20/89
;  Description       : Module to initializing the hardware timer on the PC
;                      for recording (non-DMA).
;
;  Last Modification : February 8, 1992
;
;  Additional Notes  :
;
;****************************************************************************
;            Copyright (c) 1992,  Covox, Inc.  All Rights Reserved
;****************************************************************************


IFNDEF NO_DOS_SEG
  DOSSEG
ENDIF


      INCLUDE  model.inc
      INCLUDE  timer.inc
      INCLUDE  system.inc
      INCLUDE  getxbyte.inc


      EXTRN    _sr_reset_8253_timer:PROC
      EXTRN    _sr_set_8253_timer:PROC

      .DATA

VM_USED_MASK      EQU   1    ; bit mask if VM card used
LPT_USED_MASK     EQU   2    ; bit mask if External used
VALUE_USED_MASK   EQU   16   ; bit mask if using port address passed


vmII_table        DW   22fh, 24fh, 28fh, 2cfh
adread            DW   0
dont_use_timer    DB   0
hardware_type     DB   0     ; bit flags of what hardware type is being used
sample_rate       DB   0


      .CODE

      PUBLIC   recordByteInit
      PUBLIC   recordByteUninit
      PUBLIC   recordByte

;*********************************************************************
;**
;**  recordByteInit:  Initialize the recording
;**
;**  Call to initialize the hardware before any record, sets up port numbers
;**  and varables, as well as hardware.
;**
;**  Called by C as: recordByteInit(port,rate,no_timer_flag);
;**  C declaration:  int recordByteInit(int, int, int);
;**
;**  Return:
;**   The phyical port number used
;**  Passed: 
;**   port_in: The port where sound is to come in.  If the number is 0-3
;**            then the Voice Master II ports 22f, 24f, 28f, or 2cf are used
;**            respectively. If port number between 4-6 then external Covox
;**            digitizer is used.Otherwise the value is considered to be the
;**            actual port address. (7 and 8 are reserved).
;**
;**   rate:    If 0, default (132) is used.  If greater then 209 uses 209
;**            Otherwise the value is used as is.
;**
;**   no_timer_flag: If zero uses timer2 of the IBM 8253, if not zero don't
;**            wait for timer when reading bytes.
;**************************************************************************

recordByteInit   PROC USES DS,\
      port_in : WORD, \
      rate : BYTE, \
      no_timer_flag : WORD

      mov   ax, @data
      mov   ds, ax

      mov   bx, port_in
      cmp   bx, 04
      jb    SHORT use_vmII_ports
      cmp   bx, 07
      jb    SHORT use_lpt123_ports
      mov   hardware_type, VALUE_USED_MASK   ; Set so hardware is port value passed
      mov   adread, bx
      jmp   end_translation

use_vmII_ports:
      mov   hardware_type, VM_USED_MASK      ; Set so hardware Voice Master Card
      shl   bx, 01
      mov   ax, [vmII_table + bx]
      mov   adread, ax
      mov   dx, ax
      sub   dx, 2
      out   dx, al           ; on old VM set to voice channel
      jmp   end_translation

use_lpt123_ports:
      ASSUME   es:system_data
      push  es
      mov   ax, system_data
      mov   es, ax
      sub   bx, 04           ;convert from 4-6 to 0-2
      shl   bx, 01           ;Now convert to a pointer
      mov   ax, es:[bx + lpt1]
      pop   es

      ASSUME   es:NOTHING

      mov   adread, ax
      or    ax, ax
      jz    SHORT exit       ; exit if lpt address zero, i.e. not installed

      mov   hardware_type, LPT_USED_MASK

      external_on ax         ; Set printer port address for macro

      get_x_byte  adread, ah ; clear first byte of A to D

      jmp   SHORT test_for_timer

end_translation:
      mov   dx, adread       ; clear first byte of A to D
      in    al, dx      

test_for_timer:
      mov   ax, no_timer_flag
      or    ax, ax           ; make sure use timer flag 0 or 1
      jnz   SHORT @F         ; jump if not using timer
      call  _sr_set_8253_timer
      xor   ax, ax
      jmp   SHORT set_use_flag

@@:
      mov   ax, 1

set_use_flag:
      mov   dont_use_timer, al

      xor   ah, ah
      mov   al, rate
      cmp   ax, 0
      jne   SHORT @F
      xor   ah, ah
      mov   al, 132          ; set default rate 
      jmp   SHORT rate_done

@@:
      cmp   al, 209
      jbe   SHORT rate_done
      xor   ah, ah
      mov   al, 209          ; if rate over 209 set to 209

rate_done:
      mov   sample_rate, al

exit:
      mov   ax, adread       ; return phyical port address

      ret
recordByteInit   ENDP

;*********************************************************************
;**
;**  recordByteUninit:  Uninitialize from recording, call after recordByteInit
;**
;**  Called by C as: recordByteUninit();
;**  C declaration:  int recordByteUninit(void);
;**
;**************************************************************************

recordByteUninit  PROC USES DS

      mov   ax, @data
      mov   ds, ax

      cmp   dont_use_timer, 0
      jnz   SHORT @F
      call  _sr_reset_8253_timer

@@:
      test  hardware_type, LPT_USED_MASK
      jz    SHORT @F
      external_off adread

@@:
      xor   ax, ax
      ret
recordByteUninit   ENDP

;*********************************************************************
;**
;**  recordByte:  Get byte of A/D data from device selected with 
;**       init_recording
;**
;**  C declaration:  int recordByte(void);
;**
;**  Assembly language programs that call this function assume that
;**  only AX, and DX are changed.
;**
;**************************************************************************
recordByte   PROC

      IF @DataSize
         push    DS          ; if more then one data segment
         mov     ax, @data
         mov     ds, ax
      ENDIF

      cmp   dont_use_timer, 0
      jnz   SHORT after_wait ; skip wait loop if no hardware timer to be used

wait_loop:
      in    al, timer2       ; read lsb from counter register.
      cmp   al, sample_rate  ; compare timer value to sampling/playback rate.
      ja    SHORT wait_loop  ; keep waiting if not.
      mov   al, 0            ; set timer again
      out   timer2, al                 

after_wait:
      test  hardware_type, LPT_USED_MASK
      jz    SHORT use_internal_atod
      get_x_byte adread, ah
      mov   al, ah
      jmp   SHORT get_ret

use_internal_atod:
      mov   dx, adread       ; get from internal A to D
      in    al, dx

get_ret:
      xor   ah, ah

      IF @DataSize 
         pop       DS        ; if more then one data segment
      ENDIF

      ret

recordByte   ENDP

END

