#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include "parsePrcPdb.h"
#include "util.h"

#define FMT4CC		"%c%c%c%c"
#define CVT4CC(x)	((x) >> 24) & 0xff, ((x) >> 16) & 0xff, ((x) >> 8) & 0xff, (x) & 0xff





static void usage(const char* me)
{
	fprintf(stderr, "USAGE: %s \"Db Name\" type crid Tbmp0000.bin code0001.bin .... > file.prc\n", me);
	fprintf(stderr, "\teach resource should be 8 chars long: 4 type, 4 hexchars id, then \".bin\"\n");
	fprintf(stderr, "\talternate syntax is tver03e8=filename\n");
	
	
	exit(-1);
}

static int64_t strTo4cc(const char* cc)		//negative on error
{
	uint32_t ret = 0;
	int i;
	
	for (i = 0; i < 4; i++) {
		char ch = *cc++;
		
		if (ch < 0x20 || ch >= 0x80)
			return -1;
		
		ret = (ret << 8) + (uint32_t)(uint8_t)ch;
	}
	return (uint64_t)ret;
}

static int32_t hexStrToU16(const char* cc)		//negative on error
{
	uint16_t ret = 0;
	int i;
	
	for (i = 0; i < 4; i++) {
		char ch = *cc++;
		
		if (ch >= '0' && ch <= '9')
			ch -= '0';
		else if (ch >= 'a' && ch <= 'f')
			ch -= 'a' - 10;
		else if (ch >= 'A' && ch <= 'F')
			ch -= 'A' - 10;
		else
			return -1;
		
		ret = (ret << 4) + ch;
	}
	return (uint32_t)ret;
}

static bool parseResId(const char *name, uint32_t *typP, uint16_t *idP)
{
	int64_t typ;
	int32_t id;
	
	if (strlen(name) < 8)
		return false;
	
	if ((typ = strTo4cc(name)) < 0 || (id = hexStrToU16(name + 4)) < 0)
		return false;
	
	*typP = typ;
	*idP = id;
	
	return true;
}

static FILE *nextIncludedRes(const char *arg, uint32_t *typP, uint16_t *idP, const char **pathP)
{
	const char* filePath = arg, *fileName;
	
	if (strlen(arg) >= 10 && arg[8] == '=' && parseResId(arg, typP, idP))	//new format
		return fopen(*pathP = arg + 9, "rb");
	
	fileName = strrchr(filePath, '/');
	fileName = fileName ? (fileName + 1) : filePath;
	
	if (strlen(fileName) != 12 || fileName[8] != '.' || fileName[9] != 'b' || fileName[10] != 'i' || fileName[11] != 'n')
		return false;
	
	if (!parseResId(fileName, typP, idP))
		return false;
	
	return fopen(*pathP = filePath, "rb");
}

int main(int argc, char** argv)
{
	const char* self = *argv++;
	struct PalmDb *db;
	
	argc--;
	if (argc < 3)
		usage(self);
	
	if (strlen(argv[0]) >= sizeof(db->name) - 1 || strlen(argv[1]) != 4 || strlen(argv[2]) != 4 || strTo4cc(argv[1]) < 0 || strTo4cc(argv[2]) < 0)
		usage(self);
	
	db = createPrcPdb(argv[0], PALM_DB_ATTR_RES_DB, 1, 0xf0000000, 0xf0000000, 0xf0000000, 0, strTo4cc(argv[1]), strTo4cc(argv[2]), 0);
	if (!db)
		abort();
	
	argv += 3;
	argc -= 3;
	
	while (argc--) {
		const char* filePath;
		struct Buffer buf;
		uint32_t typ;
		uint16_t id;
		int64_t len;
		FILE *f;
		
		f = nextIncludedRes(*argv++, &typ, &id, &filePath);
		if (!f)
			usage(self);
		
		if (fseek(f, 0, SEEK_END))
			abort();
		
		len = ftell(f);
		if (len < 0)
			abort();
		
		if (fseek(f, 0, SEEK_SET))
			abort();
		
		fprintf(stderr, " * including res ('"FMT4CC"', %u) from file \"%s\" (%u bytes long)\n", CVT4CC((uint32_t)typ), id, filePath, (int)len);
		
		buf.data = malloc(len + 1);
		if (!buf.data)
			abort();
		
		buf.sz = len;
		
		if (len != fread(buf.data, 1, len, f))
			abort();
		
		fclose(f);
		
		db = appendRes(db, typ, id, &buf);
		if (!db)
			abort();
	}
	
	if (!writePrcPdb(db, "/dev/stdout"))
		abort();
	freePrcPdb(db);
	
	return 0;
}
