#include <dal.h>
#include <kal.h>
#include <boot.h>
#include <NotifyMgr.h>
#include <StringMgr.h>
#include <Extensions/ExpansionMgr/VFSMgr.h>
#include <string.h>
#include "common.h"
#include "printf.h"
#include "slot.h"
#include "sd.h"



//from disassembly of Zodiac's slot drivers
#define ZOD_SLOT_NR_CUSTOM_CONTROL_CRID		CREATE_4CC('T','p','w','v')
#define ZOD_SLOT_NR_CUSTOM_CONTROL_SEL		1

//from disassembly of TX's FAT32
#define TRIM_CUSTOM_CONTROL_CRID			CREATE_4CC('e','x','v','f')
#define TRIM_CUSTOM_CONTROL_SEL				0x1000

struct TrimRequest {
	uint32_t firstSec;
	uint32_t numSec;
};



register struct GlobalsRaw ***globalsRaw asm ("r9");


struct Globals {								//limited to 0x38 bytes by 'amdi' resource (0x10 for palmos, 0x08 for pace linker stub data). change that to get more
	LocalID				self;
	uint32_t			mutex;
	uint16_t			slotRefNum;
	bool 				haveCard;	//do we have a usable card?
	uint32_t			sd[STRUCT_SD_SIZE];
};

struct GlobalsRaw {
	uint32_t palmOsData[4];
	uint32_t paceLinkerStubData[2];
	struct Globals g;
};


static __attribute__((always_inline)) struct Globals* globalsGet(void)	//let's always inline this to pretend we're pros :D
{
	return &(globalsRaw[0][MY_LIB_ID / 4]->g);
}

static bool sdPrvIsCardRO(void)
{
	struct Globals *g = globalsGet();
	union SdFlags f;
	
	f.value = sdGetFlags((struct SD*)g->sd);
	return f.RO || sdIsCardLockSwitchOn((struct SD*)g->sd);
}

static Err notifCallbackFromOurselvesCardInsertedRemoved(SysNotifyParamType *notifyParamsP)
{
	struct Globals *g = globalsGet();
	uint8_t buf[64];

	KALTaskDelay(100);
	if (sdIsCardInserted((struct SD*)g->sd)) {
		
		logi("card inserted\n");
		if (g->haveCard)
			logi("already had card - assuming spurious\n");
		else {
			sdCardPower((struct SD*)g->sd, true);
			
			logt("initing card\n");
			if (sdCardInit((struct SD*)g->sd, buf)) {
				
				logi("Card inserted and inited!\n");
				g->haveCard = true;
				ExpCardInserted(g->slotRefNum);
			}
			else {
				
				logi("card init failed\n");
				sdCardPower((struct SD*)g->sd, false);
			}
		}
	}
	else {
		logt("card removed\n");
		if (!g->haveCard)
			logi("already had no card - assuming spurious\n");
		else {
			
			ExpCardRemoved(g->slotRefNum);
			g->haveCard = false;
			sdCardPower((struct SD*)g->sd, false);
		}
	}
	
	logt("%s: out\n", __func__);
	return errNone;
}

Err impl_SlotLibOpen(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	LocalID lid;
	Err e;

	MemSet(g, sizeof(*g), 0);

	sdCardPower((struct SD*)g->sd, false);
	
	e = KALMutexCreate(&g->mutex, CREATE_4CC('s','d','D','r'));
	if (e != errNone) {
		logt("mutex create = 0x%04x\n", e);
		goto mutex_create;
	}

	logt("%s(slotLibRefNum=0x%04x)\n", __func__, slotLibRefNum);
	e = ExpSlotRegister(slotLibRefNum, &g->slotRefNum);
	if (e != errNone) {
		logt("slot register = 0x%04x\n", e);
		goto out_slot_register;
	}
	
	logt("slot registered: slot handle %u\n", g->slotRefNum);

	lid = DmFindDatabase(MY_DB_NAME);
	g->self = lid;
	if (!lid) {
		loge("Cannot find self!\n");
		e = dmErrCantFind;
		goto out_db_find;
	}
	
	e = SysNotifyRegister(lid, MY_CRID, notifCallbackFromOurselvesCardInsertedRemoved, 0, NULL);
	if (e != errNone) {
		logt("notif register = 0x%04x\n", e);
		goto out_notif_register;
	}
	
	if (!sdIsCardInserted((struct SD*)g->sd))
		logt("slot driver init: no card seen yet\n");
	else {
		logt("slot driver init: card seen - initing\n");
		SysNotifyBroadcastFromInterrupt(MY_CRID, MY_CRID, NULL);
	}
	
	logt("%s: out\n", __func__);
	return errNone;

out_notif_register:
	//nothing

out_db_find:
	ExpSlotUnregister(g->slotRefNum);
	g->slotRefNum = 0;

out_slot_register:
	KALMutexDelete(g->mutex);
	g->mutex = 0;

mutex_create:	
	logt("%s: out w/ error\n", __func__);
	return e;
}

Err impl_SlotLibClose(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	
	logt("%s\n", __func__);
	
	if (g->mutex)
		KALMutexReserve(g->mutex, -1);
	g->haveCard = false;
	
	if (g->self)
		(void)SysNotifyUnregister(g->self, MY_CRID, 0);
	
	if(g->slotRefNum)
		(void)ExpSlotUnregister(g->slotRefNum);
	
	if (g->mutex)
		KALMutexDelete(g->mutex);
	
	sdShutdown((struct SD*)g->sd);
	sdCardPower((struct SD*)g->sd, false);

	return errNone;
}

Err impl_SlotLibSleep(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	
	logt("%s\n", __func__);
	sdSleep((struct SD*)g->sd);
	return errNone;
}

Err impl_SlotLibWake(uint16_t slotLibRefNum)
{
	struct Globals *g = globalsGet();
	
	logt("%s\n", __func__);
	sdWake((struct SD*)g->sd);
	return errNone;
}
	
uint32_t impl_SlotLibLibAPIVersion(uint16_t slotLibRefNum)
{
	logt("%s\n", __func__);
	return slotDrvrAPIVersion;
}

Err impl_SlotLibCustomControl(uint16_t slotLibRefNum, uint32_t apiCreator, uint16_t apiSelector,  void *valueP, uint16_t *valueLenP)
{
	struct Globals *g = globalsGet();
	Err ret;
	
	logt("%s\n", __func__);
	
	KALMutexReserve(g->mutex, -1);
	
	if (apiCreator == TRIM_CUSTOM_CONTROL_CRID && apiSelector == TRIM_CUSTOM_CONTROL_SEL && valueP && valueLenP && !(*valueLenP % sizeof(struct TrimRequest))) {
	//TRIM
		
		const struct TrimRequest *req = (const struct TrimRequest*)valueP;
		uint32_t numReqd = *valueLenP / sizeof(struct TrimRequest);
		
		if (!g->haveCard)
			ret = expErrCardNotPresent;
		else if (sdPrvIsCardRO())
			ret = expErrCardReadOnly;
		else if (!numReqd)
			ret = errNone;
		else do {
			
			ret = sdTrim((struct SD*)g->sd, req->firstSec, req->numSec);
			if (ret != errNone)
				break;
			
			req++;
				
		} while (ret == errNone && --numReqd);
	}
	else if (apiCreator == ZOD_SLOT_NR_CUSTOM_CONTROL_CRID && apiSelector == ZOD_SLOT_NR_CUSTOM_CONTROL_SEL) {
	//TELL zodiac we are card #1. YES this logic is correct
		
		if (!valueP)
			ret = sysErrParamErr;
		else {
			*(uint16_t*)valueP = 1;
			if (valueLenP)
				*valueLenP = sizeof(uint16_t);
			ret = errNone;
		}
	}
	else {
		
		fatal("Unexpected custom control (" FMT_4CC ", %u) with %d bytes of data\n", CNV_4CC(apiCreator), apiSelector, valueLenP ? *valueLenP : -1);
		ret = expErrUnsupportedOperation;
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardPresent(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	
	KALMutexReserve(g->mutex, -1);
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	KALMutexRelease(g->mutex);
	
	return ret;
}

static void slotPrvGetManufStr(char* dst, uint8_t manuf, uint16_t oem, bool mmc)
{
	const char *ret;
	
	if (mmc) {
		
		switch (manuf) {
			case 1:		ret = "Infineon";			break;
			case 2:		ret = "Sandisk";			break;
			case 6:		ret = "Hitachi";			break;
			case 22:	ret = "Mobipocket eBook";	break;
			case 44:	ret = "PNY";				break;
			default:	ret = NULL;					break;
		}
	}
	else{	//see here: http://softgun.sourceforge.net/sdcard.shtml, https://www.cameramemoryspeed.com/sd-memory-card-faq/reading-sd-card-cid-serial-psn-internal-numbers/
		
		switch ((manuf << 16) + oem) {
			case (  1 << 16) + ('P' << 8) + 'A':	ret = "Panasonic";			break;
			case (  2 << 16) + ('T' << 8) + 'M':	ret = "Toshiba";			break;
			case (  3 << 16) + ('S' << 8) + 'D':	//fallthrough
			case (  3 << 16) + ('P' << 8) + 'T':	ret = "Sandisk";			break;
			case (  6 << 16) + ('R' << 8) + 'K':	ret = "Fuji";				break;
			case (  9 << 16) + ('A' << 8) + 'P':	ret = "ATP";				break;
			case ( 17 << 16) + ('D' << 8) + ' ':	ret = "PNY";				break;
			case ( 25 << 16) + ('D' << 8) + 'Y':	//fallthrough
			case ( 48 << 16) + ('P' << 8) + 'R':	ret = "Integral Platinum";	break;
			case ( 27 << 16) + ('S' << 8) + 'M':	ret = "Samsung";			break;
			case (116 << 16) + ('J' << 8) + 'E':	//fallthrough
			case (116 << 16) + ('J' << 8) + '\'':	//fallthrough
			case ( 28 << 16) + ('S' << 8) + 'V':	ret = "Transcend";			break;
			case ( 29 << 16) + ('A' << 8) + 'D':	ret = "Corsair";			break;
			case ( 29 << 16) + ('A' << 8) + 'F':	ret = "AData";				break;
			case ( 39 << 16) + ('P' << 8) + 'H':	ret = "Phison";				break;
			case ( 40 << 16) + ('B' << 8) + 'E':	ret = "Lexar/PNY";			break;
			case ( 49 << 16) + ('S' << 8) + 'P':	ret = "Silicon Power";		break;
			case ( 65 << 16) + ('4' << 8) + '2':	ret = "Kingston";			break;
			case (130 << 16) + ('J' << 8) + 'T':	ret = "Sony";				break;
			default:								ret = NULL;					break;
		}
	}
	
	if (ret)
		StrCopy(dst, ret);
	else {
		#define HCV(x)	(((x) > 9)?((x) + 'A' - 10):((x) + '0'))
		#define HCS(v)	HCV(((v) >> 4) & 0x0F),HCV((v) & 0x0F)
		
		StrPrintF(dst, mmc ? "?: 'm-%d'" : "?: 's-%dx%02x%02x'", manuf, HCS(oem >> 8), HCS(oem));
		
		#undef HCS
		#undef HCV
	}
}

static void slotPrvGetDeviceClass(char *dst, uint32_t numBlocks, bool isMmc, bool isNewVer, bool isHighCap)
{
	uint32_t numKb = numBlocks / 2, divShift;
	char letter;
	
	if (numKb >= (1UL << (divShift = 30)))
		letter = 'T';
	else if (numKb >= (1UL << (divShift = 20)))
		letter = 'G';
	else if (numKb >= (1UL << (divShift = 10)))
		letter = 'M';
	else {
		divShift = 0;
		letter ='K';
	}
	
	if (numKb & ((1 << divShift) - 1))
		StrPrintF(dst, "%u.%u%c ", numKb >> divShift, (numKb & ((1 << divShift) - 1)) * 10 >> divShift, letter);
	else
		StrPrintF(dst, "%u%c ", numKb >> divShift, letter);
	
	if (isMmc) {
		
		if (!isNewVer)
			StrCat(dst, "MMC");
		else if (isHighCap)
			StrCat(dst, "HS-MMC");
		else
			StrCat(dst, "MMCplus");
	}
	else {
		
		if (!isNewVer)
			StrCat(dst, "SDv1");
		else if (isHighCap)
			StrCat(dst, "SDxC");
		else
			StrCat(dst, "SDv2");
	}
}


Err impl_SlotLibCardInfo(uint16_t slotLibRefNum, uint16_t slotRefNum, ExpCardInfoType *infoP)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else if (infoP) {
		
		union SdFlags f;
		uint32_t snum;
		uint16_t oid;
		uint8_t mid;
		
		sdGetInfo((struct SD*)g->sd, &mid, &oid, &snum);
		f.value = sdGetFlags((struct SD*)g->sd);
		
		MemSet(infoP, sizeof(*infoP), 0);
		infoP->capabilityFlags = expCapabilityHasStorage | (sdPrvIsCardRO() ? expCapabilityReadOnly : 0);
		StrIToA(infoP->deviceUniqueIDStr, snum);
		slotPrvGetManufStr(infoP->manufacturerStr, mid, oid, !f.SD);
		slotPrvGetDeviceClass(infoP->deviceClassStr, sdGetNumSecs((struct SD*)g->sd), !f.SD, f.v2, f.HC);
		StrCopy(infoP->productStr, "Card");
	}
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibMediaType(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t *mediaTypeP)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	union SdFlags f;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else if (mediaTypeP) {
		
		f.value = sdGetFlags((struct SD*)g->sd);
		*mediaTypeP = f.SD ? expMediaType_SecureDigital : expMediaType_MultiMediaCard;
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardMediaType(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t *mediaTypeP)
{
	return impl_SlotLibMediaType(slotLibRefNum, slotRefNum, mediaTypeP);	//these are the same for us..
}

Err impl_SlotLibCardReserve(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardRelease(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardGetSerialPort(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t* portP)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else if (!portP)
		ret = sysErrParamErr;
	else {
		*portP = 0;
		ret = expErrUnsupportedOperation;
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Boolean impl_SlotLibCardIsFilesystemSupported(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t filesystemType)
{
	logt("%s\n", __func__);
	return filesystemType == 'vfat' || filesystemType == 'fatf';
}

Err impl_SlotLibCardMetrics(uint16_t slotLibRefNum, uint16_t slotRefNum, CardMetricsPtr cardMetricsP)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else {
		MemSet(cardMetricsP, sizeof(*cardMetricsP), 0);
		cardMetricsP->totalSectors		= sdGetNumSecs((struct SD*)g->sd);
		cardMetricsP->bytesPerSector	= slotSectorSize;
		cardMetricsP->bootIndicator		= 0xFF;
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardLowLevelFormat(uint16_t slotLibRefNum, uint16_t slotRefNum)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	logt("%s\n", __func__);
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibPowerCheck(uint16_t slotLibRefNum, uint16_t slotRefNum, uint16_t operationFlags, uint16_t readBlocks, uint16_t writeBlocks)
{
	struct Globals *g = globalsGet();
	Err ret = errNone;
	
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardSectorRead(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t sectorNumber, uint8_t *bufferP, uint32_t *numSectorsP)
{
	struct Globals *g = globalsGet();
	uint32_t cardNsec;
	Err ret = errNone;
	
	logt("%s(%u+%u)\n", __func__, sectorNumber, numSectorsP ? *numSectorsP : 0);
	
	KALMutexReserve(g->mutex, -1);
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else if (!bufferP || !numSectorsP)
		ret = sysErrParamErr;
	else {
		cardNsec = sdGetNumSecs((struct SD*)g->sd);
		
		if (sectorNumber >= cardNsec)
			*numSectorsP = 0;
		else if (sectorNumber + *numSectorsP >= cardNsec)
			*numSectorsP = cardNsec - sectorNumber;
		
		if (!*numSectorsP) {
			
			ret = errNone;
		}
		else if (*numSectorsP == 1) {
			
			if (sdSecRead((struct SD*)g->sd, sectorNumber, bufferP))
				ret = errNone;
			else {
				*numSectorsP = 0;
				ret = expErrCardBadSector;
			}
		}
		else {
			
			uint32_t left = *numSectorsP, done = 0, max = sdPrvGetMaxSectorsAtOnce((struct SD*)g->sd);
			
			if (left > max)		//let the caller sort it out...
				left = max;
			
			ret = errNone;
			if (sdReadStart((struct SD*)g->sd, sectorNumber, left)) {
				
				do {
					
					if (sdReadNext((struct SD*)g->sd, bufferP)) {
						
						done++;
						bufferP += slotSectorSize;
					}
					else {
						
						ret = expErrCardBadSector;
						break;
					}
					
				} while (--left);
			}
			if (!sdReadStop((struct SD*)g->sd))
				ret = expErrCardBadSector;
			*numSectorsP = done;
		}
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

Err impl_SlotLibCardSectorWrite(uint16_t slotLibRefNum, uint16_t slotRefNum, uint32_t sectorNumber, const uint8_t *bufferP, uint32_t *numSectorsP)
{
	static const uint8_t __attribute__((aligned(8))) zeroes[512] = {0,};	//alignment for implementations with DMA
	struct Globals *g = globalsGet();
	uint32_t cardNsec;
	Err ret = errNone;
	
	logt("%s(%u+%u)\n", __func__, sectorNumber, numSectorsP ? *numSectorsP : 0);
	
	KALMutexReserve(g->mutex, -1);
	
	if (!bufferP)
		bufferP = zeroes;
	
	if (!g->haveCard)
		ret = expErrCardNotPresent;
	else if (!numSectorsP)
		ret = sysErrParamErr;
	else if (sdPrvIsCardRO())
		ret = expErrCardReadOnly;
	else {
		
		cardNsec = sdGetNumSecs((struct SD*)g->sd);
		
		if (sectorNumber >= cardNsec)
			*numSectorsP = 0;
		else if (sectorNumber + *numSectorsP >= cardNsec)
			*numSectorsP = cardNsec - sectorNumber;
		
		if (!*numSectorsP) {
			
			ret = errNone;
		}
		else if (*numSectorsP == 1) {
			
			if (sdSecWrite((struct SD*)g->sd, sectorNumber, bufferP))
				ret = errNone;
			else {
				*numSectorsP = 0;
				ret = expErrCardBadSector;
			}
		}
		else {
			
			uint32_t left = *numSectorsP, done = 0, max = sdPrvGetMaxSectorsAtOnce((struct SD*)g->sd);;
			
			if (left > max)		//let the caller sort it out...
				left = max;
			
			ret = errNone;
			if (sdWriteStart((struct SD*)g->sd, sectorNumber, left)) {
				
				do {
					
					if (sdWriteNext((struct SD*)g->sd, bufferP)) {
						
						done++;
						if (bufferP != zeroes)
							bufferP += slotSectorSize;
					}
					else {
						
						ret = expErrCardBadSector;
						break;
					}
					
				} while (--left);
			}
			if (!sdWriteStop((struct SD*)g->sd))
				ret = expErrCardBadSector;
			*numSectorsP = done;
		}
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}
